#include "calc_engine/Lexr.hh"
#include <iostream>
#include <stdexcept>

using namespace std;

//======================================  Constructor
Lexr::Lexr(void)
  : mTrTable(kInvChar), mDebug(0)
{
    //----------------------------------  Set up the translate table
    mTrTable.set_alpha(kAlpha);
    mTrTable.set_numer(kNumer);
    mTrTable.set_type("+-/*&|><=!%", kOper);
    mTrTable.set_type("\"'", kQuote);
    mTrTable.set_type("()[]",kParen);
    mTrTable.set_type(",:", kComma);
    mTrTable.set_type(" \t\n\r", kSpace);
    mTrTable.set_type(".", kDot);
}

//======================================  Get the next token
Lexr::LxTags 
Lexr::getToken(std::istream& in, std::string& token) const {
    token.clear();
    StateID state(start);
    LxTags tkType = tkEnd;
    char qtype = 0;
    bool term = false, unget=false;
    while (!term && tkType == tkEnd) {
	char c = in.get();
	if (c == istream::traits_type::eof()) {
	    term = true;
	} else if (state == start) {
	    switch(mTrTable[c]) {
	    case kInvChar:
	        throw runtime_error("lex: Invalid character read");
	    case kAlpha:
	        state = in_symbol;  
		token += c;
		break;
	    case kNumer:
	        state = numer_int;
		token += c;
		break;
	    case kDot:
	        state = numer_fix;
		token += c;
		break;
	    case kQuote:
	        state = in_string;
		qtype = c;
		break;
	    case kOper:
	        state = in_oper;
		token += c;
		break;
	    case kParen:
	        if (c == '(')      tkType = tkOParen;
		else if (c == ')') tkType = tkCParen;
		else if (c == '[') tkType = tkOSubs;
		else if (c == ']') tkType = tkCSubs;
		else if (c == '<') tkType = tkOSubs;
		else if (c == '>') tkType = tkCSubs;
		else if (c == '{') tkType = tkOParen;
		else if (c == '}') tkType = tkCParen;
		token += c;
		break;
	    case kComma:
	        tkType = tkComma;
		token += c;
	    case kSpace:
	        break;
	    }
	} else if (state == in_string) {
	    if (qtype == c) tkType = tkString;
	    else            token += c;
	} else {
	    switch(mTrTable[c]) {
	    case kInvChar:
	        throw runtime_error("lex: Invalid character read");
	    case kAlpha:
	        switch (state) {
		case in_symbol:
		  token += c;
		  break;
		case numer_int:
		case numer_fix:
		  if (c == 'e' || c == 'E') {
		    state = numer_exp;
		    token += c;
		    break;
		  }
		case numer_exp:
		case numer_sexp:
		  throw runtime_error("Invalid character in number");
		default:
		  term  = true;
		  unget = true;
		}
		break;
	    case kNumer:
	        switch (state) {
		case numer_exp:
		    state = numer_sexp;
		case in_symbol: 
		case numer_int:
		case numer_fix:
		case numer_sexp:
	            token += c;
		    break;
		default:
		    term = true;
		    unget = true;
		}
		break;
	    case kDot:
	        switch (state) {
		case numer_int:
	            token += c;
		    state = numer_fix;
		    break;
		default:
		    term = true;
		    unget = true;
		}
		break;
	    case kQuote:
	        term = true;
		unget = true;
		break;
	    case kOper:
	        switch (state) {
		case in_oper:
		    token += c;
		    break;
		case numer_exp:
		    if (c == '+' || c == '-') {
		        token += c;
			state = numer_sexp;
			break;
		    }
		default:
		    unget = true;
		    term = true;
		}
		break;
	    default:
	        unget = true;
		term = true;
		break;
	    }
	}
    }

    //----------------------------------  Unget the character if it wasn't used
    if (unget) in.unget();

    //----------------------------------  Set the token type.
    if (term) {
        switch (state) {
	case in_symbol: 
	    tkType = tkSymbol;
	    break;
	case in_string:
	    throw runtime_error("Unterminated string");
	case in_oper:
	    tkType = tkOper;
	    break;
	case numer_int:
	case numer_fix:
	case numer_sexp:
	    tkType = tkLiteral;
	    break;
	case numer_exp:
	    throw runtime_error("Invalid numeric literal");
	default:
	    break;
	}
    }
    return tkType;
}

//======================================  Set the debug printout level
void 
Lexr::setDebug(int lvl) {
    mDebug = lvl;
}

//======================================  Assign characters to various types
void 
Lexr::addDelimChars(const char* str) {
    mTrTable.char_type(str, kComma);
}

void 
Lexr::addOpChars(const char* str) {
    mTrTable.char_type(str, kOper);
}

void 
Lexr::addParenChars(const char* str){
    mTrTable.char_type(str, kParen);
}

//======================================  Set various type groups
void 
Lexr::setDelimChars(const char* str) {
    mTrTable.set_type(str, kComma);
}

void 
Lexr::setOpChars(const char* str) {
    mTrTable.set_type(str, kOper);
}

void 
Lexr::setParenChars(const char* str){
    mTrTable.set_type(str, kParen);
}
