#ifndef DYNCONFIG_HH
#define DYNCONFIG_HH

#include "Time.hh"
#include "IIRFilter.hh"
#include <list>
#include <string>
#include <iosfwd>

class Dacc;

/**  The DynConfig class tracks the time dependent control parameters 
  *  stored in epics channels or in a specified parameter file.
  *  @memo Dynamic IFO parameter tracking.
  *  @author J. Zweizig
  *  @version 1.0; Last modified May 28, 2003
  */
class DynConfig {
public:
    /**  Construct a dynamic configuration tracker. The specified Dacc object 
      *  is used to read in epics channels. The optional file specifies 
      *  unrecorded parameters or overrides the parameters recorded in the 
      *  frame data stream.
      *  @memo constructor
      *  @param in   Data accessor to be used to read epics data.
      *  @param file Parameter override file.
      */
    DynConfig(Dacc& in, const char* file=0);

    /**  Destroy the dynamic parameter follower.
      *  @memo Destructor.
      */
    ~DynConfig(void);

    /**  Get the expiry time for the current parameter settings. The earliest
      *  parameter expiry time is returned. 
      *  \return Expiration time.
      */
    Time getExpiry(void) const;

    /**  Specify a parameter to be followed. The double precision float 
      *  specified by \a vbl is kept up to data with the value of the 
      *  parameter \a name.
      *  @memo Specify a parameter
      *  @param name Parameter name to be followed
      *  @param vbl  Reference to variable to receive parameter value.
      *  @param def  Default parameter value.
      */
    void addVbl(const std::string& name, double& vbl, double def=0.0);

    /**  Dump the current state of all requested parameters.
      *  @memo Dump parameter state
      *  @param out Output stream to which data will bw dumped.
      */
    void dumpConfig(std::ostream& out) const;

    /**  Get the current parameter values for the time specified by t0.
      *  \brief Get dynamic configuration
      *  \param t0 Time for which configuration is needed.
      */
    void getConfig(const Time& t0);

    /**  Set the debugging level.
      *  \param l Debug printout level.
      */
    void setDebug(int l);

    /**  Specify a parameter file to be used to resolve the parameter names,
      *  \param file Parameter file path.
      */
    void setFile(const char* file);

//======================================  LscEmulate control variables
private:
    Dacc&       mDacc;
    Time        mSetTime;
    Time        mExpire;
    int         mDebug;
    std::string mFile;

//======================================  Dynamic parameter database.
private:
    struct ConfData {
        ConfData(const std::string& name, double* addr, double def) 
	  : _name(name), _addr(addr), _default(def), _expire(0), _src(kDef) {}
        ConfData(const ConfData& d) 
	  : _name(d._name), _addr(d._addr), _default(d._default), 
	    _expire(d._expire), _src(d._src) {}
        std::string _name;
        double*     _addr;
        double      _default;
        Time        _expire;
	enum {kDef, kFile, kFrame} _src;
    };
    typedef std::list<ConfData> ConfList;
    typedef ConfList::iterator conf_iter;
    typedef ConfList::const_iterator const_conf_iter;
    ConfList mConfList;
};

inline Time 
DynConfig::getExpiry(void) const {
    return mExpire;
}

#endif // DYNCONFIG_HH
