#include "TSeries.hh"
#include "ParseLine.hh"
#include "Dacc.hh"
#include "DynConfig.hh"
#include <iostream>
#include <iomanip>

using namespace std;

//======================================  Dynamic configuration constructor
DynConfig::DynConfig(Dacc& d, const char* file) 
  : mDacc(d), mExpire(0)
{
    if (file) setFile(file);
}

//======================================  Destructor
DynConfig::~DynConfig() {
}

//======================================  Set values
void
DynConfig::addVbl(const string& chan, double& vbl, double init) {
    vbl = init;
    mConfList.push_back(ConfData(chan, &vbl, init));
    if (!mDacc.isChannelRead(chan.c_str())) mDacc.addChannel(chan.c_str());
}

//======================================  Set values
void
DynConfig::getConfig(const Time& t0) {
    bool checkFile(mExpire == Time(0));
    mExpire = Time(0);

    //----------------------------------  Fill undefined parameters from frame
    bool change = false;
    for (conf_iter i=mConfList.begin() ; i!=mConfList.end() ; i++) {
        if (i->_src != ConfData::kFile || i->_expire <= t0) {
	    if (i->_src == ConfData::kFile) checkFile = true;
	    const TSeries* ts = mDacc.refData(i->_name.c_str());
	    if (!ts || ts->isEmpty()) {
	        change |= (*(i->_addr) != i->_default);
	        *(i->_addr) = i->_default;
		i->_expire  = Time(0);
		i->_src     = ConfData::kDef;
	    } else if (i->_expire <= t0 || Almost(t0, i->_expire)) {
	        int N     = ts->getNSample();
		int j0    = ts->getBin(t0);
		double x  = ts->getDouble(j0);
		change   |= (x != *(i->_addr));
		while (j0 < N && ts->getDouble(j0) == x) {
		    i->_expire = ts->getBinT(++j0);
		}
		*(i->_addr) = x;
		i->_src = ConfData::kFrame;
	    }
	}
	if (!mExpire || (i->_expire != Time(0) && i->_expire < mExpire)) {
	    mExpire = i->_expire;
	}
    }

    //----------------------------------  Read file
    if (checkFile && !mFile.empty()) {
        bool readParam = false;
	Time startTime(0), stopTime(0);
        ParseLine pl(mFile.c_str());
	while(pl.getLine() >= 0) {
	    if (!pl.getCount()) {
	        continue;
	    } else if (string(pl[0]) == "Time") {
	        readParam = false;
	        startTime = Time(pl.getInt(1));
	        stopTime  = Time(pl.getInt(2));
		readParam = (startTime != Time(0) && t0 >= startTime &&
			     t0 < stopTime);
	    } else if (readParam) {
	        for (int i=0 ; i<pl.getCount() ; i++) {
		    string argi(pl[i]);
		    int argl = argi.size();
		    int eqinx = argi.find("=");
		    if (eqinx >= argl) {
		        cout << "Syntax error in parameter specification: "
			     << argi << endl;
			continue;
		    }
		    string name(pl[i], 0, eqinx);
		    double value = strtod(pl[i]+eqinx+1, 0);
		    for (conf_iter j=mConfList.begin() ; 
			 j!=mConfList.end() ; j++) {
		        if (j->_name == name) {
			    change |= (*(j->_addr) != value);
			    *(j->_addr) = value;
			    j->_src = ConfData::kFile;
			    j->_expire = stopTime;
			}
		    }
		}
		if (!mExpire || (stopTime < mExpire)) mExpire = stopTime;
	    }
	}
    }
    mSetTime = t0;

    //----------------------------------  Print constants if changed
    if (change && mDebug) dumpConfig(cout);
}

void
DynConfig::dumpConfig(ostream& out) const {
    out << "Configuration for the interval: " << mSetTime 
	<< " - " << mExpire << endl;
    for (const_conf_iter i=mConfList.begin() ; i!=mConfList.end() ; i++) {
        out << setw(-32) << setfill(' ') << i->_name;
	out << setw(10)  << *(i->_addr) << setw(0);
	if      (i->_src == ConfData::kDef)   out << " Default ";
	else if (i->_src == ConfData::kFile)  out << " File    ";
	else if (i->_src == ConfData::kFrame) out << " Frame   ";
	out << i->_expire.getS() << endl;
    }
    out << endl;
}

//======================================  Set debugging level
void
DynConfig::setDebug(int lvl) {
    mDebug = lvl;
}

//======================================  Set configuration parameter file
void
DynConfig::setFile(const char* f) {
    mFile = f;
}
