/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "LscCalib.hh"
#include "CalibChan.hh"
#include "DVecType.hh"
#include "Dacc.hh"
#include "FrWriter.hh"
#include "FrStatDataRef.hh"
#include "xsil/array.hh"
#include "xsil/ligolw.hh"
#include "xsil/param.hh"
#include "xsil/table.hh"
#include "xsil/xsil_time.hh"
#include "xsil/Xreader.hh"
#include "xsil/Xwriter.hh"
#include <string>
#include <stdexcept>
#include <fstream>
#include <memory>

using namespace std;

//======================================  Time settings
inline void
setTmin(Time& t, const Time& x) {
    if (!x) return;
    if (!t || t>x) t=x;
}

inline void
setTmax(Time& t, const Time& x) {
    if (!t || t<x) t=x;
}

//======================================  Get a string parameter
void
getParam(const xsil::xobj* x, const string& n, string& s) {
    const xsil::param* p=dynamic_cast<const xsil::param*>(x->find(n, "Param"));
    if (p) s = p->getValue();
    else   s.clear();
}

//======================================  Get a float parameter
void
getParam(const xsil::xobj* x, const string& n, float& f) {
    const xsil::param* p=dynamic_cast<const xsil::param*>(x->find(n, "Param"));
    if (p) f = strtod(p->getValue(), 0);
    else   f = 0;
}

//======================================  Get a double parameter
void
getParam(const xsil::xobj* x, const string& n, double& d) {
    const xsil::param* p=dynamic_cast<const xsil::param*>(x->find(n, "Param"));
    if (p) d = strtod(p->getValue(), 0);
    else   d = 0;
}

//======================================  Get an integer parameter
void
getParam(const xsil::xobj* x, const string& n, int& l) {
    const xsil::param* p=dynamic_cast<const xsil::param*>(x->find(n, "Param"));
    if (p) l = strtol(p->getValue(), 0, 0);
    else   l = 0;
}

//======================================  Get a Time value parameter
void
getTime(const xsil::xobj* x, const string& n, Time& t) {
    const xsil::xsil_time* p=dynamic_cast<const xsil::xsil_time*>(x->find(n, "Time"));
    if (p) t = p->getTime();
    else   t = Time(0);
}

//======================================  Get a f-series parameter
void
getFSeries(xsil::xobj* x, const string& n, FSeries& f) {

    //----------------------------------  Find the array data
    xsil::array* p=dynamic_cast<xsil::array*>(x->find(n, "Array"));
    if (!p) return;

    //----------------------------------  Get the data.
    vector<double> fv;
    p->getData(fv);
    int nDim = fv.size();

    //----------------------------------  Get the number of rows
    int nRow = nDim/3;
    if (nDim != 3*nRow) throw runtime_error("FSeries row structure invalid");
    double f0 = fv[0];
    double dF = (fv[nDim-3] - fv[0]) / double(nRow-1);
    DVecType<dComplex> dv(nRow);
    for (int i=0; i<nRow; ++i) {
        double F = f0 + dF * double(i);
	int inx = i*3;
	if (F != fv[inx]) cerr << "Frequency error, f=" << fv[inx] 
			       << " expected=" << F << endl;
	dv[i].setMArg(fv[inx+1], fv[inx+2]);
    }
    f = FSeries(f0, dF, Time(0), Interval(1.0/dF), dv);
}

//======================================  Get a t-series
void
get2TSeries(xsil::xobj* x, const string& n, TSeries& t1, TSeries& t2) {

    //----------------------------------  Find the array data
    xsil::array* p=dynamic_cast<xsil::array*>(x->find(n, "Array"));
    if (!p) return;

    //----------------------------------  Get the data.
    vector<double> tv;
    p->getData(tv);
    int nDim = tv.size();

    //----------------------------------  Get the number of rows
    int nRow = nDim/3;
    if (nDim != 3*nRow) throw runtime_error("TSeries row structure invalid");
    Time     t0 = Time(int(tv[0]));
    Interval dT = double(tv[nDim-3] - tv[0]) / double(nRow-1);
    DVecType<double> dv1(nRow);
    DVecType<double> dv2(nRow);
    for (int i=0; i<nRow; ++i) {
        Time ti = t0 + dT * double(i);
	int inx = i*3;
	if (ti != Time(long(tv[inx]))) cerr << "Time error, t=" << long(tv[inx])
					    << " expected=" << ti << endl;
	dv1[i] = tv[inx+1];
 	dv2[i] = tv[inx+2];
    }
    t1 = TSeries(t0, dT, dv1);
    t2 = TSeries(t0, dT, dv2);
}

//======================================  Get a t-series
void
getTSeries(xsil::xobj* x, const string& n, TSeries& ts) {

    //----------------------------------  Find the array data
    xsil::array* p=dynamic_cast<xsil::array*>(x->find(n, "Array"));
    if (!p) return;

    //----------------------------------  Get the data.
    vector<double> tv;
    p->getData(tv);
    int nDim = tv.size();

    //----------------------------------  Get the number of rows
    int nRow = nDim/2;
    if (nDim != 2*nRow) throw runtime_error("TSeries row structure invalid");
    Time     t0 = Time(int(tv[0]));
    Interval dT = double(tv[nDim-2] - tv[0]) / double(nRow-1);
    DVecType<double> dv(nRow);
    for (int i=0; i<nRow; ++i) {
        Time ti = t0 + dT * double(i);
	int inx = i*2;
	if (ti != Time(long(tv[inx]))) cerr << "Time error, t=" << long(tv[inx]) 
					    << " expected=" << ti << endl;
	dv[i] = tv[inx+1];
    }
    ts = TSeries(t0, dT, dv);
}

//======================================  Constructor
LscCalib::LscCalib(const std::string& name, const std::string& file, 
		   const std::string& chan, const Time& gps) 
  : mVersionID(0), mDebug(0)
{
    clear(); 
    mName = name;
    if (!file.empty()) read(file, name, chan, gps);
}


//======================================  Destructor
LscCalib::~LscCalib(void) {
}

//======================================  Clear the calibration record.
void 
LscCalib::clear(void) {
    mChannel.clear();
    mName.clear();
    mVersion.clear();
    mComment.clear();
    mVersionID = 0;

    mStartTime = Time(0);
    mDuration  = 0.0;

    mEXCChannel.clear();
    mCalLineFreq = 0;
    mCalLineAmplASQ = 0;
    mCalLineAmplEXC = 0;

    mGainChan.clear();
    mGainRefValue.clear();

    mSensingFunction.clear();
    mOpenLoopGain.clear();
    mResponseFunction.clear();

    mAlpha.Clear();
    mAlphaBeta.Clear();
}

//======================================  Make sure all fields are consistent 
void 
LscCalib::prepare(void) {

    //----------------------------------  Figure out a time period
    Time tStart(mStartTime);
    Time tEnd(tStart + mDuration);
    if (!mAlpha.empty()) {
        setTmin(tStart, mAlpha.getStartTime());
        setTmax(tEnd,   mAlpha.getEndTime());
    }	
    if (!mAlphaBeta.empty()) {
        setTmin(tStart, mAlphaBeta.getStartTime());
        setTmax(tEnd,   mAlphaBeta.getEndTime());
    }
    if (!mOpenLoopGain.empty()) {
        setTmin(tStart, mOpenLoopGain.getStartTime());
        setTmax(tEnd,   mOpenLoopGain.getEndTime());
    }
    if (!mResponseFunction.empty()) {
        setTmin(tStart, mResponseFunction.getStartTime());
        setTmax(tEnd,   mResponseFunction.getEndTime());
    }
    if (!mSensingFunction.empty()) {
        setTmin(tStart, mSensingFunction.getStartTime());
        setTmax(tEnd,   mSensingFunction.getEndTime());
    }
    if (tEnd <= tStart) tEnd = tStart + Interval(1.0);
    mStartTime = tStart;
    mDuration  = tEnd-tStart;

    //----------------------------------  Set the FSeries times if necessary
    if (!mSensingFunction.empty() || !mSensingFunction.getStartTime()) {
        mSensingFunction.setTimeSpan(mStartTime, mDuration);
    }

    if (!mResponseFunction.empty() || !mResponseFunction.getStartTime()) {
        mResponseFunction.setTimeSpan(mStartTime, mDuration);
    }

    if (!mOpenLoopGain.empty() || !mOpenLoopGain.getStartTime()) {
        mOpenLoopGain.setTimeSpan(mStartTime, mDuration);
    }

    //----------------------------------  Calculate OLG from Sensing, response
    if (mOpenLoopGain.empty() && 
	!mSensingFunction.empty() && 
	!mResponseFunction.empty()) {
        mOpenLoopGain  = mResponseFunction;
	mOpenLoopGain *= mSensingFunction;
	mOpenLoopGain += -1.0;
    }
    if (!mOpenLoopGain.empty() && 
	mSensingFunction.empty() && 
	!mResponseFunction.empty()) {
        mSensingFunction  = mOpenLoopGain;
	mSensingFunction += 1.0;
	mSensingFunction /= mResponseFunction;
    }
}

//======================================  Get the gain factors.
double
LscCalib::getCavFac(const Time& t) const {
    if (t < mAlpha.getStartTime() || t>= mAlpha.getEndTime())
        throw range_error("Cavity gain factor not specified at time");
    return mAlpha.getDouble(mAlpha.getBin(t));
}

double
LscCalib::getOLGFac(const Time& t) const {
    if (t < mAlphaBeta.getStartTime() || t>= mAlphaBeta.getEndTime())
        throw range_error("Open-loop gain factor not specified at time");
    return mAlphaBeta.getDouble(mAlphaBeta.getBin(t));
}

//======================================  Calculate the response function
FSeries 
LscCalib::getResponse(double alpha, double gamma) const {
    FSeries resp(mOpenLoopGain);
    if (gamma != 1.0) resp *= gamma;
    resp += 1.0;
    if (alpha != 1.0) resp *= 1.0/alpha;
    resp /= mSensingFunction;
    return resp;
}

//======================================  Calculate the response function
FSeries 
LscCalib::getResponse(const Time& t) const {
    double alpha = getCavFac(t);
    double gamma = getOLGFac(t);
    return getResponse(alpha, gamma);
}

//======================================  Generic file read
void 
LscCalib::read(const string& file, const string& name, 
	       const string& chan, const Time& gps) {
    string::size_type N=file.size();
    if (file.substr(N-4, 4) == ".xml") {
        readXml(file, name, chan);
    } else if (file.substr(N-4, 4) == ".gwf") {
        readFrame(file, name, chan, gps);
    } else {
        ifstream in(file.c_str());
	char idstr[5];
	in.read(idstr,5);
	in.close();
	if (string(idstr) == "IGWD") {
	    readFrame(file, name, chan, gps);
	} else if (string(idstr, 5) == "<?xml") {
	    readXml(file, name, chan);
	} else {
	    throw runtime_error(string("Unidentifiable file type: ") + file);
	}
    }
}

//======================================  Read a frame file
void 
LscCalib::readFrame(const string& fi, const string& nm, 
		    const string& ch, const Time& gps) {

    //----------------------------------  Reset contents.
    mAlpha.Clear();
    mAlphaBeta.Clear();
    mSensingFunction.clear();
    mResponseFunction.clear();

    //----------------------------------  Get the channel names.
    CalibChanList cName(ch);
    if (!cName) {
        cerr << "Channel name: " << ch << " not recognized." << endl;
        return;
    }
    mChannel = ch;

    //----------------------------------  Set up and read the frame.
    Dacc In(fi.c_str());
    In.setDebug(mDebug);
    In.setIgnoreMissingChannel(true);

    //----------------------------------  Read stuff from the history
    Time calib_gps(0);
    In.getHistory(CALFRAME_COMMENT_NAME, calib_gps, mComment);
    In.getHistory(CALFRAME_VERSION_NAME, calib_gps, mVersion);
    if (!mCalibTime) mCalibTime = calib_gps; 
    In.getStaticData();
    mSensingFunction  = In.getStaticFSeries(cName.getSensing(),   gps);
    mResponseFunction = In.getStaticFSeries(cName.getResponse(),  gps);
    mOpenLoopGain     = In.getStaticFSeries(cName.getOLoopGain(), gps);
    if (!mResponseFunction.empty()) {
        mVersionID = In.findStat(cName.getResponse(), gps).getVersion();
    } else if (!mOpenLoopGain.empty()) {
        mVersionID = In.findStat(cName.getOLoopGain(), gps).getVersion();
    }
    mAlpha            = In.getStaticTSeries(cName.getCavFac(),    gps);
    mAlphaBeta        = In.getStaticTSeries(cName.getOLoopFac(),  gps);

    //----------------------------------  Read stuff from ProcData
    In.addChannel(cName.getCavFac());
    In.addChannel(cName.getOLoopFac());
    In.addFSeries(cName.getSensing());
    In.addFSeries(cName.getResponse());
    In.addFSeries(cName.getOLoopGain());
    In.fillData(Interval(1e9));

    //----------------------------------  Get the data objects
    TSeries* ts = In.refData(cName.getCavFac());
    if (mAlpha.isEmpty()     && ts) mAlpha = *ts;
    ts = In.refData(cName.getOLoopFac());
    if (mAlphaBeta.isEmpty() && ts) mAlphaBeta = *ts;
    FSeries* fs = In.refFData(cName.getSensing());
    if (mSensingFunction.empty()  && fs) mSensingFunction  = *fs;
    fs = In.refFData(cName.getResponse());
    if (mResponseFunction.empty() && fs) mResponseFunction = *fs;
    fs = In.refFData(cName.getOLoopGain());
    if (mOpenLoopGain.empty() && fs) mOpenLoopGain     = *fs;

    //----------------------------------  Check Consistency
    prepare();
}

//======================================  Read an xml file
void 
LscCalib::readXml(const string& file, const string& name, const string& chan) {
    ifstream in(file.c_str());
    xsil::Xreader xr(in);
    readXml(xr, name, chan);
}

void 
LscCalib::readXml(xsil::Xreader& xr, const string& name, const string& chan) {
    unique_ptr<xsil::xobj> doc(xr.readDoc());

    xsil::ligolw* cDoc = dynamic_cast<xsil::ligolw*>(doc->find(name, "LIGO_LW"));
    if (!cDoc) throw runtime_error("Unable to find calibration for channel");

    //----------------------------------  parameters
    getParam(cDoc, CALXML_CHANNEL_PARAM,        mChannel);
    if (!chan.empty() && mChannel != chan) 
        throw runtime_error("File does not contain calibration for channel");
    getParam(cDoc, CALXML_VERSION_PARAM, mVersion);
    getParam(cDoc, CALXML_COMMENT_PARAM, mComment);
    getParam(cDoc, CALXML_VERSNID_PARAM, mVersionID);
    getParam(cDoc, "EXCChannel",     mEXCChannel);
    getParam(cDoc, "CalLineFreq",    mCalLineFreq);
    getParam(cDoc, "CalLineAmplASQ", mCalLineAmplASQ);
    getParam(cDoc, "CalLineAmplEXC", mCalLineAmplEXC);
    double dT(0);
    getParam(cDoc, CALXML_DURATION_PARAM, dT);
    mDuration = dT;
    getTime(cDoc, CALXML_START_TIME_PARAM, mStartTime);
    getTime(cDoc, CALXML_CALIB_TIME_PARAM, mCalibTime);

    //----------------------------------  Read in Gain channels.
    xsil::table* tb =  dynamic_cast<xsil::table*>(cDoc->find("DARMChannels", "Table"));
    if (tb) {
        while (1) {
	    string chan;
	    double refv;
	    if (tb->refStream().read(&chan, 1)) break;
	    if (tb->refStream().read(&refv, 1)) break;
	    mGainChan.push_back(chan);
	    mGainRefValue.push_back(refv);
	}
    }

    //----------------------------------  Transfer functions
    getFSeries(cDoc, "OpenLoopGain", mOpenLoopGain);
    getFSeries(cDoc, "ResponseFunction", mResponseFunction);
    getFSeries(cDoc, "SensingFunction", mSensingFunction);
    getTSeries(cDoc, "CavityFactor", mAlpha);
    getTSeries(cDoc, "OLoopFactor", mAlphaBeta);

    //----------------------------------  CheckConsistency
    prepare();
}

//======================================  Set the alpha TSeries
void 
LscCalib::setAlpha(const TSeries& a) {
    mAlpha = a;
}

//======================================  Set the alpha*beta TSeries
void 
LscCalib::setAlphaBeta(const TSeries& ab) {
    mAlphaBeta = ab;
}

//======================================  Add a calibration line
void 
LscCalib::setCalibTime(const Time& t) {
    mCalibTime = t;
}

//======================================  Add a calibration line
void 
LscCalib::setCalLine(const string& exc, float frq, float ampAsq, float ampExc){
    mEXCChannel     = exc;
    mCalLineFreq    = frq;
    mCalLineAmplASQ = ampAsq;
    mCalLineAmplEXC = ampExc;
}

//======================================  Set the debug level.
void 
LscCalib::setDebug(int lvl) {
    mDebug = lvl;
}

//======================================  Add a Gain Channel
void 
LscCalib::addGainChan(const std::string& chan, float ampl) {
    mGainChan.push_back(chan);
    mGainRefValue.push_back(ampl);
}

//======================================  Set the open loop gain function
void 
LscCalib::setOpenLoopGain(const FSeries& o) {
    mOpenLoopGain = o;
}

//======================================  Set the response function
void 
LscCalib::setResponseFunction(const FSeries& t) {
    mResponseFunction = t;
}

//======================================  Set the sensing function
void 
LscCalib::setSensingFunction(const FSeries& t) {
    mSensingFunction = t;
}

//======================================  Set the channel name
void 
LscCalib::setChannel(const std::string& c) {
    mChannel = c;
}

//======================================  Set the calibration document name
void 
LscCalib::setName(const std::string& n) {
    mName = n;
}

//======================================  Set the version description
void 
LscCalib::setVersion(const std::string& v) {
    mVersion = v;
}

//======================================  Set the version ID
void 
LscCalib::setVersionID(int id) {
    mVersionID = id;
}

//======================================  Set the sensing function
void 
LscCalib::setComment(const std::string& c) {
    mComment = c;
}

//======================================  Write an xml file
void 
LscCalib::writeXml(const string& file) {
    ofstream in(file.c_str());
    xsil::Xwriter xr(in);
    writeXml(xr);
}

//======================================  Insert FSeries into an xml document
static void
putFSeries(xsil::ligolw* doc, const char* name, const FSeries& fs) {
    if (fs.empty()) return;
    unique_ptr<xsil::array> pArray(new xsil::array(name, "float"));
    int N=fs.getNStep()+1;
    pArray->addDim("Frequency", N);
    pArray->addDim("Frequency_Modulus_Phase",3);
    vector<fComplex> dat(N);
    fs.getData(N, &dat[0]);
    vector<double> fMagPhi(3*N);
    double f0 = fs.getCenterFreq();
    double dF = fs.getFStep();
    for (int i=0; i<N; i++) {
       int inx=3*i;
       fMagPhi[inx  ] = f0 + dF*double(i);
       fMagPhi[inx+1] = dat[i].Mag();
       fMagPhi[inx+2] = dat[i].Arg();
    }
    pArray->refStream().Fill(3*N, &fMagPhi[0], 3);
    doc->addObject(pArray.release());
}

//======================================  Insert 2 TSeries into an xml document
static void
putTSeries(xsil::ligolw* doc, const char* name, const TSeries& ts) {
    long N=ts.getNSample();
    if (!N) return;
    unique_ptr<xsil::array> pArray(new xsil::array(name, "double"));
    pArray->addDim("GPS", N);
    pArray->addDim("GPS_Value",2);
    vector<double> dat(N);
    ts.getData(N, &dat[0]);
    vector<double> tAlpha(2*N);
    Time     t0 = ts.getStartTime();
    Interval dT = ts.getTStep();
    for (int i=0; i<N; i++) {
       int inx=2*i;
       tAlpha[inx  ] = (t0 + dT*double(i)).getS();
       tAlpha[inx+1] = dat[i];
    }
    pArray->refStream().Fill(2*N, &tAlpha[0], 2);
    doc->addObject(pArray.release());
}

#if 0
//======================================  Insert a TSeries into an xml document
static void
put2TSeries(xsil::ligolw* doc, const char* name, 
	    const TSeries& ts1, const TSeries& ts2) {
    if (ts1.isEmpty()) return;
    unique_ptr<xsil::array> pArray(new xsil::array(name, "double"));
    int N=ts1.getNSample();
    pArray->addDim("GPS", N);
    pArray->addDim("GPS_Alpha_AlphaBeta",3);
    vector<double> dat1(N);
    ts1.getData(N, &dat1[0]);
    vector<double> dat2(N);
    ts2.getData(N, &dat2[0]);
    vector<double> tAlphaBeta(3*N);
    Time     t0 = ts1.getStartTime();
    Interval dT = ts1.getTStep();
    for (int i=0; i<N; i++) {
       int inx=3*i;
       tAlphaBeta[inx  ] = (t0 + dT*double(i)).getS();
       tAlphaBeta[inx+1] = dat1[i];
       tAlphaBeta[inx+2] = dat2[i];
    }
    pArray->refStream().Fill(3*N, &tAlphaBeta[0], 3);
    doc->addObject(pArray.release());
}
#endif

//======================================  Write an xml file
void 
LscCalib::writeXml(xsil::Xwriter& xw) {
    prepare();
    unique_ptr<xsil::ligolw> d(new xsil::ligolw(mName.c_str()));
    if (!d.get()) throw runtime_error("Unable to construct calibration");

    if (!mChannel.empty()) 
        d->addParam(CALXML_CHANNEL_PARAM, 0, 0, mChannel.c_str());
    if (!mVersion.empty()) 
        d->addParam(CALXML_VERSION_PARAM, 0, 0, mVersion.c_str());
    if (!mComment.empty()) 
        d->addParam(CALXML_COMMENT_PARAM, 0, 0, mComment.c_str());
    d->addParam(CALXML_VERSNID_PARAM, 0, mVersionID);
    d->addTime(CALXML_START_TIME_PARAM, mStartTime);
    d->addParam(CALXML_DURATION_PARAM,  0, double(mDuration));
    d->addTime(CALXML_CALIB_TIME_PARAM, mCalibTime);

    if (!mEXCChannel.empty()) 
        d->addParam("EXCChannel", 0, 0, mEXCChannel.c_str());
    d->addParam("CalLineFreq",    0, mCalLineFreq);
    d->addParam("CalLineAmplASQ", 0, mCalLineAmplASQ);
    d->addParam("CalLineAmplEXC", 0, mCalLineAmplEXC);

    int nChan = mGainChan.size();
    if (nChan) {
        unique_ptr<xsil::table> tb(new xsil::table("DARMChannels"));
	tb->addColumn("ChannelName", "string");
	tb->addColumn("RefValue",    "float");
	for (int i=0; i<nChan; ++i) {
	    tb->refStream().Add(mGainChan[i]);
	    tb->refStream().Add(mGainRefValue[i]);
	}
	d->addObject(tb.release());
    }

    putFSeries(d.get(), "OpenLoopGain", mOpenLoopGain);
    //putFSeries(d.get(), "ResponseFunction", mResponseFunction);
    putFSeries(d.get(), "SensingFunction", mSensingFunction);
    if (!mAlpha.empty())     putTSeries(d.get(), "CavityFactor", mAlpha);
    if (!mAlphaBeta.empty()) putTSeries(d.get(), "OLoopFactor", mAlphaBeta);

    //----------------------------------  Put doc into wrapper, write it out.
    unique_ptr<xsil::ligolw> wrap(new xsil::ligolw);
    wrap->addObject(d.release());
    wrap->Spew(xw);
}

//======================================  Write a frame file
void 
LscCalib::writeFrame(const string& file, CalibChanList::cal_format fmt) {
    if (mDebug) cerr << "Write channel: " << mChannel 
		     << " calibration to frame: " << file << endl;
    prepare();
    CalibChanList cName(mChannel);
    if (!cName) throw runtime_error("No frame names for channel");
    FrWriter fw("Calibration", -1);
    fw.setDebug(mDebug);

    //----------------------------------  Open the file for output.
    fw.open(file.c_str(), true);

    //----------------------------------  Build the frame
    fw.buildFrame(mStartTime, mDuration);
    fw.addWriterHistory();
    fw.addHistory("LscCalib", Now(), "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/DMT/EasyCalibrate/LscCalib.cc 7601 2016-04-15 20:51:43Z john.zweizig@LIGO.ORG $");
    fw.addHistory(CALFRAME_VERSION_NAME, mCalibTime, mVersion);
    fw.addHistory(CALFRAME_COMMENT_NAME, mCalibTime, mComment);

    //----------------------------------  Write the standard detector
    void* det = fw.addStdDetector(mChannel);

    //----------------------------------  Write the Time series
    if (!mAlpha.isEmpty() || !mAlphaBeta.isEmpty()) {
        switch (fmt) {
	case CalibChanList::cfmt_framev2:
	    fw.addStatic(cName.getCavFac(),    mVersionID, mStartTime, 
			 mStartTime+mDuration, det, mAlpha);
	    fw.addStatic(cName.getOLoopFac(),  mVersionID, mStartTime, 
			 mStartTime+mDuration, det, mAlphaBeta);
	    break;
	case CalibChanList::cfmt_framev1:
	default:
	    fw.addProcSeries(cName.getCavFac(),   mAlpha);
	    fw.addProcSeries(cName.getOLoopFac(), mAlphaBeta);
	}
    }

    // Version, mVersion
    // Comment, mComment *** in history
    // StartTime, mStartTime ** frame span
    // Duration, mDuration ** frame span

    // EXCChannel, mEXCChannel
    // CalLineFreq, mCalLineFreq
    // CalLineAmplASQ, mCalLineAmplASQ
    // CalLineAmplEXC, mCalLineAmplEXC;

    // Gain channel table - DARMChannels
    // Column ChannelName, mGainChan[i]
    // Column RefValue, mGainRefValue[i]

    //----------------------------------  Calculate the reference response
    if ( mResponseFunction.empty() && !mSensingFunction.empty() && 
	!mOpenLoopGain.empty()) {
        mResponseFunction = getResponse(1.0, 1.0);
    }

    //----------------------------------  Write the transfer functions.
    switch (fmt) {
    case CalibChanList::cfmt_framev2:
        if (!mResponseFunction.empty()) {
	    fw.addStatic(cName.getResponse(), mVersionID, mStartTime, 
			 mStartTime+mDuration, det, mResponseFunction);
	}
        if (!mSensingFunction.empty()) {
	    fw.addStatic(cName.getSensing(),  mVersionID, mStartTime, 
			 mStartTime+mDuration, det, mSensingFunction);
	}
        if (!mOpenLoopGain.empty()) {
	    fw.addStatic(cName.getOLoopGain(),  mVersionID, mStartTime, 
			 mStartTime+mDuration, det, mOpenLoopGain);
	}
	break;
    case CalibChanList::cfmt_framev1:
    default:
        fw.addFrequencySeries(cName.getResponse(), mResponseFunction);
	fw.addFrequencySeries(cName.getSensing(), mSensingFunction);
    }

    //---------------------------------- Write out the file.
    fw.writeFrame();
    fw.close();
}
