/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef DATENV_HH
#define DATENV_HH

#ifndef __CINT__
#include "DaccAPI.hh"
#include "SigFlag.hh"
#include "TrigRaw.hh"
#endif //  !defined(__CINT__)

/**  <h2> Introduction </h2>
  *  The %DatEnv data processing base class implements the basic structure
  *  of a data processing monitor and implements most of the functions common 
  *  to all such routines. The functions provided by the environment are:
  *  <ul>
  *    <li> Parse common command line arguments.
  *    <li> Implement iteration over data strides.
  *    <li> Read frame data.
  *    <li> Copy requested data into time series (TSeries) containers.
  *    <li> Handling attention interrupts.
  *    <li> Clean up on termination.
  *  </ul>
  *
  *  <h2> Writing a %DatEnv Monitor </h2>
  *  The strategy for writing a monitor program is to create a class 
  *  (MyClass) based on %DatEnv including a constructor, destructor and
  *  ProcessData() method. An attention handler may also be provided if
  *  desired. The operations to be performed by each of these methods
  *  are:
  *
  *  <b> Constructor (MyClass(int argc, const char* argv[])) </b>
  *  <ul>
  *    <li> %Process user specific arguments
  *    <li> Define parameters (see TrigPar class).
  *    <li> Configure the process.
  *    <li> Request data channels with Dacc::addChannel().
  *  </ul>
  *
  *  <b> Destructor (~MyClass(void)) </b>
  *  <ul>
  *    <li> Save modified parameters
  *    <li> Print results and statistics.
  *  </ul>
  *
  *  <b> Epoch processing (void ProcessData(void)) </b>
  *  <ul>
  *    <li> %Process data from one epoch.
  *    <li> Generate trigger is necessary.
  *    <li> Keep statistics.
  *  </ul>
  *
  *  <b> Interrupt handler (void Attention(void)) </b>
  *  <ul>
  *    <li> Monitor dependent interrupt processing.
  *  </ul>
  *
  *  The \c EXECDAT(myclass) macro is used to generate the main
  *  function. Definition and implementation templates are available 
  *  from DatTemplate.hh and DatTemplate.cc.
  *
  *  The \c EXEROOT(myclass) macro is used to generate a main function
  *  with \c root graphics. Note that the \c "TROOT.h" and \c "TApplication.h"
  *  header files must be included in your monitor implementation file 
  *  if \c EXEROOT is used.
  *
  *  <h2> Running the Monitor </h2>
  *  The following environment and options are common to all %DatEnv-based 
  *  monitors. The specific code may implement additional options.
  *
  *  <h3>Command line options</h3>
  *  The base class constructor recognizes the following command line 
  *  arguments: 
  *  <table>
  *  <tr><th>Argument</th><th>Meaning</th></tr> 
  *  <tr><td>-debug \<lvl\>   </td>
  *      <td> Set debug level to \<lvl\> </td></tr>
  *  <tr><td>-infile \<file\> </td>
  *      <td> Read frames from \<file\> </td></tr>
  *  <tr><td>-inlist \<list\> </td>
  *      <td> Read frames from files listed in \<list\>  </td></tr>
  *  <tr><td>-maxframes \<nFr\>  </td>
  *      <td> Stop after reading \<nFr\> frames  </td></tr>
  *  <tr><td>-partition \<part\> </td>
  *      <td> Read from shared memory partition \<part\>. </td></tr>
  *  </table>
  *  If no input option is specified, the frames are read from the partition
  *  specified by the \c LIGOSMPART environment variable or, if this is not 
  *  defined, from "LIGO_Online".
  *
  *  <h3>Environment</h3>
  *  The following environment variables may be used by the monitor.
  *  
  *  <table>
  *  <tr><th>Variable</th><th>Meaning</th></tr> 
  *  <tr><td>\c DMTHTMLOUT</td>
  *      <td>Directory for any html pages to be written to the monitor_reports
  *          pages.</td></tr>
  *  <tr><td>\c DMTINPUT</td>
  *      <td>Input frame or frames to be read by the monitor. The DMTINPUT
  *          variable contens may contain glob-style wild-cards in the 
  *          last part of the path (not in the directory name(s)). A shared
  *          memory partition may be specified as the input by setting 
  *          \c DMTINPUT to "/online/<partition-name>".</td></tr>
  *  <tr><td>\c DMTPARS</td>
  *      <td>Default parameter directory</td></tr>
  *  <tr><td>\c DMTRENDOUT</td>
  *      <td>%Trend frame output directory specifier. The \c DMTRENDOUT 
  *          environment variable may contain escape sequences for 
  *          truncated-GPS subdirectory names (see Trend class). </td></tr>
  *  <tr><td>\c DMT_IGNORE_NAN</td>
  *        <td>By default, the data accessor class will return an invalid 
  *            data status and an empty time series if NaN data are found.
  *            If \c DMT_IGNORE_NAN is set the nan check will be ignored
  *            (see Dacc class).</td></tr>
  *  <tr><td>\c LIGOSMPART</td>
  *      <td>An input shared memory partition to be used if DMTINPUT is not
  *          specified.</td></tr>  
  *  </table>
  *
  *  <h3>Running under ROOT</h3>
  *  Note that running a monitor online under ROOT is not currently well 
  *  maintained (JGZ 8/7/14).
  *
  *  When running under ROOT, the base class definition and implementation 
  *  files must be loaded before the monitor can be run. If these files 
  *  aren't provided automatically by the start-up script, they may be 
  *  loaded with: 
  *  \verbatim
        gROOT->LoadMacro("DatEnv.hh");
        gROOT->LoadMacro("DatEnv.cc"); \endverbatim
  *  The monitor is run with the \c .x command or by loading it and calling 
  *  it as a function. See the note entitled "Online monitoring of LIGO Data
  *  Using ROOT" for more details.
  *  @memo Data processing monitor base class.
  *  @author John G. Zweizig
  *  @version 1.4; Last Modified March 4, 2003
  */
class DatEnv {
  public:
    /**  The class constructor gets all the command line arguments in the 
      *  usual Unix/C argument list format. It sets up the environment 
      *  for processing and opens the frame input stream as directed by 
      *  the command line arguments.
      *  @memo Class constructor.
      *  @param argc Number of arguments specified.
      *  @param argv Pointers to arguments.
      */
    DatEnv(int argc, const char *argv[]);

    /**  Close the input stream and go away.
      *  @memo Class destructor.
      */
    virtual ~DatEnv();

    /**  MainLoop reads in data and invokes ProcessData() for each data 
      *  epoch successfully read in. MainLoop exits when a SIGINT or 
      *  \c SIGTERM signal is caught, or when finish() is called by a 
      *  subsidiary function (usually by ProcessData() ). This function 
      *  is provided by the base class.
      *  \brief Loop over data epochs.
      */
    void MainLoop(void);

    /**  ProcessData performs the monitor specific processing data from
      *  a single stride. It must be defined as part of the derived monitor 
      *  class implementation.
      *  @memo Process one stride (Supplied by user).
      */
    virtual void ProcessData(void)=0;

    /**  Attention performs the monitor specific attention interrupt 
      *  processing. Its implementation is optional. Execution of 
      *  the attention handler is deferred until the processing of the 
      *  current epoch has completed. This means that there may be a 
      *  significant delay between the delivery of the SIGUSR1 signal and
      *  the invocation of the handler. The ROOT version uses the ROOT event
      *  handler as a source of When used with ROOT, the default action is
      *  to terminate the monitor. When used in the background, it defaults
      *  to no action. The execution of the attention handler is 
      *  synchronized to the 
      *  @memo Attention interrupt handler (Supplied by user).
      */
    virtual void Attention(void);

    /**  finish() stops processing by setting a flag that breaks the main 
      *  loop. Note that finish returns to the calling function (generally 
      *  the user's ProcessData() ) which must exit normally for the process 
      *  to terminate.
      *  @memo Stop data processing and go away cleanly.
      */
    void finish(void) {mActive = false;}

    /**  Debug returns the debug level as specified by the "-debug nnn" 
      *  argument. If "-debug" isn't specified in the command line the 
      *  debug level defaults to 0.
      *  \brief Get the debug level.
      *  \return Requested debug level.
      */
    int Debug(void) const {return mDebug;}

    /**  Return a constant reference to the data accessor.
      *  @memo Get the data accessor.
      *  \return Reference to the data accessor.
      */
    const DaccAPI& getDacc(void) const {return *mIn;}

    /**  Test for a %DatEnv command line argument. All DatEnv command line 
      *  arguments take a value in the following argument position.
      *  \brief Test if string is a %DatEnv command line argument.
      *  \param arg Argument keyword.
      *  \return true if the specified string is a %DatEnv argument.
      */
    bool isDatEnvArg(const char* arg) const;

    /**  Align the time stride to an integer multiple of seconds.
      *  @memo set stride alignment.
      *  @param mult boundary alignment.
      *  @param off  boundary offset
      */
    void setStrideAlignment(unsigned long mult, double off);

    /**  Test whether the specified signal has been caught.
      *  \brief Test for a specified signal.
      *  \param sig Signal number.
      *  \return true if the specified signal was caught.
      */
    bool testAttnSignal(int sig) const;

  //------------------------------------  Private methods
  private:
    /**  Test if a termination interrupt occurred.
      *  @memo Test for termination.
      */
    bool testTerm(void) const;

    /**  Test if an attention interrupt occurred.
      *  @memo Test for attention.
      */
    bool testAttn(void);

  protected:
    /**  Return a reference to the data accessor.
      *  \brief Get the data accessor.
      *  \return Reference to data accessor for this monitor.
      */
    DaccAPI& getDacc(void) {return *mIn;}

    ///  Frame Input object
    DaccAPI*  mIn;

#ifndef __CINT__
    /**  A raw trigger data identifier is encoded for each data epoch read.
      *  This field is omitted in the ROOT version.
      *  @memo Raw trigger data identifier for current epoch.
      */
    trig::TrigRaw  mRawID;
#endif  // __CINT__

  private:
    ///  Internal run flag
    bool     mActive;

#ifndef __CINT__
    ///  Termination signal handling object
    SigFlag mTerm;

    ///  Attention signal handling object
    SigFlag mAttn;
#endif // !def(__CINT__)
  
    ///  Debug Level
    int  mDebug;

    /**  Frame read error.
      *  mEOF is set when a frame read error occurs.
      */
    bool mEOF;

    /**  Attention signals.
      *  mAttnMask is set to a or of the bits corresponding to the Signals 
      *  that have been caught.
      */
    long mAttnMask;

    long mMaxFrames;

    long mAlignMult;
    double mAlignOff;
};

inline bool 
DatEnv::testAttnSignal(int sig) const {
    return ((1 << sig) & mAttnMask) != 0;
}

//
//   The EXECDAT macro generates the main function to execute a monitor
//   based on DatEnv. The argument is whatever name you have chosen
//   for your class.
//
#define EXECDAT(myclass) \
int main(int argc, const char *argv[]) { \
    myclass MyObject(argc,argv); \
    MyObject.MainLoop(); \
    return 0;}

//
//   The EXEROOT macro generates the main function to execute a monitor
//   based on DatEnv. This macro differs from EXECDAT in that it also 
//   sets up the root object structure which allows the derived class to 
//   use any root classes. The argument is whatever name you have chosen
//   for your class.
//
#define EXEROOT(myclass) \
TROOT root("DatEnv", "Graphical background monitor"); \
int main(int argc, const char* argv[]) { \
    int bc(3); const char* bv[] = {"DatEnv", "-n", "-l"}; \
    TApplication theApp("App", &bc, const_cast<char**>(bv)); \
    myclass MyObject(argc,argv); \
    MyObject.MainLoop(); \
    return 0;}

#endif     //  DATENV_HH
