/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    DMT Data processing base class implementation.
//
//-------------------------------------  Header files.
#include "DatEnv.hh"

//-------------------------------------  For backward compatibility, we
#define REF_DACC_INPUT 1
#ifdef REF_DACC_INPUT
#include "Dacc.hh"
#endif
//-----------------------------------------------------------------------

#ifndef __CINT__
#include <iostream>
#include "Interval.hh"
#include "Time.hh"
#include <stdexcept>
#include <string>
#include <cstdlib>
#include <unistd.h>

#else
#define DACC_ONLDEV "/online/"

#endif // __CINT__

using namespace std;

//======================================  DatEnv constructor.
DatEnv::DatEnv(int argc, const char *argv[]) 
  : mActive(false), mDebug(0), mEOF(false), mAttnMask(0), mMaxFrames(0),
    mAlignMult(0), mAlignOff(0)
{
    mIn = new Dacc();

    string parfile = DACC_ONLDEV;
    if (getenv("LIGOSMPART")) parfile += getenv("LIGOSMPART");
    else                      parfile += "LIGO_Online";
    const char* file = getenv("DMTINPUT");
    if (!file)  file = parfile.c_str();

    //----------------------------------  Parse the arguments
    for (int i=1 ; i<argc && argv ; i++) {
        string argi = argv[i];
        if (argi  == "-partition") {
	    if (++i >= argc) return;
	    parfile = string(DACC_ONLDEV) + argv[i];
	    getDacc().addPath(parfile);
	    file = 0;
	} else if (argi == "-infile") {
	    if (++i >= argc) return;
	    getDacc().addPath(argv[i]);
	    file = 0;
	} else if (argi == "-inlist") {
	    if (++i >= argc) return;
	    getDacc().addPathList(argv[i]);
	    file = 0;
	} else if (argi == "-debug") {
	    if (++i >= argc) mDebug = 1;
	    else             mDebug = strtol(argv[i], 0, 0);
	    getDacc().setDebug(mDebug);
	} else if (argi == "-maxframes") {
	    mMaxFrames = strtol(argv[++i], 0, 0);
	}
    }
    if (file) getDacc().addPath(file);

#ifndef __CINT__
    //----------------------------------  Handle signals
    mTerm.add(SIGTERM);
    mTerm.add(SIGHUP);
    mTerm.add(SIGINT);

    // mAttn.setMode(SigFlag::kBlock);
    mAttn.add(SIGUSR1);
    mAttn.add(SIGIO);

#else
    gROOT->SetInterrupt(kTRUE);
#endif   // !def(__CINT__)

    //----------------------------------  Open a frame stream (FrameCPP)
    // if (getDacc().open()) return;
    mActive = true;
}

//======================================  Test if DatEnv command line argument
bool
DatEnv::isDatEnvArg(const char* argc) const {
    string arg(argc);
    if (arg == "-partition")      return true;
    else if (arg == "-infile")    return true;
    else if (arg == "-inlist")    return true;
    else if (arg == "-debug")     return true;
    else if (arg == "-maxframes") return true;
    return false;
}

//--------------------------------------  DatEnv object destructor.
DatEnv::~DatEnv() 
{
    //----------------------------------  DMT has terminater
    cout << "Data environment has terminated with Term/Attn/finish/EOF =" 
	 << bool(mTerm) << "/" << bool(mAttn) << "/" << !mActive << "/" 
	 << mEOF << endl;

#ifndef __CINT__
    //----------------------------------  Release signals
    mTerm.zero();
    mAttn.zero();

#endif //  __CINT__

    //----------------------------------  Close the Input file/partition.
    mIn->close();
    delete mIn;
}

inline std::ostream&
err_hdr(const std::string& txt) {
    return cerr << txt << "[" << getpid() << "] ";
}

//--------------------------------------  Main processing loop function.
void
DatEnv::MainLoop(void) 
{
    bool first_stride(true);

    getDacc().open();
#ifdef REF_DACC_INPUT
    Dacc& refDacc(dynamic_cast<Dacc&>(*mIn));
    if (getDacc().isOnline()) refDacc.setNoWait(true);
#endif

    //----------------------------------  Loop until terminated.
    bool newFill = true;
    while(!testTerm() && mActive) {

        //------------------------------  Check for an attention interrupt.
        if (testAttn()) {
	    Attention();
	    if (!mActive) continue;
	}

        //------------------------------  Get the data identifier
	if (getDacc().testData(true)) {
	    if (getDacc().synch()) {
	        mEOF = true;
		cerr << "Synch failed" << endl;
		break;
	    }

	    //--------------------------  Force the stride alignment
	    if (newFill && mAlignMult != 0) {
	        Time t  = getDacc().getCurrentTime();
		long ts = t.getS();
		Time Tv = Time(ts - (ts%mAlignMult)) + Interval(mAlignOff);
		if (Tv < t && !Almost(Tv, t)) Tv += Interval(mAlignMult);
		if (!Almost(Tv, t)) {
		    if (Debug()) cerr << "Aligning to (" << mAlignMult << ", " 
				      << mAlignOff << ") - skip to " << Tv 
				      << endl;
		    if (getDacc().seek(Tv)) continue;
		    if (!Almost(getDacc().getCurrentTime(), Tv)) continue;
		}
	    }

	    //--------------------------  Generate a frame name
#ifndef __CINT__
	    string FrameID = getDacc().getFrameID();
	    mRawID = trig::TrigRaw(getDacc().getFile(), FrameID.c_str(), 
				   getDacc().getOffset(), getDacc().getStride());
	    if (Debug() > 2) cout << "Frame ID: " << FrameID << endl;
#endif

	    //--------------------------  Read a frame
	    int rc;
	    try {
	        rc = getDacc().fillData(Interval(0.0), newFill);
	    } catch(exception& e) {
	        err_hdr("ProcessData") << "Exception in fillData: " 
			<< e.what() << endl;
		finish();
		break;
	    }
	    if (rc == 0) {
	        try {
		    ProcessData();
		    newFill = true;
		} catch(exception& e) {
		    err_hdr("ProcessData") 
			<< "Terminated with exception " << e.what()
			<< " while analyzing stride at "
			<< getDacc().getFillTime() << endl;
		    finish();
		} catch(...) {
		    err_hdr("ProcessData") << "Caught unidentified exception."
					   << endl;
		    finish();
		}
	    } else if (rc == -8) {
	        newFill = false;
	    } else if (rc == -9) {
		err_hdr("MainLoop") << "Invalid data in structure at gps: "
				    << mIn->getFillTime().getS() << endl;
		if (first_stride) {
		    newFill = true;
		} else {
		    mEOF = true;
		    break;
		}
	    } else if (rc <= -4) {
	        err_hdr("MainLoop") << "Error while reading frame - " 
				    << Dacc::getMsgText(rc) << endl;
		mEOF = true;
		break;
	    } else if (Debug()) {
	        err_hdr("MainLoop") << "Frame missing from input stream." 
				    << endl;
		newFill = true;
	    }
	} else if (Debug()) {
	    cerr << "Caught a signal" << endl;
	}

	//-----   this should be revisited as a time count or something similar.
	if (mMaxFrames) {
#ifdef REF_DACC_INPUT
	    if (refDacc.getTotalFrames() >= mMaxFrames) {
		cout << "Maximum frame count (" << mMaxFrames << ") exceeded." 
		     << endl;
		mActive = false;
	    }
#endif
	}
    } // while(!term) - Loop over strides.
}

//======================================  Set stride alignment
void
DatEnv::setStrideAlignment(unsigned long mult, double off) {
    mAlignMult = mult;
    if (!mAlignMult) {
        mAlignOff = 0.0;
    } else {
        mAlignOff = off;
	double dMult = mAlignMult;
	mAlignOff -= double(long(mAlignOff/dMult)*dMult);
	if (mAlignOff < 0.0) mAlignOff += dMult;
    }
}


//--------------------------------------  Background interrupt handling.
#ifndef __CINT__  

void
DatEnv::Attention(void) {
}

bool 
DatEnv::testAttn(void) {
    mAttnMask = mAttn.getSigFlags();
    if (mAttnMask) mAttn.clearFlags(mAttnMask);
    return (mAttnMask != 0);
}

bool 
DatEnv::testTerm(void) const {
    return mTerm;
}

//--------------------------------------  Root interrupt handling.
#else   
void
DatEnv::Attention(void) {
    mActive = false;
}

bool 
DatEnv::testAttn(void) {
    gSystem->ProcessEvents();
    mAttnMask = gROOT->IsInterrupted() ? 1 : 0;
    return (mAttnMask != 0);
}

bool 
DatEnv::testTerm(void) {
    return false;
}
#endif
