#ifndef INSPIRAL_HH
#define INSPIRAL_HH
#include "Chirp.hh"
#include "Interval.hh"

/**  Newtonian binary in-spiral template generator.
  *  Inspiral is a template generator based on the Chirp class. It
  *  generates binary in-spiral wave-forms. So far, only the zeroeth 
  *  order (Newtonian) approximation is used.
  */
class Inspiral : public Chirp {
public:

  /**  Inspiral Constructor. 
    *  Generate an in-spiral waveform template for two objects of masses 
    *  'Mass1' and 'Mass2'. 'dT' is the time interval to be considered. 
    *  If not specified dT defaults to 60s. 't0' if specified is the 
    *  absolute critical time of the inspiral and the template extends 
    *  between t0-dT and t0. If t0 is not specified, the template extends 
    *  from 0->dT.
    */
  Inspiral(double Mass1, double Mass2, const Interval& dT = Interval(60.0), 
	   const Time& t0=Time(0, 0));

  /**  Inspiral destructor.
    */
  ~Inspiral(void) {}

  /**  Frequency domain template bin.
    *  Complex Frequency domain representation of the template at frequency
    *  'Freq'.
    */
  fComplex Fspace(double Freq, double dF=0.0) const;

  /**  Time domain template bin.
    *  Tspace returns the waveform value at time 't0'.
    */ 
  double Tspace(const Time& t0) const;

  /**  Get the t0.
    *  Return the template start time. 
    */
  Time getT0(void)   const;

  /**  Get the End time.
    *  Return the template end time. The end-time is defined to be the 
    *  time at which the template frequency reaches a maximum.
    */
  Time getTEnd(void) const;

  /**  Get the Critical time.
    *  Return the template critical time. The critical time is defined 
    *  explicitly or implicitly when the object is constructed.
    */
  Time getTc(void) const;

  /**  Time versus Frequency.
    *  TvsF(f) returns the absolute (GPS) time at which the in-spiral 
    *  frequency is exactly 'f'.
    */
  Time   TvsF(double f) const;

  /**  Waveform Frequency.
    *  freq(t) returns the in-spiral frequency at Absolute time t.
    */
  double freq(const Time& t) const;

  /**  Waveform Phase angle.
    *  phi(t) returns the phase angle at absolute time t.
    */
  double  phi(const Time& t) const;

  /**  Amplitude.
    *  Ampl(t) returns the wave-form amplitude.
    */
  double Ampl(const Time& t) const;

private:
  //---------------------------  External Parameters
  /**  Mass of first object.
    *  The mass of the lighter object in solar mass units.
    */
  double mMass1;

  /**  Mass of second object.
    *  The mass of the heavier object in solar mass units.
    */
  double mMass2;

  /**  Template start time.
    *  Absolute time of the start of template generation.
    */
  Time   mT0;

  /**  Template limit time.
    *  The template limit time is defined to be the absolute (GPS) time at 
    *  which the frequency is maximized. This is effectively the point at 
    *  which the Newtonian approximation breaks down.
    */
  Time   mTl;

  /**  Critical time.
    *  The critical time is the absolute (GPS) time at which the point-like
    *  objects merge.
    */
  Time   mTc;

  /**  Phase offset.
    *  This is the phase at Tc.
    */
  double mPhic;

  //---------------------------  Useful internal function.
  /**  Time at which the frequency is maximized.
    *  TMax searches over the interval Tmin - Tmax for a a maximum in 
    *  freq(t). It returns the Time at which the maximum is found.
    */
  Time TMax(const Time& tmin, const Time& tmax) const;

  //---------------------------  Derived constants.
  /**  Total mass.
    *  Internal variable calculated from MAss1 + Mass2.
    */
  double mMTot;

  /**  Reduced mass/Total.
    *  Internal variable calculated from Mass1*Mass2/(MTot*MTot).
    */ 
  double mEta;
};

#endif  //  INSPIRAL_HH
