/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef SENDS_THREAD_BASE
#define SENDS_THREAD_BASE
#include <pthread.h>


namespace thread {

   /**  Thread_base is a base class for threads. There is a single
     *  entry point for the thread.
     *  \brief Threaded asynchronous execution base class
     *  \author John Zweizig (john.zweizig@ligo.org)
     *  \version 1.1; Last modified May 16, 2010
     */
   class thread_base {
   public:
      /**  Construct the thread base class.
        *  \brief Thread base constructor.
        */
      thread_base(void);

      /**  Destroy the %thread_base class.
        *  \brief Thread base destructor
        */
      virtual ~thread_base(void);
 
      /**  Test whether the thread is currently executing.
        *  \brief Test the busy flag.
	*  \return True if the asynchronous thread is running.
	*/
      bool busy(void) const;

      /**  Wait for the thread to complete and store the return value
        *  pointer in the specified location.
	*  \brief Join the current thread.
	*  \param retcd Pointer to a location to receive the exit value.
	*  \return Zero if the join completed.
	*/
      int join(void** retcd) const;

      /**  Send a signal to this thread.
        *  \brief Send a signal.
	*  \param sig Signal number to be sent to the thread.
	*  \return pthread_kill return code.
	*/
      virtual int kill_thread(int sig);

      /**  Test the thread run-enable flag.
        *  \brief test the run state.
	*  \return True if thread execution is enabled.
	*/
      bool run(void) const;

      /**  Set the detached flag to the specified state. This sets the 
        *  detached flag in the thread attributes structure. The next 
	*  time a thread is spawned it will be run as a detached thread.
	*  \brief Enable detached execution
	*  \param d If true, enable detached running, else disable.
	*/
      void set_detached(bool d);

      /**  Set the minimum stack size. By default, thread_base uses 
        *  a stack size of 1MB. If the requested stack size is less 
	*  than a safe minimum (16kB), the minimum will be used.
	*  \brief Set the stack size
	*  \param ssiz Minimum stack size in bytes.
	*/
      void set_stacksize(long ssiz);

      /**  Start the thread. The run flag is set and the busy flag is set 
        *  if the thread is successfully started.
	*  \return pthread_create error code.
	*/
      virtual int start_thread(void);

      /**  Stop the thread by clearing its run-enable flag. The thread will 
        *  stop only if it checks its run-enable flag (see run()).
        *  \brief Clear the run-enable flag.
	*  \return Zero on success.
	*/
      virtual int stop_thread(void);

      /**  Method to be executed when the thread is started.
        */
      virtual void* thread_entry(void) = 0;

   private:
      /**  thread_base stub method to set-up for thread execution. The
        *  thread_stub sets the run and busy flags and then calls the 
	*  thread_entry method. Exceptions in thread_entry are caught
	*  and the busy flag is clear after thread entry is returned.
	*  \brief Set up thread environment.
	*  \return User defined return value from thread_entry.
	*/
      void* thread_stub(void);

      /**  Private thread launch function used by start_thread() to cast the
        *  argument to the thread_base class and invoke the thread_stub method.
	*  \brief launch a thread
	*  \param p thread_base pointer to the class object.
	*  \return User defined return code from thread_stub/thread_entry.
	*/
      static void* launch(void* p);

   private:
      volatile bool  mRun;
      volatile bool  mBusy;
      pthread_t      mThreadID;
      pthread_attr_t mThreadAttr;
   };

   //==================================  Inline methods
   inline bool
   thread_base::busy(void) const {
      return mBusy;
   }

   inline bool
   thread_base::run(void) const {
      return mRun;
   }
}

#endif // !defined(SENDS_THREAD_BASE)
