/**  @name Bit Manipulaton
  *  A quick and nasty collection of inline bit manipulation functions.
  *  @memo Bit twiddling functions
  *  @author J. Zweizig
  *  @version 1.0; Modified 14-Jul-2000
  */
//@{
/** Bit count data type;
  * @memo Bit count type.
  */
typedef unsigned int  bcount_t;

/** Bit mask data type;
  * @memo Bit mask type.
  */
typedef unsigned long bmask_t;

/**  Count the number of bits set. This function iterates once per set 
  *  bit and deletes the lowes bit with each iteration.
  *  @memo Count set bits.
  */
inline bcount_t
count_set_bits(bmask_t x) {
    bcount_t r=0;
    for ( ; x ; x &= x-1) r++;
    return r;
}

/**  Return the bit number (0 -> Nbit-1) of the highest set bit. A binary
  *  search technique is used.
  *  @memo Number of highest set bit.
  */
inline bcount_t 
highest_bit_set(bmask_t x) {
    bcount_t r = 0;
    for (bcount_t i=4*sizeof(bmask_t) ; x>1 && i ; i>>=1) {
        bmask_t hx = x >> i;
        if (hx) {
	    x  = hx;
	    r += i;
	}
    }
    return r;
}

/**  Returns true if the argument has exactly one bit set i.e. if the
  *  numeric value is a power of two.
  *  @memo test for power of two.
  */
inline bool 
is_power_of_2(bmask_t x) {
    return x && !(x & (x-1));
}

/**  Make a bit mask consisting of 1s in the N lowest order positions.
  *  @memo Make a mask with N set bits.
  */
inline bmask_t
make_mask(bcount_t N) {
    return (1<<N) - 1;
}

/**  Make a bit mask with 1s in the positions Nlow <= x < Nhigh. No check 
  *  is made on the asrguments and if they are not in the range 
  *  0 <= Nlow < Nhigh < Nbits, the results are unpredictable.
  *  @memo Make a mask with a range of bits set.
  */
inline bmask_t
make_mask(bcount_t Nlow, bcount_t Nhigh) {
    return (1<<Nhigh) - (1<<Nlow);
}

/**  Return the bit number (0 -> Nbit-1) of the lowest set bit. A binary
  *  search technique is used.
  *  @memo Number of lowest set bit.
  */
inline bcount_t 
lowest_bit_set(bmask_t x) {
    bcount_t r = 0;
    for (bcount_t i=4*sizeof(bmask_t) ; x>1 && i ; i>>=1) {
        if (!(x & make_mask(i))) {
	    x >>= i;
	    r  += i;
	}
    }
    return r;
}

/**  Return true if the the bits have odd parity.
  *  @memo Number of lowest set bit.
  */
inline bool
odd_parity(bmask_t v) {
    bmask_t s = v;
    for (int i=sizeof(bmask_t)*4; i; i /= 2) s ^= (s >> i);
    return (s & 1) != 0; 
}

//@}
