// (c) Microsoft Corporation 2005-2007. 
//-------------------------------------------------------------------------
// Reflection on F# values. Analyze an object to see if it the representation
// of an F# value.  Augments System.Reflection.
//-------------------------------------------------------------------------

#light


module Microsoft.FSharp.Reflection 

open System
open System.Reflection
open Microsoft.FSharp.Core
open Microsoft.FSharp.Primitives.Basics
open Microsoft.FSharp.Collections

//---------------------------------------------------------------------
// F# reified type inspection.

///Represents the one-level decomposition of an F# type according to the
///various structural types of the F# language.  
///
/// Note: Most attributes associated with a System.Type value can be accessed
/// via regular .NET reflection. However, it is useful to be able to view some
/// F# type definitions from the perspective of the F# language, in particular
/// for record, tuple and discriminated-union definitions. Although these types
/// are ultimately compiled as .NET types, the details of the encoding used is
/// often irrelevant, and this type and other items in this API help hide the
/// details of this encoding.
///
type TypeInfo =
#if CLI_AT_LEAST_2_0
    /// An F# tuple type
    | TupleType of Type list
      
    /// An F# function type
    | FunctionType of Type * Type
      
    /// An F# record type
    | RecordType of (string * Type) list
      
    /// An F# discriminated union type
    | SumType of (string * (string * Type) list) list
      
    /// An F# or .NET delegate type. The types carried in the result are the argument
    /// and return types of the Invoke method on the delegate. Delegates accepting byref
    /// arguments are returned as ObjectType.
    | DelegateType of Type list * Type
      
    /// The F# "unit" type
    | UnitType 
      
    /// An F# class, interface or struct type, or such a type from some other .NET language
    | ObjectType of Type
      
#else
    | TupleType of int
    | FunctionType of unit * unit
    | RecordType of (string * Type) list
    | SumType of (string * (string * Type) list) list
    | DelegateType of Type list * Type
    | UnitType 
    | ObjectType of Type
#endif

module Type = 
(*
        /// An F# variable type
        val (|Variable|Tuple|Function|Record|Sum|Abstract|Unit|External|): System.Type -> ...
          
*)

    /// Throw away extraneous compiler-specific information from a runtime type to return a true F# type.
    val GetTypeOfReprType : System.Type -> System.Type

    /// A convenience function
    val IsUnitType : System.Type -> bool

    /// A convenience function
    val IsListType : System.Type -> bool

    /// A convenience function
    val IsOptionType : System.Type -> bool

    /// Get the TypeInfo for a System.Type
    val GetInfo: System.Type -> TypeInfo

    /// The maximum size for which the structure of tuple types are faithfully reported by reflection
    val MaxTupleSize : int 
    


//---------------------------------------------------------------------
// F# dynamic value inspection.

///Represents the one-level decomposition of an F# value according to the
///various structural types of the F# language.  
type ValueInfo =

    /// An F# tuple value
    | TupleValue of obj list

    /// An F# function value.  The System.Type refines
    /// the set of possible syntactic closures that might
    /// correspond to this closure value.
    | FunctionClosureValue of System.Type * obj

    /// An F# record value
    | RecordValue of (string * obj) list

    /// An F# discriminated union value. The System.Type and string values refine
    /// the set of possible discriminators that may correspond to this value. 
    | ConstructorValue of System.Type * string * (string * obj) list

    /// An F# exception value.   The System.Type identifies
    /// the syntactic exception declarations that corresponds to 
    /// this exception value.
    | ExceptionValue of System.Type * (string * obj) list

    /// The F# "()" value
    | UnitValue

    /// An value of a class, interface, struct or delegate type other
    /// then the special values detected above.
    | ObjectValue of obj

//---------------------------------------------------------------------
// Families of F# values and other operations on values.


module Value = 
#if CLI_AT_LEAST_2_0
    /// Get the reified F# type of the value.  This will often be less
    /// specific than obj.GetType().
    val GetType: 'a -> System.Type
    val GetTypeInfo: 'a -> TypeInfo
    /// Get the reflective view of a value
    val GetInfo: 'a -> ValueInfo
#else
    val inline GetType: 'a -> System.Type
    val inline GetTypeInfo: 'a -> TypeInfo
    val inline GetInfo: 'a -> ValueInfo
#endif

    /// Precompute a function for reading a particular field from a record.
    /// Assumes the given type is a RecordType with a field of the given name. 
    /// If not an unspecified exception is raised during pre-computation.
    ///
    /// Using the computed function will be much faster than executing a corresponding call to Value.GetInfo
    /// because the path executed by the computed function is optimized given the knowledge that it will be
    /// used to read values of the given type.
    val GetRecordFieldReader     : Type * field:string -> (obj -> obj)

    /// Precompute a function for reading all the fields from a record. The fields are returned in the
    /// same order as the fields reported by a call to Microsoft.FSharp.Reflection.Type.GetInfo for
    /// this type.
    ///
    /// Assumes the given type is a RecordType. 
    /// If not an unspecified exception is raised during pre-computation.
    ///
    /// Using the computed function will be much faster than executing a corresponding call to Value.GetInfo
    /// because the path executed by the computed function is optimized given the knowledge that it will be
    /// used to read values of the given type.
    val GetRecordReader          : Type -> (obj -> obj[])
    (* 
    /// Precompute a function for reading a particular field from a discriminator of a sum type
    val GetSumFieldReader      : Type * discriminator:int * field:string -> (obj -> obj)
    *)
    
    /// Precompute a function for reading an integer representing the discriminator tag of a sum type.
    ///
    /// Assumes the given type is a SumType. 
    /// If not an unspecified exception is raised during pre-computation.
    ///
    /// Using the computed function will be much faster than executing a corresponding call to Value.GetInfo
    /// because the path executed by the computed function is optimized given the knowledge that it will be
    /// used to read values of the given type.
    val GetSumTagReader          : Type -> (obj -> int)

    /// Precompute a function for reading all the fields for a particular discriminator tag of a sum type
    ///
    /// Assumes the given type is a SumType where the tag is a legitimate tag for the type.
    /// If not an unspecified exception is raised during pre-computation.
    ///
    /// Tags can be mapped to and from names using the functions returned by GetSumTagConverters
    ///
    /// Using the computed function will be much faster than executing a corresponding call to Value.GetInfo
    /// because the path executed by the computed function is optimized given the knowledge that it will be
    /// used to read values of the given type.
    val GetSumRecordReader       : Type * int -> (obj -> obj[])
    
    /// Precompute a function for reading the values of a particular tuple type
    ///
    /// Assumes the given type is a TupleType.
    /// If not an unspecified exception is raised during pre-computation.
    val GetTupleReader           : Type -> (obj -> obj[])

    /// Precompute a function for constructing a record value. 
    ///
    /// Assumes the given type is a RecordType.
    /// If not an unspecified exception is raised during pre-computation.
    val GetRecordConstructor     : Type -> (obj[] -> obj)
    
    /// Precompute a function for constructing a discriminated union value for a particular tag. 
    ///
    /// Assumes the given type is a SumType where the tag is a legitimate tag for the type.
    /// If not an unspecified exception is raised during pre-computation.
    val GetSumConstructor        : Type * tag:int -> (obj[] -> obj)
    
    /// Precompute a function for reading the values of a particular tuple type
    ///
    /// Assumes the given type is a TupleType.
    /// If not an unspecified exception is raised during pre-computation.
    val GetTupleConstructor      : Type -> (obj[] -> obj)

    /// Precompute a pair of functions for converting between integer discriminator tags
    /// the names of the discriminators for the given sum type.
    ///
    /// Assumes the given type is a SumType. 
    /// If not an unspecified exception is raised during pre-computation.
    val GetSumTagConverters      : Type -> int * (int -> string) *  (string -> int)



//---------------------------------------------------------------------
// F# Module/TypeDefinition/ValueDefinition inspection


type ValueDefinition
    with 
        /// A handle that represents the compiled form of the F# value
        member CompiledHandle : System.Reflection.MemberInfo

(*
        /// The full F# compiled type of the value.  If the value definition is generic
        /// then the type is with respect to the generic type parameters of the
        /// compiledHandle for the value definition.
        member Type : System.Type

        /// Similar to the Type property, but instantiates the type of a 
        /// generic value with the given types.
        member GetInstantiatedType : System.Type[] -> System.Type

        /// Read the value. 
        /// Raises a InvalidArgumentException if the type of the object is generic,
        /// in which case GetInstantiatedValue must be used to indicate the types
        /// at which the generic value are being used.
        ///
        /// For non-generic objects the runtime type of the object 
        /// returned is compatible with the type of the value definition and
        /// is also compatible with the reified type returned by the Type property.  
        ///
        /// If the value definition is mutable then the value returned may vary as a result of
        /// modifications to the value.
        ///
        /// If the value definition is compiled as a method then this property
        /// will build a function value of the correct type that then calls the 
        /// underlying method.  
        member GetValue : unit -> obj  
    
        /// Similar to the Value property, but returns a value where the type
        /// parameters of the value have been instantiated to the given type 
        /// arguments.
        member GetInstantiatedValue : System.Type[] -> obj  

        /// Attempt to look up the raw quotation definition for the value.
        /// Raises a InvalidArgumentException if the quotation definition was not
        /// stored at the time of compilation of the value, or if the definition of
        /// the value uses an F# language construct not supported by the quotation
        /// library.
        member Definition : Quotations.Raw.expr
    
        /// Similar to the Value property, but returns a value where the type
        /// parameters of the value have been instantiated to the given type 
        /// arguments.
        member GetInstantiatedDefinition : System.Type[] -> Quotations.Raw.expr  
*)

  end
  
type TypeDefinition
    with 
        /// A handle that represents the compiled form of the F# type definition.
        /// This can be used to access all the features of the type definition, e.g. the
        /// type parameters, properties and other members. 
        member CompiledHandle : System.Type

        /// Return the GetTypeInfoOfType decomposition of a generic instance of the type.
        member TypeInfo : TypeInfo

    end

type ModuleDefinition 
    with 

        /// Fetch the list of public type definitions in the F# Module.
        /// Private types and type abbreviations are not returned.
        member TypeDefinitions: TypeDefinition list

        /// Fetch the list of public nested module definitions in the F# Module.
        member ModuleDefinitions: ModuleDefinition list

        /// Fetch the list of concrete values in the F# Module, i.e. those that 
        /// have a canonical corresponding construct in the compiled IL version of the F# code. Inlined
        /// values and private values optimized away during compilation are not returned.
        member ConcreteValues: ValueDefinition list

        /// A handle that represents the compiled form of the F# module.
        member CompiledHandle : System.Type

    end    

module Assembly = 
        
    /// Get an enumeration of handles to the F#-defined concrete 
    /// module definitions in an assembly.  Nested module definitions
    /// are not returned.
    val GetFSharpModules : System.Reflection.Assembly -> ModuleDefinition list

    /// Get handles to the F#-defined concrete type definitions 
    /// in an assembly.  Type definitions nested inside modules
    /// are not returned, nor are type definitions hidden inside modules, 
    /// and nor are type abbreviations.
    val GetFSharpTypeDefinitions   : System.Reflection.Assembly -> TypeDefinition list
    
[<Obsolete("This function has been marked for deletion in a future version of F#.  Use Microsoft.FSharp.Reflection.Assembly.GetFSharpModules instead")>]
val GetAssemblyModules : System.Reflection.Assembly -> ModuleDefinition list

[<Obsolete("This function has been marked for deletion in a future version of F#.  Use Microsoft.FSharp.Reflection.Assembly.GetFSharpTypes instead")>]
val GetAssemblyTypes   : System.Reflection.Assembly -> TypeDefinition list

[<Obsolete("This function has been marked for deletion in a future version of F#.  Use Microsoft.FSharp.Reflection.Type.GetInfo instead")>]
val GetTypeInfoOfType : System.Type -> TypeInfo

[<Obsolete("This type is been marked for deletion in a future version of F#. Consider using Microsoft.FSharp.Reflection.Type.GetInfo instead")>]
type 'a typeinfo = { result: TypeInfo }

[<Obsolete("This function has been marked for deletion in a future version of F#.  Use Microsoft.FSharp.Reflection.Type.GetInfo instead")>]
#if CLI_AT_LEAST_2_0
val typeinfoof : unit -> typeinfo< 'a >
#else
val inline typeinfoof : unit -> typeinfo< $a >
#endif

[<Obsolete("This function has been marked for deletion in a future version of F#.  Use Microsoft.FSharp.Reflection.Value.GetInfo instead")>]
val GetValueInfoOfObject : obj -> ValueInfo

[<Obsolete("This function has been marked for deletion in a future version of F#.  Use Microsoft.FSharp.Reflection.Value.GetInfo instead")>]
#if CLI_AT_LEAST_2_0
val GetValueInfo : 'a -> ValueInfo
#else
val inline GetValueInfo : 'a -> ValueInfo
#endif

[<Obsolete("This function has been marked for deletion in a future version of F#")>]
#if CLI_AT_LEAST_2_0
val GetTypeInfoOfValue : 'a -> TypeInfo
#else
val inline GetTypeInfoOfValue : 'a -> TypeInfo
#endif

[<Obsolete("This function has been marked for deletion in a future version of F#. Consider using Reflection.Type.IsUnitType instead")>]
/// A convenience function
val IsUnitType : System.Type -> bool

[<Obsolete("This function has been marked for deletion in a future version of F#. Consider using Reflection.Type.IsListType instead")>]
/// A convenience function
val IsListType : System.Type -> bool

[<Obsolete("This function has been marked for deletion in a future version of F#. Consider using Reflection.Type.IsOptionType instead")>]
/// A convenience function
val IsOptionType : System.Type -> bool

[<Obsolete("This data recognizer has been renamed to ObjectType")>]
val (|ExternalType|_|) : TypeInfo -> Type option
[<Obsolete("This data recognizer has been renamed to ObjectType")>]
val (|AbstractType|_|) : TypeInfo -> Type option

[<Obsolete("This data constructor has been renamed to ObjectType")>]
val ExternalType : Type -> TypeInfo
[<Obsolete("This data constructor has been renamed to ObjectType")>]
val AbstractType : Type -> TypeInfo

[<Obsolete("This data recognizer has been renamed to ObjectValue")>]
val (|ExternalValue|_|) : ValueInfo -> obj option
[<Obsolete("This data recognizer has been renamed to ObjectValue")>]
val (|AbstractValue|_|) : ValueInfo -> obj option

[<Obsolete("This data constructor has been renamed to ObjectValue")>]
val ExternalValue : obj -> ValueInfo
[<Obsolete("This data constructor has been renamed to ObjectValue")>]
val AbstractValue : obj -> ValueInfo


