// The purpose of this file is to connect the Caml and F# code in ilwrite.ml
// to the PDB writer COM component provided by the CLR.  Both the Microsoft
// .NET CLR and the Rotor SSCLI CLR provide such a component, though the means
// of accesssing the components vary slightly.  There is no consistent way of
// accessing debug symbol writers from managed code across multiple platforms
// (unless you are writing the binary using reflection).
//
// You might need to register diasymreader.dll for this code to function
// correctly.  That DLL contains the PDB writer COM component for the
// Microsoft.NET CLR.

#include "common.h"
#include "getcor.h"

#include <corsym.h>

#include <mscoree.h>


// Unfortunately V1.0 and V1.1. of the Microsoft .NET CLI
// use different UUIDs values for the interfaces in the SymWriter API
// but use the same symbolic names for these UUIDs.  This breaking
// change was made for really very silly reasons indeed.  Thus
// we redefine the V1 version of these GUIDs here.  V1.0 of the SSCLI
// uses the same GUIDs as V1.0 of the Microsoft .NET CLI.
// 
#define MIDL_DEFINE_GUID(type,name,l,w1,w2,b1,b2,b3,b4,b5,b6,b7,b8) \
        const type name = {l,w1,w2,{b1,b2,b3,b4,b5,b6,b7,b8}}

MIDL_DEFINE_GUID(IID, IID_V1_0_ISymUnmanagedWriter,0x2DE91396,0x3844,0x3B1D,0x8E,0x91,0x41,0xC2,0x4F,0xD6,0x72,0xEA);
MIDL_DEFINE_GUID(IID, IID_V1_1_ISymUnmanagedWriter,0xED14AA72,0x78E2,0x4884,0x84,0xE2,0x33,0x42,0x93,0xAE,0x52,0x14);
MIDL_DEFINE_GUID(IID, IID_V1_1_ISymUnmanagedWriter2,0x0B97726E,0x9E6D,0x4f05,0x9A,0x26,0x42,0x40,0x22,0x09,0x3C,0xAA);

MIDL_DEFINE_GUID(IID, IID_V2_0_IMetaDataImport2, 0xfce5efa0, 0x8bba, 0x4f8e, 0xa0, 0x36, 0x8f, 0x20, 0x22, 0xb0, 0x84, 0x66);

MIDL_DEFINE_GUID(CLSID, CLSID_CorSymWriter_Deprecated,0x108296C1,0x281E,0x11d3,0xBD,0x22,0x00,0x00,0xF8,0x08,0x49,0xBD);
MIDL_DEFINE_GUID(CLSID, CLSID_CorSymWriter_SxS,0x0AE2DEB0,0xF901,0x478B,0xBB,0x9F,0x88,0x1E,0xE8,0x06,0x67,0x88);


#undef MIDL_DEFINE_GUID


//-----------------------------------------------------------------------------
// Entry points used by both the Augmented-Managed (i.e. F#-on-windows) and 
// OCaml compialtions of this code.
//-----------------------------------------------------------------------------

HRESULT DoCoCreateInstance(bool forceUseOfSSCLI,
                           REFCLSID   rclsid,
                           REFIID     riid,
                           void     **ppv)
{

#ifndef FEATURE_PAL
    if (forceUseOfSSCLI) {
#endif
        return PAL_MyCoCreateInstance(rclsid,riid,ppv);
#ifndef FEATURE_PAL
    } else {
        return CoCreateInstance(rclsid, NULL,CLSCTX_INPROC_SERVER,riid,ppv);
    }
#endif
}



EXTAPI pdbWriteInitialize(LPCTSTR binaryName, 
                          LPCTSTR pdbName, 
                          bool forceUseOfSSCLI, 
                          int versionHint,
                          ISymUnmanagedWriter **ppRes1, 
                          IMetaDataImport **ppRes2)
{



#ifdef CAML_STUBS
#ifndef FEATURE_PAL
    IfFailRet(CoInitialize(NULL), "initializing COM");
#endif
#endif

    IMetaDataImport *pImport = NULL;
    IMetaDataDispenser *pDisp = NULL;
    IfFailRet(DoCoCreateInstance(forceUseOfSSCLI,CLSID_CorMetaDataDispenser, 
                                 IID_IMetaDataDispenser, (void **) &pDisp), "getting SSCLI CorMetaDataDispenser");
    ISymUnmanagedWriter *pWriter;
    HRESULT hr = S_OK;


    // Yes, there was a breaking change where the IID for 
    // the ISymUnmanagedWriter changed
    // between RTM and Everett for really ridiculous reasons.  Hence
    // we go looking for both IIDs.  The order of search depends on the
    // version we're compiling for 
    //    0 = unknown/v2.0
    //    1 = v1.0.3705
    //    2 = v1.1.4322
    //
    // Also, there seems to be a new CLSID to use in v2.0.  Great.

#define NUMITEMS 5
    IID CLSIDs_MSCLR_UnknownVersion[NUMITEMS] = { CLSID_CorSymWriter_SxS       , CLSID_CorSymWriter_SxS      , CLSID_CorSymWriter_SxS      , CLSID_CorSymWriter_Deprecated, CLSID_CorSymWriter_Deprecated };
    IID IIDs_MSCLR_UnknownVersion[NUMITEMS]   = { IID_V1_1_ISymUnmanagedWriter2, IID_V1_1_ISymUnmanagedWriter, IID_V1_0_ISymUnmanagedWriter, IID_V1_1_ISymUnmanagedWriter , IID_V1_0_ISymUnmanagedWriter };

    IID CLSIDs_MSCLR_v1_0_3705[NUMITEMS]      = { CLSID_CorSymWriter_Deprecated, CLSID_CorSymWriter_Deprecated , CLSID_CorSymWriter_SxS      , CLSID_CorSymWriter_SxS      , CLSID_CorSymWriter_SxS        };
    IID IIDs_MSCLR_v1_0_3705[NUMITEMS]        = { IID_V1_0_ISymUnmanagedWriter , IID_V1_1_ISymUnmanagedWriter  , IID_V1_0_ISymUnmanagedWriter, IID_V1_1_ISymUnmanagedWriter, IID_V1_1_ISymUnmanagedWriter2 };

    IID CLSIDs_MSCLR_v1_1_4322[NUMITEMS]      = { CLSID_CorSymWriter_Deprecated, CLSID_CorSymWriter_Deprecated , CLSID_CorSymWriter_SxS      , CLSID_CorSymWriter_SxS      , CLSID_CorSymWriter_SxS        };
    IID IIDs_MSCLR_v1_1_4322[NUMITEMS]        = { IID_V1_1_ISymUnmanagedWriter , IID_V1_1_ISymUnmanagedWriter  , IID_V1_1_ISymUnmanagedWriter, IID_V1_1_ISymUnmanagedWriter, IID_V1_1_ISymUnmanagedWriter2 };

    CLSID *pCLSIDs = (versionHint = 1) ? CLSIDs_MSCLR_v1_0_3705 : (versionHint = 2) ? CLSIDs_MSCLR_v1_1_4322 : CLSIDs_MSCLR_UnknownVersion; 
    IID   *pIIDs   = (versionHint = 1) ? IIDs_MSCLR_v1_0_3705   : (versionHint = 2) ? IIDs_MSCLR_v1_1_4322   : IIDs_MSCLR_UnknownVersion; 

    for (int i = 0; i < NUMITEMS; i++) 
    {
        if (SUCCEEDED (hr = DoCoCreateInstance(forceUseOfSSCLI,
                                               pCLSIDs[i],
                                               pIIDs[i],
                                               (void**)&pWriter)))
        {
            break;
        }
    }
    if (FAILED(hr))
        return hr;

    if ((versionHint > 0) || FAILED(pDisp->OpenScope(binaryName, cssAccurate, IID_V2_0_IMetaDataImport2,
                                                     (IUnknown**) &pImport)))
    {
        
        IfFailRet (pDisp->OpenScope(binaryName, cssAccurate, IID_IMetaDataImport,
                                    (IUnknown**) &pImport), "opening emitted module");
    }

    pDisp->Release();

    IfFailRet(pWriter->Initialize(pImport,pdbName,NULL, TRUE), "initializing PDB writer");

    if (ppRes1)
        *ppRes1 = pWriter;
    if (ppRes2)
        *ppRes2 = pImport;
    return S_OK;

}


EXTAPI pdbWriteClose(ISymUnmanagedWriter *pWriter,IMetaDataImport *pImport, bool forceUseOfSSCLI)
{
    IfFailRet(pWriter->Close(), "closing PDB writer");
    pWriter->Release();

    // The Rotor code in rotor/clr/src/Tools/ildbsymbols/symwrite.cpp does not release pImport correctly.
    // Thus whenever we're using that API we force an extra Release.
#ifndef FEATURE_PAL
    if (forceUseOfSSCLI)
#endif
        pImport->Release();
    pImport->Release();

    return S_OK;
}



EXTAPI pdbWriteGetDebugInfo(ISymUnmanagedWriter *pWriter, 
			    UINT32 *pCharacteristics, 
			    UINT32 *pMajorVersion, 
			    UINT32*pMinorVersion, 
			    UINT32*pType, 
			    BYTE *pDataBuffer, 
			    DWORD *pDataBufferSize)
{
    DWORD cData;
    IfFailRet (pWriter->GetDebugInfo(NULL,0,&cData,NULL), "getting size of debug info");
    if (pDataBufferSize) 
	*pDataBufferSize = cData;
    if (pDataBuffer)
    {
       IMAGE_DEBUG_DIRECTORY IID;
       IfFailRet (pWriter->GetDebugInfo(&IID,cData,NULL,pDataBuffer), "getting debug info");
       if (pCharacteristics) *pCharacteristics = IID.Characteristics;
       if (pMajorVersion) *pMajorVersion = IID.MajorVersion;
       if (pMinorVersion) *pMinorVersion = IID.MinorVersion;
       if (pType) *pType = IID.Type;
    }
    return S_OK;
}



EXTAPI pdbWriteSetUserEntryPoint(ISymUnmanagedWriter *pWriter,ULONG32 tok)
{
    IfFailRet(pWriter->SetUserEntryPoint(tok), "writing PDB entrypoint");
    return S_OK;
}

EXTAPI pdbWriteDefineDocument(ISymUnmanagedWriter *pWriter,LPCTSTR urlw,ISymUnmanagedDocumentWriter **ppRes)
{
    ISymUnmanagedDocumentWriter *pDocWriter;
    IfFailRet(pWriter->DefineDocument(urlw, &CorSym_LanguageType_ILAssembly /* CorSym_LanguageType_CSharp */ , &CorSym_LanguageVendor_Microsoft, &CorSym_DocumentType_Text, &pDocWriter), "defining document");

    if (ppRes) 
	*ppRes = pDocWriter;
    return S_OK;
}

EXTAPI pdbWriteOpenMethod(ISymUnmanagedWriter *pWriter,ULONG32 tok)
{
    IfFailRet(pWriter->OpenMethod(tok), "opening method");
    return S_OK;
}

EXTAPI pdbWriteCloseMethod(ISymUnmanagedWriter *pWriter)
{
    IfFailRet(pWriter->CloseMethod(), "closing method");
    return S_OK;
}


EXTAPI pdbWriteOpenScope(ISymUnmanagedWriter *pWriter,ULONG32 start)
{
    ULONG32 scopeId = 0;
    IfFailRet(pWriter->OpenScope(start,&scopeId), "opening scope");
    return S_OK;
}


EXTAPI pdbWriteCloseScope(ISymUnmanagedWriter *pWriter,ULONG32 end)
{
    IfFailRet(pWriter->CloseScope(end), "closing scope");
    return S_OK;

}

static BYTE c = ELEMENT_TYPE_I4;

EXTAPI pdbWriteDefineLocalVariable(ISymUnmanagedWriter *pWriter,LPCTSTR wname, BYTE *pSig, int nSig, ULONG32 attrib)
{
    IfFailRet(pWriter->DefineLocalVariable(wname,
                                             0,
                                             nSig,pSig,
                                             ADDR_IL_OFFSET, 
                                             attrib, 
                                             0, 
                                             0, 
                                             0, 
                                             0), "defining local");
    return S_OK;
}

EXTAPI pdbWriteSetMethodRange(ISymUnmanagedWriter *pWriter,
			      ISymUnmanagedDocumentWriter * pDocWriter1, 
			      ULONG32 line1, ULONG32 col1, 
			      ISymUnmanagedDocumentWriter *pDocWriter2, 
			      ULONG32 line2, ULONG32 col2)
{
    // Ignore E_NOTIMPL errors: this is not implemented on Rotor
    IfImplementedAndFailRet(pWriter->SetMethodSourceRange(pDocWriter1, line1,col1, pDocWriter2, line2,col2), "setting method range");
    return S_OK;
}



EXTAPI pdbWriteDefineSequencePoints(ISymUnmanagedWriter *pWriter,
				    ISymUnmanagedDocumentWriter *pDocWriter, 
				    ULONG32 spCount,
				    ULONG32 *offsets,
				    ULONG32 *lines,
				    ULONG32 *columns,
				    ULONG32 *endLines,
				    ULONG32 *endColumns)
{
    if (spCount == 0) 
        return S_OK;

    IfFailRet(pWriter->DefineSequencePoints(pDocWriter, spCount, offsets, lines, columns, endLines, endColumns), "defining sequence points");
    return S_OK;
}


#ifdef CAML_STUBS

//-----------------------------------------------------------------------------
// OCaml stubs for the above
//-----------------------------------------------------------------------------

extern "C"
CAMLprim value pdbWriteInitializeCaml(value binary, value pdb, value forceUseOfSSCLI,value versionHint)
{

    CAMLparam3(binary, pdb, forceUseOfSSCLI);
    CAMLlocal1(result);

    WCHAR binaryName[_MAX_PATH];
    MultiByteToWideChar(CP_ACP, 0, String_val(binary), -1, binaryName, _MAX_PATH);

    WCHAR pdbName[_MAX_PATH];
    MultiByteToWideChar(CP_ACP, 0, String_val(pdb), -1, pdbName, _MAX_PATH);

    IMetaDataImport *pImport = NULL;
    ISymUnmanagedWriter *pWriter = NULL;
    IfFailThrow(pdbWriteInitialize(binaryName, pdbName, Bool_val(forceUseOfSSCLI), Int_val(versionHint), &pWriter, &pImport), "pdbWriteInitialize");
    result = alloc(2,0 ); // each pdb_reader holds a two pointer values 
    Store_field(result,0,copy_nativeint((long) pWriter));
    Store_field(result,1,copy_nativeint((long) pImport));

    CAMLreturn(result);

}


extern "C"
CAMLprim value pdbWriteCloseCaml(value pdbw, value forceUseOfSSCLI)
{
    CAMLparam2(pdbw, forceUseOfSSCLI);
    ISymUnmanagedWriter *pWriter = (ISymUnmanagedWriter *) Nativeint_val(Field(pdbw,0));
    IMetaDataImport *pImport = (IMetaDataImport *) Nativeint_val(Field(pdbw,1));
    IfFailThrow(pdbWriteClose(pWriter, pImport, Bool_val(forceUseOfSSCLI)),"closing PDB writer");

    CAMLreturn(Atom(0));

}



extern "C"
CAMLprim value pdbWriteGetDebugInfoCaml(value pdbw)
{
    CAMLparam1(pdbw);
    CAMLlocal2(data,result);
    
    ISymUnmanagedWriter *pWriter = (ISymUnmanagedWriter *) Nativeint_val(Field(pdbw,0));

    UINT32 Characteristics = 0;
    UINT32 MajorVersion = 0;
    UINT32 MinorVersion = 0;
    UINT32 Type = 0;
    DWORD cData;

    IfFailThrow(pdbWriteGetDebugInfo(pWriter,NULL, NULL,NULL,NULL, NULL, &cData), "getting size of debug info");

    data = alloc_string(cData);
    char *dataval = String_val(data);

    IfFailThrow(pdbWriteGetDebugInfo(pWriter,&Characteristics, &MajorVersion, &MinorVersion, &Type, (BYTE *)dataval, &cData), "getting debug info");
    
    result = alloc(5,0 ); 

    Store_field(result,0,copy_int32(Characteristics));
    Store_field(result,1,copy_int32(MajorVersion));
    Store_field(result,2,copy_int32(MinorVersion));
    Store_field(result,3,copy_int32(Type));
    Store_field(result,4,data);
    CAMLreturn(result);

}



extern "C"
CAMLprim value pdbWriteSetUserEntryPointCaml(value pdbw, value tok)
{
    CAMLparam2(pdbw, tok);
    ISymUnmanagedWriter *pWriter = (ISymUnmanagedWriter *) Nativeint_val(Field(pdbw,0));

    IfFailThrow(pdbWriteSetUserEntryPoint(pWriter,Int32_val(tok)), "writing PDB entrypoint");

    CAMLreturn(Atom(0));

}

extern "C"
CAMLprim value pdbWriteDefineDocumentCaml(value pdbw, value url)
{
    CAMLparam2(pdbw, url);
    ISymUnmanagedWriter *pWriter = (ISymUnmanagedWriter *) Nativeint_val(Field(pdbw,0));

    WCHAR urlw[_MAX_PATH];
    MultiByteToWideChar(CP_ACP, 0, String_val(url), -1, urlw, _MAX_PATH);

    ISymUnmanagedDocumentWriter *pDocWriter;
    IfFailThrow(pdbWriteDefineDocument(pWriter,urlw, &pDocWriter), "defining document");

    CAMLlocal1(result);
    result = alloc(1,0 ); // each pdb_document_reader holds a single pointer value
    Store_field(result,0,copy_nativeint((long) pDocWriter));

    CAMLreturn(result);

}
extern "C"
CAMLprim value pdbWriteOpenMethodCaml(value pdbw, value tok)
{
    CAMLparam2(pdbw, tok);
    ISymUnmanagedWriter *pWriter = (ISymUnmanagedWriter *) Nativeint_val(Field(pdbw,0));
    IfFailThrow(pdbWriteOpenMethod(pWriter,Int32_val(tok)), "opening method");
    CAMLreturn(Atom(0));

}

extern "C"
CAMLprim value pdbWriteCloseMethodCaml(value pdbw)
{
    CAMLparam1(pdbw);
    ISymUnmanagedWriter *pWriter = (ISymUnmanagedWriter *) Nativeint_val(Field(pdbw,0));
    IfFailThrow(pdbWriteCloseMethod(pWriter), "closing method");
    CAMLreturn(Atom(0));

}


extern "C"
CAMLprim value pdbWriteOpenScopeCaml(value pdbw,value start)
{
    CAMLparam2(pdbw,start);
    ISymUnmanagedWriter *pWriter = (ISymUnmanagedWriter *) Nativeint_val(Field(pdbw,0));
    IfFailThrow(pdbWriteOpenScope(pWriter,Int_val(start)), "opening scope");
    CAMLreturn(Atom(0));
}


extern "C"
CAMLprim value pdbWriteCloseScopeCaml(value pdbw,value end)
{
    CAMLparam2(pdbw,end);
    ISymUnmanagedWriter *pWriter = (ISymUnmanagedWriter *) Nativeint_val(Field(pdbw,0));
    IfFailThrow(pdbWriteCloseScope(pWriter,Int_val(end)), "closing scope");
    CAMLreturn(Atom(0));

}

extern "C"
CAMLprim value pdbWriteDefineLocalVariableCaml(value pdbw, value nm, value sig, value attrib)
{
    CAMLparam4(pdbw,nm,sig,attrib);
    WCHAR wname[_MAX_PATH];
    MultiByteToWideChar(CP_ACP, 0, String_val(nm), -1, wname, _MAX_PATH);

    BYTE * pSig = (BYTE *) String_val(sig);
    ULONG32 nSig = string_length(sig);

    ISymUnmanagedWriter *pWriter = (ISymUnmanagedWriter *) Nativeint_val(Field(pdbw,0));
    IfFailThrow(pdbWriteDefineLocalVariable(pWriter,wname,pSig,nSig,Int32_val(attrib)), "defining local variable in debug information");
    CAMLreturn(Atom(0));
}

extern "C"
CAMLprim value pdbWriteSetMethodRangeCaml_native(value pdbw, value pdbdw1, value line1, value col1, value pdbdw2, value line2, value col2)
{
    CAMLparam5(pdbw,pdbdw1,line1,col1,pdbdw2);
    CAMLxparam2(line2,col2);
    ISymUnmanagedWriter *pWriter = (ISymUnmanagedWriter *) Nativeint_val(Field(pdbw,0));
    ISymUnmanagedDocumentWriter *pDocWriter1 = (ISymUnmanagedDocumentWriter *) Nativeint_val(Field(pdbdw1,0));
    ISymUnmanagedDocumentWriter *pDocWriter2 = (ISymUnmanagedDocumentWriter *) Nativeint_val(Field(pdbdw2,0));

    IfFailThrow(pdbWriteSetMethodRange(pWriter,pDocWriter1, Int_val(line1),Int_val(col1), pDocWriter2, Int_val(line2),Int_val(col2)), "setting method range");
    CAMLreturn(Atom(0));
}


extern "C"
CAMLprim value pdbWriteSetMethodRangeCaml_bytecode(value *argv, int argn)
{
    return pdbWriteSetMethodRangeCaml_native(argv[0], argv[1], argv[2],argv[3], argv[4], argv[5], argv[6]);
}




extern "C"
CAMLprim value pdbWriteDefineSequencePointsCaml(value pdbw, value pdbdw, value sps)
{
    CAMLparam3(pdbw,pdbdw,sps);

    ISymUnmanagedWriter *pWriter = (ISymUnmanagedWriter *) Nativeint_val(Field(pdbw,0));
    ISymUnmanagedDocumentWriter *pDocWriter = (ISymUnmanagedDocumentWriter *) Nativeint_val(Field(pdbdw,0));

    ULONG32 spCount = Wosize_val(sps);

    if (spCount == 0) 
        CAMLreturn(Atom(0));

    ULONG32 *offsets = (ULONG32 *) alloca(sizeof(ULONG32) * spCount);
    ULONG32 *lines = (ULONG32 *) alloca(sizeof(ULONG32) * spCount);
    ULONG32 *columns = (ULONG32 *) alloca(sizeof(ULONG32) * spCount);
    ULONG32 *endLines = (ULONG32 *) alloca(sizeof(ULONG32) * spCount);
    ULONG32 *endColumns = (ULONG32 *) alloca(sizeof(ULONG32) * spCount);

    CAMLlocal1(sp);
    for (int i = 0; i < spCount; i++) 
    {
        sp = Field(sps,i);
        offsets[i] = Int_val(Field(sp,0));
        lines[i] = Int_val(Field(sp,1));
        columns[i] = Int_val(Field(sp,2));
        endLines[i] = Int_val(Field(sp,3));
        endColumns[i] = Int_val(Field(sp,4));
    }

    IfFailThrow(pdbWriteDefineSequencePoints(pWriter,pDocWriter, spCount, offsets, lines, columns, endLines, endColumns), "defining sequence points");
    CAMLreturn(Atom(0));
}


#endif // CAML_STUBS


#ifdef CAML_STUBS

//-----------------------------------------------------------------------------
// A couple of random extra functions needed by the OCaml version of the 
// compiler in order to avoid a dependency on the Unix library.
//-----------------------------------------------------------------------------

extern "C"
CAMLprim value absilWriteGetTimeStampCaml(value pdbw)
{
    CAMLparam1(pdbw);
    time_t now;
    time(&now);
    CAMLreturn(copy_int32((DWORD) now));

}

static BOOL LUtilGetFileTime(LPCTSTR pszFilename, FILETIME* pFileTime)
{
    WIN32_FIND_DATA fd;
    HANDLE hFind = FindFirstFile(pszFilename, &fd);
    if(hFind != INVALID_HANDLE_VALUE)
    {
        *pFileTime = fd.ftLastWriteTime;
        FindClose(hFind);
        return TRUE;
    }
    return FALSE;
}




extern "C"
CAMLprim value fsharpServiceGetFileTimeStampCaml(value file)
{
    CAMLparam1(file);
    WCHAR filew[_MAX_PATH];
    MultiByteToWideChar(CP_ACP, 0, String_val(file), -1, filew, _MAX_PATH);

    FILETIME time;
    if (!LUtilGetFileTime(filew,&time))
        failwith("fsharpServiceGetFileTimeStamp: could not get file time");
    CAMLreturn(copy_int64((ULONGLONG( time.dwHighDateTime )<<32)|time.dwLowDateTime ));
}
#endif // CAML_STUBS

