(* (c) Microsoft Corporation. All rights reserved  *)

(* From "Algorithms", Cormen, Leiserson & Rivest p 252. *)
(* Also http://homepages.ius.edu/jholly/c343/notes/lrotate.htm *)

(*F# 
module Microsoft.Research.AbstractIL.Internal.Pmap 

type ('key,'a) t (*F# = Map.t<'key,'a> F#*)

let empty = Map.empty
let is_empty m = Map.is_empty m
let add k v m = Map.add k v m
let find k m = Map.find k m
let remove k m = Map.remove k m 
let mem k m = Map.mem k m 
let iter f m = Map.iter f m 
let exists f m = Map.exists f m 
let map f m = Map.map f m 
let filter f m = Map.filter f m 
let mapi f m = Map.mapi f m 
let fold f m x = Map.fold f m x
let tryfind f x = Map.tryfind f x
F#*)

(*IF-OCAML*)
type ('key,'a) t = E | N of 'key * 'a * ('key, 'a) t *  ('key, 'a) t * int

let empty = E

let height  = function
  | E -> 0
  | N (_,_,_,_,h) -> h

let is_empty = function
  | E -> true
  | N _ -> false

let mk l k v r = 
  let hl = height l in 
  let hr = height r in 
  let m = if hl < hr then hr else hl in 
  N(k,v,l,r,m+1)

let rebalance t1 k v t2 =
  let t1h = height t1 in 
  if  height t2 > t1h + 2 then (* right is heavier than left *)
    match t2 with 
      N(t2k,t2v,t2l,t2r,t2h) -> 
	(* one of the nodes must have height > height t1 + 1 *)
	if height t2l > t1h + 1 then  (* balance left: combination *)
	  match t2l with 
	  | N(t2lk,t2lv,t2ll,t2lr,t2lh) ->
	      mk (mk t1 k v t2ll) t2lk t2lv (mk t2lr t2k t2v t2r) 
	  | _ -> failwith "rebalance"
	else (* rotate left *)
	  mk (mk t1 k v t2l) t2k t2v t2r
    | _ -> failwith "rebalance"
  else
    let t2h = height t2 in 
    if  t1h > t2h + 2 then (* left is heavier than right *)
      match t1 with 
	N(t1k,t1v,t1l,t1r,t1h) -> 
	(* one of the nodes must have height > height t2 + 1 *)
	  if height t1r > t2h + 1 then 
	  (* balance right: combination *)
	    match t1r with 
	    | N(t1rk,t1rv,t1rl,t1rr,t1rh) ->
		mk (mk t1l t1k t1v t1rl) t1rk t1rv (mk t1rr k v t2)
	    | _ -> failwith "rebalance"
	  else
	    mk t1l t1k t1v (mk t1r k v t2)
      | _ -> failwith "rebalance"
    else mk t1 k v t2

let rec add k v = function 
    E -> N (k,v,E,E,1)
  | N (k2,v2,l,r,h) -> 
      let c = compare k k2 in 
      if c < 0 then rebalance (add k v l) k2 v2 r
      else if c = 0 then N(k,v,l,r,h)
      else rebalance l k2 v2 (add k v r) 

let rec find k = function
    E -> raise Not_found
  | N(k2,v2,l,r,_) -> 
      let c = compare k k2 in 
      if c < 0 then find k l
      else if c = 0 then v2
      else find k r

let rec tryfind k = function
    E -> None
  | N(k2,v2,l,r,_) -> 
      let c = compare k k2 in 
      if c < 0 then tryfind k l
      else if c = 0 then Some v2
      else tryfind k r

let rec splice_out_succ = function
  | E -> failwith "internal error: Map.splice_out_succ_or_pred"
  | N (k2,v2,l,r,_) ->
      match l with 
      |	E -> k2,v2,r
      | N _ -> let k3,v3,l' = splice_out_succ l in k3,v3,mk l' k2 v2 r

let rec remove k = function 
  | E -> E
  | N (k2,v2,l,r,_) -> 
      let c = compare k k2 in 
      if c < 0 then rebalance (remove k l) k2 v2 r
      else if c = 0 then 
	match l,r with 
	| E,_ -> r
	| _,E -> l
	| _, N(rk,rv,rl,rr,_) -> 
	    let sk,sv,r' = splice_out_succ r in 
	    mk l sk sv r'
      else rebalance l k2 v2 (remove k r) 

let rec mem k = function
    E -> false
  | N(k2,v2,l,r,_) -> 
      let c = compare k k2 in 
      if c < 0 then mem k l
      else if c = 0 then true
      else mem k r

let rec iter f = function
  | E -> ()
  | N(k2,v2,l,r,_) -> iter f l; f k2 v2; iter f r

let rec exists f = function
  | E -> false
  | N(k2,v2,l,r,_) -> f k2 v2 or exists f l or exists f r

let rec map f = function
  | E -> E
  | N(k2,v2,l,r,h) -> N(k2,f v2, map f l, map f r,h)

let rec mapi f = function
  | E -> E
  | N(k2,v2,l,r,h) -> N(k2,f k2 v2, mapi f l, mapi f r,h)

let rec fold f m x = 
  match m with 
  | E -> x
  | N(k2,v2,l,r,h) -> fold f r (f k2 v2 (fold f l x))

let partition1 f k v (acc1,acc2) = if f k v then (add k v acc1,acc2) else (acc1,add k v acc2) 
    
let rec partition_aux f s acc = 
    match s with 
    | E -> acc
    | N(k,v,l,r,_) -> 
        let acc = partition_aux f r acc in 
        let acc = partition1 f k v acc in
        partition_aux f l acc

let partition f s = partition_aux f s (empty,empty)

let filter1 f k v acc = if f k v then add k v acc else acc 
let rec filter_aux f s acc = 
    match s with 
    | E -> acc
    | N(k,v,l,r,_) -> filter1 f k v acc

let filter f s = filter_aux f s empty


(*ENDIF-OCAML*)

let tryfind_multi k map = match tryfind k map with Some res -> res | None -> []

