
// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 

#light

open System
open System.Reflection
open System.IO
open Microsoft.FSharp.Compatibility
open Printf

type search = 
    | ExactMatch 
    | NamespaceMatch 

let kind = ref ExactMatch
let interfaces = ref false
let methods = ref false
let basetypes = ref false
let modinfo = ref false
let includes = ref []
let deep = ref false
  
let findDLLs (dir:string) =
    if (Directory.Exists dir) then
        [ for file in Directory.GetFiles(dir, "*.dll") -> file ]
    else
        eprintf "Directory %s does not exist!\n" dir;
        []

let print_indented ind s = 
    for i = 1 to ind do printf " "; done;
    printfn "%s" s

let rec dumpType ind shallow (t:Type) =
    let desc = 
        if t.IsClass then "class"
        else if t.IsInterface then "interface" 
        else if t.IsValueType then "struct" 
        else if t.IsArray then "array" 
        else "??" 

    print_indented ind (desc^" "^t.FullName);
    
    if not shallow then 
        if !kind = ExactMatch || !modinfo then 
            print_indented (ind+2) ("Module: "^ t.Module.FullyQualifiedName );
            
        if !kind = ExactMatch || !interfaces then 
            for intf in t.GetInterfaces() do
                dumpType (ind+2) true intf;
                
    if !basetypes then 
        match t.BaseType with 
        | null -> ()
        | baseType -> dumpType 0 shallow baseType
     
let searchFile (pat:string) file = 
    // Load the module - expect to fail if it is not a .NET module 
    match (try Some (Assembly.LoadFrom(file)) with _ -> None) with
    | Some a ->
        eprintf "Searching Module %s\n" file;
        match a.GetModules() with 
        | null -> ()
        | modules ->
            let pat = pat.ToUpper() 
            for m in modules do 
                let types = 
                    try m.GetTypes() 
                    with _ -> eprintf "Types module %s failed to load\n" m.Name; [| |]
                    
                for t in types do 
                    let name = t.FullName.ToUpper() 
                    let show = 
                        match !kind with 
                        | ExactMatch -> name = pat
                        | NamespaceMatch -> t.Namespace <> null && t.Namespace.ToUpper() = pat
                    
                    if show then dumpType 0 false t
    | None -> ()

let searchDir pat dir = 
    for dll in findDLLs dir do
        searchFile pat dll

let search pat =
    
    let mscorlib = Assembly.Load("mscorlib.dll") 
    eprintf "Searching System Libraries";  
    
    let dirFrameworks = Path.GetDirectoryName(mscorlib.Location) 
    
    searchDir pat dirFrameworks;
    eprintf "Searching the current directory...";  
    searchDir pat ".";
    
    for dir in !includes do 
        eprintf "Searching directory %s\n" dir;      
        searchDir pat dir

let usage =
    [ "-x", Arg.Unit (fun _ -> kind := ExactMatch), "Exact Type Name (including namespace)";
      "-n",  Arg.Unit (fun _ ->  kind := NamespaceMatch), "Search for namespace";
      "-I",  Arg.String (fun s -> includes := !includes @ [s]), "Extra search directory";
      "-w",  Arg.Set deep, "Match the name anywhere the namespace";
      "-i",  Arg.Set interfaces, "Show interfaces";
      "-r",  Arg.Set basetypes, "Show base types";
      "-a",  Arg.Unit (fun () -> interfaces := true), "Show all";
      "-l",  Arg.Set modinfo, "Show module information"; ]

let _ = 
    Arg.parse usage search "findtype <options> name1 name2 ... "
