// (c) Microsoft Corporation 2005-2007.

#light

namespace Microsoft.FSharp.Core

open Microsoft.FSharp.Core
open Microsoft.FSharp.Core.Operators
open Microsoft.FSharp.Collections

/// Compatibility module for ML string processing.  Richer string operations
/// are available via the member functions on strings and other functionality in
///  <a href="http://msdn2.microsoft.com/en-us/library/system.string.aspx">System.String</a> 
/// and <a href="http://msdn2.microsoft.com/library/system.text.regularexpressions.aspx">System.Text.RegularExpressions</a> types.
[<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
module String = 

    /// Compare the given strings using ordinal comparison
    val compare: string -> string -> int

    /// Return a new string made by concatenating the given strings
    /// with separator 'sep', i.e. 'a1 + sep + ... + sep + aN'
    val concat: sep:string -> strings:string list -> string

    /// Returns the character at the specified position in the string
    val get: string -> int -> char

    /// Return a substring of length 'length' starting index 'start'.
    val sub: string -> start:int -> length:int -> string

    ///Apply the given function to each character in the string
    val iter: (char -> unit) -> string -> unit

    ///Build a new string whose characters are the results of applying the given function
    ///to each of the characters of the input string.
    val map: (char -> char) -> string -> string

    ///Build a new string whose characters are the results of applying the given function
    ///to each of the characters of the input string and concatenating the resulting
    ///strings.
    val map_concat: (char -> string) -> string -> string

    ///Test if all characters in the string satisfy the given predicate.
    val for_all: (char -> bool) -> string -> bool

    ///Test if any character of the string satisfies the given predicate.
    val exists: (char -> bool) -> string -> bool

    /// Return the length of the string.
    val length: string -> int

    /// Return a new string with all characters converted to lowercase
    val lowercase: string -> string

    /// Return a string of the given length containing repetitions of the given character
    val make: int -> char -> string

    /// Return s string of length 1 contianing the given character
    val of_char: char -> string

    /// Return true is the given string contains the given character
    val contains: string -> char -> bool

    /// Return true is the given string contains the given character in the
    /// range specified by the given start index and the given length
    val contains_between: string -> start:int -> length:int -> char -> bool

    /// Return true is the given string contains the given character in the
    /// range from the given start index to the end of the string.
    val contains_from: string -> int -> char -> bool

    /// Return the first index of the given character in the
    /// string.  Raise [[Not_found]]/[[KeyNotFoundException]] if
    /// the string does not contain the given character.
    val index: string -> char -> int

    /// Return the first index of the given character in the
    /// range from the given start position to the end of the string.  
    /// Raise [[Not_found]]/[[KeyNotFoundException]] if
    /// the string does not contain the given character.
    val index_from: string -> start:int -> char -> int

    /// Return true if the string contains the given character prior to the given index
    val rcontains_from: string -> start:int -> char -> bool

    /// Return the index of the first occurrence of the given character 
    /// from the end of the string proceeding backwards
    val rindex: string -> char -> int

    /// Return the index of the first occurrence of the given character 
    /// starting from the given index proceeding backwards.
    val rindex_from: string -> start:int -> char -> int

    /// Return a string with the first character converted to uppercase.
    val capitalize: string -> string

    /// Return a string with the first character converted to lowercase.
    val uncapitalize: string -> string

    /// Return a string with all characters converted to uppercase.
    val uppercase: string -> string

    #if CLI_AT_MOST_1_1
    #else
    /// Split the string using the given list of separator characters.
    /// Trimming is also performed at both ends of the string and any empty
    /// strings that result from the split are discarded.
    val split: char list -> (string -> string list)

    /// Removes all occurrences of a set of characters specified in a
    /// list from the beginning and end of this instance.
    val trim: char list -> (string -> string)
    #endif
