/*
	FATSort, utility for sorting FAT directory structures
	Copyright (C) 2004 Boris Leidner <fatsort(at)formenos.de>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

/*
	This file contains the main function of fatsort.
*/

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <assert.h>
#include <errno.h>
#include <locale.h>
#include <time.h>

// project includes
#include "endianness.h"
#include "signal.h"
#include "FAT_fs.h"
#include "options.h"
#include "errors.h"
#include "sort.h"
#include "clusterchain.h"
#include "misc.h"
#include "platform.h"
#include "mallocv.h"

// program information
#define INFO_PROGRAM "FATSort Utility"
#define INFO_VERSION "1.1.1"
#define INFO_AUTHOR "Boris Leidner <fatsort(at)formenos.de>"
#define INFO_DATE "2004-2013"
#define INFO_HEADER INFO_PROGRAM " " INFO_VERSION " by " INFO_AUTHOR ", " INFO_DATE "."
#define INFO_LICENSE	"License GPLv2: GNU GPL version 2 (see LICENSE.txt)\n" \
			"This is free software: you are free to change and redistribute it.\n" \
			"There is NO WARRANTY, to the extent permitted by law.\n"

#define INFO_USAGE "\n"	"Usage: fatsort [options] device\n" \
			"\n" \
			"Options:\n" \
			"\t-c\t Ignore case of file names\n" \
			"\t-f\t Force sorting even if file system is mounted\n" \
			"\t-h\t Print some help\n" \
			"\t-i\t Print file system information only\n" \
			"\t-I pfx\t Ignore file name prefix\n" \
			"\t-l\t Print current order of files only\n" \
			"\t-o flag\t Sort order of files where flag is one of\n" \
			"\t\t\td : directories first (default)\n" \
			"\t\t\tf : files first\n" \
			"\t\t\ta : files and directories are not differentiated\n" \
			"\t-n\t Natural order sorting\n" \
			"\t-q\t Be quiet\n" \
			"\t-r\t Sort in reverse order\n" \
			"\t-R\t Sort in random order\n" \
			"\t-t\t Sort by last modification date and time\n" \
			"\t-v\t Print version information\n" \
			"\n" \
			"\tThe following options can be specified multiple times:\n" \
			"\n" \
			"\t-d dir\t Sort directory dir only\n" \
			"\t-D dir\t Sort directory dir and all subdirectories\n" \
			"\t-x dir\t Don't sort directory dir\n" \
			"\t-X dir\t Don't sort directory dir and its subdirectories\n" \
			"\n" \
			"Device must be a FAT12, FAT16 or FAT32 file system.\n" \
			"\n" \
			"Example: fatsort /dev/sda\n" \
			"\n" \
			"NOTE: THE FILESYSTEM MUST BE CONSISTENT, OTHERWISE YOU MAY DAMAGE IT!\n" \
			"IF SOMEONE ELSE HAS ACCESS TO THE DEVICE HE MIGHT EXPLOIT FATSORT WITH\n" \
			"A FORGED CORRUPT FILESYSTEM! USE THIS PROGRAM AT YOUR OWN RISK!\n"

int32_t printFSInfo(char *filename) {
/*
	print file system information
*/

	assert(filename != NULL);

	u_int32_t value, clen;
	int32_t usedClusters, badClusters;
	int32_t i;
	struct sClusterChain *chain;

	struct sFileSystem fs;

	printf("\t- File system information -\n");

	if (openFileSystem(filename, FS_MODE_RO, &fs)) {
		myerror("Failed to open file system!");
		return -1;
	}

	usedClusters=0;
	badClusters=0;
	for (i=2; i<fs.clusters; i++) {
		getFATEntry(&fs, i, &value);
		if ((value & 0x0FFFFFFF) != 0) usedClusters++;
		if (fs.FATType == FATTYPE_FAT32) {
			if ((value & 0x0FFFFFFF) == 0x0FFFFFF7) badClusters++;
		} else if (fs.FATType == FATTYPE_FAT16) {
			if (value == 0x0000FFF7) badClusters++;
		} else if (fs.FATType == FATTYPE_FAT12) {
			if (value == 0x00000FF7) badClusters++;
		}
	}

	printf("Device:\t\t\t\t\t%s\n", fs.path);
	printf("Type:\t\t\t\t\tFAT%d\n", (int) fs.FATType);
	printf("Sector size:\t\t\t\t%d bytes\n", (int) fs.sectorSize);
	printf("FAT size:\t\t\t\t%d sectors (%d bytes)\n", (int) fs.FATSize, (int) (fs.FATSize * fs.sectorSize));
	printf("Number of FATs:\t\t\t\t%d %s\n", fs.bs.BS_NumFATs, (checkFATs(&fs) ? "- WARNING: FATs are different!" : ""));
	printf("Cluster size:\t\t\t\t%d bytes\n", (int) fs.clusterSize);
	printf("Max. cluster chain length:\t\t%d clusters\n", (int) fs.maxClusterChainLength);
	printf("Data clusters (total / used / bad):\t%d / %d / %d\n", (int) fs.clusters, (int) usedClusters, (int) badClusters);
	printf("FS size:\t\t\t\t%.2f MiBytes\n", (float) fs.FSSize / (1024.0*1024));
	if (fs.FATType == FATTYPE_FAT32) {
		if (getFATEntry(&fs, SwapInt32(fs.bs.FATxx.FAT32.BS_RootClus), &value) == -1) {
			myerror("Failed to get FAT entry!");
			closeFileSystem(&fs);
			return -1;
		}
		printf("FAT32 root first cluster:\t\t0x%x\nFirst cluster data offset:\t\t0x%lx\nFirst cluster FAT entry:\t\t0x%x\n",
			(unsigned int) SwapInt32(fs.bs.FATxx.FAT32.BS_RootClus),
			(unsigned long) getClusterOffset(&fs, SwapInt32(fs.bs.FATxx.FAT32.BS_RootClus)), (unsigned int) value);
	} else if (fs.FATType == FATTYPE_FAT12) {
		printf("FAT12 root directory Entries:\t\t%u\n", SwapInt16(fs.bs.BS_RootEntCnt));
	} else if (fs.FATType == FATTYPE_FAT16) {
		printf("FAT16 root directory Entries:\t\t%u\n", SwapInt16(fs.bs.BS_RootEntCnt));
	}

	if (OPT_MORE_INFO) {
		printf("\n\t- FAT -\n");
		printf("Cluster \tFAT entry\tChain length\n");
		for (i=0; i<fs.clusters; i++) {
			getFATEntry(&fs, i, &value);

			clen=0;
			if ((value & 0x0FFFFFFF ) != 0) {
				if ((chain=newClusterChain()) == NULL) {
					myerror("Failed to generate new ClusterChain!");
					return -1;
				}
				clen=getClusterChain(&fs, i, chain);
				freeClusterChain(chain);
			}

			printf("%08x\t%08x\t%u\n", i, value, clen);

		}

	}

	closeFileSystem(&fs);

	return 0;

}

int main(int argc, char *argv[]) {
/*
	parse arguments and options and start sorting
*/

	// initialize rng
	srand(time(0));

	// use locale from environment
	if (setlocale(LC_ALL, "") == NULL) {
		myerror("Could not set locale!");
		return -1;
	}

	// initialize blocked signals
	init_signal_handling();
	char *filename;

	if (parse_options(argc, argv) == -1) {
		myerror("Failed to parse options!");
		return -1;
	}

	if (OPT_HELP) {
		printf(INFO_HEADER "\n\n" INFO_LICENSE INFO_USAGE);
		return 0;
	} else if (OPT_VERSION) {
		printf(INFO_HEADER "\n\n" INFO_LICENSE);
		return 0;
	} else if (optind < argc -1) {
		myerror("Too many arguments!");
		myerror("Use -h for more help.");
		return -1;
	} else if (optind == argc) {
		myerror("Device must be given!");
		myerror("Use -h for more help.");
		return -1;
	}

	filename=argv[optind];

	if (OPT_INFO) {
		infomsg(INFO_HEADER "\n\n");
		if (printFSInfo(filename) == -1) {
			myerror("Failed to print file system information");
			return -1;
		}
	} else {
		infomsg(INFO_HEADER "\n\n");
		if (sortFileSystem(filename) == -1) {
			myerror("Failed to sort file system!");
			return -1;
		}
	}
	
	freeOptions();
	
	// report mallocv debugging information
	REPORT_MEMORY_LEAKS

	return 0;
}
