note
	description: "[
		Check whether branded product is correctly activated,
		decrement remaining executions counter otherwise.
		When counter reaches 0, product must be activated.
		]"
	legal: "See notice at end of class."
	status: "See notice at end of class."
	date: "$Date: 2021-02-19 10:48:41 +0000 (Fri, 19 Feb 2021) $"
	revision: "$Revision: 96850 $"

class
	BRANDED_LICENSE_CHECKER

inherit
	BRANDED_LICENSE_CONSTANTS

create
	make

feature {NONE} -- Initialization

	make (a_edition_name: READABLE_STRING_8; a_major_minor_version: READABLE_STRING_8; a_platform: READABLE_STRING_8; a_install_location, a_user_location: PATH)
		do
			edition_name := a_edition_name
			major_minor_version := a_major_minor_version
			platform := a_platform
			location := a_install_location
			user_location := a_user_location
			create on_license_validated_actions
		end

feature -- Access

	location, user_location: PATH

	edition_name: IMMUTABLE_STRING_8

	major_minor_version: IMMUTABLE_STRING_8
			-- Version MM.mm of current product.

	platform: IMMUTABLE_STRING_8
			-- Current platform.	

	can_run: BOOLEAN
			-- Are we authorized to run this product?

	has_license: BOOLEAN
		local
			fut: FILE_UTILITIES
		do
			Result := fut.file_path_exists (location) or else fut.file_path_exists (user_location)
		end

	is_licensed: BOOLEAN
			-- Is product fully licensed?

	is_expired: BOOLEAN
			-- Is license expired?

	is_invalid_version: BOOLEAN
			-- Is invalid for current MM.mm `major_minor_version` version?

	is_invalid_platform: BOOLEAN
			-- Is invalid for current `platform`?

	is_evaluating: BOOLEAN
			-- Is product in evaluation mode?

	on_license_validated_actions: ACTION_SEQUENCE [TUPLE [BRANDED_LICENSE]]
			-- Actions triggered when a license is accepted.

feature -- Basic Operations

	check_activation
			-- Check whether product can be started.
			-- Decrement remaining execution count
			-- if not activated.
			-- Set `can_run' to `True' if product can
			-- be started, `False' otherwise.
		local
			l_app: EV_APPLICATION
			win: like validator_window
		do
			check_license
			if is_evaluating or is_expired then
				create l_app
				create win.make (Current)
				validator_window := win
				win.set_actions (agent set_can_run, agent l_app.destroy)
				win.show
				l_app.launch
			elseif attached associated_license as lic then
				on_license_validated_actions.call ([lic])
			end
			launch_gc_cycle
		end

	check_activation_while_running (a_next_action: PROCEDURE)
			-- Check whether product can be started.
			-- Decrement remaining execution count
			-- if not activated.
			-- Set `can_run' to `True' if product can
			-- be started, `False' otherwise.
		local
			win: like validator_window
		do
			check
				app_started: attached (create {EV_ENVIRONMENT}).application as app and then not app.is_destroyed
			end
			check_license
			if is_evaluating or is_expired then
				create win.make (Current)
				validator_window := win
				win.set_actions (agent set_can_run, a_next_action)
				win.show
			else
				a_next_action.call
			end
			launch_gc_cycle
		end

	check_license
			-- Check license information and set `can_run', `is_licensed' and `is_evaluating'
			-- to their proper value?
		local
			lic: like license_at
		do
			can_run := False
			is_licensed := False
			is_evaluating := not is_licensed
			lic := license_at (location)
			if lic = Void then
				lic := license_at (user_location)
			end
			if lic /= Void then
				is_licensed := True
				is_evaluating := not is_licensed

				is_invalid_version := not lic.is_valid_for_version (major_minor_version)
				is_invalid_platform := not lic.is_valid_for_platform (platform)
				is_expired := lic.is_expired
				if is_expired or is_invalid_version or is_invalid_platform then
					can_run := False
				else
					can_run := True
					on_license_validated_actions.call ([lic])
				end
			end
		end

	is_valid_license_file (a_location: PATH): BOOLEAN
		do
			Result := attached license_at (a_location) as lic and then
				not lic.is_expired and then
				lic.is_valid_for_version (major_minor_version) and then
				lic.is_valid_for_platform (platform)
		end

	license_at (a_location: PATH): detachable BRANDED_LICENSE
		local
			retried: BOOLEAN
		do
			if not retried then
				create Result.make_with_path (a_location, secret)
				if not Result.is_valid then
					Result := Void
				end
			end
		rescue
			retried := True
			retry
		end

	associated_license: detachable BRANDED_LICENSE
		require
			is_licensed
		do
			Result := license_at (location)
			if Result = Void then
				Result := license_at (user_location)
			end
		end

feature {NONE} -- Implementation

	validator_window: detachable BRANDED_LICENSE_VALIDATOR_WINDOW
			-- Reference to window to avoid being garbage collected.

	report_engine_not_initialized
			-- Action to be performed when engine is not initialized.
		local
			l_warning: EV_WARNING_DIALOG
			l_app: EV_APPLICATION
			l_screen: EV_SCREEN
			l_accelerator: EV_ACCELERATOR
			l_button: EV_BUTTON
		do
			l_app := (create {EV_ENVIRONMENT}).application
			if l_app = Void then
				create l_app
			end
			create l_warning.make_with_text (not_initialized_error_message)
			create l_screen
			l_warning.set_position ((l_screen.width - l_warning.width) // 2,
				(l_screen.height - l_warning.height) // 3)

			l_warning.set_buttons_and_actions (<<"Quit">>, <<agent die>>)
			create l_accelerator
			l_accelerator.set_key (create {EV_KEY}.make_with_code ({EV_KEY_CONSTANTS}.Key_enter))
			l_accelerator.actions.extend (agent die)
			l_warning.accelerators.extend (l_accelerator)
			l_button := l_warning.button ("Quit")
			l_warning.set_default_push_button (l_button)
			l_warning.set_default_cancel_button (l_button)
			l_warning.show
			l_app.launch
		end

	not_initialized_error_message: STRING =
			-- Standard message when product has not been installed correctly with license information.
"[
The installation of this product has been corrupted.
Please run the installation program again.

If this problem still persists, please send a problem
report to Eiffel Software support through
http://support.eiffel.com.
]"

	launch_gc_cycle
			-- Launch a GC cycle to move objects around that way it
			-- is harder to hack the value of `is_licensed'.
		local
			mem: MEMORY
		do
			create mem
			mem.collect
			mem.full_collect
			mem.full_coalesce
		end

	set_can_run
			-- Set `can_run' with `l_can_run'.
		do
			-- TODO: maybe allow a few execution, or trial period.
			-- But for branded license, this is not standard or enterprise, so no need for trial.
			can_run := is_licensed
			is_evaluating := not is_licensed
		ensure
			can_run_set: can_run
		end

	die
			-- Kill current application.
		local
			l_exception: EXCEPTIONS
		do
			create l_exception
			l_exception.die (-1)
		end

note
	copyright:	"Copyright (c) 1984-2006, Eiffel Software"
	license:	"GPL version 2 see http://www.eiffel.com/licensing/gpl.txt)"
	licensing_options:	"http://www.eiffel.com/licensing"
	copying: "[
			This file is part of Eiffel Software's Eiffel Development Environment.
			
			Eiffel Software's Eiffel Development Environment is free
			software; you can redistribute it and/or modify it under
			the terms of the GNU General Public License as published
			by the Free Software Foundation, version 2 of the License
			(available at the URL listed under "license" above).
			
			Eiffel Software's Eiffel Development Environment is
			distributed in the hope that it will be useful,	but
			WITHOUT ANY WARRANTY; without even the implied warranty
			of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
			See the	GNU General Public License for more details.
			
			You should have received a copy of the GNU General Public
			License along with Eiffel Software's Eiffel Development
			Environment; if not, write to the Free Software Foundation,
			Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
		]"
	source: "[
			 Eiffel Software
			 356 Storke Road, Goleta, CA 93117 USA
			 Telephone 805-685-1006, Fax 805-685-6869
			 Website http://www.eiffel.com
			 Customer support http://support.eiffel.com
		]"

end -- class BRANDED_LICENSE_CHECKER
