note
	description: "Summary description for {BRANDED_LICENSE}."
	date: "$Date: 2020-07-13 23:05:58 +0000 (Mon, 13 Jul 2020) $"
	revision: "$Revision: 96836 $"

class
	BRANDED_LICENSE

create
	make_empty,
	make_with_path

create {BRANDED_LICENSE_GENERATOR}
	make

feature {NONE} -- Initialization

	make_empty
		do
			create edition_name.make_from_string ({STRING_32} "custom")
			create start_date.make_now_utc
			create end_date.make_now_utc
			license_key := "FAKE-BRANDED-LICENSE"
			is_valid := False
		end

	make_with_path (a_file_location: PATH; a_secret_key: READABLE_STRING_8)
		local
			f: RAW_FILE
			tok: STRING
			l_jwt_loader: JWT_LOADER
			s: READABLE_STRING_GENERAL
		do
			create f.make_with_path (a_file_location)
			if f.exists and then f.is_access_readable then
				create tok.make (f.count)
				f.open_read
				from
				until
					f.exhausted or f.end_of_file
				loop
					f.read_stream_thread_aware (1024)
					tok.append (f.last_string)
				end
				f.close

				create l_jwt_loader
				if
					attached l_jwt_loader.token (tok, {JWT_ALG_HS256}.name, a_secret_key, Void) as jwt and then
					not jwt.has_error and then
					attached jwt.claimset as l_claimset and then
					attached iso_date_time (l_claimset.string_32_claim ("start-date")) as l_start_dt and then
					attached iso_date_time (l_claimset.string_32_claim ("end-date")) as l_end_dt and then
					attached l_claimset.string_32_claim ("license-key") as l_lic_key and then
					attached l_claimset.string_32_claim ("edition-name") as l_edition
				then
					is_valid := True
					edition_name := l_edition
					start_date := l_start_dt
					end_date  := l_end_dt
					license_key := l_lic_key
					s := l_claimset.string_32_claim ("edition-title")
					if s /= Void then
						edition_title := s
					else
						edition_title := Void
					end
					s := l_claimset.string_32_claim ("terms")
					if s /= Void then
						terms_of_use := s
					else
						terms_of_use := Void
					end
					s := l_claimset.string_32_claim ("version")
					if s /= Void then
						version := s
					else
						version := Void
					end
					s := l_claimset.string_32_claim ("platform")
					if s /= Void then
						platform := s
					else
						platform := Void
					end
				else
					make_empty
					is_valid := False
				end
			else
				make_empty
				is_valid := False
			end
		end

	make (a_edition_name: READABLE_STRING_GENERAL; a_license_key: READABLE_STRING_GENERAL; a_start_date, a_end_date: DATE_TIME)
		do
			edition_name := a_edition_name
			start_date := a_start_date
			end_date := a_end_date
			license_key := a_license_key
			is_valid := True
		end

feature -- Conversion

	to_string (a_secret_key: detachable READABLE_STRING_8): STRING
		local
			jws: JWS
			tb: STRING_TABLE [READABLE_STRING_GENERAL]
			dt: DATE_TIME
			f: RAW_FILE
		do
			create tb.make (6)
			tb["edition-name"] := edition_name
			tb["license-key"] := license_key
			tb["start-date"] := date_time_to_iso (start_date)
			tb["end-date"] := date_time_to_iso (end_date)

			if attached terms_of_use as l_terms then
				tb["terms"] := l_terms
			end
			if attached edition_title as l_edition_title then
				tb["edition-title"] := l_edition_title
			end
			if attached version as l_version then
				tb["version"] := l_version
			end
			if attached platform as l_platform then
				tb["platform"] := l_platform
			end

			create jws.make_with_claims (tb)
			jws.set_algorithm_to_hs256

			if a_secret_key = Void then
				Result := jws.claimset.string
			else
				jws.set_algorithm_to_hs256
				Result := jws.encoded_string (a_secret_key)
			end
		end

	save_to (a_location: PATH; a_secret_key: READABLE_STRING_8)
		local
			f: RAW_FILE
		do
			create f.make_with_path (a_location)
			f.create_read_write
			f.put_string (to_string (a_secret_key))
			f.close
		end

feature -- Access

	license_key: IMMUTABLE_STRING_32

	edition_name: IMMUTABLE_STRING_32

	start_date, end_date: DATE_TIME
			-- UTC date time.	

feature -- Access: optional

	terms_of_use: detachable READABLE_STRING_32

	edition_title: detachable IMMUTABLE_STRING_32

	version: detachable IMMUTABLE_STRING_32
			-- Limited to the `version` MM.mm of the product.

	platform: detachable IMMUTABLE_STRING_32
		-- Limited to the `platform` value.

feature -- Query

	edition_title_or_name: IMMUTABLE_STRING_32
		do
			if attached edition_title as l_title then
				Result := l_title
			else
				Result := edition_name
			end
		end

feature -- Element change

	set_edition_name (a_name: READABLE_STRING_GENERAL)
		do
			edition_name := a_name
		end

	set_edition_title (a_title: detachable READABLE_STRING_GENERAL)
		do
			if a_title = Void then
				edition_title := Void
			else
				edition_title := a_title
			end
		end

	set_dates (a_start_date, a_end_date: DATE_TIME)
		do
			start_date := a_start_date
			end_date := a_end_date
		end

	set_terms_of_use (a_txt: detachable READABLE_STRING_GENERAL)
		do
			if a_txt = Void then
				terms_of_use := Void
			else
				terms_of_use := a_txt
			end
		end

	set_version (v: detachable READABLE_STRING_GENERAL)
		do
			if v = Void then
				version := Void
			else
				version := v
			end
		end

	set_platform (pf: detachable READABLE_STRING_GENERAL)
		do
			if pf = Void then
				platform := Void
			else
				platform := pf
			end
		end

feature -- Status

	is_valid: BOOLEAN

	is_expired: BOOLEAN
		local
			l_now: DATE_TIME
		do
			create l_now.make_now_utc
			Result := l_now > end_date
		end

	is_valid_for_version (v: READABLE_STRING_GENERAL): BOOLEAN
		do
			if attached version as l_lic_version and then not l_lic_version.is_whitespace then
				Result := l_lic_version.is_case_insensitive_equal_general (v)
			else
				Result := True
			end
		end

	is_valid_for_platform (pf: READABLE_STRING_GENERAL): BOOLEAN
		do
			if attached platform as l_platform and then not l_platform.is_whitespace then
				Result := l_platform.is_case_insensitive_equal_general (pf)
			else
				Result := True
			end
		end

feature -- Conversion

	iso_date_time (s: detachable READABLE_STRING_GENERAL): detachable DATE_TIME
			-- 2020-03-19T23:00:44
		local
			y,mo,d,h,mi,sec: INTEGER
			i, j, pos: INTEGER
			err: BOOLEAN
		do
			if s /= Void then
				pos := s.index_of ('T', 1)
				if pos > 0 then
					i := 1
					j := s.index_of ('-', i)
					if j > i and j < pos then
						y := s.substring (i, j - 1).to_integer
						i := j + 1
						j := s.index_of ('-', i)
						if j > i and j < pos then
							mo := s.substring (i, j - 1).to_integer
							d := s.substring (j + 1, pos - 1).to_integer
						else
							err := True
						end
					else
						err := True
					end
					i := pos + 1
					j := s.index_of (':', i)
					if j > i then
						h := s.substring (i, j - 1).to_integer
						i := j + 1
						j := s.index_of (':', i)
						if j > i then
							mi := s.substring (i, j - 1).to_integer
							sec := s.substring (j + 1, s.count).to_integer
						else
							err := True
						end
					else
						err := True
					end
					if not err then
						create Result.make (y, mo, d, h, mi, sec)
					end
				end
			end
		end

	date_time_to_iso (dt: DATE_TIME): STRING
		do
			Result := dt.formatted_out ("yyyy-[0]mm-[0]dd [0]hh:[0]mi:[0]ss")
			Result.replace_substring_all (" ", "T")
		end

end
