#include <set>
#include <cmath>
#include <vector>
#include <fstream>
#include <sstream>
#include <cstdlib>
#include <iostream>
#include "sentence.hpp"

namespace maeda {


namespace {
static Row ROOT = Row("ROOT", "ROOT", "ROOT", "ROOT",
                      "", Sentence::kInvalidIndex, "", "0", "");
} // namespace

bool ReadFromEdaDependencyStream(std::istream &in, Sentence *sentence) {
  std::string line;

  while (std::getline(in, line) && line.empty())
    ;

  if (!in) {
    return false;
  }

  // ID=ʹԤȽꡢ⤷ʤskip
  size_t found = line.find("ID=");
  if (found == std::string::npos) {
    std::cerr << "Error: wrong format near " << line << std::endl;
    exit(1);
  } else {
    sentence->id = line.substr(3, line.length() - 3);
    std::getline(in, line);
  }

  sentence->rows.push_back(ROOT);

  do {
    if (line.empty()) {
      break;
    } else {
      std::istringstream ss(line);
      int id, head;
      std::string form, postag, netag, cluster("0");
      ss >> id >> head >> form >> postag >> cluster >> netag;
      //id ϼΤƤ
      sentence->rows.push_back(
          Row(form, form, postag, postag, "_", head, "L", cluster, netag));
    }
  } while (std::getline(in, line));

  return true;
}


bool ReadFromCoNLLDependencyStream(std::istream &in, Sentence *sentence) {
  std::string line;

  while (std::getline(in, line) && line.empty())
    ;

  if (!in) { return false; }

  sentence->rows.push_back(ROOT);

  do {
    if (line.empty()) {
      break;
    } else {
      std::istringstream ss(line);
      int id;
      std::string head, form, lemma, cpostag, postag, feats, deprel;
      ss >> id >> form >> lemma >> cpostag >> postag;
      ss >> feats >> head >> deprel;
      //id ϼΤƤ
      int h = 0;
      if (head == "_" || head == "NULL") {
        h = -1;
      } else {
        h = atoi(head.c_str());
      }

      sentence->rows.push_back(
          Row(form, lemma, cpostag, postag, feats, h, deprel, "0", ""));
    }
  } while (std::getline(in, line));

  return true;
}


void WriteToEdaDependencyStream(std::ostream &out, const Sentence &sentence) {
  out << "ID=" << sentence.id << std::endl;
  int id = 1;
  for (std::vector<Row>::const_iterator it = sentence.rows.begin() + 1;
       it != sentence.rows.end(); ++it) {
    out << id++
        << " " << it->head
        << " " << it->form
        << " " << it->postag
        << " " << it->cluster;
    if (it->netag != "") { out << " " << it->netag; }
    out << std::endl;
  }
  out << std::endl;
}


void WriteToCoNLLDependencyStream(std::ostream &out,
                                  const Sentence &sentence) {
  int id = 1;
  for (std::vector<Row>::const_iterator it = sentence.rows.begin() + 1;
       it != sentence.rows.end(); ++it) {
    out << id++
        << " " << it->form
        << " " << it->lemma
        << " " << it->cpostag
        << " " << it->postag
        << " " << it->feats
        << " " << it->head
        << " " << it->deprel
        << std::endl;
  }
  out << std::endl;
}


bool ReadFromKyteaOutputStream(std::istream &in, Sentence *sentence) {
  std::string line;
  if (std::getline(in, line)) {
    sentence->rows.push_back(ROOT);
    std::string unit;
    std::stringstream ss(line);
    while (ss >> unit) {
      std::string form, postag;
      std::replace(unit.begin(), unit.end(), '/', ' ');
      std::stringstream tokenizer(unit);
      tokenizer >> form >> postag;
      sentence->rows.push_back(Row(form, form,
                                   postag, postag, "_", 0, "_", "0", ""));
    }
    return true;
  } else {
    return false;
  }
}


bool GetExcludedIndicesByNE(const Sentence &sentence,
                            std::vector<size_t> *indices) {
  // ߤtagindexȽ
  // ʸΰֺǸindexpushʤ (ܸhead final)
  for (size_t i = 1; i < sentence.rows.size(); i++) {
    const std::string &prev_netag = sentence.rows[i-1].netag;
    const std::string &curr_netag = sentence.rows[i].netag;
    const char prev_state = prev_netag[prev_netag.size() - 1];
    const char curr_state = curr_netag[curr_netag.size() - 1];
    switch (curr_state) {
    case 'B':
    case 'O':
      // .. B B .. do nothing
      // .. I B .. do nothing
      // .. O B .. do nothing

      // .. B O .. do nothing
      // .. I O .. do nothing
      // .. O O .. do nothing
      break;
    case 'I':
      // .. B I .. push
      // .. I I .. push
      // .. O I .. error!
      if (prev_state == 'B' || prev_state == 'I') {
        indices->push_back(i-1);
      } else { // prev_state == 'O'
        return false;
      }
      break;
    default:
      return false;
    }
  }
  return true;
}



} //maeda

