!>=========================================================================
!!
!! Module:
!!
!! (1) hdf5_io_m     Originally by JIM     Last Modified 12/2014 (FHJ)
!!
!!     Routines to read and write wavefunctions in HDF5 format.
!!     The code is generated through repeated inclusion of a file with
!!     different preprocessor definitions each time. Consult the resulting
!!     .p.f file for clarity.
!!
!!=========================================================================

#include "f_defs.h"

module hdf5_io_m
  use global_m
#ifdef HDF5
  use h5lt
  use hdf5
  
  implicit none
  
  private
  
  public :: &
    hdf5_require_version        , &
    hdf5_require_flavor         , &
    hdf5_create_dset            , &
    hdf5_create_group           , &
    hdf5_read_int               , &
    hdf5_write_int              , &
    hdf5_read_int_array         , &
    hdf5_write_int_array        , &
    hdf5_read_int_hyperslab     , &
    hdf5_write_int_hyperslab    , &
    hdf5_read_double            , &
    hdf5_write_double           , &
    hdf5_read_double_array      , &
    hdf5_write_double_array     , &
    hdf5_read_double_hyperslab  , &
    hdf5_write_double_hyperslab , &
    hdf5_read_logical           , &
    hdf5_write_logical          , &
    hdf5_read_logical_array     , &
    hdf5_write_logical_array    , &
    hdf5_read_logical_hyperslab , &
    hdf5_write_logical_hyperslab

contains


!> Make file a particular hdf5 file has the correct version number
subroutine hdf5_require_version(loc_id, dset_name, req_version, fname, allow_greater)
  integer(HID_T), intent(in) :: loc_id !< HDF5 file id
  character(LEN=*), intent(in) :: dset_name !< HDF5 dataset name
  integer, intent(in) :: req_version !< version to require
  character(LEN=*), intent(in) :: fname !< file name, for debugging purposes
  !> allow the file to have a version greater than req_version? Defaults to .false.
  logical, intent(in), optional :: allow_greater
  
  integer :: file_version, errcode
  logical :: allow_greater_

  PUSH_SUB(hdf5_require_version)

  file_version = -1
  call hdf5_read_int(loc_id, dset_name, file_version, errcode)
  allow_greater_ = .false.
  if (present(allow_greater)) allow_greater_ = allow_greater
  if (file_version<req_version .or. &
    (file_version/=req_version.and..not.allow_greater_) .or. errcode/=0) then
    if (peinf%inode==0) then
      write(0,*)
      write(0,*) 'ERROR: Incorrect version in file ', trim(fname),' while reading ',trim(dset_name)
      write(0,*) '       Expecting: ', req_version
      write(0,*) '       Got: ', file_version
      write(0,*) '       Errcode: ', errcode
      write(0,*) 'Your file was probably generated by an older version of BerkeleyGW and'
      write(0,*) 'is now obsolete. Consult the documentation and use the appropriate converter.'
      write(0,*)
    endif
    call die("Wrong version for file '"+trim(fname)+"'.", only_root_writes=.true.)
  endif

  POP_SUB(hdf5_require_version)

end subroutine hdf5_require_version


!> Make file a particular hdf5 file has the correct flavor number
subroutine hdf5_require_flavor(loc_id, dset_name, req_flavor, fname)
  integer(HID_T), intent(in) :: loc_id !< HDF5 file id
  character(LEN=*), intent(in) :: dset_name !< HDF5 dataset name
  integer, intent(in) :: req_flavor !< flavor to require
  character(LEN=*), intent(in) :: fname !< file name, for debugging purposes
  
  integer :: file_flavor, errcode

  PUSH_SUB(hdf5_require_flavor)

  file_flavor = -1
  call hdf5_read_int(loc_id, dset_name, file_flavor, errcode)
  if (file_flavor/=req_flavor.or.errcode/=0) then
    if (peinf%inode==0) then
      write(0,*)
      write(0,*) 'ERROR: Incorrect flavor in file ', trim(fname), ' while reading ',trim(dset_name)
      write(0,*) '       Expecting: ', req_flavor
      write(0,*) '       Got: ', file_flavor
      write(0,*) '       Errcode: ', errcode
      write(0,*) 'You are probably linking the wrong file or running the BerkeleyGW binary with'
      write(0,*) 'the wrong flavor.'
      write(0,*)
    endif
    call die("Wrong flavor in file "+trim(fname)+"'.", only_root_writes=.true.)
  endif

  POP_SUB(hdf5_require_flavor)

end subroutine hdf5_require_flavor


!> Creates an empty dataset
subroutine hdf5_create_dset(loc_id, dset_name, dtype, dims, errcode)
  integer(HID_T), intent(in) :: loc_id !< HDF5 file id
  character(LEN=*), intent(in) :: dset_name !< HDF5 dataset name
  integer(HID_T), intent(in) :: dtype
  integer, intent(in) :: dims(:)
  integer, intent(out) :: errcode
  
  integer(HSIZE_T) :: hdims(size(dims))
  integer(HID_T) :: dset_id
  integer(HID_T) :: dspace

  PUSH_SUB(hdf5_create_dset)

  hdims(:) = dims(:)
  call h5screate_simple_f(size(dims), hdims, dspace, errcode)
  call h5dcreate_f(loc_id, dset_name, dtype, dspace, dset_id, errcode)
  call h5dclose_f(dset_id, errcode)
  call h5sclose_f(dspace, errcode)

  POP_SUB(hdf5_create_dset)

end subroutine hdf5_create_dset


!> Creates an empty group
subroutine hdf5_create_group(loc_id, group_name, errcode)
  integer(HID_T), intent(in) :: loc_id !< HDF5 file id
  character(LEN=*), intent(in) :: group_name !< HDF5 group name
  integer, intent(out) :: errcode
  
  integer(HID_T) :: group_id

  PUSH_SUB(hdf5_create_group)

  call h5gcreate_f(loc_id, group_name, group_id, errcode)
  call h5gclose_f(group_id, errcode)
  
  POP_SUB(hdf5_create_group)

end subroutine hdf5_create_group

#define TYPE_INT
#define READ
#include "hdf5_io_inc.f90"
#undef READ
#include "hdf5_io_inc.f90"
#undef TYPE_INT

#define TYPE_DOUBLE
#define READ
#include "hdf5_io_inc.f90"
#undef READ
#include "hdf5_io_inc.f90"
#undef TYPE_DOUBLE

#define TYPE_LOGICAL
#define READ
#include "hdf5_io_inc.f90"
#undef READ
#include "hdf5_io_inc.f90"
#undef TYPE_LOGICAL

#endif

end module hdf5_io_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
