/*
   Copyright (C) 2004-2006,2010 Benjamin Redelings

This file is part of BAli-Phy.

BAli-Phy is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

BAli-Phy is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with BAli-Phy; see the file COPYING.  If not see
<http://www.gnu.org/licenses/>.  */

///
/// \file hmm.H
///
/// \brief This file defines the generic HMM class.
///

#ifndef HMM_H
#define HMM_H

#include "mytypes.H"
#include "logsum.H"

// Can I make Q into a general function, backed by a matrix in some cases?

/// A Hidden Markov Model - represented by the transition matrix and accessory functions

class HMM {
  /// Is each state part of the silent network?
  std::vector<int> silent_network_;

  /// Ordered list of states that are part of silent networks
  std::vector<int> silent_network_states;

  /// Ordered list of states that are not part of silent networks
  std::vector<int> non_silent_network;

  /// An ordered list of states, for DP
  std::vector<int> order_;

  efloat_t generalize_P_one(std::vector<int>::const_iterator,int) const;

  void find_and_index_silent_network_states();

public:
  /// The 'temperature' parameter - e.g. scale log(P) => log(P)/T
  double B;

  /// The transition matrix for the HMM
  Matrix Q;

  /// The 'generalized' transition matrix - silent loops removed
  Matrix GQ;

  void update_GQ();

  bool connected_Q(int S1,int S2) const {return Q(S1,S2) != 0.0;}

  bool connected(int S1,int S2) const {return GQ(S1,S2) != 0.0;}

  /// Probabilities of starting in each state
  std::vector<double> start_P;

  /// bitmasks of whether states emit features in each sequence
  std::vector<int> state_emit;

  /// Is state S a silent state?
  bool silent(int S) const {return not state_emit[S];}

  /// Is state S part of a loop of silent states?  FIXME: s/network/cycle/...
  bool silent_network(int S) const;

  /// What is the n-th state, in the correct order for DP?
  int order(int i) const {assert(i <= nstates()+1);  return order_[i];}

  /// Number of states, minus the end state
  int nstates() const {return state_emit.size()-1;}

  /// The index of the end state
  int endstate() const {return nstates();}

  /// Convert a path to a generalized path
  std::vector<int> generalize(const std::vector<int>& path) const;

  /// Randomly sample a path from a generalized path g_path
  std::vector<int> ungeneralize(const std::vector<int>& g_path) const;

  /// Calculates the (log) probability that path would be generated by its generalized dual
  efloat_t generalize_P(const std::vector<int>& path) const;

  /// Calculates the (log) product of each move, according to the generalized HMM
  efloat_t path_Q_path(const std::vector<int>& path) const;

  /// Calculates the (log) product of each move, according to the generalized HMM
  efloat_t path_GQ_path(const std::vector<int>& g_path) const;

  /// Construct an HMM from the emission characteristics of the states, the start probabilities, and the transition matrix
  HMM(const std::vector<int>&,const std::vector<double>&,const Matrix&,double);

  virtual ~HMM() {}
};

#endif
