/*
 * Electric(tm) VLSI Design System
 *
 * File: iopadso.c
 * Input/output aid: PADS output
 * Written by: Neil Levine, NAL Consulting for Intel Corp.
 *
 * Copyright (c) 2001 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

/*
 * Modified 8/21/00 NL so that electric does not crash when a net has no name.
 * Modified 10/9/00 NL to not print out netname more than once for multiple parts for EAGLE netlist.
 * Modified 12/1/00 NL to work with release 6.02.
 */

/*
 * ----------------- PADS NETLIST FORMAT ----------------------
        PADS-POWERPCB-V2

        *PART*
        LED1 LED@LED
        R1 RES100@RES100
        
        *NET*
        *SIGNAL* A0 12,00 0 0 0 -2
        LED1.1 R1.2
        *SIGNAL* A1 12.00 0 0 0 -2
        LED1.2 R1.1
        
        *END*
*/


#include "config.h"
#if IOPADS

#include "global.h"
#include "eio.h"

#define NOPADSNET ((PADSNET *)-1)

typedef struct Inextpadsnet
{
	char *netname;
	char *packagename;
	char *pinname;
	struct Inextpadsnet *nextpadsnet;
} PADSNET;

PADSNET *io_padsfirstnet;

/* prototypes for local routines */
void io_padsdumpfacet(NODEPROTO *curfacet, char *prefix);
void io_padsmarkfacets(NODEPROTO *curfacet);

INTSML io_writepadslibrary(LIBRARY *lib)
{
#define MAX_SIGNAL_LINE_LENGTH  75

	char file[100], *truename;
	REGISTER INTBIG sorted, SignalLineLength;
	REGISTER char *name;
	REGISTER NODEPROTO *curfacet;
	REGISTER PADSNET *en, *lasten, *nexten;
	REGISTER PORTPROTO *pp;

	/* create the proper disk file for the PADS netlist */
	curfacet = lib->curnodeproto;
	if (curfacet == NONODEPROTO)
	{
		ttyputerr("Must be editing a facet to generate PADS output");
		return(1);
	}

	(void)strcpy(file, curfacet->cell->cellname);
	(void)strcat(file, ".asc");
	name = truepath(file);
	io_fileout = xcreate(name, io_filetypepads, "PADS File", &truename);
	if (io_fileout == NULL)
	{
		if (truename != 0) ttyputerr("Cannot write %s", truename);
		return(1);
	}

	xprintf(io_fileout, "!PADS-POWERPCB-V2\n\n*CLUSTER* ITEM\n\n\n*PART*\n");

	/* recurse through the circuit, gathering network connections */
	io_padsfirstnet = NOPADSNET;
	for(pp = curfacet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		pp->temp1 = 0;

	io_padsdumpfacet(curfacet, "");  /* crashes in here somewhere if not net name NL */

	xprintf(io_fileout, "\n*NET*\n");

	/* sort the signals */
	sorted = 0;
	while (sorted == 0)
	{
		sorted = 1;
		lasten = NOPADSNET;
		for(en = io_padsfirstnet; en != NOPADSNET; en = en->nextpadsnet)
		{
			nexten = en->nextpadsnet;
			if (nexten == NOPADSNET) break;
			if (namesame(en->netname, nexten->netname) > 0)
			{
				if (lasten == NOPADSNET) io_padsfirstnet = nexten; else
					lasten->nextpadsnet = nexten;
				en->nextpadsnet = nexten->nextpadsnet;
				nexten->nextpadsnet = en;
				en = nexten;
				sorted = 0;
			}
			lasten = en;
		}
	}

	/* write networks */
	lasten = NOPADSNET;
	for(en = io_padsfirstnet; en != NOPADSNET; en = en->nextpadsnet)
	{       
		if (lasten == NOPADSNET || namesame(lasten->netname, en->netname) != 0)
		{
			if (lasten != NOPADSNET) xprintf(io_fileout, "\n");
			xprintf(io_fileout, "*SIGNAL* %s 12.00 0 0 0 -2\n", en->netname);
			SignalLineLength = 0;			/* used to track length of line for multi pin nets */
		}

		/* write "Part" */
		xprintf(io_fileout, "%s.", en->packagename);
		SignalLineLength += strlen(en->packagename) + 1;

		/* write "Pad" (an attribute with a number) */
		xprintf(io_fileout, "%s ", en->pinname);
		SignalLineLength += strlen(en->pinname) + 1;

		if(SignalLineLength > MAX_SIGNAL_LINE_LENGTH)           /* start next line */
		{
			xprintf(io_fileout, "\n");
			SignalLineLength = 0;
		}
		lasten = en;
	}

	if (lasten != NOPADSNET) xprintf(io_fileout, "\n\n*END*\n");

	/* free memory */
	while (io_padsfirstnet != NOPADSNET)
	{
		en = io_padsfirstnet;
		io_padsfirstnet = en->nextpadsnet;
		efree(en->netname);
		efree(en->packagename);
		efree(en->pinname);
		efree((char *)en);
	}

	/* clean up */
	xclose(io_fileout);

	/* tell the user that the file is written */
	ttyputmsg(_("%s written"), truename);
	return(0);
}

/*
 * Routine to write out all facets in the current library that are marked (temp1 != 0).
 */
void io_padsdumpfacet(NODEPROTO *curfacet, char *prefix)
{
	REGISTER char *name;
	char line[50], *newprefix, *tempname;
	REGISTER NODEPROTO *np, *subnp, *subcnp;
	REGISTER NETWORK *net;
	REGISTER NODEINST *ni;
	REGISTER VARIABLE *varpart, *var, *varpackagetype;
	REGISTER PORTARCINST *pi;
	REGISTER PORTPROTO *pp;
	REGISTER PADSNET *en;
	char CurRefDes[128], CurPackage[128];
	char LastRefDes[128], LastPackage[128];
	int RefDesSame, PackageSame;

	CurRefDes[0] = 0;			/* Current reference designator string empty */                                                         
	CurPackage[0] = 0;			/* Current package string empty */
	LastRefDes[0] = 0;			/* Last reference designator string empty */
	LastPackage[0] = 0;			/* Last  package string empty */

	/* mark those facets that will be included in this dump */
	io_padsmarkfacets(curfacet);

	/* write the symbols */
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		if (np->temp1 == 0) continue;

		/* if this is the main entry facet, pickup net names */
		for(net = np->firstnetwork; net != NONETWORK; net = net->nextnetwork)
			net->temp1 = 0;
		if (np == curfacet)
		{
			for(pp = curfacet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			{
				if (pp->temp1 == 0) continue;
				pp->network->temp1 = pp->temp1;
			}
		}

		/* dump symbols in this facet */
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			/* see if this node has a part name */
			if (ni->proto->primindex != 0) continue;
			varpart = getval((INTBIG)ni, VNODEINST, -1, "ATTR_PART");	/* cm - changed case on attribute */
			if (varpart == NOVARIABLE)
			{
				subnp = ni->proto;
				subcnp = contentsview(subnp);
				if (subcnp != NONODEPROTO) subnp = subcnp;

				/* compute new prefix for nets in the subfacet */
				(void)initinfstr();
				(void)addstringtoinfstr(prefix);
				var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name);
				if (var == NOVARIABLE)
				{
					sprintf(line, "NODE%ld.", (INTBIG)ni);
					(void)addstringtoinfstr(line);
				} else
				{
					(void)addstringtoinfstr((char *)var->addr);
					(void)addtoinfstr('.');
				}
				allocstring(&newprefix, returninfstr(), el_tempcluster);

				/* store net names on ports for use inside the subfacet */
				for(pp = subnp->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					pp->temp1 = 0;
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				{
					pp = equivalentport(ni->parent, pi->proto, subnp);
					if (pp == NOPORTPROTO || pp->temp1 != 0) continue;
					net = pi->conarcinst->network;
					(void)initinfstr();
					(void)addstringtoinfstr(prefix);
					(void)addstringtoinfstr(net->netname);
					allocstring(&tempname, returninfstr(), io_aid->cluster);
					pp->temp1 = (INTBIG)tempname;
				}

				/* recurse */
				io_padsdumpfacet(subnp, newprefix);

				/* remark the facets at the current level and clean up memory */
				io_padsmarkfacets(curfacet);
				efree(newprefix);
				for(pp = subnp->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					if (pp->temp1 != 0) efree((char *)pp->temp1);
			} else
			{
				/* found package at bottom of hierarchy */
				varpackagetype = getval((INTBIG)ni, VNODEINST, -1, "ATTR_pkg_type");

				sprintf(CurRefDes, "%s%s", prefix, describevariable(varpart, -1, -1));	/* format current ref des to string */
				if (varpackagetype == NOVARIABLE)
				{
					sprintf(CurPackage, "%s", ni->proto->cell->cellname);		/* current package to string */
				} else
				{
					sprintf(CurPackage, "%s", describevariable(varpackagetype, -1, -1));	/* current package to string */
				}
				RefDesSame = (strcmp(CurRefDes, LastRefDes) == 0);				/* convert to booleans */
				PackageSame = (strcmp(CurPackage, LastPackage) == 0);

				if(RefDesSame && PackageSame)
				{
					/* don't need to write the line, warning or exchange current and last values */
				} else
				{
					xprintf(io_fileout, "%s %s@%s\n", CurRefDes, CurPackage, CurPackage);	/* write the line */
					if(RefDesSame && !PackageSame)
						ttyputerr(_("Possible hetero error, RefDes same, package different"));	/* warning */
					strcpy(LastRefDes, CurRefDes);                                                                                         /* save current values to last */
					strcpy(LastPackage, CurPackage);
				}

				/* record all networks */
				name = describevariable(varpart, -1, -1);
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				{
					/* get "Pad" information */
					(void)initinfstr();
					(void)addstringtoinfstr("ATTRP_");
					(void)addstringtoinfstr(pi->proto->protoname);
					(void)addstringtoinfstr("_pad");

					var = getval((INTBIG)ni, VNODEINST, -1, returninfstr());
					if (var == NOVARIABLE) continue;

					/* create a new net connection */
					en = (PADSNET *)emalloc(sizeof (PADSNET), io_aid->cluster);
					if (en == 0) return;
					net = pi->conarcinst->network;
					if (net->temp1 != 0)
					{
						(void)allocstring(&en->netname, (char *)net->temp1, io_aid->cluster);
					} else
					{
						(void)initinfstr();
						(void)addstringtoinfstr(prefix);

						/* test for existence of at least one netname */
						if (net->namecount)(void)addstringtoinfstr(net->netname); else	/* @@@ Modified 8/21/00 NL */
							(void)addstringtoinfstr("**NO NET NAME**");					/* @@@ Added 8/21/00 NL */

						(void)allocstring(&en->netname, returninfstr(), io_aid->cluster);
					}

					(void)initinfstr();
					(void)addstringtoinfstr(prefix);
					(void)addstringtoinfstr(name);
					(void)allocstring(&en->packagename, returninfstr(), io_aid->cluster);
					(void)allocstring(&en->pinname, describevariable(var, -1, -1), io_aid->cluster);
					en->nextpadsnet = io_padsfirstnet;
					io_padsfirstnet = en;
				}
			}
		}
	}
}

/*
 * Routine to mark the facets in the current library that should be included
 * along with "curfacet".  The facets are marked by setting their "temp1" field
 * nonzero.  These facets include all that are part of a multipage schematic.
 */
void io_padsmarkfacets(NODEPROTO *curfacet)
{
	REGISTER NODEPROTO *np;

	/* mark those facets that will be included in this dump */
	for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		np->temp1 = 0;
		if ((curfacet->cellview->viewstate&MULTIPAGEVIEW) != 0 ||
			namesamen(curfacet->cellview->viewname, "schematic-page-", 15) == 0)
		{
			/* original facet is a multipage: allow all multipage in this cell */
			if (curfacet->cell != np->cell) continue;
		} else
		{
			/* original is not multipage: allow only it */
			if (curfacet != np) continue;
		}
		np->temp1 = 1;
	}
}

#endif  /* IOPADS - at top */
