/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bsestream.h"
#include	<string.h>



/* --- stream prototypes --- */
static BseErrorType bse_null_stream_open	   (BseStream	   *stream,
						    gboolean	    read_access,
						    gboolean	    write_access);
static void	    bse_null_stream_close	   (BseStream	   *stream);
static BseErrorType bse_null_stream_start	   (BseStream	   *stream);
static BseErrorType bse_null_stream_suspend	   (BseStream	   *stream);
static gboolean	    bse_null_stream_would_block	   (BseStream	   *stream,
						    guint	    n_values);
static BseErrorType bse_null_stream_read	   (BseStream	   *stream,
						    guint	    n_values,
						    BseSampleValue *values);
static BseErrorType bse_null_stream_write	   (BseStream	   *stream,
						    guint	    n_values,
						    BseSampleValue *values);
static BseErrorType bse_null_stream_set_attribs	   (BseStream	   *stream,
						    BseStreamAttribMask mask,
						    BseStreamAttribs *attribs);

/* --- structures --- */
struct _BseStreamPrivate
{
};

/* --- variables --- */
static const BseStreamAttribMask BSE_NULL_SA_MASK = (BSE_SA_N_CHANNELS |
						     BSE_SA_PLAY_FREQUENCY |
						     BSE_SA_REC_FREQUENCY |
						     BSE_SA_FRAGMENT_SIZE);


/* --- functions --- */
BseStream*
bse_null_stream_new (void)
{
  BseStream *stream;

  stream = g_new0 (BseStream, 1);

  stream->type_name = "NULL";
  stream->ref_count = 1;
  stream->name = NULL;

  stream->opened = FALSE;
  stream->readable = TRUE;
  stream->writable = TRUE;
  stream->suspended = TRUE;
  stream->can_block = FALSE;

  stream->private = NULL;

  stream->max_channels = 1;
  stream->min_play_frequency = 8000;
  stream->max_play_frequency = 48000;
  stream->min_record_frequency = 8000;
  stream->max_record_frequency = 48000;
  stream->min_fragment_size = 0;
  stream->max_fragment_size = 0;

  stream->attribs.n_channels = 1;
  stream->attribs.play_frequency = stream->max_play_frequency;
  stream->attribs.record_frequency = stream->max_record_frequency;
  stream->attribs.fragment_size = 0;
  
  stream->open = bse_null_stream_open;
  stream->close = bse_null_stream_close;
  stream->start = bse_null_stream_start;
  stream->suspend = bse_null_stream_suspend;
  stream->would_block = bse_null_stream_would_block;
  stream->read = bse_null_stream_read;
  stream->write = bse_null_stream_write;
  stream->set_attribs = bse_null_stream_set_attribs;

  return stream;
}

static BseErrorType
bse_null_stream_open (BseStream *stream,
		      gboolean	 read_access,
		      gboolean	 write_access)
{
  stream->readable = TRUE;
  stream->writable = TRUE;
  stream->can_block = FALSE;

  stream->attribs.n_channels = CLAMP (stream->attribs.n_channels, 1, stream->max_channels);
  stream->attribs.play_frequency = CLAMP (stream->attribs.play_frequency,
					  stream->min_play_frequency,
					  stream->max_play_frequency);
  stream->attribs.record_frequency = CLAMP (stream->attribs.record_frequency,
					    stream->min_record_frequency,
					    stream->max_record_frequency);
  stream->attribs.fragment_size = CLAMP (stream->attribs.fragment_size,
					 stream->min_fragment_size,
					 stream->max_fragment_size);

  stream->private = NULL;

  return BSE_ERROR_NONE;
}

static void
bse_null_stream_close (BseStream *stream)
{
  stream->private = NULL;
}

static BseErrorType
bse_null_stream_start (BseStream *stream)
{
  return BSE_ERROR_NONE;
}

static BseErrorType
bse_null_stream_suspend (BseStream *stream)
{
  return BSE_ERROR_NONE;
}

static gboolean
bse_null_stream_would_block (BseStream *stream,
			     guint	n_values)
{
  return FALSE;
}

static BseErrorType
bse_null_stream_read (BseStream	     *stream,
		      guint	      n_values,
		      BseSampleValue *values)
{
  memset (values, 0, sizeof (BseSampleValue) * n_values);

  return BSE_ERROR_NONE;
}

static BseErrorType
bse_null_stream_write (BseStream      *stream,
		       guint	       n_values,
		       BseSampleValue *values)
{
  return BSE_ERROR_NONE;
}

static BseErrorType
bse_null_stream_set_attribs (BseStream	 *stream,
			     BseStreamAttribMask value_mask,
			     BseStreamAttribs *attrib_values)
{
  BseErrorType error;

  value_mask &= BSE_NULL_SA_MASK;

  error = BSE_ERROR_NONE;

  if (attrib_values != &stream->attribs)
    g_memmove (attrib_values, &stream->attribs, sizeof (*attrib_values));

  return error;
}
