# 1 "src/octLexer.mll"
 
(** The lexer for string to build text structures. *)
open Common
open Types
open OctParser
open Errors
open Lexing

(* Convert lexing position into error position *)
let position p =
  { line = p.pos_lnum;
    column = p.pos_cnum - p.pos_bol; }

(* Fetch the current lexing location *)
let curr_loc lexbuf =
  { start = position lexbuf.lex_start_p;
    finish = position lexbuf.lex_curr_p; }

let dummy_loc =
  let dummy_pos =
    { line = -1;
      column = -1; }
  in
    { start = dummy_pos;
      finish = dummy_pos; }

let incr_line lexbuf =
  let pos = lexbuf.lex_curr_p in
    lexbuf.lex_curr_p <-
      { pos with
        pos_lnum = pos.pos_lnum + 1;
        pos_bol = pos.pos_cnum; }

(* To buffer verbatim strings and code sections *)

let string_buffer = Buffer.create 32

let buffer_empty = ref true

let reset_string_buffer () =
  Buffer.reset string_buffer;
  buffer_empty := true

let buffer_char c =
  Buffer.add_char string_buffer c;
  buffer_empty := false

let buffer_lexeme lexbuf =
  let s = lexeme lexbuf in
    Buffer.add_string string_buffer s;
    buffer_empty := false

let get_raw_buffered_string () = Buffer.contents string_buffer

let remove_opening_blanks s =
  let length = String.length s in
  let rec loop i =
    if i >= length then "" else
    match s.[i] with
    | ' ' | '\009' | '\012' -> loop (i + 1)
    | '\010' ->
        String.sub s (i + 1) (length - (i + 1))
    | '\013' ->
        let j = i + 1 in
          if j < length && s.[j] = '\010' then
            String.sub s (j + 1) (length - (j + 1))
          else
            String.sub s j (length - j)
    | _ -> String.sub s i (length - i)
  in
    loop 0

let remove_closing_blanks s =
  let length = String.length s in
  let rec loop i =
    if i < 0 then "" else
    match s.[i] with
    | ' ' | '\009' | '\012' -> loop (i - 1)
    | '\010' ->
        let j = i - 1 in
          if j >= 0 && s.[j] = '\013' then
            String.sub s 0 j
          else
            String.sub s 0 i
    | _ -> String.sub s 0 (i + 1)
  in
    loop (length - 1)

let get_buffered_string () =
  get_raw_buffered_string ()
  |> remove_opening_blanks
  |> remove_closing_blanks

let buffer_not_empty () = not !buffer_empty

(* To store the position of the beginning of a
   verbatim string or code section *)
let start_loc = ref (dummy_pos, dummy_pos)

let set_start_loc lexbuf =
  start_loc := (lexbuf.lex_start_p, lexbuf.lex_curr_p)

let get_start_loc () =
  let start_p, curr_p = !start_loc in
  { start = position start_p;
    finish = position curr_p; }

let use_start_loc lexbuf =
  let start_p, _ = !start_loc in
  lexbuf.lex_start_p <- start_p

(* To store the positions of nested code sections *)
let inner_start_locs = ref [];;

let push_inner_start_loc lexbuf =
  inner_start_locs := (curr_loc lexbuf) :: !inner_start_locs

let pop_inner_start_loc () =
  match !inner_start_locs with
  | [] -> None
  | l :: rest ->
      inner_start_locs := rest;
      Some l

(* To store the format of a target *)
let target_format = ref None;;

(* To store the kind of a reference *)
let ref_kind = ref RK_element;;

(* To store the start of a see description *)
let see_loc = ref dummy_loc;;

let set_see_loc lexbuf =
  see_loc := curr_loc lexbuf

let get_see_loc () = !see_loc

(* To store the modules of a module list *)
let module_list_modules = ref [];;

let reset_module_list () =
  module_list_modules := [];;

let add_module md =
  module_list_modules := md :: !module_list_modules

let get_module_list () =
  List.rev !module_list_modules

(* Hash table of styles (initialized below) *)
let style_table = Hashtbl.create 19

(* Hash table of reference kinds (initialized below) *)
let ref_kind_table = Hashtbl.create 19

(* Hash table of tags (initialized below) *)
let tag_table = Hashtbl.create 19


# 163 "src/octLexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\209\255\210\255\009\000\002\000\005\000\212\255\213\255\
    \214\255\031\000\001\000\247\255\003\000\252\255\104\000\039\001\
    \015\000\255\255\000\002\064\002\025\003\235\255\236\255\237\255\
    \242\003\179\000\240\255\050\004\245\255\002\005\219\005\027\006\
    \244\006\196\007\250\255\157\008\241\255\242\255\221\008\182\009\
    \246\009\207\010\015\011\232\011\248\255\244\255\001\000\004\000\
    \009\000\014\000\008\000\019\000\190\000\096\001\020\000\007\000\
    \020\000\021\000\024\000\234\255\027\000\064\000\082\000\062\000\
    \226\255\092\000\172\000\176\000\181\000\135\000\001\001\134\001\
    \182\000\174\000\185\000\186\000\161\001\233\255\174\000\190\000\
    \178\000\165\001\225\255\139\003\231\255\167\001\229\255\227\255\
    \031\002\192\000\179\000\032\002\223\255\215\255\061\001\061\001\
    \050\001\056\003\221\255\057\003\219\255\140\003\217\255\116\005\
    \232\255\141\003\230\255\228\255\145\003\065\001\087\001\146\003\
    \224\255\216\255\109\001\120\001\112\001\147\003\222\255\151\003\
    \220\255\152\003\218\255\223\012\252\255\184\013\006\000\254\255\
    \015\000\248\013\209\014\117\005\250\255\251\255\252\255\253\255\
    \013\000\254\255\017\000\121\005\122\005\015\000\253\255\254\255\
    \255\255\123\005\150\005\020\000\253\255\254\255\255\255\254\000\
    \101\007\025\000\253\255\254\255\255\255\209\015\252\255\210\016\
    \028\000\254\255\030\000\138\017\250\255\063\000\147\017\034\000\
    \251\255\252\255\021\000\032\000\032\002\255\255\254\255\253\255\
    \204\017\250\255\193\001\214\017\038\000\251\255\252\255\026\000\
    \013\000\242\003\255\255\254\255\253\255\025\018\250\255\153\003\
    \035\018\043\000\251\255\252\255\253\255\254\255\050\004\255\255\
    \092\018\250\255\017\004\132\018\045\000\251\255\252\255\038\000\
    \005\000\219\005\255\255\254\255\253\255\196\018\250\255\129\005\
    \233\018\047\000\251\255\252\255\237\003\048\007\255\255\131\001\
    \207\001\247\001\247\001\019\007\254\255\045\002\058\002\058\002\
    \020\007\253\255\017\019\251\255\143\005\062\019\048\000\252\255\
    \253\255\254\255\049\007\255\255\074\019\250\255\138\019\049\000\
    \251\255\250\005\253\255\254\255\052\007\255\255\020\000\252\255\
    \022\000\254\255\255\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\045\000\044\000\045\000\255\255\255\255\
    \255\255\045\000\009\000\255\255\045\000\255\255\002\000\045\000\
    \045\000\255\255\001\000\255\255\001\000\255\255\255\255\255\255\
    \017\000\016\000\255\255\012\000\255\255\006\000\004\000\255\255\
    \017\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\016\000\255\255\016\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\002\000\003\000\255\255\
    \000\000\255\255\002\000\255\255\255\255\255\255\255\255\255\255\
    \005\000\255\255\000\000\255\255\003\000\003\000\255\255\255\255\
    \255\255\255\255\003\000\003\000\255\255\255\255\255\255\255\255\
    \003\000\003\000\255\255\255\255\255\255\255\255\255\255\002\000\
    \003\000\255\255\000\000\255\255\255\255\005\000\005\000\005\000\
    \255\255\255\255\005\000\005\000\005\000\255\255\255\255\255\255\
    \255\255\255\255\005\000\005\000\005\000\255\255\255\255\005\000\
    \005\000\005\000\255\255\255\255\255\255\255\255\255\255\005\000\
    \005\000\005\000\255\255\255\255\255\255\255\255\005\000\255\255\
    \255\255\255\255\005\000\005\000\005\000\255\255\255\255\005\000\
    \005\000\005\000\255\255\255\255\255\255\255\255\255\255\005\000\
    \005\000\005\000\255\255\255\255\005\000\005\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\004\000\004\000\004\000\255\255\
    \255\255\255\255\004\000\255\255\255\255\255\255\005\000\005\000\
    \255\255\003\000\255\255\255\255\005\000\255\255\255\255\255\255\
    \002\000\255\255\255\255";
  Lexing.lex_default =
   "\003\000\000\000\000\000\003\000\255\255\255\255\000\000\000\000\
    \000\000\255\255\255\255\000\000\255\255\000\000\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\000\000\000\000\000\000\
    \255\255\255\255\000\000\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\000\000\255\255\000\000\255\255\000\000\000\000\
    \255\255\255\255\255\255\255\255\000\000\000\000\255\255\255\255\
    \255\255\255\255\000\000\255\255\000\000\255\255\000\000\255\255\
    \000\000\255\255\000\000\000\000\255\255\255\255\255\255\255\255\
    \000\000\000\000\255\255\255\255\255\255\255\255\000\000\255\255\
    \000\000\255\255\000\000\124\000\000\000\255\255\255\255\000\000\
    \255\255\255\255\255\255\132\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\255\255\140\000\140\000\255\255\000\000\000\000\
    \000\000\146\000\146\000\255\255\000\000\000\000\000\000\152\000\
    \152\000\255\255\000\000\000\000\000\000\159\000\000\000\159\000\
    \255\255\000\000\255\255\166\000\000\000\255\255\166\000\255\255\
    \000\000\000\000\255\255\255\255\255\255\000\000\000\000\000\000\
    \179\000\000\000\255\255\179\000\255\255\000\000\000\000\255\255\
    \255\255\255\255\000\000\000\000\000\000\192\000\000\000\255\255\
    \192\000\255\255\000\000\000\000\000\000\000\000\255\255\000\000\
    \203\000\000\000\255\255\203\000\255\255\000\000\000\000\255\255\
    \255\255\255\255\000\000\000\000\000\000\216\000\000\000\255\255\
    \216\000\255\255\000\000\000\000\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\000\000\237\000\000\000\255\255\237\000\255\255\000\000\
    \000\000\000\000\255\255\000\000\246\000\000\000\246\000\255\255\
    \000\000\255\255\000\000\000\000\255\255\000\000\000\001\000\000\
    \000\001\000\000\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\004\000\006\000\004\000\004\000\005\000\004\000\006\000\
    \127\000\050\000\255\255\255\255\050\000\255\255\255\255\137\000\
    \128\000\142\000\138\000\128\000\051\000\138\000\148\000\051\000\
    \004\000\058\000\004\000\154\000\058\000\012\000\161\000\162\000\
    \050\000\255\255\162\000\007\000\168\000\008\000\255\255\128\000\
    \181\000\138\000\187\000\051\000\255\255\194\000\255\255\205\000\
    \058\000\218\000\239\000\248\000\009\000\000\000\162\000\000\000\
    \015\000\002\001\001\001\255\255\255\255\255\255\107\000\063\000\
    \165\000\255\255\063\000\165\000\000\000\121\000\053\000\017\000\
    \119\000\106\000\114\000\109\000\000\000\060\000\059\000\108\000\
    \057\000\055\000\058\000\011\000\016\000\010\000\063\000\165\000\
    \211\000\051\000\054\000\056\000\255\255\255\255\255\255\052\000\
    \050\000\000\000\017\000\049\000\017\000\121\000\046\000\061\000\
    \119\000\048\000\114\000\109\000\047\000\060\000\002\000\108\000\
    \057\000\055\000\058\000\014\000\064\000\013\000\045\000\255\255\
    \044\000\051\000\054\000\056\000\255\255\062\000\255\255\052\000\
    \050\000\027\000\017\000\049\000\017\000\029\000\046\000\061\000\
    \069\000\048\000\175\000\069\000\047\000\023\000\174\000\188\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\026\000\212\000\063\000\062\000\000\000\069\000\
    \101\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\028\000\063\000\087\000\022\000\021\000\
    \101\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\030\000\024\000\
    \024\000\024\000\024\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\040\000\103\000\103\000\
    \103\000\103\000\103\000\103\000\103\000\103\000\103\000\103\000\
    \099\000\094\000\089\000\073\000\078\000\075\000\088\000\076\000\
    \001\000\000\000\079\000\080\000\081\000\074\000\090\000\091\000\
    \154\000\255\255\070\000\153\000\000\000\070\000\000\000\000\000\
    \000\000\000\000\025\000\000\000\255\000\000\000\255\255\000\000\
    \099\000\094\000\089\000\073\000\078\000\075\000\088\000\076\000\
    \156\000\070\000\079\000\080\000\081\000\074\000\090\000\091\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\086\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\000\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\000\000\000\000\095\000\096\000\097\000\110\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\070\000\072\000\095\000\096\000\097\000\110\000\
    \071\000\069\000\076\000\111\000\068\000\076\000\081\000\065\000\
    \085\000\081\000\067\000\085\000\115\000\066\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \116\000\076\000\070\000\072\000\117\000\081\000\229\000\085\000\
    \071\000\069\000\178\000\111\000\068\000\178\000\000\000\065\000\
    \000\000\000\000\067\000\000\000\115\000\066\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\077\000\
    \116\000\178\000\000\000\082\000\117\000\084\000\229\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\155\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\225\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \088\000\091\000\000\000\088\000\091\000\000\000\019\000\000\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\000\000\226\000\227\000\000\000\225\000\088\000\
    \091\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\226\000\227\000\093\000\092\000\018\000\
    \173\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\173\000\230\000\173\000\231\000\232\000\
    \000\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\173\000\230\000\173\000\231\000\232\000\
    \000\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\000\000\000\000\000\000\000\000\000\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\000\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\000\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\000\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\000\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\097\000\099\000\000\000\097\000\099\000\000\000\019\000\
    \000\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\000\000\000\000\000\000\000\000\000\000\
    \097\000\099\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\000\000\000\000\098\000\100\000\
    \020\000\000\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\085\000\101\000\105\000\085\000\
    \101\000\105\000\108\000\111\000\117\000\108\000\111\000\117\000\
    \119\000\121\000\191\000\119\000\121\000\191\000\000\000\000\000\
    \000\000\000\000\000\000\085\000\101\000\105\000\000\000\000\000\
    \000\000\108\000\111\000\117\000\000\000\000\000\000\000\119\000\
    \121\000\191\000\000\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\000\000\000\000\000\000\
    \000\000\084\000\102\000\104\000\000\000\000\000\000\000\113\000\
    \112\000\118\000\000\000\000\000\000\000\120\000\122\000\000\000\
    \000\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \000\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \000\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\024\000\202\000\000\000\223\000\202\000\000\000\000\000\
    \031\000\000\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\000\000\000\000\000\000\000\000\
    \224\000\202\000\186\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\186\000\000\000\186\000\
    \224\000\024\000\000\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\186\000\000\000\186\000\
    \000\000\000\000\199\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\199\000\000\000\199\000\
    \000\000\000\000\000\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\199\000\000\000\199\000\
    \000\000\000\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\000\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\000\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\033\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\000\000\000\000\000\000\
    \000\000\033\000\000\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\105\000\138\000\137\000\
    \105\000\138\000\136\000\142\000\255\255\148\000\141\000\255\255\
    \147\000\000\000\215\000\000\000\000\000\215\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\105\000\138\000\000\000\133\000\
    \236\000\000\000\000\000\236\000\134\000\000\000\000\000\000\000\
    \255\255\215\000\150\000\255\255\103\000\103\000\103\000\103\000\
    \103\000\103\000\103\000\103\000\103\000\103\000\000\000\236\000\
    \000\000\135\000\104\000\000\000\000\000\000\000\000\000\144\000\
    \255\255\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\000\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\000\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\024\000\249\000\000\000\000\000\249\000\000\000\
    \000\000\031\000\000\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\000\000\000\000\000\000\
    \000\000\000\000\249\000\210\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\210\000\000\000\
    \210\000\000\000\024\000\000\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\210\000\000\000\
    \210\000\000\000\000\000\000\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\000\000\000\000\
    \000\000\143\000\255\255\149\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\255\255\000\000\
    \000\000\000\000\000\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\000\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\000\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\227\000\232\000\000\000\227\000\
    \232\000\000\000\031\000\000\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\000\000\000\000\
    \000\000\000\000\000\000\227\000\232\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\000\000\
    \000\000\228\000\233\000\032\000\000\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\255\255\
    \222\000\243\000\255\255\000\000\253\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\222\000\243\000\222\000\243\000\253\000\
    \000\000\253\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\222\000\243\000\222\000\243\000\253\000\
    \000\000\253\000\000\000\000\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\000\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\033\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\034\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\000\000\
    \000\000\000\000\000\000\033\000\000\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\000\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\000\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\035\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\038\000\000\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\036\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\000\000\000\000\000\000\035\000\000\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\000\000\037\000\000\000\000\000\000\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \000\000\000\000\000\000\000\000\000\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\000\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\000\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\038\000\000\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \036\000\000\000\000\000\000\000\000\000\000\000\000\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\000\000\000\000\000\000\000\000\039\000\000\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\000\000\000\000\037\000\000\000\000\000\000\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\000\000\000\000\000\000\000\000\000\000\000\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\000\000\000\000\000\000\000\000\000\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\000\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\000\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\000\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\000\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\042\000\000\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\000\000\000\000\000\000\000\000\041\000\000\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\000\000\000\000\000\000\000\000\000\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\000\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\000\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\000\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\000\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\042\000\000\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\000\000\000\000\000\000\000\000\043\000\
    \000\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\000\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\000\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \128\000\127\000\000\000\128\000\126\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\128\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\000\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\000\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\129\000\000\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\000\000\000\000\000\000\000\000\125\000\
    \000\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\000\000\000\000\000\000\000\000\000\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\000\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\000\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\000\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\000\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\000\000\000\000\000\000\000\000\000\000\000\000\129\000\
    \000\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\000\000\000\000\000\000\000\000\
    \130\000\000\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \000\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \000\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\162\000\161\000\158\000\162\000\160\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\162\000\000\000\158\000\158\000\000\000\000\000\000\000\
    \000\000\158\000\158\000\000\000\000\000\158\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\158\000\000\000\000\000\000\000\
    \000\000\158\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\158\000\158\000\158\000\000\000\
    \000\000\158\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\158\000\000\000\158\000\000\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \158\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \158\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\158\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\000\000\000\000\
    \000\000\000\000\255\255\255\255\000\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\165\000\168\000\000\000\165\000\167\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\000\000\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\165\000\000\000\000\000\000\000\000\000\164\000\
    \000\000\000\000\000\000\255\255\000\000\164\000\000\000\164\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\164\000\000\000\
    \000\000\255\255\164\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\255\255\255\255\000\000\178\000\181\000\000\000\
    \178\000\180\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \255\255\000\000\255\255\255\255\000\000\164\000\172\000\164\000\
    \000\000\000\000\000\000\000\000\178\000\000\000\255\255\255\255\
    \255\255\183\000\000\000\000\000\000\000\000\000\255\255\177\000\
    \000\000\177\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \170\000\255\255\000\000\255\255\000\000\171\000\000\000\164\000\
    \177\000\255\255\000\000\000\000\177\000\000\000\255\255\000\000\
    \255\255\000\000\255\255\000\000\000\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\191\000\194\000\000\000\191\000\193\000\177\000\
    \185\000\177\000\000\000\000\000\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\000\000\
    \000\000\191\000\000\000\000\000\000\000\000\000\190\000\000\000\
    \000\000\000\000\177\000\255\255\190\000\000\000\190\000\184\000\
    \255\255\177\000\000\000\000\000\255\255\000\000\255\255\000\000\
    \255\255\255\255\000\000\255\255\000\000\190\000\000\000\000\000\
    \000\000\190\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\255\255\000\000\202\000\205\000\000\000\
    \202\000\204\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\197\000\198\000\196\000\000\000\
    \000\000\000\000\000\000\000\000\202\000\000\000\255\255\255\255\
    \255\255\201\000\000\000\000\000\000\000\000\000\000\000\201\000\
    \000\000\201\000\169\000\000\000\000\000\255\255\255\255\190\000\
    \255\255\255\255\000\000\255\255\190\000\000\000\190\000\000\000\
    \201\000\255\255\000\000\000\000\201\000\000\000\255\255\000\000\
    \255\255\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\201\000\
    \209\000\207\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\182\000\215\000\218\000\000\000\
    \215\000\217\000\201\000\000\000\000\000\000\000\255\255\208\000\
    \000\000\201\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \255\255\255\255\000\000\000\000\215\000\000\000\000\000\000\000\
    \000\000\214\000\000\000\000\000\000\000\000\000\000\000\214\000\
    \000\000\214\000\255\255\255\255\000\000\255\255\255\255\000\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\255\255\
    \220\000\255\255\000\000\000\000\214\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\255\255\000\000\255\255\000\000\
    \000\000\195\000\236\000\239\000\000\000\236\000\238\000\214\000\
    \221\000\214\000\000\000\255\255\000\000\255\255\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\236\000\000\000\000\000\000\000\000\000\235\000\000\000\
    \000\000\000\000\214\000\000\000\235\000\000\000\235\000\214\000\
    \000\000\214\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\000\000\235\000\000\000\000\000\
    \000\000\235\000\000\000\249\000\248\000\000\000\249\000\247\000\
    \000\000\000\000\000\000\000\000\206\000\000\000\255\255\255\255\
    \000\000\000\000\000\000\255\255\255\255\000\000\255\255\000\000\
    \000\000\255\255\249\000\255\255\235\000\242\000\235\000\245\000\
    \000\000\000\000\000\000\000\000\000\000\245\000\000\000\245\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\255\255\000\000\245\000\235\000\
    \000\000\000\000\245\000\000\000\235\000\000\000\241\000\000\000\
    \000\000\000\000\000\000\255\255\255\255\000\000\255\255\255\255\
    \000\000\255\255\255\255\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\245\000\252\000\245\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\255\255\255\255\000\000\255\255\
    \000\000\255\255\000\000\255\255\000\000\000\000\000\000\000\000\
    \245\000\000\000\000\000\000\000\219\000\245\000\255\255\251\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\000\000\255\255\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\240\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\250\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\004\000\000\000\000\000\004\000\005\000\
    \126\000\050\000\003\000\003\000\050\000\003\000\003\000\136\000\
    \128\000\141\000\138\000\128\000\051\000\138\000\147\000\051\000\
    \000\000\058\000\004\000\153\000\058\000\000\000\160\000\162\000\
    \050\000\003\000\162\000\000\000\167\000\000\000\003\000\128\000\
    \180\000\138\000\184\000\051\000\003\000\193\000\003\000\204\000\
    \058\000\217\000\238\000\247\000\000\000\255\255\162\000\255\255\
    \000\000\254\000\254\000\000\001\000\001\003\000\050\000\063\000\
    \165\000\003\000\063\000\165\000\255\255\046\000\009\000\016\000\
    \047\000\051\000\048\000\049\000\255\255\055\000\058\000\049\000\
    \056\000\054\000\057\000\000\000\000\000\000\000\063\000\165\000\
    \208\000\009\000\009\000\054\000\003\000\003\000\003\000\009\000\
    \009\000\255\255\016\000\009\000\016\000\046\000\009\000\060\000\
    \047\000\009\000\048\000\049\000\009\000\055\000\000\000\049\000\
    \056\000\054\000\057\000\000\000\063\000\000\000\010\000\003\000\
    \012\000\009\000\009\000\054\000\003\000\061\000\003\000\009\000\
    \009\000\014\000\016\000\009\000\016\000\014\000\009\000\060\000\
    \069\000\009\000\170\000\069\000\009\000\014\000\171\000\183\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\207\000\062\000\061\000\255\255\069\000\
    \065\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\062\000\069\000\014\000\014\000\
    \065\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \066\000\067\000\068\000\072\000\073\000\074\000\068\000\075\000\
    \000\000\255\255\078\000\079\000\080\000\072\000\089\000\090\000\
    \151\000\003\000\070\000\151\000\255\255\070\000\255\255\255\255\
    \255\255\255\255\025\000\255\255\254\000\255\255\000\001\255\255\
    \066\000\067\000\068\000\072\000\073\000\074\000\068\000\075\000\
    \151\000\070\000\078\000\079\000\080\000\072\000\089\000\090\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\070\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\255\255\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\255\255\255\255\094\000\095\000\096\000\109\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\053\000\053\000\094\000\095\000\096\000\109\000\
    \053\000\053\000\076\000\110\000\053\000\076\000\081\000\053\000\
    \085\000\081\000\053\000\085\000\114\000\053\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \115\000\076\000\053\000\053\000\116\000\081\000\223\000\085\000\
    \053\000\053\000\178\000\110\000\053\000\178\000\255\255\053\000\
    \255\255\255\255\053\000\255\255\114\000\053\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\076\000\
    \115\000\178\000\255\255\081\000\116\000\085\000\223\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\151\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\224\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\018\000\
    \088\000\091\000\255\255\088\000\091\000\255\255\018\000\255\255\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\255\255\225\000\226\000\255\255\224\000\088\000\
    \091\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\225\000\226\000\088\000\091\000\018\000\
    \172\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\172\000\229\000\172\000\230\000\231\000\
    \255\255\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\172\000\229\000\172\000\230\000\231\000\
    \255\255\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\255\255\255\255\255\255\255\255\255\255\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\255\255\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\255\255\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\255\255\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\255\255\
    \019\000\019\000\019\000\019\000\019\000\019\000\019\000\019\000\
    \020\000\097\000\099\000\255\255\097\000\099\000\255\255\020\000\
    \255\255\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\255\255\255\255\255\255\255\255\255\255\
    \097\000\099\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\255\255\255\255\097\000\099\000\
    \020\000\255\255\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\083\000\101\000\105\000\083\000\
    \101\000\105\000\108\000\111\000\117\000\108\000\111\000\117\000\
    \119\000\121\000\191\000\119\000\121\000\191\000\255\255\255\255\
    \255\255\255\255\255\255\083\000\101\000\105\000\255\255\255\255\
    \255\255\108\000\111\000\117\000\255\255\255\255\255\255\119\000\
    \121\000\191\000\255\255\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\255\255\255\255\255\255\
    \255\255\083\000\101\000\105\000\255\255\255\255\255\255\108\000\
    \111\000\117\000\255\255\255\255\255\255\119\000\121\000\255\255\
    \255\255\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \255\255\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \255\255\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\024\000\202\000\255\255\220\000\202\000\255\255\255\255\
    \024\000\255\255\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\255\255\255\255\255\255\255\255\
    \220\000\202\000\185\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\185\000\255\255\185\000\
    \220\000\024\000\255\255\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\185\000\255\255\185\000\
    \255\255\255\255\198\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\198\000\255\255\198\000\
    \255\255\255\255\255\255\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\198\000\255\255\198\000\
    \255\255\255\255\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\255\255\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\255\255\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\255\255\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\029\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\255\255\255\255\255\255\
    \255\255\029\000\255\255\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\103\000\131\000\131\000\
    \103\000\131\000\131\000\139\000\140\000\145\000\139\000\140\000\
    \145\000\255\255\215\000\255\255\255\255\215\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\103\000\131\000\255\255\131\000\
    \236\000\255\255\255\255\236\000\131\000\255\255\255\255\255\255\
    \146\000\215\000\145\000\146\000\103\000\103\000\103\000\103\000\
    \103\000\103\000\103\000\103\000\103\000\103\000\255\255\236\000\
    \255\255\131\000\103\000\255\255\255\255\255\255\255\255\139\000\
    \140\000\255\255\255\255\255\255\255\255\146\000\255\255\255\255\
    \255\255\255\255\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\255\255\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\255\255\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\030\000\249\000\255\255\255\255\249\000\255\255\
    \255\255\030\000\255\255\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\255\255\255\255\255\255\
    \255\255\255\255\249\000\209\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\209\000\255\255\
    \209\000\255\255\030\000\255\255\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\209\000\255\255\
    \209\000\255\255\255\255\255\255\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\255\255\255\255\
    \255\255\139\000\140\000\145\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\146\000\255\255\
    \255\255\255\255\255\255\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\255\255\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\255\255\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\255\255\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\255\255\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\032\000\227\000\232\000\255\255\227\000\
    \232\000\255\255\032\000\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\255\255\255\255\
    \255\255\255\255\255\255\227\000\232\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\255\255\
    \255\255\227\000\232\000\032\000\255\255\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\152\000\
    \221\000\242\000\152\000\255\255\252\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\152\000\
    \255\255\255\255\255\255\221\000\242\000\221\000\242\000\252\000\
    \255\255\252\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\221\000\242\000\221\000\242\000\252\000\
    \255\255\252\000\255\255\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\033\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\255\255\
    \255\255\255\255\255\255\033\000\255\255\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\152\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\255\255\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\255\255\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\035\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\035\000\255\255\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \255\255\255\255\255\255\255\255\035\000\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \255\255\255\255\035\000\255\255\255\255\255\255\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \255\255\255\255\255\255\255\255\255\255\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\255\255\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\255\255\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\255\255\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\255\255\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\039\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\039\000\255\255\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\255\255\255\255\255\255\255\255\255\255\255\255\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\255\255\255\255\255\255\255\255\039\000\255\255\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\255\255\255\255\039\000\255\255\255\255\255\255\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\255\255\255\255\255\255\255\255\255\255\255\255\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\255\255\255\255\255\255\255\255\255\255\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\255\255\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\255\255\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\255\255\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\255\255\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\041\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\041\000\255\255\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\255\255\255\255\255\255\255\255\041\000\255\255\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\255\255\255\255\255\255\255\255\255\255\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\255\255\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\255\255\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\255\255\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\255\255\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\043\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\043\000\255\255\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\255\255\255\255\255\255\255\255\043\000\
    \255\255\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\255\255\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\255\255\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \123\000\123\000\255\255\123\000\123\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\123\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\255\255\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\255\255\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\125\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\125\000\255\255\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\255\255\255\255\255\255\255\255\125\000\
    \255\255\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\255\255\255\255\255\255\255\255\255\255\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\255\255\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\255\255\
    \125\000\125\000\125\000\125\000\125\000\125\000\125\000\125\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\255\255\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\255\255\
    \129\000\129\000\129\000\129\000\129\000\129\000\129\000\129\000\
    \130\000\255\255\255\255\255\255\255\255\255\255\255\255\130\000\
    \255\255\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\255\255\255\255\255\255\255\255\
    \130\000\255\255\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \255\255\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \255\255\130\000\130\000\130\000\130\000\130\000\130\000\130\000\
    \130\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\255\255\157\000\157\000\255\255\255\255\255\255\
    \255\255\157\000\157\000\255\255\255\255\157\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\157\000\255\255\255\255\255\255\
    \255\255\157\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\157\000\157\000\157\000\255\255\
    \255\255\157\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\157\000\255\255\157\000\255\255\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \157\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \157\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\157\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\255\255\159\000\159\000\255\255\255\255\
    \255\255\255\255\159\000\159\000\255\255\255\255\159\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\159\000\255\255\255\255\
    \255\255\255\255\159\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\159\000\159\000\159\000\
    \255\255\255\255\159\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\159\000\255\255\159\000\
    \255\255\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\255\255\163\000\163\000\255\255\163\000\163\000\
    \255\255\255\255\255\255\255\255\166\000\166\000\255\255\166\000\
    \166\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\159\000\163\000\255\255\255\255\255\255\255\255\163\000\
    \255\255\255\255\255\255\166\000\255\255\163\000\255\255\163\000\
    \166\000\255\255\255\255\255\255\255\255\255\255\166\000\255\255\
    \166\000\255\255\255\255\255\255\255\255\255\255\163\000\255\255\
    \255\255\159\000\163\000\255\255\255\255\255\255\255\255\166\000\
    \255\255\255\255\159\000\166\000\255\255\176\000\176\000\255\255\
    \176\000\176\000\255\255\255\255\255\255\255\255\255\255\179\000\
    \179\000\255\255\179\000\179\000\255\255\163\000\163\000\163\000\
    \255\255\255\255\255\255\255\255\176\000\255\255\166\000\166\000\
    \166\000\176\000\255\255\255\255\255\255\255\255\179\000\176\000\
    \255\255\176\000\255\255\179\000\255\255\255\255\255\255\255\255\
    \163\000\179\000\255\255\179\000\255\255\163\000\255\255\163\000\
    \176\000\166\000\255\255\255\255\176\000\255\255\166\000\255\255\
    \166\000\255\255\179\000\255\255\255\255\255\255\179\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\189\000\189\000\255\255\189\000\189\000\176\000\
    \176\000\176\000\255\255\255\255\192\000\192\000\255\255\192\000\
    \192\000\179\000\179\000\179\000\255\255\255\255\255\255\255\255\
    \255\255\189\000\255\255\255\255\255\255\255\255\189\000\255\255\
    \255\255\255\255\176\000\192\000\189\000\255\255\189\000\176\000\
    \192\000\176\000\255\255\255\255\179\000\255\255\192\000\255\255\
    \192\000\179\000\255\255\179\000\255\255\189\000\255\255\255\255\
    \255\255\189\000\255\255\255\255\255\255\255\255\255\255\192\000\
    \255\255\255\255\255\255\192\000\255\255\200\000\200\000\255\255\
    \200\000\200\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\189\000\189\000\189\000\255\255\
    \255\255\255\255\255\255\255\255\200\000\255\255\192\000\192\000\
    \192\000\200\000\255\255\255\255\255\255\255\255\255\255\200\000\
    \255\255\200\000\163\000\255\255\255\255\203\000\203\000\189\000\
    \203\000\203\000\255\255\166\000\189\000\255\255\189\000\255\255\
    \200\000\192\000\255\255\255\255\200\000\255\255\192\000\255\255\
    \192\000\255\255\255\255\255\255\203\000\255\255\255\255\255\255\
    \255\255\203\000\255\255\255\255\255\255\255\255\255\255\203\000\
    \255\255\203\000\255\255\255\255\255\255\255\255\255\255\200\000\
    \200\000\200\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \203\000\255\255\255\255\255\255\203\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\176\000\213\000\213\000\255\255\
    \213\000\213\000\200\000\255\255\255\255\255\255\179\000\200\000\
    \255\255\200\000\255\255\255\255\255\255\255\255\255\255\203\000\
    \203\000\203\000\255\255\255\255\213\000\255\255\255\255\255\255\
    \255\255\213\000\255\255\255\255\255\255\255\255\255\255\213\000\
    \255\255\213\000\216\000\216\000\255\255\216\000\216\000\255\255\
    \255\255\255\255\203\000\255\255\255\255\255\255\255\255\203\000\
    \213\000\203\000\255\255\255\255\213\000\255\255\255\255\255\255\
    \255\255\216\000\255\255\255\255\255\255\255\255\216\000\255\255\
    \255\255\255\255\255\255\255\255\216\000\255\255\216\000\255\255\
    \255\255\189\000\234\000\234\000\255\255\234\000\234\000\213\000\
    \213\000\213\000\255\255\192\000\255\255\216\000\255\255\255\255\
    \255\255\216\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\234\000\255\255\255\255\255\255\255\255\234\000\255\255\
    \255\255\255\255\213\000\255\255\234\000\255\255\234\000\213\000\
    \255\255\213\000\255\255\255\255\216\000\216\000\216\000\237\000\
    \237\000\255\255\237\000\237\000\255\255\234\000\255\255\255\255\
    \255\255\234\000\255\255\244\000\244\000\255\255\244\000\244\000\
    \255\255\255\255\255\255\255\255\200\000\255\255\237\000\216\000\
    \255\255\255\255\255\255\237\000\216\000\255\255\216\000\255\255\
    \255\255\237\000\244\000\237\000\234\000\234\000\234\000\244\000\
    \255\255\255\255\255\255\255\255\255\255\244\000\255\255\244\000\
    \255\255\255\255\237\000\255\255\255\255\255\255\237\000\255\255\
    \255\255\255\255\255\255\255\255\203\000\255\255\244\000\234\000\
    \255\255\255\255\244\000\255\255\234\000\255\255\234\000\255\255\
    \255\255\255\255\255\255\246\000\246\000\255\255\246\000\246\000\
    \255\255\237\000\237\000\237\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\244\000\244\000\244\000\
    \255\255\255\255\246\000\255\255\255\255\255\255\255\255\246\000\
    \255\255\255\255\255\255\255\255\237\000\246\000\255\255\246\000\
    \255\255\237\000\255\255\237\000\255\255\255\255\255\255\255\255\
    \244\000\255\255\255\255\255\255\213\000\244\000\246\000\244\000\
    \255\255\255\255\246\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\246\000\246\000\246\000\
    \255\255\216\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \246\000\255\255\255\255\255\255\255\255\246\000\255\255\246\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\234\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\237\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\244\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\246\000";
  Lexing.lex_base_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\010\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \009\000\217\000\025\001\233\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\084\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\094\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\104\000\028\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\114\000\
    \031\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_backtrk_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\004\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\015\000\000\000\015\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_default_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_trans_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\009\000\000\000\000\000\000\000\
    \000\000\000\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \012\000\000\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \012\000\000\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\000\000";
  Lexing.lex_check_code =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\255\255\255\255\255\255\
    \255\255\255\255\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\255\255\255\255\255\255\255\255\
    \255\255\025\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\103\000\103\000\103\000\103\000\103\000\103\000\
    \103\000\103\000\103\000\103\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \255\255\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \041\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\255\255\255\255\255\255\255\255\
    \041\000\255\255\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\255\255\255\255\255\255\255\255\
    \255\255\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \255\255\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \255\255\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \255\255\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \043\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\255\255\255\255\255\255\255\255\
    \043\000\255\255\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \255\255\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \255\255\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\255\255";
  Lexing.lex_code =
   "\255\003\255\255\002\255\000\003\255\004\255\255\005\255\255\000\
    \003\002\004\001\005\255\007\255\255\006\255\255\000\006\255\000\
    \007\255";
}

let rec main lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 8 (-1); __ocaml_lex_main_rec lexbuf 0
and __ocaml_lex_main_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 177 "src/octLexer.mll"
                                            chr
# 1891 "src/octLexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 240 "src/octLexer.mll"
    ( Char (String.make 1 chr) )
# 1895 "src/octLexer.ml"

  | 1 ->
let
# 241 "src/octLexer.mll"
                tag
# 1901 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_curr_pos in
# 242 "src/octLexer.mll"
    ( try
        let f = Hashtbl.find tag_table tag in
          set_start_loc lexbuf;
          f lexbuf
      with Not_found -> Custom tag )
# 1909 "src/octLexer.ml"

  | 2 ->
# 248 "src/octLexer.mll"
    ( BEGIN )
# 1914 "src/octLexer.ml"

  | 3 ->
# 250 "src/octLexer.mll"
    ( END )
# 1919 "src/octLexer.ml"

  | 4 ->
# 252 "src/octLexer.mll"
    ( reset_string_buffer ();
      set_start_loc lexbuf;
      verb lexbuf )
# 1926 "src/octLexer.ml"

  | 5 ->
let
# 194 "src/octLexer.mll"
                                   fmt
# 1932 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 2) (lexbuf.Lexing.lex_curr_pos + -1) in
# 256 "src/octLexer.mll"
    ( reset_string_buffer ();
      set_start_loc lexbuf;
      target_format := Some fmt;
      target lexbuf )
# 1939 "src/octLexer.ml"

  | 6 ->
# 261 "src/octLexer.mll"
    ( reset_string_buffer ();
      set_start_loc lexbuf;
      target_format := None;
      target lexbuf )
# 1947 "src/octLexer.ml"

  | 7 ->
# 266 "src/octLexer.mll"
    ( raise (LexerError(curr_loc lexbuf, Unmatched_target)) )
# 1952 "src/octLexer.ml"

  | 8 ->
# 268 "src/octLexer.mll"
    ( reset_string_buffer ();
      set_start_loc lexbuf;
      code lexbuf )
# 1959 "src/octLexer.ml"

  | 9 ->
# 272 "src/octLexer.mll"
    ( raise (LexerError(curr_loc lexbuf, Unmatched_code)) )
# 1964 "src/octLexer.ml"

  | 10 ->
# 274 "src/octLexer.mll"
    ( reset_string_buffer ();
      set_start_loc lexbuf;
      pre_code lexbuf )
# 1971 "src/octLexer.ml"

  | 11 ->
# 278 "src/octLexer.mll"
    ( raise (LexerError(curr_loc lexbuf, Unmatched_pre_code)) )
# 1976 "src/octLexer.ml"

  | 12 ->
# 280 "src/octLexer.mll"
    ( reset_string_buffer ();
      set_start_loc lexbuf;
      ref_kind := RK_element;
      reference lexbuf )
# 1984 "src/octLexer.ml"

  | 13 ->
let
# 284 "src/octLexer.mll"
                      lbl
# 1990 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 2) (lexbuf.Lexing.lex_curr_pos + -1) in
# 285 "src/octLexer.mll"
    ( if lbl = "indexlist" then Special_Ref SRK_index_list
      else Ref(RK_element, lbl) )
# 1995 "src/octLexer.ml"

  | 14 ->
let
# 205 "src/octLexer.mll"
                         kind
# 2001 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 2) (lexbuf.Lexing.lex_curr_pos + -1) in
# 288 "src/octLexer.mll"
    ( reset_string_buffer ();
      set_start_loc lexbuf;
      if kind = "modules" then begin
        reset_module_list ();
        module_list lexbuf
      end else begin
        let kind =
          try
            Hashtbl.find ref_kind_table kind
          with Not_found -> RK_custom kind
        in
          ref_kind := kind;
          reference lexbuf
      end )
# 2018 "src/octLexer.ml"

  | 15 ->
# 303 "src/octLexer.mll"
    ( reset_string_buffer ();
      set_start_loc lexbuf;
      ref_kind := RK_link;
      reference lexbuf )
# 2026 "src/octLexer.ml"

  | 16 ->
let
# 207 "src/octLexer.mll"
                               num
# 2032 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_mem.(0)
and
# 208 "src/octLexer.mll"
                                 lbl
# 2037 "src/octLexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(2) lexbuf.Lexing.lex_mem.(1) in
# 308 "src/octLexer.mll"
    ( Title (int_of_string num, lbl) )
# 2041 "src/octLexer.ml"

  | 17 ->
let
# 309 "src/octLexer.mll"
                  style
# 2047 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_curr_pos in
# 310 "src/octLexer.mll"
    ( try
        Hashtbl.find style_table style
      with Not_found -> Style (SK_custom style) )
# 2053 "src/octLexer.ml"

  | 18 ->
# 314 "src/octLexer.mll"
    ( Item true )
# 2058 "src/octLexer.ml"

  | 19 ->
# 316 "src/octLexer.mll"
    ( Style SK_superscript )
# 2063 "src/octLexer.ml"

  | 20 ->
# 318 "src/octLexer.mll"
    ( Style SK_subscript )
# 2068 "src/octLexer.ml"

  | 21 ->
# 320 "src/octLexer.mll"
    ( reset_string_buffer ();
      set_start_loc lexbuf;
      html_code lexbuf )
# 2075 "src/octLexer.ml"

  | 22 ->
# 324 "src/octLexer.mll"
    ( raise (LexerError(curr_loc lexbuf, Unmatched_html_code)) )
# 2080 "src/octLexer.ml"

  | 23 ->
let
# 221 "src/octLexer.mll"
                                              num
# 2086 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 2) lexbuf.Lexing.lex_mem.(0)
and
# 221 "src/octLexer.mll"
                                                      tag
# 2091 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_mem.(0) in
# 326 "src/octLexer.mll"
    ( HTML_Title(tag, int_of_string num) )
# 2095 "src/octLexer.ml"

  | 24 ->
let
# 222 "src/octLexer.mll"
                                                  num
# 2101 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 3) lexbuf.Lexing.lex_mem.(0) in
# 328 "src/octLexer.mll"
    ( HTML_END_Title (int_of_string num) )
# 2105 "src/octLexer.ml"

  | 25 ->
let
# 217 "src/octLexer.mll"
                                  tag
# 2111 "src/octLexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 330 "src/octLexer.mll"
    ( HTML_Bold (String.make 1 tag))
# 2115 "src/octLexer.ml"

  | 26 ->
# 332 "src/octLexer.mll"
    ( HTML_END_BOLD )
# 2120 "src/octLexer.ml"

  | 27 ->
let
# 219 "src/octLexer.mll"
                                    tag
# 2126 "src/octLexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 334 "src/octLexer.mll"
    ( HTML_Italic (String.make 1 tag))
# 2130 "src/octLexer.ml"

  | 28 ->
# 336 "src/octLexer.mll"
    ( HTML_END_ITALIC )
# 2135 "src/octLexer.ml"

  | 29 ->
let
# 231 "src/octLexer.mll"
                                                                                 tag
# 2141 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_start_pos + 7) in
# 338 "src/octLexer.mll"
    ( HTML_Center tag)
# 2145 "src/octLexer.ml"

  | 30 ->
# 340 "src/octLexer.mll"
    ( HTML_END_CENTER )
# 2150 "src/octLexer.ml"

  | 31 ->
let
# 233 "src/octLexer.mll"
                                                             tag
# 2156 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_start_pos + 5) in
# 342 "src/octLexer.mll"
    ( HTML_Left tag)
# 2160 "src/octLexer.ml"

  | 32 ->
# 344 "src/octLexer.mll"
    ( HTML_END_LEFT )
# 2165 "src/octLexer.ml"

  | 33 ->
let
# 235 "src/octLexer.mll"
                                                                       tag
# 2171 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_start_pos + 6) in
# 346 "src/octLexer.mll"
    ( HTML_Right tag)
# 2175 "src/octLexer.ml"

  | 34 ->
# 348 "src/octLexer.mll"
    ( HTML_END_RIGHT )
# 2180 "src/octLexer.ml"

  | 35 ->
let
# 223 "src/octLexer.mll"
                                           tag
# 2186 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_start_pos + 3) in
# 350 "src/octLexer.mll"
    ( HTML_List tag)
# 2190 "src/octLexer.ml"

  | 36 ->
# 352 "src/octLexer.mll"
    ( HTML_END_LIST )
# 2195 "src/octLexer.ml"

  | 37 ->
let
# 225 "src/octLexer.mll"
                                           tag
# 2201 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_start_pos + 3) in
# 354 "src/octLexer.mll"
    ( HTML_Enum tag)
# 2205 "src/octLexer.ml"

  | 38 ->
# 356 "src/octLexer.mll"
    ( HTML_END_ENUM )
# 2210 "src/octLexer.ml"

  | 39 ->
let
# 227 "src/octLexer.mll"
                                           tag
# 2216 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_start_pos + 3) in
# 358 "src/octLexer.mll"
    ( HTML_Item tag)
# 2220 "src/octLexer.ml"

  | 40 ->
# 360 "src/octLexer.mll"
    ( HTML_END_ITEM )
# 2225 "src/octLexer.ml"

  | 41 ->
# 362 "src/octLexer.mll"
    ( MINUS )
# 2230 "src/octLexer.ml"

  | 42 ->
# 364 "src/octLexer.mll"
    ( PLUS )
# 2235 "src/octLexer.ml"

  | 43 ->
# 366 "src/octLexer.mll"
    ( incr_line lexbuf;
      NEWLINE )
# 2241 "src/octLexer.ml"

  | 44 ->
# 369 "src/octLexer.mll"
    ( BLANK )
# 2246 "src/octLexer.ml"

  | 45 ->
# 371 "src/octLexer.mll"
    ( Char (lexeme lexbuf) )
# 2251 "src/octLexer.ml"

  | 46 ->
# 372 "src/octLexer.mll"
                         ( EOF )
# 2256 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_main_rec lexbuf __ocaml_lex_state

and identifier lexbuf =
   __ocaml_lex_identifier_rec lexbuf 123
and __ocaml_lex_identifier_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 376 "src/octLexer.mll"
    ( identifier lexbuf )
# 2268 "src/octLexer.ml"

  | 1 ->
# 378 "src/octLexer.mll"
    ( incr_line lexbuf;
      identifier lexbuf )
# 2274 "src/octLexer.ml"

  | 2 ->
let
# 380 "src/octLexer.mll"
           id
# 2280 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 381 "src/octLexer.mll"
    ( use_start_loc lexbuf;
      id )
# 2285 "src/octLexer.ml"

  | 3 ->
# 384 "src/octLexer.mll"
    ( raise (LexerError(curr_loc lexbuf, Expected_ident)) )
# 2290 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_identifier_rec lexbuf __ocaml_lex_state

and see lexbuf =
   __ocaml_lex_see_rec lexbuf 131
and __ocaml_lex_see_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 388 "src/octLexer.mll"
    ( see lexbuf )
# 2302 "src/octLexer.ml"

  | 1 ->
# 390 "src/octLexer.mll"
    ( incr_line lexbuf;
      see lexbuf )
# 2308 "src/octLexer.ml"

  | 2 ->
# 393 "src/octLexer.mll"
    ( reset_string_buffer ();
      set_see_loc lexbuf;
      see_url lexbuf )
# 2315 "src/octLexer.ml"

  | 3 ->
# 397 "src/octLexer.mll"
    ( reset_string_buffer ();
      set_see_loc lexbuf;
      see_file lexbuf )
# 2322 "src/octLexer.ml"

  | 4 ->
# 401 "src/octLexer.mll"
    ( reset_string_buffer ();
      set_see_loc lexbuf;
      see_doc lexbuf )
# 2329 "src/octLexer.ml"

  | 5 ->
# 405 "src/octLexer.mll"
    ( raise (LexerError(curr_loc lexbuf, Expected_see)) )
# 2334 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_see_rec lexbuf __ocaml_lex_state

and see_url lexbuf =
   __ocaml_lex_see_url_rec lexbuf 139
and __ocaml_lex_see_url_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 409 "src/octLexer.mll"
    ( use_start_loc lexbuf;
      See_url (get_raw_buffered_string ()) )
# 2347 "src/octLexer.ml"

  | 1 ->
# 412 "src/octLexer.mll"
    ( raise (LexerError(get_see_loc (), Unterminated_see_url)) )
# 2352 "src/octLexer.ml"

  | 2 ->
# 414 "src/octLexer.mll"
    ( incr_line lexbuf;
      buffer_lexeme lexbuf;
      see_url lexbuf )
# 2359 "src/octLexer.ml"

  | 3 ->
# 418 "src/octLexer.mll"
    ( buffer_lexeme lexbuf;
      see_url lexbuf )
# 2365 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_see_url_rec lexbuf __ocaml_lex_state

and see_file lexbuf =
   __ocaml_lex_see_file_rec lexbuf 145
and __ocaml_lex_see_file_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 423 "src/octLexer.mll"
    ( use_start_loc lexbuf;
      See_file (get_raw_buffered_string ()) )
# 2378 "src/octLexer.ml"

  | 1 ->
# 426 "src/octLexer.mll"
    ( raise (LexerError(get_see_loc (), Unterminated_see_file)) )
# 2383 "src/octLexer.ml"

  | 2 ->
# 428 "src/octLexer.mll"
    ( incr_line lexbuf;
      buffer_lexeme lexbuf;
      see_file lexbuf )
# 2390 "src/octLexer.ml"

  | 3 ->
# 432 "src/octLexer.mll"
    ( buffer_lexeme lexbuf;
      see_file lexbuf )
# 2396 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_see_file_rec lexbuf __ocaml_lex_state

and see_doc lexbuf =
   __ocaml_lex_see_doc_rec lexbuf 151
and __ocaml_lex_see_doc_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 437 "src/octLexer.mll"
    ( use_start_loc lexbuf;
      See_doc (get_raw_buffered_string ()) )
# 2409 "src/octLexer.ml"

  | 1 ->
# 440 "src/octLexer.mll"
    ( raise (LexerError(get_see_loc (), Unterminated_see_doc)) )
# 2414 "src/octLexer.ml"

  | 2 ->
# 442 "src/octLexer.mll"
    ( incr_line lexbuf;
      buffer_lexeme lexbuf;
      see_doc lexbuf )
# 2421 "src/octLexer.ml"

  | 3 ->
# 446 "src/octLexer.mll"
    ( buffer_lexeme lexbuf;
      see_doc lexbuf )
# 2427 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_see_doc_rec lexbuf __ocaml_lex_state

and version lexbuf =
   __ocaml_lex_version_rec lexbuf 157
and __ocaml_lex_version_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 451 "src/octLexer.mll"
    ( version lexbuf )
# 2439 "src/octLexer.ml"

  | 1 ->
# 453 "src/octLexer.mll"
    ( incr_line lexbuf;
      version lexbuf )
# 2445 "src/octLexer.ml"

  | 2 ->
let
# 455 "src/octLexer.mll"
             v
# 2451 "src/octLexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 456 "src/octLexer.mll"
    ( use_start_loc lexbuf;
      v )
# 2456 "src/octLexer.ml"

  | 3 ->
# 459 "src/octLexer.mll"
    ( raise (LexerError(curr_loc lexbuf, Expected_version)) )
# 2461 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_version_rec lexbuf __ocaml_lex_state

and verb lexbuf =
   __ocaml_lex_verb_rec lexbuf 163
and __ocaml_lex_verb_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 177 "src/octLexer.mll"
                                            chr
# 2474 "src/octLexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 463 "src/octLexer.mll"
    ( buffer_char chr; verb lexbuf )
# 2478 "src/octLexer.ml"

  | 1 ->
# 465 "src/octLexer.mll"
    ( raise (LexerError(curr_loc lexbuf, Nested_verbatim)) )
# 2483 "src/octLexer.ml"

  | 2 ->
# 467 "src/octLexer.mll"
    ( use_start_loc lexbuf;
      Verb (get_buffered_string ()) )
# 2489 "src/octLexer.ml"

  | 3 ->
# 470 "src/octLexer.mll"
    ( raise (LexerError(get_start_loc (), Unterminated_verbatim)) )
# 2494 "src/octLexer.ml"

  | 4 ->
# 472 "src/octLexer.mll"
    ( incr_line lexbuf;
      buffer_lexeme lexbuf;
      verb lexbuf )
# 2501 "src/octLexer.ml"

  | 5 ->
# 476 "src/octLexer.mll"
    ( buffer_lexeme lexbuf; verb lexbuf )
# 2506 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_verb_rec lexbuf __ocaml_lex_state

and target lexbuf =
   __ocaml_lex_target_rec lexbuf 176
and __ocaml_lex_target_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 177 "src/octLexer.mll"
                                            chr
# 2519 "src/octLexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 480 "src/octLexer.mll"
    ( buffer_char chr; target lexbuf )
# 2523 "src/octLexer.ml"

  | 1 ->
# 482 "src/octLexer.mll"
    ( raise (LexerError(curr_loc lexbuf, Nested_target)) )
# 2528 "src/octLexer.ml"

  | 2 ->
# 484 "src/octLexer.mll"
    ( use_start_loc lexbuf;
      Target(!target_format, get_buffered_string ()) )
# 2534 "src/octLexer.ml"

  | 3 ->
# 487 "src/octLexer.mll"
    ( raise (LexerError(get_start_loc (), Unterminated_target)) )
# 2539 "src/octLexer.ml"

  | 4 ->
# 489 "src/octLexer.mll"
    ( incr_line lexbuf;
      buffer_lexeme lexbuf;
      target lexbuf )
# 2546 "src/octLexer.ml"

  | 5 ->
# 493 "src/octLexer.mll"
    ( buffer_lexeme lexbuf; target lexbuf )
# 2551 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_target_rec lexbuf __ocaml_lex_state

and code lexbuf =
   __ocaml_lex_code_rec lexbuf 189
and __ocaml_lex_code_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 177 "src/octLexer.mll"
                                            chr
# 2564 "src/octLexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 497 "src/octLexer.mll"
    ( buffer_char chr; code lexbuf )
# 2568 "src/octLexer.ml"

  | 1 ->
# 499 "src/octLexer.mll"
    ( push_inner_start_loc lexbuf;
      buffer_lexeme lexbuf;
      code lexbuf )
# 2575 "src/octLexer.ml"

  | 2 ->
# 503 "src/octLexer.mll"
    ( match pop_inner_start_loc () with
      | None ->
          use_start_loc lexbuf;
          Code(get_raw_buffered_string ())
      | Some _ ->
          buffer_lexeme lexbuf;
          code lexbuf )
# 2586 "src/octLexer.ml"

  | 3 ->
# 511 "src/octLexer.mll"
    ( match pop_inner_start_loc () with
      | None -> raise (LexerError(get_start_loc (), Unterminated_code))
      | Some l -> raise (LexerError(l, Unterminated_code)) )
# 2593 "src/octLexer.ml"

  | 4 ->
# 515 "src/octLexer.mll"
    ( incr_line lexbuf;
      buffer_lexeme lexbuf;
      code lexbuf )
# 2600 "src/octLexer.ml"

  | 5 ->
# 519 "src/octLexer.mll"
    ( buffer_lexeme lexbuf; code lexbuf )
# 2605 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_code_rec lexbuf __ocaml_lex_state

and pre_code lexbuf =
   __ocaml_lex_pre_code_rec lexbuf 200
and __ocaml_lex_pre_code_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 177 "src/octLexer.mll"
                                            chr
# 2618 "src/octLexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 523 "src/octLexer.mll"
    ( buffer_char chr; pre_code lexbuf )
# 2622 "src/octLexer.ml"

  | 1 ->
# 525 "src/octLexer.mll"
    ( raise (LexerError(curr_loc lexbuf, Nested_pre_code)) )
# 2627 "src/octLexer.ml"

  | 2 ->
# 527 "src/octLexer.mll"
    ( use_start_loc lexbuf;
      Pre_Code (get_buffered_string ()) )
# 2633 "src/octLexer.ml"

  | 3 ->
# 530 "src/octLexer.mll"
    ( raise (LexerError(get_start_loc (), Unterminated_pre_code)) )
# 2638 "src/octLexer.ml"

  | 4 ->
# 532 "src/octLexer.mll"
    ( incr_line lexbuf;
      buffer_lexeme lexbuf;
      pre_code lexbuf )
# 2645 "src/octLexer.ml"

  | 5 ->
# 536 "src/octLexer.mll"
    ( buffer_lexeme lexbuf; pre_code lexbuf )
# 2650 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_pre_code_rec lexbuf __ocaml_lex_state

and html_code lexbuf =
   __ocaml_lex_html_code_rec lexbuf 213
and __ocaml_lex_html_code_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 177 "src/octLexer.mll"
                                            chr
# 2663 "src/octLexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 540 "src/octLexer.mll"
    ( buffer_char chr; html_code lexbuf )
# 2667 "src/octLexer.ml"

  | 1 ->
# 542 "src/octLexer.mll"
    ( raise (LexerError(curr_loc lexbuf, Nested_html_code))  )
# 2672 "src/octLexer.ml"

  | 2 ->
# 544 "src/octLexer.mll"
    ( use_start_loc lexbuf;
      Code(get_raw_buffered_string ()) )
# 2678 "src/octLexer.ml"

  | 3 ->
# 547 "src/octLexer.mll"
    ( raise (LexerError(get_start_loc (), Unterminated_html_code)) )
# 2683 "src/octLexer.ml"

  | 4 ->
# 549 "src/octLexer.mll"
    ( incr_line lexbuf;
      buffer_lexeme lexbuf;
      html_code lexbuf )
# 2690 "src/octLexer.ml"

  | 5 ->
# 553 "src/octLexer.mll"
    ( buffer_lexeme lexbuf; html_code lexbuf )
# 2695 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_html_code_rec lexbuf __ocaml_lex_state

and reference lexbuf =
   __ocaml_lex_reference_rec lexbuf 234
and __ocaml_lex_reference_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 177 "src/octLexer.mll"
                                            chr
# 2708 "src/octLexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 557 "src/octLexer.mll"
    ( buffer_char chr; reference lexbuf )
# 2712 "src/octLexer.ml"

  | 1 ->
# 559 "src/octLexer.mll"
    ( use_start_loc lexbuf;
      Ref(!ref_kind, get_buffered_string ()) )
# 2718 "src/octLexer.ml"

  | 2 ->
# 562 "src/octLexer.mll"
    ( raise (LexerError(get_start_loc (), Unterminated_ref)) )
# 2723 "src/octLexer.ml"

  | 3 ->
# 564 "src/octLexer.mll"
    ( incr_line lexbuf;
      buffer_lexeme lexbuf;
      reference lexbuf )
# 2730 "src/octLexer.ml"

  | 4 ->
# 568 "src/octLexer.mll"
    ( buffer_lexeme lexbuf; reference lexbuf )
# 2735 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_reference_rec lexbuf __ocaml_lex_state

and module_list lexbuf =
   __ocaml_lex_module_list_rec lexbuf 244
and __ocaml_lex_module_list_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 177 "src/octLexer.mll"
                                            chr
# 2748 "src/octLexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 572 "src/octLexer.mll"
    ( buffer_char chr; module_list lexbuf )
# 2752 "src/octLexer.ml"

  | 1 ->
# 574 "src/octLexer.mll"
    ( if buffer_not_empty () then add_module (get_buffered_string ());
      use_start_loc lexbuf;
      Special_Ref(SRK_module_list (get_module_list ())) )
# 2759 "src/octLexer.ml"

  | 2 ->
# 578 "src/octLexer.mll"
    ( raise (LexerError(get_start_loc (), Unterminated_ref)) )
# 2764 "src/octLexer.ml"

  | 3 ->
# 580 "src/octLexer.mll"
    ( if buffer_not_empty () then begin
        add_module (get_buffered_string ());
        reset_string_buffer ()
      end;
      module_list lexbuf )
# 2773 "src/octLexer.ml"

  | 4 ->
# 586 "src/octLexer.mll"
    ( incr_line lexbuf;
      if buffer_not_empty () then begin
        add_module (get_buffered_string ());
        reset_string_buffer ()
      end;
      module_list lexbuf )
# 2783 "src/octLexer.ml"

  | 5 ->
# 593 "src/octLexer.mll"
    ( buffer_lexeme lexbuf; module_list lexbuf )
# 2788 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_module_list_rec lexbuf __ocaml_lex_state

and read_ref lexbuf =
   __ocaml_lex_read_ref_rec lexbuf 254
and __ocaml_lex_read_ref_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 596 "src/octLexer.mll"
      ( MINUS )
# 2800 "src/octLexer.ml"

  | 1 ->
# 597 "src/octLexer.mll"
      ( DOT )
# 2805 "src/octLexer.ml"

  | 2 ->
# 598 "src/octLexer.mll"
               ( Ref_part (Lexing.lexeme lexbuf) )
# 2810 "src/octLexer.ml"

  | 3 ->
# 599 "src/octLexer.mll"
      ( EOF )
# 2815 "src/octLexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_read_ref_rec lexbuf __ocaml_lex_state

;;

# 601 "src/octLexer.mll"
 

(* Initialize style hash table *)
let _ =
  List.iter
    (fun (kwd, tok) -> Hashtbl.add style_table kwd tok)
    [ ("b", Style SK_bold);
      ("e", Style SK_emphasize);
      ("C", Style SK_center);
      ("L", Style SK_left);
      ("R", Style SK_right);
      ("i", Style SK_italic);
      ("ul", LIST);
      ("ol", ENUM);
      ("li", Item false); ]

(* Initialize reference kind hash table *)
let _ =
  List.iter
    (fun (kind, tok) -> Hashtbl.add ref_kind_table kind tok)
    [ ("val", RK_value);
      ("type", RK_type);
      ("exception", RK_exception);
      ("module", RK_module);
      ("modtype", RK_module_type);
      ("class", RK_class);
      ("classtype", RK_class_type);
      ("attribute", RK_attribute);
      ("method", RK_method);
      ("section", RK_section);
      ("recfield", RK_recfield);
      ("const", RK_const); ]

(* Initialize tag hash table *)
let _ =
  List.iter
    (fun (tag, tok) -> Hashtbl.add tag_table tag tok)
    [ ("author", fun _ -> AUTHOR);
      ("deprecated", fun _ -> DEPRECATED);
      ("param", fun lexbuf -> Param (identifier lexbuf));
      ("raise", fun lexbuf -> Raise (identifier lexbuf));
      ("return", fun _ -> RETURN);
      ("inline", fun _ -> INLINE);
      ("see", fun lexbuf -> See (see lexbuf));
      ("since", fun lexbuf -> Since (version lexbuf));
      ("before", fun lexbuf -> Before (version lexbuf));
      ("version", fun lexbuf -> Version (version lexbuf));
      ("canonical", fun lexbuf -> Canonical (identifier lexbuf)); ]


# 2873 "src/octLexer.ml"
