// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

#include <string.h>

#define OPENSSL_SUPPRESS_DEPRECATED
#include <openssl/engine.h>

static const char* g_engineId = "dntest";
static const char* g_engineName = "DotNet Test ENGINE";

// Public key and its private key are named `first`.
// Their namespaces (dntest_load_privkey vs dntest_load_pubkey) allow to use
// independent unrelated keys.
// Managed tests validate we're actually loading public key when public is requested
// and that private key is used when private is requested.

// Public key in PKCS#1 format
static char g_pubRsaKey[] = {
    0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xBF,0x67,0x16,0x84,0x85,0x21,0x5A,0x6A,0xB8,
    0x9B,0xCA,0xB9,0x33,0x1F,0x6F,0x5F,0x36,0x0F,0x43,0x00,0xBE,0x5C,0xF2,0x82,0xF7,
    0x70,0x42,0x95,0x7E,0xA2,0x02,0x90,0x8B,0x22,0x79,0xF3,0x4A,0x42,0x6D,0x62,0xF5,
    0x9D,0x6C,0x10,0x56,0xE3,0x6D,0xC9,0xF6,0xEE,0xA9,0xAE,0xB1,0xB3,0x1F,0x81,0x22,
    0xF5,0x83,0xEE,0x9C,0xAE,0x2A,0x86,0xA4,0x71,0x44,0x90,0x5D,0xF0,0x54,0x41,0xB0,
    0xA5,0xF2,0x9E,0x03,0xC5,0xAC,0x18,0x88,0xD9,0x37,0x44,0xD8,0x96,0x38,0xD8,0x3A,
    0xC3,0x77,0x74,0xB3,0x39,0xE4,0xAF,0xB3,0x49,0xC7,0x14,0xB1,0x22,0x38,0xB0,0xF8,
    0x1A,0x71,0x38,0x0F,0x05,0x1C,0x58,0x5C,0xB2,0x74,0x34,0xFA,0x54,0x4B,0xDA,0xC6,
    0x79,0xE1,0xE1,0x65,0x81,0xD0,0xE9,0x02,0x03,0x01,0x00,0x01
};

// Private key in PKCS#1 format for public key above
char g_priRsaKey[] = {
    0x30,0x82,0x02,0x5C,0x02,0x01,0x00,0x02,0x81,0x81,0x00,0xBF,0x67,0x16,0x84,0x85,
    0x21,0x5A,0x6A,0xB8,0x9B,0xCA,0xB9,0x33,0x1F,0x6F,0x5F,0x36,0x0F,0x43,0x00,0xBE,
    0x5C,0xF2,0x82,0xF7,0x70,0x42,0x95,0x7E,0xA2,0x02,0x90,0x8B,0x22,0x79,0xF3,0x4A,
    0x42,0x6D,0x62,0xF5,0x9D,0x6C,0x10,0x56,0xE3,0x6D,0xC9,0xF6,0xEE,0xA9,0xAE,0xB1,
    0xB3,0x1F,0x81,0x22,0xF5,0x83,0xEE,0x9C,0xAE,0x2A,0x86,0xA4,0x71,0x44,0x90,0x5D,
    0xF0,0x54,0x41,0xB0,0xA5,0xF2,0x9E,0x03,0xC5,0xAC,0x18,0x88,0xD9,0x37,0x44,0xD8,
    0x96,0x38,0xD8,0x3A,0xC3,0x77,0x74,0xB3,0x39,0xE4,0xAF,0xB3,0x49,0xC7,0x14,0xB1,
    0x22,0x38,0xB0,0xF8,0x1A,0x71,0x38,0x0F,0x05,0x1C,0x58,0x5C,0xB2,0x74,0x34,0xFA,
    0x54,0x4B,0xDA,0xC6,0x79,0xE1,0xE1,0x65,0x81,0xD0,0xE9,0x02,0x03,0x01,0x00,0x01,
    0x02,0x81,0x81,0x00,0x84,0xED,0x88,0x62,0xF2,0xBE,0xAE,0x37,0xCE,0x0C,0x4C,0xA7,
    0x80,0x8C,0xC5,0x61,0x5F,0x7F,0x0B,0xEE,0x99,0x46,0x9E,0x1A,0x3C,0xD4,0x97,0x39,
    0x91,0xDF,0xDC,0x5E,0x1C,0x73,0x0E,0x34,0xDC,0x0E,0xF4,0x3F,0x35,0x0B,0x66,0x80,
    0x96,0x87,0x8E,0xB9,0x24,0x28,0xAE,0x69,0xA7,0xFA,0x19,0xD8,0x2A,0xBA,0x4E,0x2D,
    0x4A,0x5D,0x5F,0x24,0x3D,0x4B,0x73,0x46,0x73,0x4D,0x70,0x5C,0x4C,0x49,0x4F,0xE2,
    0xB3,0x6E,0x2E,0x35,0xC3,0x9E,0xE0,0x8B,0xFB,0x11,0x72,0xF5,0xAB,0x08,0x4A,0xF4,
    0x4B,0xD4,0xD0,0x37,0x02,0xD0,0x4E,0x64,0x69,0xF0,0x26,0xEF,0x37,0x49,0xCB,0xED,
    0x3E,0xCB,0x31,0x07,0x46,0xCF,0x49,0xDA,0x3C,0x27,0x85,0xCC,0x17,0xD5,0x42,0x15,
    0xEF,0x18,0xF3,0xED,0x02,0x41,0x00,0xD0,0x63,0xF8,0x9E,0x01,0xEB,0x68,0x1C,0xEA,
    0xCB,0x78,0x1F,0xE8,0x07,0xF8,0x7C,0x70,0x2B,0x52,0x2A,0x76,0xB7,0xD0,0xE0,0x6D,
    0xA4,0x4B,0xB7,0xD6,0x20,0x2D,0x5E,0x9F,0x3E,0x7B,0xE5,0xBC,0xCC,0x3B,0x32,0xB9,
    0xB2,0x93,0xAB,0x62,0xF5,0x0A,0x84,0x17,0xC2,0xFA,0x9D,0x6A,0x76,0xE4,0x65,0xAA,
    0x96,0x2A,0xB6,0x1A,0x8A,0x9A,0x13,0x02,0x41,0x00,0xEB,0x21,0x8F,0x00,0xB7,0x31,
    0x7C,0xC6,0x25,0xDF,0x2D,0xFB,0x71,0x81,0x1D,0xC5,0xDA,0x91,0xD9,0xA2,0xAD,0x85,
    0x92,0x82,0xDC,0xA6,0xBA,0x3B,0x4C,0x67,0x48,0x97,0xE9,0xD0,0x3D,0x9E,0x5F,0xD2,
    0xA9,0xFD,0x4C,0xE7,0xD9,0xA3,0xE5,0xB7,0x9E,0x94,0x84,0x29,0xC2,0x15,0x61,0xE7,
    0x14,0x1D,0x90,0xBC,0xA7,0x57,0x33,0xD2,0x48,0x93,0x02,0x40,0x0D,0x07,0xD3,0x49,
    0xFE,0x10,0xBC,0x47,0xE2,0x9E,0xAA,0x7A,0x44,0x46,0x0B,0x51,0xAC,0xA9,0xE8,0xCF,
    0x62,0xF1,0x07,0x8C,0xA1,0x0E,0x7E,0xF5,0x95,0xDC,0x19,0x3A,0x2B,0x76,0xFA,0xC4,
    0x58,0xD3,0xE4,0x77,0xBD,0x88,0xDF,0x16,0xFE,0x6F,0x18,0x23,0x3E,0x61,0x20,0xCE,
    0xAB,0x13,0x98,0x20,0x8B,0x54,0x2C,0x83,0x8A,0x91,0x54,0x25,0x02,0x40,0x78,0x82,
    0x61,0x9D,0x97,0x46,0xA8,0xD1,0x91,0x95,0x7A,0x26,0xB5,0xFC,0xDB,0xFA,0x8C,0xD4,
    0x55,0xBB,0xF7,0xBD,0x4E,0xE2,0xFD,0x1E,0x02,0xB2,0xE3,0xAC,0xC7,0xDA,0xFC,0x3D,
    0xFB,0x66,0xD1,0x6B,0xD2,0x2D,0xFD,0x9D,0x92,0xC1,0x5A,0xBA,0x2A,0x6F,0xA9,0xF1,
    0x11,0x05,0x0E,0x81,0x75,0xA0,0xD5,0x8E,0xAB,0x21,0x99,0x70,0xBC,0x3B,0x02,0x40,
    0x4D,0xBF,0x36,0xE5,0xDC,0xBF,0x02,0x7A,0xD4,0xED,0x57,0x2E,0x6F,0x5F,0x83,0x83,
    0xC0,0x8C,0xD5,0x83,0x8C,0x0C,0xAE,0x16,0xFA,0x58,0xEE,0x5C,0x5A,0x38,0x8B,0x28,
    0x7F,0x9C,0x58,0x64,0x7D,0x58,0x60,0x9B,0x03,0x91,0x2A,0x10,0xD0,0xC7,0x72,0xA3,
    0x25,0x9D,0x39,0x65,0x1C,0xD1,0xEE,0xB3,0xA2,0x0C,0x5F,0x9A,0xE5,0x8E,0x18,0xC0
};

static EVP_PKEY* load_priv(BIO* bio)
{
    RSA* rsaKey = d2i_RSAPrivateKey_bio(bio, NULL);
    if (!rsaKey)
    {
        printf("%s: Error loading RSA Private Key\n", g_engineId);
        return NULL;
    }

    EVP_PKEY* key = EVP_PKEY_new();
    if (!EVP_PKEY_assign_RSA(key, rsaKey))
    {
        printf("%s: Error assigning RSA Private Key to EVP_PKEY\n", g_engineId);

        // Assignment has failed we need to free both keys separately
        RSA_free(rsaKey);
        EVP_PKEY_free(key);
        return NULL;
    }

    // Per documentation EVP_PKEY_assign_RSA sets the referenced RSA key to supplied key internally
    // and so key will be freed when the parent pkey is freed.
    return key;
}

static EVP_PKEY* load_pub(BIO* bio)
{
    RSA *rsaKey = d2i_RSAPublicKey_bio(bio, NULL);
    if (!rsaKey)
    {
        printf("%s: Error loading RSA Public Key\n", g_engineId);
        return NULL;
    }

    EVP_PKEY* key = EVP_PKEY_new();
    if (!EVP_PKEY_assign_RSA(key, rsaKey)) 
    {
        printf("%s: Error assigning RSA Public Key to EVP_PKEY\n", g_engineId);

        // Assignment has failed we need to free both keys separately
        RSA_free(rsaKey);
        EVP_PKEY_free(key);
        return NULL;
    }

    // Per documentation EVP_PKEY_assign_RSA sets the referenced RSA key to supplied key internally
    // and so key will be freed when the parent pkey is freed.
    return key;
}

static EVP_PKEY* load_key(
    const char* keyId,
    const char* key,
    int keyLength,
    EVP_PKEY* (*load_func)(BIO* bio))
{
    printf("%s: Key load request for '%s'. Key length=%d\n", g_engineId, keyId, keyLength);
    EVP_PKEY* ret = NULL;
    if (keyId != NULL)
    {
        if (strcmp(keyId, "first") == 0)
        {
            BIO* bio = BIO_new_mem_buf(key, keyLength);
            if (bio != NULL)
            {
                ret = load_func(bio);
                BIO_free(bio);
            }
        }
        else
        {
            printf("%s: Key load request failed for '%s'\n", g_engineId, keyId);
        }
    }

    return ret;
}

static EVP_PKEY* dntest_load_privkey(
    ENGINE* engine,
    const char* keyId,
    UI_METHOD* ui_method,
    void* callback_data)
{
    printf("%s: Request to load private key for %s\n", g_engineId, keyId);
    return load_key(keyId, g_priRsaKey, sizeof(g_priRsaKey), load_priv);
}

static EVP_PKEY* dntest_load_pubkey(
    ENGINE* engine,
    const char* keyId,
    UI_METHOD* ui_method,
    void* callback_data)
{
    printf("%s: Request to load public key for %s\n", g_engineId, keyId);
    return load_key(keyId, g_pubRsaKey, sizeof(g_pubRsaKey), load_pub);
}

static int bind(ENGINE* engine, const char* id)
{
    int ret = 1;

    if (!ENGINE_set_id(engine, g_engineId) ||
        !ENGINE_set_name(engine, g_engineName) ||
        !ENGINE_set_load_privkey_function(engine, dntest_load_privkey) ||
        !ENGINE_set_load_pubkey_function(engine, dntest_load_pubkey))
    {
        ret = 0;
    }

    return ret;
}

IMPLEMENT_DYNAMIC_BIND_FN(bind)
IMPLEMENT_DYNAMIC_CHECK_FN()
