import { Poll } from '@lumino/polling';
import { ISignal } from '@lumino/signaling';
import { ServerConnection } from '../serverconnection';
import * as KernelSpec from './kernelspec';
import * as restapi from './restapi';
import { BaseManager } from '../basemanager';
/**
 * An implementation of a kernel spec manager.
 */
export declare class KernelSpecManager extends BaseManager implements KernelSpec.IManager {
    /**
     * Construct a new kernel spec manager.
     *
     * @param options - The default options for kernel.
     */
    constructor(options?: KernelSpecManager.IOptions);
    /**
     * The server settings for the manager.
     */
    readonly serverSettings: ServerConnection.ISettings;
    /**
     * Test whether the manager is ready.
     */
    get isReady(): boolean;
    /**
     * A promise that fulfills when the manager is ready.
     */
    get ready(): Promise<void>;
    /**
     * Get the most recently fetched kernel specs.
     */
    get specs(): restapi.ISpecModels | null;
    /**
     * A signal emitted when the specs change.
     */
    get specsChanged(): ISignal<this, restapi.ISpecModels>;
    /**
     * A signal emitted when there is a connection failure.
     */
    get connectionFailure(): ISignal<this, Error>;
    /**
     * Dispose of the resources used by the manager.
     */
    dispose(): void;
    /**
     * Force a refresh of the specs from the server.
     *
     * @returns A promise that resolves when the specs are fetched.
     *
     * #### Notes
     * This is intended to be called only in response to a user action,
     * since the manager maintains its internal state.
     */
    refreshSpecs(): Promise<void>;
    /**
     * Execute a request to the server to poll specs and update state.
     */
    protected requestSpecs(): Promise<void>;
    private _isReady;
    private _connectionFailure;
    private _pollSpecs;
    private _ready;
    private _specs;
    private _specsChanged;
}
/**
 * The namespace for `KernelManager` class statics.
 */
export declare namespace KernelSpecManager {
    /**
     * The options used to initialize a KernelManager.
     */
    interface IOptions extends BaseManager.IOptions {
        /**
         * When the manager stops polling the API. Defaults to `when-hidden`.
         */
        standby?: Poll.Standby;
    }
}
