## Blockmap lib

Create Blockmap indexes for dirs

```kotlin
// specify where metadata will be stored
val indexManager = IndexManager("path/to/BLOCKMAP_DATA")
val localDirs = indexManager.register(localPaths)
```

Create a release to upload to a server or to use with a local one

```kotlin
createRelease(newPath, "path/to/BLOCKMAP_RELEASE")
```

Create a new release with blockmap indexes and partial downloading
```kotlin
// specify how to retrieve release: URL + path
val desiredVersion = DesiredVersion("https://s3.us-east-2.amazonaws.com" ,"cwm-213.4198-unpacked/release")
DirMerger(localDirs, desiredVersion, outputPath).merge()
```

### Notes
- FileCache: uses a strategy similar to the branch predictor
- Each LocalCopyTask has its FileCache. We can remove synchronized parts.
  - Num of simultaneously opened files: WRITE_PARALLEL_LEVEL * FILE_CACHE_SIZE (5 * 2 = 10)
- We create RandomAccessFile with proper size in advance
  - It is harder to verify resulted file size. Maybe count local bytes + net bytes?

### TODO
- Remove compressed chunk index and place it in BlockMap index
  - use sha-256 and PGP sign
  - describe compressed chunks offset and len in BlockMap Index
- We shouldn't depend on compression algo
  - describe it in the release?
  - create an interface for choosing compression
  - fix tests after this
    - IndexManagerTests
    - Merge with corrupted release tests
- Use apache client for requests
  - check that it polls connections
- Think about how to handle failed requests. The current approach is to repeat 10 times.
- Add statistics for
  - num of requests
  - num of bytes retrieved from the internet
  - num of file misses in
- Research chunk sizes: in our codebase it less than in the original paper
- HANDLE SYMLINKS!!!!!!1!1
- Check that block params are the same as yours
- Use transactions when you fail
  - outputPath/temp dir and move to parent on completion
- Download the whole file when hashes don't match
  - How will you do it since the release is compressed? Do you download zipped release?
- make logging not via println



### The whole example

```kotlin
package com.jetbrains.rd.platform.codeWithMe.blockmap

import java.io.File


private fun merge(localPaths: List<String>, outputPath: String) {
  val indexManager = IndexManager("/Users/denis.konoplev/BLOCKMAP_DATA")

  val localDirs = indexManager.register(localPaths)

  val desiredVersion = DesiredVersion("https://s3.us-east-2.amazonaws.com" ,"cwm-213.4198-unpacked/release")
  DirMerger(localDirs, desiredVersion, outputPath).merge()
}


private fun mergeRunner() {
  val localPaths = listOf("/Users/denis.konoplev/Downloads/CodeWithMeGuest-213.1495",
        "/Users/denis.konoplev/Downloads/CodeWithMeGuest-213.1497")
  val outputPath = "/Users/denis.konoplev/Downloads/JDK_result"
  
  File(outputPath).deleteRecursively()
  merge(localPaths, outputPath)
  File(outputPath).deleteRecursively()
}

fun main() {
  val release = true
  if (release) {
    val newPath = "/Users/denis.konoplev/Downloads/CodeWithMeGuest-213.1498"
    createRelease(newPath, "/Users/denis.konoplev/BLOCKMAP_RELEASE")
  }
  mergeRunner()
}
```