/* tree representation
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#ifndef __TREEREP_H__
#define __TREEREP_H__

/* the operation types for node_op */
typedef enum { op_undefined,
    op_and, op_or, op_xor,
    op_smaller, op_larger, op_smaller_or_equal, op_larger_or_equal,
    op_equal, op_not_equal, op_minus, op_plus,
    op_shift_left, op_shift_right,
    op_times, op_divide, op_modulo,
    op_mnot, op_mminus, op_mplus,
    op_test_end, op_test_assert, op_check
} op_t;

extern string op_name[];
extern string op_label[];

extern int op_inverted[];

extern boolean op_commutates[];

boolean is_monop(op_t x);

int operator_index(string s, boolean monop);


/* the assembler codes for a node_asm */
typedef enum { opcode_undefined,
    opcode_addlw, opcode_andlw, opcode_call, opcode_clrwdt,
    opcode_goto,
    opcode_iorlw, opcode_movlw, opcode_option, opcode_retfie,
    opcode_retlw,
    opcode_return, opcode_sleep, opcode_sublw, opcode_tris,
    opcode_xorlw,
    opcode_addwf, opcode_andwf, opcode_clrf, opcode_clrw, opcode_comf,
    opcode_addwfc, opcode_lfsr,
    opcode_decf, opcode_decfsz, opcode_incf, opcode_incfsz,
    opcode_retiw,
    opcode_iorwf, opcode_movf, opcode_movwf, opcode_nop,
    opcode_rlf, opcode_rrf, opcode_subwf, opcode_swapf, opcode_xorwf,
    opcode_rlncf, opcode_rrncf, opcode_subfwb, opcode_subwfb,
    opcode_bcf, opcode_bsf, opcode_btfsc, opcode_btfss,
    opcode_skpz, opcode_skpnz, opcode_setz, opcode_clrz,
    opcode_skpc, opcode_skpnc, opcode_setc, opcode_clrc,
    opcode_skpdc, opcode_skpndc, opcode_setdc, opcode_clrdc,
    opcode_movfw, opcode_tstf, opcode_bank, opcode_page, opcode_hpage,
    opcode_mullw, opcode_mulwf, opcode_cpfseq, opcode_cpfsgt,
    opcode_cpfslt,
    opcode_dcfsnz, opcode_infsnz,
    opcode_negf, opcode_setf, opcode_tstfsz, opcode_btg,
    opcode_daw, opcode_pop, opcode_push, opcode_reset,
    opcode_tblrd, opcode_tblrd_postinc, opcode_tblrd_postdec, opcode_tblrd_preinc,
    opcode_tblwt, opcode_tblwt_postinc, opcode_tblwt_postdec, opcode_tblwt_preinc,
    opcode_movlb,
    opcode_bc, opcode_bn, opcode_bnc, opcode_bnn, opcode_bnov,
    opcode_bnz, opcode_bov, opcode_bra, opcode_bz, opcode_rcall,
    opcode_a2nd, opcode_illegal,
    opcode_f2nd,
    opcode_bankA, opcode_bankB
} opcode_t;
#define opcode_value \
   ( ( target_cpu == pic_16 ) ? opcode_value_16 : ( \
      ( target_cpu == pic_14 ) ? opcode_value_14 : opcode_value_12 ))
extern int opcode_value_16[];
extern int opcode_value_12[];

extern int opcode_value_14[];

extern string opcode_name[];

/* opcode characterisation */
#define field_file         1
#define field_dest         2
#define field_label        4
#define field_const        8
#define field_tris        16
#define field_bit         32
#define field_fsr         64
#define field_flabel     128
#define field_fd          ( field_file | field_dest )
#define field_fb          ( field_file | field_bit )
#define field_ff          ( field_fsr | field_flabel )
extern int opcode_flags[];

extern boolean code_has(int c, int f);

#define dest_w 'w'
#define dest_f 'f'

#define mode_none 0
#define mode_in   1
#define mode_out  2

extern boolean mode_has(int c, int m);


/* assembler opcodes corresponding to a few high-level operators */
#define opcode_from_const \
   ( ( target_cpu == pic_16 ) ? opcode_from_const_16 : ( \
    ( target_cpu == pic_14 ) ? opcode_from_const_14 : opcode_from_const_12 ))
extern opcode_t opcode_from_const_12[];

extern opcode_t opcode_from_const_14[];

extern opcode_t opcode_from_const_16[];

extern opcode_t opcode_from_store[];

/* the node types */
typedef enum { node_undefined,
    node_chain, node_decl, node_ref, node_value,
    node_procedure, node_var, node_type, node_const,
    node_assign, node_if, node_while, node_for,
    node_op, node_asm, node_label, node_precall,
    node_return, node_call, node_test, node_org,
    node_error, node_w
} node_t;

extern string node_name[];

/* types of references */
typedef enum { ref_undefined,
    ref_const,
    ref_var,
    ref_own,                    /* ref to virtual var from within the put or get */
    ref_formal,                 /* ref to or from the parameters inside the subprogram */
    ref_actual                  /* ref to or from the parameters in the call */
} ref_t;

extern string ref_name[];


/********** node data:
 * all: nr, loc, kind, name
 * node_chain( first->*, next->[chain] )
 * node_assign( first->ref, next->exp, type )
 * node_ref( first->decl )
 * node_if( condition->exp, first->chain, next->chain )
 * node_while( condition->exp, first->chain )
 * node_for( var, start, end, step, first->chain, type )
 * node_op( op, first->exp, next->exp, type )
 * node_asm( opcode, first, next )
 * node_call( first->label )
 * node_label( first )
 * node_ref( first->object )
 * node_type( )
 * node_var( type, value )
 * node_const( type, value )
 * node_procedure( first, next, type )
 * object = type | const | var | procedure | function
 * exp    = chain | op | ref
 *
 * Only during parsing the node_decl( prev_decl ) are a backward tree,
 * last_decl points to the current lowest decl of the current block,
 * and only the first decl of a block has first==true.
 */
extern int generation;
typedef struct node_struct *tree;
typedef struct node_struct {
    int generation;             /* used by recursive tree walkers */
    int data;                   /* used by recursive tree walkers */
    int nr;                     /* number to identify an individual node */
    loc_t loc;                  /* source location which created the node */
    node_t kind;                /* kind of node */

    tree first;                 /* left subtree */
    tree next;                  /* right subtree */
    tree condition;             /* condition for node_if and node_while */
    tree prev_decl;             /* previous declaration for a node_decl */
    tree type;                  /* type when node has a type */
    tree address;               /* mapping to target address */
    tree ret;                   /* return instruction pointer */
    tree actual;                /* for arguments: used during call parsing */
    tree var;                   /* for node_for only */
    tree start;                 /* for node_for only */
    tree end;                   /* for node_for only */
    tree step;                  /* for node_for only */
    tree value;                 /* the current value of a node, if any */
    tree impl;                  /* implementation of an operator */
    tree result;                /* result var for functions and operators */
    tree get;                   /* for variables only */
    tree put;                   /* for variables only */
    tree transfer1;             /* pre transfer code for a virtual */
    tree transfer2;             /* post transfer code for a virtual */
    tree tlabel;                /* transfer entry point for a put */
    tree call_label;            /* sx_12 call entry point */

    tree master1;               /* variable which inherits byte address */
    tree master2;               /* variable which inherits bit address */
    tree uncle;                 /* variable which inherits everything */

    tree jtaddress;             /* index into the jump table */

    int x;                      /* for a node_value: the value */
    int large;                  /* for node_type : the largest value */
    int bits;                   /* for node_type : the nr of bits */
    op_t op;                    /* operator for node_op */
    char *name;                 /* for node_type, node_var, node_func, node_proc */
    char *label;                /* for node_proc */
    boolean known;              /* indicates whether a value is known */
    boolean is_volatile;        /* indicates whether a store is volatile */
    int used;                   /* indicates how often the node is used */
    opcode_t opcode;            /* for node_asm: the opcode */
    char dest;                  /* for node_asm: the destination (f/w) */
    int mode;                   /* for parameters: in, out or in | out */
    boolean fixed;              /* for fixed-address variables */
    boolean first_decl;         /* true for first decl of a block */
    boolean indirect;           /* for node_call and node_ref */
    ref_t ref;                  /* node_ref: kind of reference */
    boolean last_page;          /* force this code to the last page */
    boolean is_virtual;         /* for virtual procedures / functions */
    boolean is_argument;        /* for arguments only */
    boolean is_chained;         /* for chained procedures & calls */
    boolean is_target;          /* for node_ref */
    boolean is_gathered;        /* for gathering the virtual procedures */
    boolean has_return;         /* for procedures and unctions */
    boolean has_transfer;       /* indicates whether a virtual is also used virtual */
    boolean is_interrupt;       /* for a node_procedure only */
    boolean is_raw;             /* for an interrupt only */
    boolean sacred;             /* protects page and bank opcodes */
    boolean fizzle;             /* marks removed instructions */
    int free_bit;               /* for ralloc */
    int free_byte;              /* for ralloc */
    int bank;                   /* for removing bank instructions */
    boolean pass_in_w;          /* in-w parameter passing */
    boolean read_allowed;
    boolean write_allowed;
    boolean generated;

    tree chain;                 /* for the garbage-collection */
    tree duplicate;             /* for cloning */
    boolean done;
} node;

#define for_each_subtree
#define expand_for_all_subtrees   \
   for_each_subtree( first      ) \
   for_each_subtree( next       ) \
   for_each_subtree( condition  ) \
   for_each_subtree( prev_decl  ) \
   for_each_subtree( type       ) \
   for_each_subtree( address    ) \
   for_each_subtree( ret        ) \
   for_each_subtree( actual     ) \
   for_each_subtree( var        ) \
   for_each_subtree( start      ) \
   for_each_subtree( end        ) \
   for_each_subtree( step       ) \
   for_each_subtree( value      ) \
   for_each_subtree( impl       ) \
   for_each_subtree( result     ) \
   for_each_subtree( get        ) \
   for_each_subtree( put        ) \
   for_each_subtree( transfer1  ) \
   for_each_subtree( transfer2  ) \
   for_each_subtree( tlabel     ) \
   for_each_subtree( call_label ) \
   for_each_subtree( jtaddress  ) \


#define assert_kind( loc, p, x ){ \
   assert_pointer( loc, p ); \
   if( p->kind != x ){ \
      snark_node( loc, p ); \
   } \
} \

/* allocate a new node of the indicated kind */
extern int node_nr;
extern tree node_root;
extern tree free_nodes;
tree new_node(loc_t loc, int kind);


/* forward */
void show_subtree(tree p);
#define trace_subtree( p ) { \
   log(( m , "tree start" )); \
   show_subtree( p ); \
   log(( m , "tree end" )); \
}

void mark_reachable(tree p);


void garbage_collection(tree p);


int variable_address(tree p);

int constant_value(tree p);

boolean node_is_constant(tree p, int x);

boolean node_is_some_constant(tree p);


/* global nodes: jal build-in types */
extern tree type_universal, type_bit, type_byte, type_char, type_word, type_long;

/* the transfer variables */
extern tree transfer_bit, transfer_byte;

/* global nodes: some constants */
extern tree const_zero, const_one, const_two;

tree transfer_variable(tree t);

tree universal_result(op_t op);


/********** allocators **********/

tree new_chain(tree q);


tree new_chain2(tree q1, tree q2);

tree new_chain3(tree q1, tree q2, tree q3);

tree new_chain4(tree q1, tree q2, tree q3, tree q4);

tree new_chain5(tree q1, tree q2, tree q3, tree q4, tree q5);


tree new_chain6(tree q1, tree q2, tree q3, tree q4, tree q5, tree q6);


tree new_chain7(tree q1, tree q2, tree q3, tree q4, tree q5, tree q6, tree q7);

tree new_chain8(tree q1, tree q2, tree q3, tree q4, tree q5, tree q6, tree q7, tree q8);

tree new_chain9(tree q1, tree q2, tree q3, tree q4, tree q5, tree q6, tree q7, tree q8, tree q9);

tree new_chain10(tree q1, tree q2, tree q3, tree q4, tree q5, tree q6, tree q7, tree q8, tree q9,
                 tree q10);


tree new_label(loc_t loc, string m, int n, string o);
tree new_error(loc_t loc);
tree new_value(tree type, int value);
tree new_org(int value);
tree new_const(tree v);
tree new_address_value(int value1, int value2);
tree new_decl(loc_t loc, tree first);
tree new_return(loc_t loc, tree first);
tree new_w(loc_t loc, int mode);
tree new_call(loc_t loc, tree q);
tree new_precall(loc_t loc, tree q);

#define chip_without_code_pages ( \
   ( target_chip == t_16f84 ) \
   || ( target_chip == t_16c84 ) \
   || ( target_chip == t_12c508 ) \
   || ( target_cpu == pic_16 ))

#define chip_without_memory_banks ( \
   ( target_chip == t_12c508 ) \
   || ( target_chip == t_16c84 ) \  
   || ( target_chip == t_16f84 ))

#define chip_with_call_vectors ( \
   ( target_chip == t_sx18 ) \
   || ( target_chip == t_sx28 ) \
   || ( target_chip == t_12c509a ) \
   || ( target_chip == t_12c508 ) \
   || ( target_chip == t_12ce674 ))

tree new_icall(loc_t loc, tree q, int x);

tree new_var(loc_t loc, char *name, tree type);

/* this one is not used! */
tree new_proc(char *name, tree first, tree next);

tree new_ref(loc_t loc, ref_t ref, tree t, tree proc);

tree new_indirect(loc_t loc, tree t);

tree new_op(loc_t loc, tree t, op_t op, tree left, tree right);

tree new_asm(loc_t loc, opcode_t opcode, tree arg1, int arg2);

#endif
