//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// G4SmoothTrajectoryPoint class implementation
//
// Contact:
//   Questions and comments to this code should be sent to
//     Katsuya Amako  (e-mail: Katsuya.Amako@kek.jp)
//     Makoto  Asai   (e-mail: asai@slac.stanford.edu)
//     Takashi Sasaki (e-mail: Takashi.Sasaki@kek.jp)
// --------------------------------------------------------------------

#include "G4SmoothTrajectoryPoint.hh"

#include "G4AttDef.hh"
#include "G4AttDefStore.hh"
#include "G4AttValue.hh"
#include "G4UnitsTable.hh"

// #define G4ATTDEBUG
#ifdef G4ATTDEBUG
#  include "G4AttCheck.hh"
#endif

G4Allocator<G4SmoothTrajectoryPoint>*& aSmoothTrajectoryPointAllocator()
{
  G4ThreadLocalStatic G4Allocator<G4SmoothTrajectoryPoint>* _instance = nullptr;
  return _instance;
}

G4SmoothTrajectoryPoint::G4SmoothTrajectoryPoint(G4ThreeVector pos) { fPosition = pos; }

G4SmoothTrajectoryPoint::G4SmoothTrajectoryPoint(
  G4ThreeVector pos, std::vector<G4ThreeVector>* auxiliaryPoints)
  : fPosition(pos), fAuxiliaryPointVector(auxiliaryPoints)
{}

G4SmoothTrajectoryPoint::G4SmoothTrajectoryPoint(const G4SmoothTrajectoryPoint& right)
  : G4VTrajectoryPoint(),
    fPosition(right.fPosition), fAuxiliaryPointVector(right.fAuxiliaryPointVector)
{}

G4SmoothTrajectoryPoint::~G4SmoothTrajectoryPoint() { delete fAuxiliaryPointVector; }

const std::map<G4String, G4AttDef>* G4SmoothTrajectoryPoint::GetAttDefs() const
{
  G4bool isNew;
  std::map<G4String, G4AttDef>* store =
    G4AttDefStore::GetInstance("G4SmoothTrajectoryPoint", isNew);
  if (isNew) {
    G4String Pos("Pos");
    (*store)[Pos] = G4AttDef(Pos, "Step Position", "Physics", "G4BestUnit", "G4ThreeVector");
    G4String Aux("Aux");
    (*store)[Aux] =
      G4AttDef(Aux, "Auxiliary Point Position", "Physics", "G4BestUnit", "G4ThreeVector");
  }
  return store;
}

std::vector<G4AttValue>* G4SmoothTrajectoryPoint::CreateAttValues() const
{
  auto values = new std::vector<G4AttValue>;

  if (fAuxiliaryPointVector != nullptr) {
    for (const auto& iAux : *fAuxiliaryPointVector) {
      values->push_back(G4AttValue("Aux", G4BestUnit(iAux, "Length"), ""));
    }
  }

  values->push_back(G4AttValue("Pos", G4BestUnit(fPosition, "Length"), ""));

#ifdef G4ATTDEBUG
  G4cout << G4AttCheck(values, GetAttDefs());
#endif

  return values;
}
