/*******************************************************************************
* File:   		wepserver.c			                                           
* Date:   		2004-05-21												       
* Author: 		Fernando Tarn					    						   
* Last Modified:	2004-5-23					       						   
*																		       
* Description: Manage server mode.										       
*																		       
*  This program is free software; you can redistribute it and/or modify	       
*  it under the terms of the GNU General Public License as published by	       
*  the Free Software Foundation; either version 2 of the License, or	       
*  (at your option) any later version.									       
*																		       
*  This program is distributed in the hope that it will be useful,		       
*  but WITHOUT ANY WARRANTY; without even the implied warranty of	    	   
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	       	   
*  GNU Library General Public License for more details.			       		   
*																		       
*  You should have received a copy of the GNU General Public License	       
*  along with this program; if not, write to the Free Software			       
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
********************************************************************************/

#include <sys/socket.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <netdb.h>

#ifndef __CYGWIN__
#include <resolv.h>
#endif

#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <strings.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>


#include "messages.h"
#include "wepserver.h"
#include "md5_digest.h"
#include "log.h"

//Some conf values
#define BLOCK_SIZE "4294967296"

#ifndef __CYGWIN__
#define DEFAULT_LOG_FILE "/var/tmp/wepdecrypt_s.log"
#else
#define DEFAULT_LOG_FILE "wd_server.log"
#endif

#define PORT 3617
#define VERBOSITY 2
#define MAXBUF 99

//Verbosity levels
#define LOW 1
#define MEDIUM 2
#define HIGH 3

int socket_server;
unsigned int current_key64[5], current_key128[13];
unsigned int block_key[5];
hosts_list list;
server_data s_data;

void parse_path_file(char * packet_file){
	int i,j=0,cont=0;
	
	for(i=0;i<strlen(packet_file);i++)
		if (packet_file[i] == '/') cont++;
	i=0;
	if (cont !=0){
		while (j < cont){
			s_data.path[i] = packet_file[i];
			if (packet_file[i] == '/') j++;
			i++;
		}
		s_data.path[i]='\0'; j=0;
	}
	else strncpy(s_data.path,"./",2);
	for(;i<strlen(packet_file);i++){
		s_data.file_name[j] = packet_file[i];
		j++;
	}
	s_data.file_name[j]='\0';		
}

int parse_config_file(char * login, int * port){
	int i,j, verbosity;
	char block_size[15];
	char path[35], line[75], conf_setting[20], conf_value[55], log_file[55];
	FILE *fp;
	
	//set path
#ifndef __CYGWIN__
	if (strncmp(login, "root", 4) == 0) sprintf(path,"/root/.wepdecrypt");
	else sprintf(path,"/home/%s/.wepdecrypt", login);
#else
	sprintf(path,"wepdecrypt.conf", login);	
#endif
	

	(*port) = -1; block_size[0] = '\0'; verbosity = -1; log_file[0] = '\0';
	
	//create config file
	if ((fp = fopen(path,"r")) == NULL){
		sleep(0.3);
		if ((fp = fopen(path,"w+")) == NULL){			
			return 0;
		}
		fprintf(fp,"#Wepdecrypt configuration file\n");
		fprintf(fp,"#Server listening port\nPort=%i\n\n#Blocksize can be set up to 1099511627775\nBlocksize=%s\n\n#Full path to server logfile\nLogfile=%s\n\n#Verbosity levels: 1 - LOW 2 - MEDIUM 3 - HIGH\nVerbosity=%i\n\n", PORT, BLOCK_SIZE, DEFAULT_LOG_FILE, VERBOSITY);		
		(*port) = PORT;
		if (!BN_dec2bn(&s_data.block_size, BLOCK_SIZE)) return 0;
		s_data.verbosity = VERBOSITY;
		strncpy(s_data.log_file, DEFAULT_LOG_FILE, strlen(DEFAULT_LOG_FILE)+1);
		fclose(fp);
		return 1;
	}
	//parse config file
	else{
		while(!feof(fp)){			
			i = 0; j = 0;
			while (i < 74){
				line[i] = fgetc(fp);
				if (line[i] == '\n') break;
				i++;
			}
			line[i] = '\0';			
			if (line[0] == '#' || line[0] == '\n' || line[0] <= '\0') continue;
			i = 0;			
			while ((line[i] != '=') && (i < 19)){
				conf_setting[i] = line[i];
				i++;
			}			
			conf_setting[i] = '\0'; i++; 			
			while ((line[i] != '\0') && (i < 75) && (j < 54)){				
				conf_value[j] = line[i];
				j++; i++;
			}
			conf_value[j] = '\0';
						
			if (strncmp(conf_setting,"Port",4) == 0) (*port) = atoi(conf_value);
			else if (strncmp(conf_setting,"Blocksize",9) == 0) strncpy(block_size, conf_value, strlen(conf_value)+1);
			else if (strncmp(conf_setting,"Verbosity",9) == 0) verbosity = atoi(conf_value);
			else if (strncmp(conf_setting,"Logfile",7) == 0) strncpy(log_file, conf_value, strlen(conf_value)+1);
			else if (strncmp(conf_setting,"Guidirlog",9) == 0);
			else{
				//Bad data in config file
				fclose(fp);
				return 0;
			}
		}		
	}
	fclose(fp);
	if (log_file[0] != '\0') strncpy(s_data.log_file, log_file, strlen(log_file)+1);
	if (((*port) == -1) || block_size[0] == 0 || verbosity == -1 || log_file[0] == '\0' || verbosity < LOW || verbosity > HIGH){
		return 0;
		}	
	if (!BN_dec2bn(&s_data.block_size, block_size)) return 0;
	s_data.verbosity = verbosity;		
	return 1;
}

int parse_client_data(const unsigned char * client_recv, client_data * c_data, int size){	
	//Make sure client data is ok , and do the casting
	if (size < sizeof(client_data)){
		c_data->option = BAD_DATA;
		return -1;
	}
	else {		
		memcpy(c_data,client_recv,sizeof(client_data));
		if (c_data->option != GET_KEYS && c_data->option != GET_FILE && c_data->option != DECRYPTED_KEYS && c_data->option != DECRYPTED_BLOCK){
			c_data->option = BAD_DATA;
			return -1;
		}
		return 0;
	}
}
	
void store_decrypted_key(char * log_file, client_data c_data, int mode){
	server_log_key(log_file, c_data.decrypted_key, c_data.decrypted_bssid, mode);		
}

void update_current_key(unsigned int * current_key, const unsigned int * end_key, int mode){
	int i;
	if (mode == 64){
		if ((end_key[0] == 255) && (end_key[1] == 255) && (end_key[2] == 255) && (end_key[3] == 255) && (end_key[4] == 255)){
			for (i=0;i<5; i++) current_key[i] = 255;
			return;
		}
		for(i=0;i<5;i++){
			if (i != 4) current_key[i] = end_key[i];
			else current_key[i] = end_key[i]+1;
		}
		i = 4;
		while (current_key[i] > 255){
			current_key[i] = 0;
			current_key[i-1] += 1;
			i--;
		}
	}
	else{
		if ((end_key[0] == 255) && (end_key[1] == 255) && (end_key[2] == 255) && (end_key[3] == 255) && (end_key[4] == 255) && (end_key[5] == 255) && (end_key[6] == 255) &&\
			(end_key[7] == 255) && (end_key[8] == 255) && (end_key[9] == 255) && (end_key[10] == 255) && (end_key[11] == 255) && (end_key[12] == 255)){
			for (i=0;i<13; i++) current_key[i] = 255;
			return;
		}
		for(i=0;i<13;i++){
			if (i != 12) current_key[i] = end_key[i];
			else current_key[i] = end_key[i]+1;
		}
		i = 12;
		while (current_key[i] > 255){
			current_key[i] = 0;
			current_key[i-1] += 1;
			i--;
		}
	}
}

void set_init_key(unsigned int * init_key, const unsigned int * current_key, int mode){
	int i;
	if (mode == 64)
		for (i=0;i<5;i++) init_key[i] = current_key[i];
	else
		for (i=0;i<13;i++) init_key[i] = current_key[i];
}

/*void set_block(void){
	int i, j = 3, l=1;
	long number = s_data.block_size, divisor, k;
	while (number != 0){
		divisor = 1;
		for ( i = 0; i < j ;i ++) divisor *= 256;
		k = number / divisor;
		if (k >= 1 && j != 0){
			block_key[l] = k;
			number -= divisor * k;
		}
		else if (j!=0) block_key[i]=0;
		if (j==0){
			block_key[l] = number;
			number = 0;
		}
		j--; l++;
	}
}
*/
int set_block(void){
	int i, j = 4, l=0, ret=1;
	char log_message[100];
	BIGNUM *number, *div, *max_value, *k, *one, *zero;
	BN_CTX *ctx;

	div=BN_new();
	k=BN_new();
	max_value=BN_new();
	one=BN_new();
	number=BN_new();
	zero=BN_new();
	ctx = BN_CTX_new();
	
	if (!BN_dec2bn(&max_value, "1099511627775")) ret=0;
	if (!BN_dec2bn(&one, "1")) ret=0;
	if (!BN_dec2bn(&one, "0")) ret=0;
	if (BN_copy(number, s_data.block_size) == NULL) ret=0;

	if (BN_cmp(number, max_value) == 1){
		sprintf(log_message, "Server Error: Blocksize is too big");
		server_log_message(s_data.log_file, log_message);
    	ret=0;
	}

	if (BN_cmp(number, zero) <= 0){
		sprintf(log_message, "Server Error: Blocksize must be higher than 0");
		server_log_message(s_data.log_file, log_message);
		ret=0;
	}
	
	while (!BN_is_zero(number) && ret != 0){
		BN_dec2bn(&div, "1");
		for ( i = 0; i < j ;i ++) if(!BN_lshift(div, div, 8)) ret=0;
		if (!BN_div(k, NULL, number, div, ctx)) ret=0;
		BN_cmp(k, one);
		if ((BN_cmp(k, one) >= 0) && j != 0){
			block_key[l] = atoi(BN_bn2dec(k));
			if (!BN_mul(div, div, k, ctx)) ret=0;
			if (!BN_sub(number, number, div)) ret=0;
		}		
		if (j==0){
			block_key[l] = atoi(BN_bn2dec(k));
			if (!BN_dec2bn(&number, "0")){
				ret=0;
				break;
			}
		}
		j--; l++;
	}

	BN_clear_free(div);
	BN_clear_free(max_value);
	BN_clear_free(k);
	BN_clear_free(one);
	BN_clear_free(number);
	BN_clear_free(zero);
	BN_CTX_free(ctx);
	
	return ret;
}

void set_end_key(const int * init_key, int * end_key, int mode){
	int i,j,k;	
	if (mode == 64){
		for (i=0;i<5;i++) end_key[i] = 0;
		for (i=4;i>=0;i--){
			end_key[i] += init_key[i] + block_key[i];
			if (end_key[i] > 255 && (i != 0)){
				end_key[i-1] += 1;
				end_key[i] -= 256;
			}
			else if (end_key[i] > 255 && (i == 0)){
				for(j=0;j<5;j++) end_key[j] = 255;
				break;
			}
		}
	}
	else{
		for(i=0;i<8;i++) end_key[i] = init_key[i];
		for(i=8;i<13;i++) end_key[i] = 0;
		k=4;		
		for(i=12;i>=0;i--){			
			if (k != -1) end_key[i] += init_key[i] + block_key[k];			
			if (end_key[i] > 255 && (i != 0)){
				end_key[i-1] += 1;
				end_key[i] -= 256;				
			}
			else if (end_key[i] > 255 && (i == 0)){
				for(j=0;j<13;j++) end_key[j] = 255;
				break;
			}
			if (k != -1) k--;
		}		
	}
}

void set_sent_data(sent_data * st_data, int mode, const unsigned int * init_key, const unsigned int * end_key, int message){
	int i;
	st_data->mode = mode;
	st_data->message = message;
	if (mode == 64 && init_key != NULL && end_key != NULL)
		for (i=0;i<5;i++){
			st_data->init_key[i] = init_key[i];
			st_data->end_key[i]  = end_key[i];
		}
	else if (mode == 128 && init_key != NULL && end_key != NULL){
		for (i=0;i<13;i++){
			st_data->init_key[i] = init_key[i];
			st_data->end_key[i]  = end_key[i];			
		}
	}
	else{
		st_data->init_key[0] = st_data->end_key[0] = '\0';
	}
}

void delete_hostlist(void){
	hosts_list temp;

	while (list != NULL){
		temp = list;
		list = list->next;
		free(temp);
	}
}

int check_list(const unsigned char * client_address, int mode, const char * key){
	hosts_list temp;
	temp = list;

	while(temp != NULL){
		if (strncmp(client_address,temp->client_address,16) == 0){
			if (mode == 0) return 1;
			else if ((mode == 64) && (strncmp(temp->end_key, key, 14) == 0)) return 1;
			else if ((mode == 128) && (strncmp(temp->end_key, key, 35) == 0)) return 1;
			return 0;
		}
		temp = temp->next;
	}
	return 0;
}

void add_new_client(const unsigned char * client_address, const int * init_key, const int * end_key, int mode){
	hosts_list temp = malloc(sizeof(struct hosts_list));

	strncpy(temp->client_address,client_address,strlen(client_address));
	temp->assigned_blocks = 1;
	switch(mode){
		case 64:
			sprintf(temp->start_key,"%.2X:%.2X:%.2X:%.2X:%.2X", init_key[0], init_key[1], init_key[2], init_key[3], init_key[4]);
			sprintf(temp->end_key,"%.2X:%.2X:%.2X:%.2X:%.2X", end_key[0], end_key[1], end_key[2], end_key[3], end_key[4]);
			break;
		case 128:
			sprintf(temp->start_key,"%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X", init_key[0], init_key[1], init_key[2], init_key[3], init_key[4]\
				, init_key[5], init_key[6], init_key[7], init_key[8], init_key[9], init_key[10], init_key[11], init_key[12]);
			sprintf(temp->end_key,"%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X", end_key[0], end_key[1], end_key[2], end_key[3], end_key[4]\
				, end_key[5], end_key[6], end_key[7], end_key[8], end_key[9], end_key[10], end_key[11], end_key[12]);
			break;
	}
	temp->next = NULL;

	if (list == NULL) list = temp;
	else{
		temp->next = list;
		list = temp;
	}
}

void update_client_list(const char * client_address, unsigned int * init_key, unsigned int * end_key, int mode){
	hosts_list temp;
	temp = list;

	while(temp != NULL){
		if (strncmp(client_address,temp->client_address,16) == 0) break;
		temp = temp->next;
	}	

	if (temp == NULL) return;
	else{
		switch(mode){
			case 64:
				sprintf(temp->start_key,"%.2X:%.2X:%.2X:%.2X:%.2X", init_key[0], init_key[1], init_key[2], init_key[3], init_key[4]);
				sprintf(temp->end_key,"%.2X:%.2X:%.2X:%.2X:%.2X", end_key[0], end_key[1], end_key[2], end_key[3], end_key[4]);
				break;
			case 128:
				sprintf(temp->start_key,"%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X", init_key[0], init_key[1], init_key[2], init_key[3], init_key[4]\
					, init_key[5], init_key[6], init_key[7], init_key[8], init_key[9], init_key[10], init_key[11], init_key[12]);
				sprintf(temp->end_key,"%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X", end_key[0], end_key[1], end_key[2], end_key[3], end_key[4]\
					, end_key[5], end_key[6], end_key[7], end_key[8], end_key[9], end_key[10], end_key[11], end_key[12]);
				break;
		}
		temp->assigned_blocks++;
	}
}

void server_init(char * packet_file, int network_count, int cracking_mode, const unsigned char * init_key){
	int i, port, value=1;
	char log_message[100];

#ifndef __CYGWIN__
	char login[15];
#endif

	unsigned char digest[16];
	struct sockaddr_in server_conf;

	if (network_count) s_data.network_count = network_count;
	if (cracking_mode) s_data.mode = cracking_mode;
	s_data.log_file[0] = '\0';

#ifndef __CYGWIN__	
	getlogin_r(login,15);
	parse_path_file(packet_file);
#endif

#ifndef __CYGWIN__
	if(!parse_config_file(login, &port)){
#else
	if(!parse_config_file(NULL, &port)){
#endif
		sprintf(log_message,"Server Error: Error in config file\n");
		if (s_data.log_file[0] != '\0'){			
			server_log_init(s_data.log_file, 0, "-","-", 0, "-", 0);			
			server_log_message(s_data.log_file, log_message);
			server_log_stop(s_data.log_file);
		}
		else{
			/*server_log_init(DEFAULT_LOG_FILE, 0, "-","-", 0, "-", 0);
			server_log_message(DEFAULT_LOG_FILE, log_message);
			server_log_stop(DEFAULT_LOG_FILE);*/
			fprintf(stdout,"%s", log_message);
		}
		exit(1);
	}

#ifndef __CYGWIN__
	server_log_init(s_data.log_file, s_data.mode, s_data.path, s_data.file_name, port, BN_bn2dec(s_data.block_size), s_data.network_count);
#else
	server_log_init(s_data.log_file, s_data.mode, NULL, packet_file, port, BN_bn2dec(s_data.block_size), s_data.network_count);
#endif
	

	if(!make_md5_digest(s_data.path, s_data.file_name, digest)){
		sprintf(log_message,"Server Error: Error making md5 digest\n");
		server_log_message(s_data.log_file, log_message);
		server_log_stop(s_data.log_file);
		exit(1);
	}
	
	copy_md5_digest(s_data.digest, digest);
	
	if ((socket_server = socket(PF_INET, SOCK_STREAM, 0)) < 0){
		sprintf(log_message, "Server Error: Socket call failed: %s\n", strerror(errno));
		server_log_message(s_data.log_file, log_message);
		server_log_stop(s_data.log_file);
		exit(1);
	}

	if ( setsockopt(socket_server, SOL_SOCKET, SO_REUSEADDR, &value, sizeof(value)) != 0){
		sprintf(log_message, "\nServer error: Setsockopt call failed: %s\n", strerror(errno));
		server_log_message(s_data.log_file, log_message);
	}	
	
	server_conf.sin_family = AF_INET;
	server_conf.sin_port   = htons(port);
	server_conf.sin_addr.s_addr = INADDR_ANY;

	if (bind(socket_server, (struct sockaddr *) &server_conf, sizeof(server_conf)) != 0){
		sprintf(log_message, "Server error: Bind call failed: %s\n", strerror(errno));
		close(socket_server);
		server_log_message(s_data.log_file, log_message);		
		server_log_stop(s_data.log_file);
		exit(1);
	}

	if (listen(socket_server, 20) != 0){
		sprintf(log_message, "Server Error: Listen call failed: %s\n", strerror(errno));
		close(socket_server);
		server_log_message(s_data.log_file, log_message);
		server_log_stop(s_data.log_file);		
		exit(1);
	}

	list = NULL;
	
	for (i=0;i<5;i++) current_key64[i]  = 0;
	for (i=0;i<13;i++) current_key128[i] = 0;
		
	if (init_key != NULL && s_data.mode == 64)
		for (i=0;i<5;i++) current_key64[i]  = init_key[i];
	
	if (init_key != NULL && s_data.mode == 128)
		for (i=0;i<13;i++) current_key128[i]  = init_key[i];
			
	for (i=0;i<5;i++) block_key[i] = 0;

	if (!set_block()){
		sprintf(log_message, "Server Error: Couldn't set key_block\n");
		close(socket_server);
		server_log_message(s_data.log_file, log_message);
		server_log_stop(s_data.log_file);		
		exit(1);
	}
}

void server_run(void){
	int size, connection_socket, read_bytes, checked_data;
	int rd, fd; 
	int i = 0, decrypted_keys = 0, all_blocks_assigned = 0 , all_blocks_decrypted = 0;
	long int assigned_blocks = 0;
	unsigned int init_key[13], end_key[13];
	unsigned char client_recv[100], file[50], data[100], log_message[100], client_address[17];
	pid_t fork_c;
	struct sockaddr_in client_conf;	
	client_data c_data;
	sent_data st_data;
		
	size = sizeof(client_conf);
	
	do{
		if ((connection_socket = accept(socket_server, (struct sockaddr *) &client_conf, &size)) <= 0){
			sprintf(log_message,"Server Error: Accept call failed: %s\n", strerror(errno));
			server_log_message(s_data.log_file, log_message);
			server_stop();
			exit(1);
		}
		
		bzero(client_recv, 100);
		
		sprintf(client_address,"%s", inet_ntoa(client_conf.sin_addr)); 
		
		read_bytes = recv(connection_socket,client_recv,MAXBUF,0);
		
		if (read_bytes < 0){
			sprintf(log_message,"Server Error: No data read from %s client", client_address);
			server_log_message(s_data.log_file, log_message);
		}
		
		checked_data = parse_client_data(client_recv, &c_data, read_bytes);
		
		if (s_data.verbosity == LOW ) sprintf(log_message, "Connection[%d]: %s", i+1, client_address);
		else
			sprintf(log_message, "Connection[%d]: %s:%d", i+1, client_address, ntohs(client_conf.sin_port));		
		i = i + 1;
		
		server_log_message(s_data.log_file, log_message);
		
		switch (c_data.option){
			case GET_KEYS:
				sprintf(log_message,"Client option: GET_KEYS");
				server_log_message(s_data.log_file, log_message);
				if (compare_md5_digests(s_data.digest, c_data.digest) != 0){
					set_sent_data(&st_data, 0,NULL, NULL, HASHES_DOESNT_MATCH);
					send(connection_socket,(unsigned char *) & st_data,sizeof(st_data),0);
					break;
				}
				switch (s_data.mode){
					case 64:
						if (all_blocks_assigned){
							set_sent_data(&st_data,s_data.mode, NULL, NULL, NO_MORE_BLOCKS);
							break;
						}
						set_init_key(init_key, current_key64, 64);
						set_end_key(init_key, end_key, 64);
						update_current_key(current_key64, end_key, 64);						
						if (s_data.verbosity == MEDIUM || s_data.verbosity == HIGH){
							sprintf(log_message, "Init key: %.2X:%.2X:%.2X:%.2X:%.2X\nEnd key: %.2X:%.2X:%.2X:%.2X:%.2X", init_key[0], init_key[1], init_key[2], init_key[3], init_key[4]\
							, end_key[0], end_key[1], end_key[2], end_key[3], end_key[4]);							
							server_log_message(s_data.log_file, log_message);
						}
						if (!check_list(client_address, 0, NULL)) add_new_client(client_address, init_key, end_key, 64);
						else update_client_list(client_address, init_key, end_key, 64);						
						set_sent_data(&st_data,s_data.mode, init_key, end_key, ALL_OK);
						if ((current_key64[0] == 255) && (current_key64[1] == 255) && (current_key64[2] == 255) && (current_key64[3] == 255) && (current_key64[4] == 255)){
							all_blocks_assigned = 1;
						}
						break;
					case 128:
						if (all_blocks_assigned){
							set_sent_data(&st_data,s_data.mode, NULL, NULL, NO_MORE_BLOCKS);
							break;
						}
						set_init_key(init_key, current_key128, 128);
						set_end_key(init_key, end_key, 128);						
						update_current_key(current_key128, end_key, 128);						
						if (s_data.verbosity == MEDIUM || s_data.verbosity == HIGH){
							sprintf(log_message, "Init Key %.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X\nEnd Key %.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X", init_key[0], init_key[1], init_key[2], init_key[3], init_key[4]\
								, init_key[5], init_key[6], init_key[7], init_key[8], init_key[9], init_key[10], init_key[11], init_key[12]\
								, end_key[0], end_key[1], end_key[2], end_key[3], end_key[4], end_key[5], end_key[6], end_key[7], end_key[8]\
								, end_key[9], end_key[10], end_key[11], end_key[12]);
							server_log_message(s_data.log_file, log_message);
						}
						if (!check_list(client_address, 0, NULL)) add_new_client(client_address, init_key, end_key, 128);
						else update_client_list(client_address, init_key, end_key, 128);
						set_sent_data(&st_data,s_data.mode, init_key, end_key, ALL_OK);
						if ((current_key128[0] == 255) && (current_key128[1] == 255) && (current_key128[2] == 255) && (current_key128[3] == 255) && (current_key128[4] == 255) && (current_key128[5] == 255) &&\
							(current_key128[6] == 255) && (current_key128[7] == 255) && (current_key128[8] == 255) && (current_key128[9] == 255) && (current_key128[10] == 255) && (current_key128[11] == 255) &&\
							(current_key128[12] == 255)){							
							all_blocks_assigned = 1;
						}
						break;
					default:						
						break;
				}
			send(connection_socket,(unsigned char *) & st_data,sizeof(st_data),0);
			assigned_blocks++;
			break;
		case GET_FILE:
			//The client wants the file so fork the connection and send the file
			sprintf(log_message,"Client option: GET_FILE");
			server_log_message(s_data.log_file, log_message);		
			fork_c = fork();
			switch(fork_c){
				case -1:
					sprintf(log_message,"Server Error: Couldn't fork the connection");
					server_log_message(s_data.log_file, log_message);
					break;
				case 0:
					sprintf(file,"%s%s",s_data.path, s_data.file_name);					
					if ((fd=open(file,O_RDONLY)) == -1){
						sprintf(log_message,"Error openning file to send");
						server_log_message(s_data.log_file, log_message);
						close(connection_socket);
						exit(1);
						break;						
					}
					while ((rd = read(fd, data, 100)) > 0)
						send(connection_socket,data,rd,0);					
					close(fd);
					close(connection_socket);
					exit(0);		
				case 1:
					break;
			}
			break;
		case DECRYPTED_KEYS:
			sprintf(log_message,"Client option: DECRYPTED_KEYS");
			server_log_message(s_data.log_file, log_message);
			if (compare_md5_digests(s_data.digest, c_data.digest) != 0){
				set_sent_data(&st_data,0, NULL, NULL, HASHES_DOESNT_MATCH);
				send(connection_socket,(unsigned char *) & st_data,sizeof(st_data),0);
				break;
			}
			else{
				store_decrypted_key(s_data.log_file, c_data, s_data.mode);
				decrypted_keys++;
			}
			break;
		case DECRYPTED_BLOCK:
			if (s_data.verbosity == MEDIUM || s_data.verbosity == HIGH){
				sprintf(log_message, "Client option: DECRYPTED_BLOCK");
				server_log_message(s_data.log_file, log_message);
			}
			
			if (compare_md5_digests(s_data.digest, c_data.digest) != 0){
				set_sent_data(&st_data,0, NULL, NULL, HASHES_DOESNT_MATCH);
				send(connection_socket,(unsigned char *) & st_data,sizeof(st_data),0);
				break;
			}
			
			if (s_data.mode == 64){
				if (!check_list(client_address, s_data.mode, "FF:FF:FF:FF:FF")) break;
			}
			else if (s_data.mode == 128){				
				if (!check_list(client_address, s_data.mode, "FF:FF:FF:FF:FF:FF:FF:FF:FF:FF:FF:FF")) break;
			}
			
			all_blocks_decrypted = 1;			
			break;
		case BAD_DATA:
			sprintf(log_message,"Client option: BAD_DATA");
			server_log_message(s_data.log_file, log_message);
			//set_sent_data(&st_data,NONE, NULL, NULL, BAD_DATA_RECEIVED);
			//send(connection_socket,(unsigned char *) & st_data,sizeof(st_data),0);
			break;
		}
		
		close(connection_socket);				
	}while(decrypted_keys < s_data.network_count && !all_blocks_decrypted);
}

void server_stop(){
	unsigned char log_message[100];
	hosts_list temp;
	temp = list;
	
	if (s_data.verbosity == HIGH){
		server_log_message(s_data.log_file, "\nLogged clients:");
		while (temp != NULL){
			sprintf(log_message, "HOST: %s\nINIT KEY: %s\nEND KEY:%s\nASSIGNED BLOCKS: %i", temp->client_address, temp->start_key, temp->end_key, temp->assigned_blocks);
			server_log_message(s_data.log_file, log_message);
			temp = temp->next;
		}
	}
	
	server_log_stop(s_data.log_file);	
	
	delete_hostlist();
	close(socket_server);
}
