# JerryScript types

## jerry_init_flag_t

Enum that contains the following elements:

 - JERRY_INIT_EMPTY - empty flag set
 - JERRY_INIT_SHOW_OPCODES - dump byte-code to log after parse
 - JERRY_INIT_SHOW_REGEXP_OPCODES - dump regexp byte-code to log after compilation
 - JERRY_INIT_MEM_STATS - dump memory statistics

 *`JERRY_INIT_MEM_STATS_SEPARATE` and `JERRY_INIT_DEBUGGER`, which were deprecated in version 2.0, have been completely removed in version 3.0*.

## jerry_log_level_t

Enum that represents the configurable logging levels.
The levels are ordered by severity, with the most serious levels listed first..

 - JERRY_LOG_LEVEL_ERROR - the engine will terminate after the message is printed
 - JERRY_LOG_LEVEL_WARNING - a request is aborted, but the engine continues its operation
 - JERRY_LOG_LEVEL_DEBUG - debug messages from the engine, low volume
 - JERRY_LOG_LEVEL_TRACE - detailed info about engine internals, potentially high volume

*Introduced in version 3.0*.

## jerry_type_t

Enum that contains JerryScript API value types:

 - JERRY_TYPE_NONE - no type information
 - JERRY_TYPE_UNDEFINED - undefined type
 - JERRY_TYPE_NULL - null type
 - JERRY_TYPE_BOOLEAN - boolean type
 - JERRY_TYPE_NUMBER - number type
 - JERRY_TYPE_STRING - string type
 - JERRY_TYPE_OBJECT - object type
 - JERRY_TYPE_FUNCTION - function type
 - JERRY_TYPE_EXCEPTION - exception/abort type
 - JERRY_TYPE_SYMBOL - symbol type
 - JERRY_TYPE_BIGINT - bigint type

*Introduced in version 2.0*.

*Updated in version 2.4*: Added `JERRY_TYPE_BIGINT` value.

## jerry_object_type_t

Enum that contains JerryScript **object** value types:

 - JERRY_OBJECT_TYPE_NONE - Non-object type
 - JERRY_OBJECT_TYPE_GENERIC - Generic JavaScript object without any internal property
 - JERRY_OBJECT_TYPE_MODULE_NAMESPACE - Module namespace object
 - JERRY_OBJECT_TYPE_ARRAY - Array object
 - JERRY_OBJECT_TYPE_PROXY - Proxy object
 - JERRY_OBJECT_TYPE_SCRIPT - Script object (see [jerry_parse](#jerry_parse))
 - JERRY_OBJECT_TYPE_MODULE - Module object (see [jerry_parse](#jerry_parse))
 - JERRY_OBJECT_TYPE_PROMISE - Promise object
 - JERRY_OBJECT_TYPE_DATAVIEW - Dataview object
 - JERRY_OBJECT_TYPE_FUNCTION - Function object (see [jerry_function_type](#jerry_function_type))
 - JERRY_OBJECT_TYPE_TYPEDARRAY - %TypedArray% object (see [jerry_typedarray_type](#jerry_typedarray_type))
 - JERRY_OBJECT_TYPE_ITERATOR - Iterator object (see [jerry_iterator_type](#jerry_typedarray_type))
 - JERRY_OBJECT_TYPE_CONTAINER - Container object (see [jerry_container_type](#jerry_container_type))
 - JERRY_OBJECT_TYPE_ERROR - Error object (see [jerry_error_type](#jerry_error_type))
 - JERRY_OBJECT_TYPE_ARRAYBUFFER - ArrayBuffer object
 - JERRY_OBJECT_TYPE_SHARED_ARRAYBUFFER - SharedArrayBuffer object
 - JERRY_OBJECT_TYPE_ARGUMENTS - Arguments object
 - JERRY_OBJECT_TYPE_BOOLEAN - Boolean object
 - JERRY_OBJECT_TYPE_DATE - Date object
 - JERRY_OBJECT_TYPE_NUMBER - Number object
 - JERRY_OBJECT_TYPE_REGEXP - RegExp object
 - JERRY_OBJECT_TYPE_STRING - String object
 - JERRY_OBJECT_TYPE_SYMBOL - Symbol object
 - JERRY_OBJECT_TYPE_GENERATOR - Generator object
 - JERRY_OBJECT_TYPE_BIGINT - BigInt object
 - JERRY_OBJECT_TYPE_WEAKREF - WeakRef object

*Introduced in version 2.4*.

*Updated in version 3.0*: Added `JERRY_OBJECT_TYPE_MODULE_NAMESPACE`, `JERRY_OBJECT_TYPE_PROMISE`, `JERRY_OBJECT_TYPE_DATAVIEW`,
                                       `JERRY_OBJECT_TYPE_ERROR`, `JERRY_OBJECT_TYPE_SHARED_ARRAYBUFFER`, `JERRY_OBJECT_TYPE_WEAKREF`,
                                       `JERRY_OBJECT_TYPE_SCRIPT`, and `JERRY_OBJECT_TYPE_MODULE` values.

## jerry_function_type_t

Enum that contains JerryScript **function** value types:

 - JERRY_FUNCTION_TYPE_NONE - Non-function type
 - JERRY_FUNCTION_TYPE_GENERIC - Generic JavaScript function
 - JERRY_FUNCTION_TYPE_ACCESSOR - Accessor function
 - JERRY_FUNCTION_TYPE_BOUND - Bound function
 - JERRY_FUNCTION_TYPE_ARROW - Arrow function
 - JERRY_FUNCTION_TYPE_GENERATOR - Generator function object

*Introduced in version 2.4*.

## jerry_iterator_type_t

Enum that contains JerryScript **iterator** value types:

 - JERRY_ITERATOR_TYPE_NONE - Non-iterator type
 - JERRY_ITERATOR_TYPE_ARRAY - Array iterator
 - JERRY_ITERATOR_TYPE_STRING - String iterator
 - JERRY_ITERATOR_TYPE_MAP - Map iterator
 - JERRY_ITERATOR_TYPE_SET - Set iterator

*Introduced in version 2.4*.

## jerry_proxy_custom_behavior_t

These option bits allow specializing Proxies with non-standard behaviour.
These flags are recommended only for those trusted Proxies, whose handlers
produce correct results.

- JERRY_PROXY_SKIP_RESULT_VALIDATION - skip result validation for `[[GetPrototypeOf]]`, `[[SetPrototypeOf]]`, `[[IsExtensible]]`,
                                       `[[PreventExtensions]]`, `[[GetOwnProperty]]`, `[[DefineOwnProperty]]`, `[[HasProperty]]`,
                                       `[[Get]]`, `[[Set]]`, `[[Delete]]` and `[[OwnPropertyKeys]]`

*Introduced in version 3.0*.

## jerry_property_filter_t

Enum that contains JerryScript **property filter** options bits:

 - JERRY_PROPERTY_FILTER_ALL - List all property keys independently from key type or property value attributes (equivalent to Reflect.ownKeys call)
 - JERRY_PROPERTY_FILTER_TRAVERSE_PROTOTYPE_CHAIN - Include keys from the objects' prototype chain as well
 - JERRY_PROPERTY_FILTER_EXCLUDE_NON_CONFIGURABLE - Exclude property key if the property is non-configurable
 - JERRY_PROPERTY_FILTER_EXCLUDE_NON_ENUMERABLE - Exclude property key if the property is non-enumerable
 - JERRY_PROPERTY_FILTER_EXCLUDE_NON_WRITABLE - Exclude property key if the property is non-writable
 - JERRY_PROPERTY_FILTER_EXCLUDE_STRINGS - Exclude property key if it is a string
 - JERRY_PROPERTY_FILTER_EXCLUDE_SYMBOLS - Exclude property key if it is a symbol
 - JERRY_PROPERTY_FILTER_EXCLUDE_INTEGER_INDICES - Exclude property key if it is an integer index
 - JERRY_PROPERTY_FILTER_INTEGER_INDICES_AS_NUMBER - By default integer index property keys are converted to string. Enabling this flags keeps integer index property keys as numbers

*Introduced in version 2.4*.

*Updated in version 3.0*: The following values have been modified due to typos in their names:
                                       `JERRY_PROPERTY_FILTER_EXLCUDE_NON_CONFIGURABLE`, `JERRY_PROPERTY_FILTER_EXLCUDE_NON_ENUMERABLE`,
                                       `JERRY_PROPERTY_FILTER_EXLCUDE_NON_WRITABLE`, `JERRY_PROPERTY_FILTER_EXLCUDE_STRINGS`,
                                       `JERRY_PROPERTY_FILTER_EXLCUDE_SYMBOLS`, `JERRY_PROPERTY_FILTER_EXLCUDE_INTEGER_INDICES`.

## jerry_error_t

Possible types of an Error object:

 - JERRY_ERROR_COMMON - Error
 - JERRY_ERROR_EVAL - EvalError
 - JERRY_ERROR_RANGE - RangeError
 - JERRY_ERROR_REFERENCE - ReferenceError
 - JERRY_ERROR_SYNTAX - SyntaxError
 - JERRY_ERROR_TYPE - TypeError
 - JERRY_ERROR_URI - URIError
 - JERRY_ERROR_AGGREGATE - AggregateError

There is also a special value `JERRY_ERROR_NONE` which is not an error type
this value can only be returned by the [jerry_error_type](#jerry_error_type).

*Updated in version 2.0*: The `JERRY_ERROR_NONE` was added to be used by the [jerry_error_type](#jerry_error_type) method.

## jerry_feature_t

Possible compile time enabled feature types:

 - JERRY_FEATURE_CPOINTER_32_BIT - 32 bit compressed pointers
 - JERRY_FEATURE_ERROR_MESSAGES - error messages
 - JERRY_FEATURE_JS_PARSER - js-parser
 - JERRY_FEATURE_HEAP_STATS - memory statistics
 - JERRY_FEATURE_PARSER_DUMP - parser byte-code dumps
 - JERRY_FEATURE_REGEXP_DUMP - regexp byte-code dumps
 - JERRY_FEATURE_SNAPSHOT_SAVE - saving snapshot files
 - JERRY_FEATURE_SNAPSHOT_EXEC - executing snapshot files
 - JERRY_FEATURE_DEBUGGER - debugging
 - JERRY_FEATURE_VM_EXEC_STOP - stopping ECMAScript execution
 - JERRY_FEATURE_VM_THROW - capturing ECMAScript throws
 - JERRY_FEATURE_JSON - JSON support
 - JERRY_FEATURE_TYPEDARRAY - Typedarray support
 - JERRY_FEATURE_DATE - Date support
 - JERRY_FEATURE_REGEXP - RegExp support
 - JERRY_FEATURE_LINE_INFO - line info available
 - JERRY_FEATURE_LOGGING - logging
 - JERRY_FEATURE_SYMBOL - symbol support
 - JERRY_FEATURE_DATAVIEW - DataView support
 - JERRY_FEATURE_PROXY - Proxy support
 - JERRY_FEATURE_MAP - Map support
 - JERRY_FEATURE_SET - Set support
 - JERRY_FEATURE_WEAKMAP - WeakMap support
 - JERRY_FEATURE_WEAKSET - WeakSet support
 - JERRY_FEATURE_BIGINT - BigInt support
 - JERRY_FEATURE_REALM - realm support
 - JERRY_FEATURE_GLOBAL_THIS - GlobalThisValue support
 - JERRY_FEATURE_PROMISE_CALLBACK - Promise callback support
 - JERRY_FEATURE_MODULE - Module support
 - JERRY_FEATURE_WEAKREF - WeakRef support
 - JERRY_FEATURE_FUNCTION_TO_STRING - function toString support

*Introduced in version 2.0*.

*Updated in version 2.3*: Added `JERRY_FEATURE_WEAKMAP`, `JERRY_FEATURE_WEAKSET` values.

*Updated in version 2.4*: Added `JERRY_FEATURE_BIGINT`, `JERRY_FEATURE_REALM` values.

*Updated in version 3.0*: Added `JERRY_FEATURE_VM_THROW`, `JERRY_FEATURE_GLOBAL_THIS`,
                                `JERRY_FEATURE_PROMISE_CALLBACK`, and `JERRY_FEATURE_MODULE`,
                                `JERRY_FEATURE_WEAKREF`, and `JERRY_FEATURE_FUNCTION_TO_STRING` values.
                          Removed `JERRY_FEATURE_PROMISE.`

## jerry_container_type_t

Container object types:

 - JERRY_CONTAINER_TYPE_INVALID - Invalid container
 - JERRY_CONTAINER_TYPE_MAP - Map type
 - JERRY_CONTAINER_TYPE_SET - Set type
 - JERRY_CONTAINER_TYPE_WEAKMAP - WeakMap type
 - JERRY_CONTAINER_TYPE_WEAKSET - WeakSet type

 *Introduced in version 2.3*.

## jerry_well_known_symbol_t

Well-known symbols:

 - JERRY_SYMBOL_ASYNC_ITERATOR - @@asyncIterator well-known symbol
 - JERRY_SYMBOL_HAS_INSTANCE - @@hasInstance well-known symbol
 - JERRY_SYMBOL_IS_CONCAT_SPREADABLE - @@isConcatSpreadable well-known symbol
 - JERRY_SYMBOL_ITERATOR - @@iterator well-known symbol
 - JERRY_SYMBOL_MATCH - @@match well-known symbol
 - JERRY_SYMBOL_REPLACE - @@replace well-known symbol
 - JERRY_SYMBOL_SEARCH - @@search well-known symbol
 - JERRY_SYMBOL_SPECIES - @@species well-known symbol
 - JERRY_SYMBOL_SPLIT - @@split well-known symbol
 - JERRY_SYMBOL_TO_PRIMITIVE - @@toPrimitive well-known symbol
 - JERRY_SYMBOL_TO_STRING_TAG - @@toStringTag well-known symbol
 - JERRY_SYMBOL_UNSCOPABLES - @@unscopables well-known symbol
 - JERRY_SYMBOL_MATCH_ALL - @@matchAll well-known symbol

 *Introduced in version 2.4*.

## jerry_regexp_flags_t

RegExp object optional flags:

  - JERRY_REGEXP_FLAG_GLOBAL - global match; find all matches rather than stopping after the first match
  - JERRY_REGEXP_FLAG_IGNORE_CASE - ignore case
  - JERRY_REGEXP_FLAG_MULTILINE - multiline; treat beginning and end characters (^ and $) as working
    over
    multiple lines (i.e., match the beginning or end of each line (delimited by \n or \r), not only the
    very beginning or end of the whole input string)
  - JERRY_REGEXP_FLAG_STICKY - The sticky flag indicates that it matches only from the index indicated
    by the lastIndex property
  - JERRY_REGEXP_FLAG_UNICODE - The unicode flag enables various Unicode-related features
  - JERRY_REGEXP_FLAG_DOTALL -The dotall flag indicates that the dot special character (".") should
    additionally match the following line terminator ("newline") characters in a string;


*Introduced in version 2.0*.

*Updated in version 2.4*: Added `JERRY_REGEXP_FLAG_STICKY`, `JERRY_REGEXP_FLAG_UNICODE` , `JERRY_REGEXP_FLAG_DOTALL` values.


## jerry_parse_option_enable_feature_t

Option bits for [jerry_parse_options_t](#jerry_parse_options_t).

 - JERRY_PARSE_NO_OPTS - No options passed
 - JERRY_PARSE_STRICT_MODE - Enable strict mode
 - JERRY_PARSE_MODULE - Parse source as an ECMAScript module
 - JERRY_PARSE_HAS_ARGUMENT_LIST - `argument_list` field is valid, this also means that function parsing will be done
 - JERRY_PARSE_HAS_SOURCE_NAME - `source_name` field is valid
 - JERRY_PARSE_HAS_START - `start_line` and `start_column` fields are valid
 - JERRY_PARSE_HAS_USER_VALUE - `user_value` field is valid

*Introduced in version 3.0*.

Using both `JERRY_PARSE_MODULE` and `JERRY_PARSE_HAS_ARGUMENT_LIST` is an invalid combination and will result in
an exception during parsing.

**See also**

- [jerry_parse_options_t](#jerry_parse_options_t)


## jerry_gc_mode_t

Set garbage collection operational mode

 - JERRY_GC_PRESSURE_LOW - free unused objects
 - JERRY_GC_PRESSURE_HIGH - free as much memory as possible

The difference between `JERRY_GC_PRESSURE_LOW` and `JERRY_GC_PRESSURE_HIGH`
is that the former keeps memory allocated for performance improvements such
as property hash tables for large objects. The latter frees all possible
memory blocks but the performance may drop after the garbage collection.

*Introduced in version 2.0*.


## jerry_frame_type_t

List of backtrace frame types returned by
[jerry_frame_type](#jerry_frame_type).

 - JERRY_BACKTRACE_FRAME_JS - indicates that the frame is created for a JavaScript function/method

*Introduced in version 3.0*.


## jerry_generate_snapshot_opts_t

Flags for [jerry_generate_snapshot](#jerry_generate_snapshot):

 - JERRY_SNAPSHOT_SAVE_STATIC - generate static snapshot (see below)

**Generate static snapshots**
Snapshots contain literal pools, and these literal pools contain references
to constant literals (strings, numbers, etc.). When a snapshot is executed,
these literals are converted to jerry values and the literal pool entries
are changed to their corresponding jerry value. To support this conversion,
the literals and literal pools are copied into RAM even if the
`JERRY_SNAPSHOT_EXEC_COPY_DATA` option is passed to
[jerry_exec_snapshot](#jerry_exec_snapshot). This non-negligible memory
consumption can be avoided by using static snapshots. The literals of
these snapshots are limited to magic strings and 28 bit signed integers,
so their constant pools do not need to be loaded into the memory.
Hence these snapshots can be executed from ROM.

***Important note:*** The [jerry_exec_snapshot](#jerry_exec_snapshot)
function rejects static snapshots unless the `JERRY_SNAPSHOT_EXEC_ALLOW_STATIC`
option bit is set. The caller must also ensure that the same magic
strings are set by [jerry_register_magic_strings](#jerry_register_magic_strings)
when the snapshot is generated and executed. Furthermore the
`JERRY_SNAPSHOT_EXEC_COPY_DATA` option is not allowed.

*Introduced in version 2.0*.

*Updated in version 3.0*: The `JERRY_SNAPSHOT_SAVE_STRICT` value is removed, `JERRY_PARSE_STRICT_MODE` should be used instead.


## jerry_exec_snapshot_opts_t

Flags for [jerry_exec_snapshot](#jerry_exec_snapshot) functions:

 - JERRY_SNAPSHOT_EXEC_COPY_DATA - copy snapshot data into memory (see below)
 - JERRY_SNAPSHOT_EXEC_ALLOW_STATIC - allow executing static snapshots
 - JERRY_SNAPSHOT_EXEC_LOAD_AS_FUNCTION - load snapshot as function instead of executing it
 - JERRY_SNAPSHOT_EXEC_HAS_SOURCE_NAME - `source_name` field is valid
                                      in [jerry_exec_snapshot_option_values_t](#jerry_exec_snapshot_option_values_t)
 - JERRY_SNAPSHOT_EXEC_HAS_USER_VALUE - `user_value` field is valid
                                        in [jerry_exec_snapshot_option_values_t](#jerry_exec_snapshot_option_values_t)

*Updated in version 3.0*: The `JERRY_SNAPSHOT_EXEC_LOAD_AS_FUNCTION` value is added,
                                       which replaces the `jerry_load_function_snapshot` function.

**Copy snapshot data into memory**

By default the snapshot buffer is expected to be present in memory until
[jerry_cleanup](#jerry_cleanup) is called. For example `static const` buffers
compiled into the application binary satisfy this requirement.

If the snapshot buffer is freed after [jerry_exec_snapshot](#jerry_exec_snapshot)
is called the `JERRY_SNAPSHOT_EXEC_COPY_DATA` must be passed to copy the necessary
parts of the snapshot buffer into memory.

The `JERRY_SNAPSHOT_EXEC_COPY_DATA` option is not allowed for static snapshots.

*Introduced in version 2.0*.


## jerry_char_t

**Summary**

Jerry's char value

**Prototype**

```c
typedef uint8_t jerry_char_t;
```

## jerry_size_t

**Summary**

Jerry's size

**Prototype**

```c
typedef uint32_t jerry_size_t;
```

## jerry_length_t

**Summary**

Jerry's length

**Prototype**

```c
typedef uint32_t jerry_length_t;
```

## jerry_encoding_t

Supported string encodings.

 - JERRY_ENCODING_CESU8 - cesu-8 encoding
 - JERRY_ENCODING_UTF8 - utf-8 encoding

*Introduced in version 3.0*.

## jerry_value_t

**Summary**

A JerryScript value can be undefined, null, boolean, number, string, object, or an exception value. Exception values
represent thrown exceptions during execution. Exception values cannot be passed as an argument to regular API function,
only to those that work with exception values specifically.

Returned and created values by the API functions must be freed with
[jerry_value_free](#jerry_value_free) when they are no longer needed.

**Prototype**

```c
typedef uint32_t jerry_value_t;
```


## jerry_context_data_manager_t

**Summary**

Structure that defines how a context data item will be initialized and deinitialized. JerryScript zeroes out the memory
for the item by default, and if the `init_cb` field is not NULL, it will be called with the pointer to the memory as
an additional custom initializer. The `deinit_cb` (if non-`NULL`) is called during a call to `jerry_cleanup ()` to run
any custom deinitialization *before* the VM has been fully cleaned up. The `finalize_cb` (if non-`NULL`) is also called
during a call to `jerry_cleanup ()` to run any custom deinitialization *after* the VM has been fully cleaned up.
If bytes_needed field is 0, no buffer is allocated for the manager, callback functions are called with NULL pointer.

**Prototype**

```c
typedef struct
{
  /**
   * Callback responsible for initializing a context item, or NULL to zero out the memory. This is called lazily, the
   * first time jerry_context_data () is called with this manager.
   *
   * @param [in] data The buffer that JerryScript allocated for the manager. The buffer is zeroed out. The size is
   * determined by the bytes_needed field. The buffer is kept alive until jerry_cleanup () is called.
   */
  void (*init_cb) (void *data);

  /**
   * Callback responsible for deinitializing a context item, or NULL. This is called as part of jerry_cleanup (),
   * right *before* the VM has been cleaned up. This is a good place to release strong references to jerry_value_t's
   * that the manager may be holding.
   * Note: because the VM has not been fully cleaned up yet, jerry_object_native_info_t free_cb's can still get called
   * *after* all deinit_cb's have been run. See finalize_cb for a callback that is guaranteed to run *after* all
   * free_cb's have been run.
   *
   * @param [in] data The buffer that JerryScript allocated for the manager.
   */
  void (*deinit_cb) (void *data);

  /**
   * Callback responsible for finalizing a context item, or NULL. This is called as part of jerry_cleanup (),
   * right *after* the VM has been cleaned up and destroyed and jerry_... APIs cannot be called any more. At this point,
   * all values in the VM have been cleaned up. This is a good place to clean up native state that can only be cleaned
   * up at the very end when there are no more VM values around that may need to access that state.
   *
   * @param [in] data The buffer that JerryScript allocated for the manager. After returning from this callback,
   * the data pointer may no longer be used.
   */
  void (*finalize_cb) (void *data);

  /**
   * Number of bytes to allocate for this manager. This is the size of the buffer that JerryScript will allocate on
   * behalf of the manager. The pointer to this buffer is passed into init_cb, deinit_cb and finalize_cb. It is also
   * returned from the jerry_context_data () API.
   */
  size_t bytes_needed;
} jerry_context_data_manager_t;
```

*Introduced in version 2.0*.


## jerry_context_alloc_t

**Summary**

Function type for allocating buffer for JerryScript context.

**Prototype**

```c
typedef void *(*jerry_context_alloc_t) (size_t size, void *cb_data_p);
```

- `size` - allocation size
- `cb_data_p` - pointer to user data

**Introduced in version 2.0*.*
*External context allocation has been moved to the port API in version 3.0.*


## jerry_context_t

**Summary**

An opaque declaration of the JerryScript context structure.

**Prototype**

```c
typedef struct jerry_context_t jerry_context_t;
```

*Introduced in version 2.0*.

## jerry_container_op_t

Enum that contains the supported container operation types
  - JERRY_CONTAINER_OP_ADD - Set/WeakSet add operation
  - JERRY_CONTAINER_OP_GET - Map/WeakMap get operation
  - JERRY_CONTAINER_OP_SET - Map/WeakMap set operation
  - JERRY_CONTAINER_OP_HAS - Set/WeakSet/Map/WeakMap has operation
  - JERRY_CONTAINER_OP_DELETE - Set/WeakSet/Map/WeakMap delete operation
  - JERRY_CONTAINER_OP_SIZE - Set/WeakSet/Map/WeakMap size operation
  - JERRY_CONTAINER_OP_CLEAR - Set/Map clear operation

*Introduced in version 3.0*.

## jerry_binary_op_t

Enum that contains the supported binary operation types
 - JERRY_BIN_OP_EQUAL - equal comparison (==)
 - JERRY_BIN_OP_STRICT_EQUAL - strict equal comparison (===)
 - JERRY_BIN_OP_LESS - less relation (<)
 - JERRY_BIN_OP_LESS_EQUAL - less or equal relation (<=)
 - JERRY_BIN_OP_GREATER - greater relation (>)
 - JERRY_BIN_OP_GREATER_EQUAL - greater or equal relation (>=)
 - JERRY_BIN_OP_INSTANCEOF - instanceof operation
 - JERRY_BIN_OP_ADD - addition operator (+)
 - JERRY_BIN_OP_SUB - subtraction operator (-)
 - JERRY_BIN_OP_MUL - multiplication operator (*)
 - JERRY_BIN_OP_DIV - division operator (/)
 - JERRY_BIN_OP_REM - remainder operator (%)

*Introduced in version 2.0*.

**See also**

- [jerry_binary_op](#jerry_binary_op)

## jerry_property_descriptor_flags_t

Enum that contains the flags of property descriptors.

 - JERRY_PROP_NO_OPTS - empty property descriptor
 - JERRY_PROP_IS_CONFIGURABLE - [[Configurable]]
 - JERRY_PROP_IS_ENUMERABLE - [[Enumerable]]
 - JERRY_PROP_IS_WRITABLE - [[Writable]]
 - JERRY_PROP_IS_CONFIGURABLE_DEFINED - Is [[Configurable]] defined?
 - JERRY_PROP_IS_ENUMERABLE_DEFINED - Is [[Enumerable]] defined?
 - JERRY_PROP_IS_WRITABLE_DEFINED - Is [[Writable]] defined?
 - JERRY_PROP_IS_VALUE_DEFINED - Is [[Value]] defined?
 - JERRY_PROP_IS_GET_DEFINED - Is [[Get]] defined?
 - JERRY_PROP_IS_SET_DEFINED - Is [[Set]] defined?
 - JERRY_PROP_SHOULD_THROW - Should throw in case of an exception, instead of returning with false

*Introduced in version 3.0*.

**See also**

- [jerry_property_descriptor_t](#jerry_property_descriptor_t)

## jerry_parse_options_t

**Summary**

Various configuration options for parsing functions such as [jerry_parse](#jerry_parse).

**Prototype**

```c
typedef struct
{
  uint32_t options; /**< combination of jerry_parse_option_enable_feature_t values */
  jerry_value_t argument_list; /**< function argument list if JERRY_PARSE_HAS_ARGUMENT_LIST is set in options
                                *   Note: must be string value */
  jerry_value_t source_name; /**< source name string (usually a file name)
                              *   if JERRY_PARSE_HAS_SOURCE_NAME is set in options
                              *   Note: must be string value */
  uint32_t start_line; /**< start line of the source code if JERRY_PARSE_HAS_START is set in options */
  uint32_t start_column; /**< start column of the source code if JERRY_PARSE_HAS_START is set in options */
  jerry_value_t user_value; /**< user value assigned to all functions created by this script including eval
                             *   calls executed by the script if JERRY_PARSE_HAS_USER_VALUE is set in options */
} jerry_parse_options_t;
```

*Introduced in version 3.0*.

**See also**

- [jerry_parse](#jerry_parse)
- [jerry_generate_snapshot](#jerry_generate_snapshot)
- [jerry_exec_snapshot](#jerry_exec_snapshot)
- [jerry_parse_option_enable_feature_t](#jerry_parse_option_enable_feature_t)

## jerry_property_descriptor_t

**Summary**

Description of ECMA property descriptor. This struct can be used
for the [jerry_object_define_own_prop](#jerry_object_define_own_prop) method to
configure how the property should be registered.

The naming scheme is similar to the JavaScript `Object.defineProperty` method.

Fields should be used in pairs. That is if the `is_value_defined` is set to `true`
the `value` field should contain the value for the property.

**Prototype**

```c
typedef struct
{
  uint16_t flags; /**< any combination of jerry_property_descriptor_flags_t bits */
  jerry_value_t value; /**< [[Value]] */
  jerry_value_t getter; /**< [[Get]] */
  jerry_value_t setter; /**< [[Set]] */
} jerry_property_descriptor_t;
```

*Updated in version 3.0*: The `is_value_defined`, `is_get_defined`, `is_set_defined`,
                              `is_writable_defined`, `is_writable`, `is_enumerable_defined`,
                              `is_enumerable`, `is_configurable_defined`, and `is_configurable`
                              fields are replaced by the `flags` field.

**See also**

- [jerry_property_descriptor_flags_t](#jerry_property_descriptor_flags_t)
- [jerry_object_define_own_prop](#jerry_object_define_own_prop)

## jerry_frame_location_t

**Summary**

Source code location data retrieved by
[jerry_frame_location](#jerry_frame_location).

**Prototype**

```c
typedef struct
{
  jerry_value_t source_name; /**< source name */
  jerry_size_t line; /**< line index */
  jerry_size_t column; /**< column index */
} jerry_frame_location_t;
```

*Renamed in version 3.0, it was previously known as `jerry_backtrace_location_t` in earlier versions.*

## jerry_frame_t

**Summary**

Backtrace frame data passed to the [jerry_backtrace_cb_t](#jerry_backtrace_cb_t)
handler. This is an internal data structure which fields can be accessed by helper functions
such as [jerry_frame_location](#jerry_frame_location).

**Prototype**

```c
/**
 * Internal data structure for jerry_frame_t definition.
 */
struct jerry_frame_internal_t;

/**
 * Backtrace frame data passed to the jerry_backtrace_cb_t handler.
 */
typedef struct jerry_frame_internal_t jerry_frame_t;
```

*Introduced in version 3.0*.

## jerry_heap_stats_t

**Summary**

Description of JerryScript heap memory stats.
It is for memory profiling.

**Prototype**

```c
typedef struct
{
  size_t version /**< the version of the stats struct */
  size_t size; /**< heap total size */
  size_t allocated_bytes; /**< currently allocated bytes */
  size_t peak_allocated_bytes; /**< peak allocated bytes */
  size_t reserved[4]; /**< padding for future extensions */
} jerry_heap_stats_t;
```

*Introduced in version 2.0*.

**See also**

- [jerry_heap_stats](#jerry_heap_stats)

## jerry_call_info_t

**Summary**

Call related information passed to [jerry_external_handler_t](#jerry_external_handler_t).

**Prototype**

```c
typedef struct jerry_call_info_t
{
  jerry_value_t function; /**< invoked function object */
  jerry_value_t this_value; /**< this value passed to the function */
  jerry_value_t new_target; /**< current new target value, undefined for non-constructor calls */
} jerry_call_info_t;
```

*Introduced in version 3.0*. Contents of this struct replaces the `jerry_get_new_target` function.

**See also**

- [jerry_external_handler_t](#jerry_external_handler_t)

## jerry_external_handler_t

**Summary**

Type of an external function handler

**Prototype**

```c
typedef jerry_value_t (*jerry_external_handler_t) (const jerry_call_info_t *call_info_p,
                                                   const jerry_value_t args_p[],
                                                   const jerry_length_t args_count);
```

- `call_info_p` - pointer to a [jerry_call_info_t](#jerry_call_info_t)
                  structure which holds call related information.
- `args_p` - the function arguments, array of JavaScript values.
- `args_count` - the number of arguments.
- return value
  - The function's return value. If there is no return value, use [jerry_undefined()](#jerry_undefined).

**See also**

- [jerry_function_external](#jerry_function_external)

## jerry_value_free_callback_t

**Summary**

Native free callback of generic value types.

*Note*:
- Referred values by this method must have at least 1 reference. (Correct API usage satisfies this condition)

**Prototype**

```c
typedef void (*jerry_value_free_callback_t) (void *native_p);
```

*Introduced in version 3.0*.

**See also**

- [jerry_string_external_sz](#jerry_string_external_sz)
- [jerry_string_external](#jerry_string_external)
- [jerry_arraybuffer_external](#jerry_arraybuffer_external)

## jerry_object_native_free_cb_t

**Summary**

Native free callback of an object. The callback receives both the memory pointer and the type
information passed to [jerry_object_set_native_ptr](#jerry_object_set_native_ptr).

*Note*:
- Referred values by this method must have at least 1 reference. (Correct API usage satisfies this condition)

**Prototype**

```c
typedef void (*jerry_object_native_free_cb_t) (void *native_p, struct jerry_object_native_info_t *info_p);
```

- `native_p` - native pointer passed to [jerry_object_set_native_ptr](#jerry_object_set_native_ptr).
- `info_p` - native type info passed to [jerry_object_set_native_ptr](#jerry_object_set_native_ptr).

*Introduced in version 2.0*: Renamed from `jerry_object_free_callback_t`.

*Updated in version 2.2*: API calls are once again allowed. (See note)

*Updated in version 3.0*: `info_p` argument is added and renamed, it was previously known as `jerry_object_native_free_callback_t` in earlier versions.

**See also**

- [jerry_object_native_info_t](#jerry_object_native_info_t)

## jerry_external_string_free_cb_t

**Summary**

Free callback for external strings. See
[jerry_string_external_on_free](#jerry_string_external_on_free)
for more information.

**Prototype**

```c
typedef void (*jerry_external_string_free_cb_t) (jerry_char_t *string_p,
                                                 jerry_size_t string_size,
                                                 void *user_p);
```

- `string_p` - external string pointer
- `string_size` - size of external string
- `user_p` - pointer passed when the external string is created

*Renamed in version 3.0, it was previously known as `jerry_external_string_free_callback_t` in earlier versions.*

**See also**

- [jerry_string_external_on_free](#jerry_string_external_on_free)
- [jerry_string_external_sz](#jerry_string_external_sz)
- [jerry_string_external](#jerry_string_external)

## jerry_error_object_created_cb_t

**Summary**

Decorator callback for Error objects. This native callback is called every time when an Error
object is created and the decorator can create or update any properties of the newly created
Error object.

*Note*:
- The callback function cannot be called recursively, so the Error objects created
  when the callback is running are not updated.

**Prototype**

```c
typedef void (*jerry_error_object_created_cb_t) (const jerry_value_t error_object, void *user_p);
```

- `error_object` - the newly created Error object.
- `user_p` - pointer passed to [jerry_error_on_created](#jerry_error_on_created).

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_error_object_created_callback_t` in earlier versions.*

**See also**

- [jerry_error_on_created](#jerry_error_on_created)

## jerry_module_state_t

An enum representing the current status of a module

 - JERRY_MODULE_STATE_INVALID - Return value for jerry_module_state when its argument is not a module
 - JERRY_MODULE_STATE_UNLINKED - Module is currently unlinked
 - JERRY_MODULE_STATE_LINKING - Module is currently being linked
 - JERRY_MODULE_STATE_LINKED - Module has been linked (its dependencies has been resolved)
 - JERRY_MODULE_STATE_EVALUATING - Module is currently being evaluated
 - JERRY_MODULE_STATE_EVALUATED - Module has been evaluated (its source code has been executed)
 - JERRY_MODULE_STATE_ERROR - An exception has been encountered before the evaluated state is reached

*Introduced in version 3.0*.

## jerry_module_resolve_cb_t

**Summary**

Callback which is called by [jerry_module_link](#jerry_module_link) to get the referenced module.

*Note*:
- If realms are enabled, the returned module should be created in the current
  realm (see: [jerry_current_realm](#jerry_current_realm))

**Prototype**

```c
typedef jerry_value_t (*jerry_module_resolve_cb_t) (const jerry_value_t specifier,
                                                    const jerry_value_t referrer,
                                                    void *user_p);
```

- `specifier` - a module specifier string (usually used as a path to the module)
- `referrer` - a module object which contains the `specifier` in its source code
- `user_p` - pointer passed to [jerry_module_link](#jerry_module_link).
- return value
  - a module object - if it can be resolved successfully
  - an exception - otherwise

*Renamed in version 3.0, it was previously known as `jerry_module_resolve_callback_t` in earlier versions.*

**See also**
- [jerry_module_link](#jerry_module_link)
- [jerry_current_realm](#jerry_current_realm)

## jerry_module_import_cb_t

**Summary**

Callback which is called when an import is resolved dynamically to get the referenced module.

*Note*:
- If the function returns with a promise, the import call returns with this promise. The
  application should try to resolve the requested module later. If the module is evaluated
  successfully, the returned promise should be resolved with the namespace object of the
  module. Otherwise, the returned promise should be rejected with an exception.
- If the function returns with a resolved module, a promise is created and resolved with the
  namespace object of the module. The import call returns with the resolved promise.
- If the function returns with an exception, a promise is created and rejected with the
  returned exception. The import call returns with the rejected promise.
- All other return values are considered invalid. In this case the import call returns
  with a rejected promise. The rejected promise has a fixed error message, it does not
  specify the reason of the fail.
- If realms are enabled, the returned module should be created in the current
  realm (see: [jerry_current_realm](#jerry_current_realm))

**Prototype**

```c
typedef jerry_value_t (*jerry_module_import_cb_t) (const jerry_value_t specifier,
                                                   const jerry_value_t user_value,
                                                   void *user_p);
```

- `specifier` - a module specifier string (usually used as a path to the module)
- `user_value` - the user value assigned to the script (see [jerry_parse_options_t](#jerry_parse_options_t))
- `user_p` - pointer passed to [jerry_module_on_import](#jerry_module_on_import).
- return value
  - promise or resolved module - if the operation is successful
  - an exception - otherwise

*Renamed in version 3.0, it was previously known as `jerry_module_import_callback_t` in earlier versions.*

**See also**
- [jerry_module_on_import](#jerry_module_on_import)
- [jerry_current_realm](#jerry_current_realm)

## jerry_module_state_changed_cb_t

**Summary**

Callback which is called after the module enters into linked, evaluated or error state.

**Prototype**

```c
typedef void (*jerry_module_state_changed_cb_t) (jerry_module_state_t new_state,
                                                 const jerry_value_t module,
                                                 const jerry_value_t value,
                                                 void *user_p);
```

- `new_state` - new state of the module.
- `module` - a module whose state is changed
- `value` - depends on the state: undefined for linked, module script result for evaluated,
            and exception value for error state.
- `user_p` - pointer passed to [jerry_module_on_state_changed](#jerry_module_on_state_changed).

*Renamed in version 3.0, it was previously known as `jerry_module_state_changed_callback_t` in earlier versions.*

**See also**
- [jerry_module_on_state_changed](#jerry_module_on_state_changed)

## jerry_module_import_meta_cb_t

**Summary**

Callback which is called when an import.meta expression of a module is evaluated the first time.
The object returned by import.meta is passed as the `meta_object` argument to the callback, and
the callback can set the initial status of the object (e.g. add properties or set prototype).

**Prototype**

```c
typedef void (*jerry_module_import_meta_cb_t) (const jerry_value_t module,
                                               const jerry_value_t meta_object,
                                               void *user_p);
```

- `module` - module whose import.meta object is requested.
- `meta_object` - import.meta object created for the module.
- `user_p` - pointer passed to [jerry_module_on_import_meta](#jerry_module_on_import_meta).

*Renamed in version 3.0, it was previously known as `jerry_module_import_meta_callback_t` in earlier versions.*

**See also**
- [jerry_module_on_import_meta](#jerry_module_on_import_meta)

## jerry_native_module_evaluate_cb_t

**Summary**

Callback which is called by [jerry_module_evaluate](#jerry_module_evaluate) to evaluate the native module.

Note:
 - Native pointers can be used to assign private data to a native module,
   see [jerry_object_set_native_ptr](#jerry_object_set_native_ptr)

**Prototype**

```c
typedef jerry_value_t (*jerry_native_module_evaluate_cb_t) (const jerry_value_t native_module);
```

- `native_module` - a native module
- return value
  - any non-exception value - if the module is evaluated successfully
  - an exception - otherwise

*Renamed in version 3.0, it was previously known as `jerry_native_module_evaluate_callback_t` in earlier versions.*

**See also**
- [jerry_module_evaluate](#jerry_module_evaluate)

## jerry_backtrace_cb_t

**Summary**

Callback function which is called by [jerry_backtrace_capture](#jerry_backtrace_capture)
for each stack frame.

**Prototype**

```c
typedef bool (*jerry_backtrace_cb_t) (jerry_frame_t *frame_p, void *user_p);
```

- `frame_p` - pointer to [jerry_frame_t](#jerry_frame_t) data.
- `user_p` - pointer passed to [jerry_backtrace_capture](#jerry_backtrace_capture).
- return value
  - true, to continue capturing more frames
  - false, to end the stack capturing

*Renamed in version 3.0, it was previously known as `jerry_backtrace_callback_t` in earlier versions.*

**See also**

- [jerry_backtrace_capture](#jerry_backtrace_capture)
- [jerry_frame_t](#jerry_frame_t)

## jerry_object_native_info_t

**Summary**

Type information for native pointers. Since each native pointer has a type information,
multiple native pointers can be assigned to an object, and these can be updated or
deleted independently.

The type information has a free callback, which is called when the object is freed
by the garbage collector. If the callback is NULL, the application is not notified
about the destruction of the object.

The buffer pointed by the native pointer can have a fixed number of jerry values,
which refer to other values as long as the object is alive. The starting byte
offset and the number of these values are specified by `offset_of_references` and
`number_of_references` fields respectively. Before a buffer is attached to an
object by [jerry_object_set_native_ptr](#jerry_object_set_native_ptr),
the values must be initialized to undefined by
[jerry_native_ptr_init](#jerry_native_ptr_init).
When a buffer is no longer attached to any object, the
[jerry_native_ptr_free](#jerry_native_ptr_free)
must be called to release the values. A single buffer can be attached to any
number of living objects. When a buffer is currently attached to at least
one object, the references can be updated by
[jerry_native_ptr_set](#jerry_native_ptr_set).
However, if the buffer is no longer attached to an object, the finalize function
must be called even if the buffer is reattached to another object later. In this
case, calling the init function after the finalization is optional, because the
finalize function also initializes all values to undefined.

**Prototype**

```c
typedef struct
{
  jerry_object_native_free_cb_t free_cb; /**< the free callback of the native pointer */
  uint16_t number_of_references; /**< the number of value references which are marked by the garbage collector */
  uint16_t offset_of_references; /**< byte offset indicating the start offset of value
                                  *   references in the user allocated buffer */
} jerry_object_native_info_t;
```

*Introduced in version 2.0*.

*Updated in version 3.0*: Added `number_of_references`, and `offset_of_references` fields.

**See also**

- [jerry_object_set_native_ptr](#jerry_object_set_native_ptr)
- [jerry_object_get_native_ptr](#jerry_object_get_native_ptr)
- [jerry_object_delete_native_ptr](#jerry_object_delete_native_ptr)
- [jerry_native_ptr_init](#jerry_native_ptr_init)
- [jerry_native_ptr_free](#jerry_native_ptr_free)
- [jerry_native_ptr_set](#jerry_native_ptr_set)

## jerry_object_property_foreach_cb_t

**Summary**

Function type used as a callback for the [jerry_object_foreach](#jerry_object_foreach)
method. A function with this type must return "true" to continue the iteration or "false" to finish the
iteration on the object's properties.

**Prototype**

```c
typedef bool (*jerry_object_property_foreach_cb_t) (const jerry_value_t property_name,
                                                    const jerry_value_t property_value,
                                                    void *user_data_p);
```

- `property_name` - a property name, this is not always a string.
- `property_value` - the value for the given property.
- `user_data_p` - optional user data pointer supplied via the (jerry_object_foreach)[#jerry_object_foreach] method.
- return value
  - true, to continue the iteration
  - false, to stop the iteration

*Renamed in version 3.0, it was previously known as `jerry_object_property_foreach_t` in earlier versions.*

**See also**

- [jerry_object_foreach](#jerry_object_foreach)

## jerry_foreach_live_object_cb_t

**Summary**

Function type used as a callback for the (jerry_foreach_live_object)[#jerry_foreach_live_object] method.
A function with this type must return "true" to continue the iteration or "false" to finish the
iteration on the object's properties.

**Prototype**

```c
typedef bool (*jerry_foreach_live_object_cb_t) (const jerry_value_t object,
                                                void *user_data_p);
```

- `object` - the current JavaScript object in the for-each iteration.
- `user_data_p` - optional user data pointer supplied via the (jerry_foreach_live_object)[#jerry_foreach_live_object] method.
- return value
  - true, to continue the iteration
  - false, to stop the iteration

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_objects_foreach_t` in earlier versions.*

**See also**

- [jerry_foreach_live_object](#jerry_foreach_live_object)

## jerry_foreach_live_object_with_info_cb_t

**Summary**

Function type used as a callback for the (jerry_foreach_live_object_with_info)[#jerry_foreach_live_object_with_info]
method. A function with this type must return "true" to continue the iteration or "false" to finish the
iteration on the object's properties.

**Prototype**

```c
typedef bool (*jerry_foreach_live_object_with_info_cb_t) (const jerry_value_t object,
                                                          void *object_data_p,
                                                          void *user_data_p);
```

- `object` - the current JavaScript object in the for-each iteration.
- `object_data_p` - the current object's native data pointer.
- `user_data_p` - optional user data pointer supplied via the (jerry_foreach_live_object_with_info)[#jerry_foreach_live_object_with_info] method.
- return value
  - true, to continue the iteration
  - false, to stop the iteration

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_objects_foreach_by_native_info_t` in earlier versions.*

**See also**

- [jerry_foreach_live_object_with_info](#jerry_foreach_live_object_with_info)

## jerry_halt_cb_t

**Summary**

Callback which is called periodically by the engine, and polls whether the ECMAScript execution should be stopped. If
the callback returns with an undefined value the ECMAScript execution continues. Otherwise the result is thrown by the
engine as an exception, and execution returns to the caller. The callback function might be called again even if it
threw an exception. In this case the function must throw the same exception again.

**Prototype**

```c
typedef jerry_value_t (*jerry_halt_cb_t) (void *user_p);
```

*Introduced in version 2.0*.

**See also**

- [jerry_halt_handler](#jerry_halt_handler)

## jerry_throw_cb_t

**Summary**

Callback which is called when a value is thrown in an ECMAScript code. The callback
should not change the `exception_value`. The callback is not called again until the value
is caught.

Note:
 - The engine considers exceptions thrown by external functions as never caught. The
   application can maintain a status flag to ignore the next call of the callback
   if necessary.
   See: [jerry_function_external](#jerry_function_external)

**Prototype**

```c
typedef void (*jerry_throw_cb_t) (const jerry_value_t exception_value, void *user_data_p);
```
 - `exception_value` - the thrown exception
 - `user_data_p` -  optional user data pointer

*Renamed in version 3.0, it was previously known as `jerry_vm_throw_callback_t` in earlier versions.*

**See also**

- [jerry_on_throw](#jerry_on_throw)


## jerry_string_iterate_cb_t

**Summary**

**Prototype**

```c
typedef void (*jerry_string_iterate_cb_t) (uint32_t value, void *user_data_p);
```

- `value` - encoded byte value
- `user_data_p` - optional user data pointer supplied via the (jerry_string_iterate)[#jerry_string_iterate] method.

*Introduced in version 3.0*.

**See also**

- [jerry_string_iterate](#jerry_string_iterate)


## jerry_promise_state_t

Enum which describes the state of a Promise.

Possible values:

 - JERRY_PROMISE_STATE_NONE - Invalid/Unknown state (possibly called on a non-promise object).
 - JERRY_PROMISE_STATE_PENDING - Promise is in "Pending" state.
 - JERRY_PROMISE_STATE_FULFILLED - Promise is in "Fulfilled" state.
 - JERRY_PROMISE_STATE_REJECTED - Promise is in "Rejected" state.

*Introduced in version 2.2*.

**See also**

- [jerry_promise_result](#jerry_promise_result)

## jerry_promise_event_type_t

Event types for [jerry_promise_event_cb_t](#jerry_promise_event_cb_t) callback function.
The description of the `object` and `value` arguments are provided for each type.

Possible values:

- JERRY_PROMISE_EVENT_CREATE - A new Promise object is created.
  - object - the new Promise object
  - value - parent Promise for `then` chains, undefined otherwise.
- JERRY_PROMISE_EVENT_RESOLVE - Called when a Promise is about to be resolved.
  - object - the Promise object
  - value - value for resolving.
- JERRY_PROMISE_EVENT_REJECT - Called when a Promise is about to be rejected.
  - object - the Promise object
  - value - value for rejecting.
- JERRY_PROMISE_EVENT_RESOLVE_FULFILLED - Called when a resolve is called on a fulfilled Promise.
  - object - the Promise object
  - value - value for resolving
- JERRY_PROMISE_EVENT_REJECT_FULFILLED - Called when a reject is called on a fulfilled Promise.
  - object - the Promise object
  - value - value for rejecting
- JERRY_PROMISE_EVENT_REJECT_WITHOUT_HANDLER - Called when a Promise is rejected without a handler.
  - object - the Promise object
  - value - value for rejecting
- JERRY_PROMISE_EVENT_CATCH_HANDLER_ADDED - Called when a catch handler is added to a rejected
                                            Promise which did not have a catch handler before.
  - object - the Promise object
  - value - undefined
- JERRY_PROMISE_EVENT_BEFORE_REACTION_JOB - Called before executing a Promise reaction job.
  - object - the Promise object
  - value - undefined
- JERRY_PROMISE_EVENT_AFTER_REACTION_JOB - Called after a Promise reaction job is completed.
  - object - the Promise object
  - value - undefined
- JERRY_PROMISE_EVENT_ASYNC_AWAIT - Called when an async function awaits the result of a Promise object.
  - object - internal object representing the execution status
  - value - the Promise object
- JERRY_PROMISE_EVENT_ASYNC_BEFORE_RESOLVE - Called when an async function is continued with resolve.
  - object - internal object representing the execution status
  - value - value for resolving
- JERRY_PROMISE_EVENT_ASYNC_BEFORE_REJECT - Called when an async function is continued with reject.
  - object - internal object representing the execution status
  - value - value for rejecting
- JERRY_PROMISE_EVENT_ASYNC_AFTER_RESOLVE - Called when an async function resolve is completed.
  - object - internal object representing the execution status
  - value - value for resolving
- JERRY_PROMISE_EVENT_ASYNC_AFTER_REJECT - Called when an async function reject is completed.
  - object - internal object representing the execution status
  - value - value for rejecting

*Introduced in version 3.0*.

**See also**

- [jerry_promise_event_cb_t](#jerry_promise_event_cb_t)
- [jerry_promise_on_event](#jerry_promise_on_event)


## jerry_promise_event_filter_t

Filter types for [jerry_promise_on_event](#jerry_promise_on_event) callback function.
The callback is only called for those events which are enabled by the filters. The events are
described in [jerry_promise_event_type_t](#jerry_promise_event_type_t).

Possible values:

- JERRY_PROMISE_EVENT_FILTER_DISABLE - Disable reporting of all events.
- JERRY_PROMISE_EVENT_FILTER_CREATE - Enables the following event:
  - JERRY_PROMISE_EVENT_CREATE
- JERRY_PROMISE_EVENT_FILTER_RESOLVE - Enables the following event:
  - JERRY_PROMISE_EVENT_RESOLVE
- JERRY_PROMISE_EVENT_FILTER_REJECT - Enables the following event:
  - JERRY_PROMISE_EVENT_REJECT
- JERRY_PROMISE_EVENT_FILTER_ERROR - Enables the following events:
  - JERRY_PROMISE_EVENT_RESOLVE_FULFILLED
  - JERRY_PROMISE_EVENT_REJECT_FULFILLED
  - JERRY_PROMISE_EVENT_REJECT_WITHOUT_HANDLER
  - JERRY_PROMISE_EVENT_CATCH_HANDLER_ADDED
- JERRY_PROMISE_EVENT_FILTER_REACTION_JOB - Enables the following events:
  - JERRY_PROMISE_EVENT_BEFORE_REACTION_JOB
  - JERRY_PROMISE_EVENT_AFTER_REACTION_JOB
- JERRY_PROMISE_EVENT_FILTER_ASYNC_MAIN - Enables the following events:
  - JERRY_PROMISE_EVENT_ASYNC_AWAIT
- JERRY_PROMISE_EVENT_FILTER_ASYNC_REACTION_JOB - Enables the following events:
  - JERRY_PROMISE_EVENT_ASYNC_BEFORE_RESOLVE
  - JERRY_PROMISE_EVENT_ASYNC_BEFORE_REJECT
  - JERRY_PROMISE_EVENT_ASYNC_AFTER_RESOLVE
  - JERRY_PROMISE_EVENT_ASYNC_AFTER_REJECT

*Introduced in version 3.0*.

**See also**

- [jerry_promise_event_type_t](#jerry_promise_event_type_t)
- [jerry_promise_on_event](#jerry_promise_on_event)


## jerry_promise_event_cb_t

**Summary**

Notification callback for tracking Promise and async function operations. The arguments
passed to the callback depends on the `event_type` which is detailed in the
description of [jerry_promise_event_type_t](#jerry_promise_event_type_t).

**Prototype**

```c
typedef void (*jerry_promise_event_cb_t) (jerry_promise_event_type_t event_type,
                                          const jerry_value_t object, const jerry_value_t value,
                                          void *user_p);
```

- `event_type` - type of the event notification.
- `object` - object corresponding to the event.
- `value` - optional value argument.
- `user_data_p` - optional user data pointer supplied via the (jerry_promise_on_event)[#jerry_promise_on_event] method.

*Renamed in version 3.0, it was previously known as `jerry_promise_callback_t` in earlier versions.*

**See also**

- [jerry_promise_event_type_t](#jerry_promise_event_type_t)
- [jerry_promise_on_event](#jerry_promise_on_event)

## jerry_typedarray_type_t

Enum which describes the TypedArray types.
Possible values:

 - JERRY_TYPEDARRAY_UINT8 - represents the Uint8Array TypedArray
 - JERRY_TYPEDARRAY_UINT8CLAMPED - represents the Uint8ClampedArray TypedArray
 - JERRY_TYPEDARRAY_INT8 - represents the Int8Array TypedArray
 - JERRY_TYPEDARRAY_UINT16 - represents the Uint16Array TypedArray
 - JERRY_TYPEDARRAY_INT16 - represents the Int16Array TypedArray
 - JERRY_TYPEDARRAY_UINT32 - represents the Uint32Array TypedArray
 - JERRY_TYPEDARRAY_INT32 - represents the Int32Array TypedArray
 - JERRY_TYPEDARRAY_FLOAT32 - represents the Float32Array TypedArray
 - JERRY_TYPEDARRAY_FLOAT64 - represents the Float64Array TypedArray
 - JERRY_TYPEDARRAY_BIGINT64 - represents the BigInt64Array TypedArray
 - JERRY_TYPEDARRAY_BIGUINT64 - represents the BigUint64Array TypedArray
 - JERRY_TYPEDARRAY_INVALID - represents an invalid TypedArray

API functions can return the `JERRY_TYPEDARRAY_INVALID` value if the
TypedArray support is not in the engine.

*Introduced in version 2.0*.

*Updated in version 2.4*: Added `JERRY_TYPEDARRAY_BIGINT64`, `JERRY_TYPEDARRAY_BIGUINT64` values.

**See also**

- [jerry_typedarray_type](#jerry_typedarray_type)


## jerry_exec_snapshot_option_values_t

**Summary**

Various configuration options for [jerry_exec_snapshot](#jerry_exec_snapshot)

**Prototype**

```c
typedef struct
{
  jerry_value_t source_name; /**< source name string (usually a file name)
                              *   if JERRY_SNAPSHOT_EXEC_HAS_SOURCE_NAME is set in exec_snapshot_opts
                              *   Note: non-string values are ignored */
  jerry_value_t user_value; /**< user value assigned to all functions created by this script including
                             *   eval calls executed by the script if JERRY_SNAPSHOT_EXEC_HAS_USER_VALUE
                             *   is set in exec_snapshot_opts */
} jerry_exec_snapshot_option_values_t;
```

*Introduced in version 3.0*.

**See also**

- [jerry_exec_snapshot](#jerry_exec_snapshot)

## jerry_source_info_enabled_fields_t

Enum which describes the enabled fields of [jerry_source_info_t](#jerry_source_info_t).
Possible values:

 - JERRY_SOURCE_INFO_HAS_SOURCE_CODE - source_code field is valid
 - JERRY_SOURCE_INFO_HAS_FUNCTION_ARGUMENTS - function_arguments field is valid
 - JERRY_SOURCE_INFO_HAS_SOURCE_RANGE - both source_range_start and source_range_length fields are valid

*Introduced in version 3.0*.

**See also**

- [jerry_source_info_t](#jerry_source_info_t)
- [jerry_source_info](#jerry_source_info)

## jerry_source_info_t

**Summary**

Source related information of a script/module/function.

**Prototype**

```c
typedef struct
{
  uint32_t enabled_fields; /**< combination of jerry_source_info_enabled_fields_t values */
  jerry_value_t source_code; /**< script source code or function body */
  jerry_value_t function_arguments; /**< function arguments */
  uint32_t source_range_start; /**< start position of the function in the source code */
  uint32_t source_range_length; /**< source length of the function in the source code */
} jerry_source_info_t;
```

*Introduced in version 3.0*.

**See also**

- [jerry_source_info_enabled_fields_t](#jerry_source_info_enabled_fields_t)
- [jerry_source_info](#jerry_source_info)

## jerry_arraybuffer_type_t

**Summary**

Enum that contains the JerryScript type of an array buffer:

 - JERRY_ARRAYBUFFER_TYPE_ARRAYBUFFER - the object is an array buffer object
 - JERRY_ARRAYBUFFER_TYPE_SHARED_ARRAYBUFFER - the object is a shared array buffer object

*Introduced in version 3.0*.

**See also**

- [jerry_arraybuffer_allocate_cb_t](#jerry_arraybuffer_allocate_cb_t)
- [jerry_arraybuffer_free_cb_t](#jerry_arraybuffer_free_cb_t)

## jerry_arraybuffer_allocate_cb_t

**Summary**

Callback for allocating the backing store of array buffer or shared array buffer objects.

*Note*:
- The value referenced by `arraybuffer_user_p` is always NULL unless the buffer is created by
  [jerry_arraybuffer_external](#jerry_arraybuffer_external) or
  [jerry_shared_arraybuffer_external](#jerry_shared_arraybuffer_external).
  The value referenced by `arraybuffer_user_p` can be changed, and the new value is passed to
  [jerry_arraybuffer_free_cb_t](#jerry_arraybuffer_free_cb_t).

**Prototype**

```c
typedef uint8_t *(*jerry_arraybuffer_allocate_cb_t) (jerry_arraybuffer_type_t buffer_type,
                                                     uint32_t buffer_size,
                                                     void **arraybuffer_user_p,
                                                     void *user_p);
```

- `buffer_type` - type of the array buffer object, see: [jerry_arraybuffer_type_t](#jerry_arraybuffer_type_t).
- `buffer_size` - size of the requested buffer.
- `arraybuffer_user_p` - [in/out] user pointer assigned to the array buffer or shared array buffer object.
- `user_p` - user pointer passed to [jerry_arraybuffer_allocator](#jerry_arraybuffer_allocator)
- return value
  - Pointer to the buffer, if the allocation is successful, NULL otherwise.

*Renamed in version 3.0, it was previously known as `jerry_arraybuffer_allocate_t` in earlier versions.*

**See also**

- [jerry_arraybuffer_allocator](#jerry_arraybuffer_allocator)

## jerry_arraybuffer_free_cb_t

**Summary**

Callback for freeing the backing store of array buffer or shared array buffer objects.

*Note*:
- The value passed to `arraybuffer_user_p` is always NULL unless the buffer is created by
  [jerry_arraybuffer_external](#jerry_arraybuffer_external) or
  [jerry_shared_arraybuffer_external](#jerry_shared_arraybuffer_external),
  or the value is modified by [jerry_arraybuffer_allocate_cb_t](#jerry_arraybuffer_allocate_cb_t).

**Prototype**

```c
typedef void (*jerry_arraybuffer_free_cb_t) (jerry_arraybuffer_type_t buffer_type,
                                             uint8_t *buffer_p,
                                             uint32_t buffer_size,
                                             void *arraybuffer_user_p,
                                             void *user_p);
```

- `buffer_type` - type of the array buffer object, see: [jerry_arraybuffer_type_t](#jerry_arraybuffer_type_t).
- `buffer_p` - pointer to the allocated buffer.
- `buffer_size` - size of the allocated buffer.
- `arraybuffer_user_p` - [in/out] user pointer assigned to the array buffer or shared array buffer object.
- `user_p` - user pointer passed to [jerry_arraybuffer_allocator](#jerry_arraybuffer_allocator)

*Renamed in version 3.0, it was previously known as `jerry_arraybuffer_free_t` in earlier versions.*

**See also**

- [jerry_arraybuffer_allocator](#jerry_arraybuffer_allocator)


# General engine functions

## jerry_init

**Summary**

Initializes the JerryScript engine, making it possible to run JavaScript code and perform operations
on JavaScript values. This is required for almost all API functions.

**Prototype**

```c
void
jerry_init (jerry_init_flag_t flags)
```

`flags` - combination of various engine configuration flags [jerry_init_flag_t](#jerry_init_flag_t).

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_SHOW_OPCODES | JERRY_INIT_SHOW_REGEXP_OPCODES);

  // ...

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_init_flag_t](#jerry_init_flag_t)
- [jerry_cleanup](#jerry_cleanup)


## jerry_cleanup

**Summary**

Finish JavaScript engine execution, freeing memory and JavaScript values.

*Note*:
- JavaScript values, received from engine, will be inaccessible after the cleanup.

**Prototype**

```c
void
jerry_cleanup (void);
```

**See also**

- [jerry_init](#jerry_init)


## jerry_context_data

**Summary**

Retrieve a pointer to the item stored within the current context by the given manager.

*Note*:
- Since internally the pointer to a manager's context data item is linked to the next
  such pointer in a linked list, it is inadvisable to invoke too many different managers,
  because doing so will increase the time it takes to retrieve a manager's context data
  item, degrading performance. For example, try to keep the number of managers below five.

**Prototype**

```c
void *
jerry_context_data (const jerry_context_data_manager *manager_p);
```

- `manager_p`: the manager of this context data item.
- return value: the item created by `manager_p` when `jerry_context_data ()` was first called, or a new item created
  by `manager_p`, which will be stored for future identical calls to `jerry_context_data ()`, and which will be
  deinitialized using the `deinit_cb` callback provided by `manager_p` when the context will be destroyed.

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_get_context_data` in earlier versions.*

**Example**

[doctest]: # (test="compile")

```c
#include "jerryscript.h"

typedef struct
{
  int my_data1;
  double my_data2;
  char *my_data3;
} my_context_data_t;

/* Define how context items will be initialized. */
static void
my_context_data_new (void *user_data_p)
{
  my_context_data_t *my_data_p = (my_context_data_t *) user_data_p;

  /*
   * Initialize my_data_p. JerryScript will store it on the current context and return it whenever
   * jerry_context_data () is called with a pointer to my_manager as defined below.
   */
}

/* Define how context items will be deinitialized */
static void
my_context_data_free (void *user_data_p)
{
  my_context_data_t *my_data_p = ((my_context_data_t *) user_data_p);

  /* Perform any necessary cleanup on my_data. JerryScript will free the pointer after this function completes. */
}

/* Wrap the creation and destruction functions into a manager */
static const jerry_context_data_manager_t my_manager =
{
  .init_cb = my_context_data_new,
  .deinit_cb = my_context_data_free,
  .bytes_needed = sizeof (my_context_data_t)
};

/*
 * Then, in some function in your code, you can retrieve an item of type my_context_data_t from the currently active
 * context such that JerryScript will create and store such an item if one was not previously created
 */
static void
someplace_in_the_code (void)
{
  my_context_data_t *my_data = (my_context_data_t *) jerry_context_data (&my_manager);
  /* Perform useful things using the data found in my_data */
}
```


## jerry_register_magic_strings

**Summary**

Registers an external magic string array.

*Note*:
- The strings in the array must be sorted by size at first, then lexicographically.
- The maximum number of external magic strings is limited to 2147483648 (UINT32_MAX / 2).
  If there are more than 2147483648 external magic strings the extra is cropped.

**Prototype**

```c
void
jerry_register_magic_strings (const jerry_char_t * const *ex_str_items_p,
                              uint32_t count,
                              const jerry_length_t *str_lengths_p);
```

- `ex_str_items_p` - character arrays, representing external magic strings' contents
- `count` - number of elements in `ext_str_items_p` array
- `str_lengths_p` - array of lengths for each magic string

*Updated in version 2.0*: The first function argument type was changed.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  // must be static, because 'jerry_register_magic_strings' does not copy
  // the items must be sorted by size at first, then lexicographically
  static const jerry_char_t * const magic_string_items[] = {
                                                             (const jerry_char_t *) "magicstring1",
                                                             (const jerry_char_t *) "magicstring2",
                                                             (const jerry_char_t *) "magicstring3"
                                                           };
  uint32_t num_magic_string_items = (uint32_t) (sizeof (magic_string_items) / sizeof (jerry_char_t *));

  // must be static, because 'jerry_register_magic_strings' does not copy
  static const jerry_length_t magic_string_lengths[] = {
                                                         12,
                                                         12,
                                                         12
                                                       };
  jerry_register_magic_strings (magic_string_items, num_magic_string_items, magic_string_lengths);
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_get_literals_from_snapshot](#jerry_get_literals_from_snapshot)


## jerry_heap_stats

**Summary**

Get heap memory stats.

*Note*:
- The engine must be initialized with the `JERRY_INIT_MEM_STATS` option to allow
  heap statistic collections. See [jerry_init](#jerry_init)
- This API depends on a build option (`JERRY_MEM_STATS`) and can be checked
  in runtime with the `JERRY_FEATURE_HEAP_STATS` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
bool
jerry_heap_stats (jerry_heap_stats_t *out_stats_p);
```

- `out_stats_p` - out parameter, that provides the heap statistics.
- return value
  - true, if stats were written into the `out_stats_p` pointer.
  - false, otherwise. Usually it is because the `JERRY_FEATURE_HEAP_STATS` feature is not enabled.

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_get_memory_stats` in earlier versions.*

**Example**

```c
jerry_init (JERRY_INIT_MEM_STATS);
// ...

jerry_heap_stats_t stats = {0};
bool get_stats_ret = jerry_heap_stats (&stats);
```

**See also**

- [jerry_init](#jerry_init)


## jerry_heap_gc

**Summary**

Performs garbage collection.

**Prototype**

```c
void
jerry_heap_gc (jerry_gc_mode_t mode);
```

- `mode` - operational mode, see [jerry_gc_mode_t](#jerry_gc_mode_t)

*Updated in version 2.0*: Added `mode` argument.

*Renamed in version 3.0, it was previously known as `jerry_gc` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t object_value = jerry_object ();
  jerry_value_free (object_value);

  jerry_heap_gc (JERRY_GC_PRESSURE_LOW);

  jerry_cleanup ();
}
```

**See also**

- [jerry_gc_mode_t](#jerry_gc_mode_t)
- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)

# Parser and executor functions

Functions to parse and run JavaScript source code.

## jerry_parse

**Summary**

Parse a script, module, or function and create a compiled code using a character string.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_parse (const jerry_char_t *source_p,
             size_t source_size,
             const jerry_parse_options_t *options_p);
```

- `source_p` - string, containing source code to parse (must be a valid UTF8 string).
- `source_size` - size of the string, in bytes.
- `options_p` - additional parsing options, can be NULL if not used
- return value
  - function object value, if script was parsed successfully,
  - thrown exception, otherwise

*Updated in version 2.0*: Added `source_name_p`, and `source_name_length` arguments.

*Updated in version 3.0*: The `source_name_p`, `source_name_length`, and `parse_opts` arguments are replaced by `options_p`.
                                       This function replaces the `jerry_parse_function` method.

**Example 1**

[doctest]: # (name="02.API-REFERENCE-parse-simple.c")

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  /* Parsing a script. */
  const jerry_char_t script[] = "print ('Hello, World!');";

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_STRICT_MODE | JERRY_PARSE_HAS_SOURCE_NAME | JERRY_PARSE_HAS_START;
  parse_options.source_name = jerry_string_sz ("hello.js");
  /* This example script is extracted from the middle of a file. */
  parse_options.start_line = 10;
  parse_options.start_column = 1;

  jerry_value_t parsed_code = jerry_parse (script, sizeof (script) - 1, &parse_options);
  jerry_value_free (parse_options.source_name);

  /* Run the "parsed_code" script with "jerry_run". */

  jerry_value_free (jerry_run (parsed_code));
  jerry_value_free (parsed_code);

  jerry_cleanup ();
  return 0;
}
```

**Example - function parsing**

[doctest]: # (name="02.API-REFERENCE-parse-function.c")

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  /* Specify the argument list to parse a function. */
  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_HAS_ARGUMENT_LIST;
  parse_options.argument_list = jerry_string_sz ("a, b");

  const jerry_char_t function_code[] = "return a + b;";
  jerry_value_t parsed_function = jerry_parse (function_code, sizeof (function_code) - 1, &parse_options);
  jerry_value_free (parse_options.argument_list);

  /* Use the "parsed_function" as a normal JavaScript function. */

  jerry_value_t args[] = {
    jerry_number (3),
    jerry_number (4),
  };
  jerry_size_t argc = sizeof (args) / sizeof (args[0]);
  jerry_value_t call_result = jerry_call (parsed_function,
                                          jerry_undefined(),
                                          args,
                                          argc);

  /* use the function result */
  jerry_value_free (call_result);
  jerry_value_free (parsed_function);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_parse_value](#jerry_parse_value)
- [jerry_run](#jerry_run)
- [jerry_parse_options_t](#jerry_parse_options_t)

## jerry_parse_value

**Summary**

Parse a script, module, or function and create a compiled code using a string value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_parse_value (const jerry_value_t source_value,
                   const jerry_parse_options_t *options_p);
```

- `source_value` - string value, containing source code to parse (only string values are accepted).
- `options_p` - additional parsing options, can be NULL if not used
- return value
  - function object value, if script was parsed successfully,
  - thrown exception, otherwise

*Introduced in version 3.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t script_value = jerry_string_sz ("print ('Hello, World!');");

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_STRICT_MODE | JERRY_PARSE_HAS_SOURCE_NAME | JERRY_PARSE_HAS_START;
  parse_options.source_name = jerry_string_sz ("hello.js");
  /* This example script is extracted from the middle of a file. */
  parse_options.start_line = 10;
  parse_options.start_column = 1;

  jerry_value_t parsed_code = jerry_parse_value (script_value, &parse_options);
  jerry_value_free (parse_options.source_name);
  jerry_value_free (script_value);
  jerry_value_free (parsed_code);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_parse](#jerry_parse)
- [jerry_run](#jerry_run)
- [jerry_parse_options_t](#jerry_parse_options_t)

## jerry_run

**Summary**

Run a Script or Module created by [jerry_parse](#jerry_parse).

*Note*:
- The code should be previously parsed with `jerry_parse`.
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_run (const jerry_value_t func_val);
```

- `func_val` - function to run
- return value
  - result of bytecode, if run was successful
  - thrown exception, otherwise

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  const jerry_char_t script[] = "print ('Hello, World!');";

  /* Initialize engine */
  jerry_init (JERRY_INIT_EMPTY);

  /* Setup Global scope code */
  jerry_value_t parsed_code = jerry_parse (script, sizeof (script) - 1, NULL);

  if (!jerry_value_is_exception (parsed_code))
  {
    /* Execute the parsed source code in the Global scope */
    jerry_value_t ret_value = jerry_run (parsed_code);

    /* Returned value must be freed */
    jerry_value_free (ret_value);
  }

  /* Parsed source code must be freed */
  jerry_value_free (parsed_code);

  /* Cleanup engine */
  jerry_cleanup ();
}
```

**See also**

- [jerry_parse](#jerry_parse)


## jerry_eval

**Summary**

Perform JavaScript `eval` function call (ECMA-262 v5.1 sec-15.1.2.1).

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_eval (const jerry_char_t *source_p,
            size_t source_size,
            uint32_t parse_opts);
```

- `source_p` - source code to evaluate, it must be a valid utf8 string.
- `source_size` - length of the source code
- `parse_opts` - combination of [jerry_parse_option_enable_feature_t](#jerry_parse_option_enable_feature_t) flags.
                 The following flags are allowed: JERRY_PARSE_STRICT_MODE
- return value - result of eval, may be an exception value.

**Example**

```c
{
  jerry_value_t ret_val = jerry_eval (str_to_eval,
                                      strlen (str_to_eval),
                                      JERRY_PARSE_NO_OPTS);
}
```

**See also**

- [jerry_function_external](#jerry_function_external)
- [jerry_external_handler_t](#jerry_external_handler_t)

## jerry_run_jobs

**Summary**

Run enqueued Promise tasks until the first thrown exception or until all tasks get executed.

*Important Note*: The task queue is not guaranteed to be empty after the function call has returned a value.
Therefore, this function is best used in a while loop, handling each exception it returns, until the queue is empty.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_run_jobs (void)
```

- return value - result of last executed job, may be an exception value.

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_run_all_enqueued_jobs` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "new Promise(function(f,r) { f('Hello, World!'); }).then(function(x) { print(x); });";

  jerry_value_t parsed_code = jerry_parse (script, sizeof (script) - 1, NULL);
  jerry_value_t script_value = jerry_run (parsed_code);

  jerry_value_t job_value;
  while (true)
  {
    job_value = jerry_run_jobs ();

    if (jerry_value_is_exception (job_value))
    {
      if (jerry_value_is_abort (job_value))
      {
        // Terminate the engine
      }

      // Handle the exception here
    }
    else
    {
      // The queue should be empty now
      break;
    }
  }

  jerry_value_free (job_value);
  jerry_value_free (script_value);
  jerry_value_free (parsed_code);

  jerry_cleanup ();
  return 0;
}
```


# Get the global context

## jerry_current_realm

**Summary**

Get the Global object.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_current_realm (void);
```

- return value - api value of global object

*Renamed in version 3.0, it was previously known as `jerry_get_global_object` in earlier versions.*

**Example**

```c
{
  jerry_value_t glob_obj_val = jerry_current_realm ();

  ... // Do something with global object, ex: add properties

  jerry_value_free (glob_obj_val);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_object_define_own_prop](#jerry_object_define_own_prop)


# Checker functions

Functions to check the type of an API value ([jerry_value_t](#jerry_value_t)).

## jerry_value_is_abort

**Summary**

Returns whether the given `jerry_value_t` is an abort exception value.

**Prototype**

```c
bool
jerry_value_is_abort (const jerry_value_t value);
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an abort exception
  - false, otherwise

*Introduced in version 2.0*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_abort (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)
- [jerry_value_is_exception](#jerry_value_is_exception)

## jerry_value_is_array

**Summary**

Returns whether the given `jerry_value_t` is an array.

**Prototype**

```c
bool
jerry_value_is_array (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an array
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_array (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)

## jerry_value_is_arraybuffer

**Summary**

Returns whether the given `jerry_value_t` is an ArrayBuffer object.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
bool
jerry_value_is_arraybuffer (const jerry_value_t value)
```

- `value` - api value to check.
- return value
  - true, if the given `jerry_value_t` is an ArrayBuffer object.
  - false, otherwise

*Introduced in version 2.0*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_arraybuffer (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_arraybuffer](#jerry_arraybuffer)
- [jerry_arraybuffer_external](#jerry_arraybuffer_external)

## jerry_value_is_shared_arraybuffer

**Summary**

Returns whether the given `jerry_value_t` is a SharedArrayBuffer object.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
bool
jerry_value_is_shared_arraybuffer (const jerry_value_t value);
```

- `value` - api value to check.
- return value
    - true, if the given `jerry_value_t` is a SharedArrayBuffer object.
    - false, otherwise

*Introduced in version 3.0*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_shared_arraybuffer (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_shared_arraybuffer](#jerry_shared_arraybuffer)
- [jerry_shared_arraybuffer_external](#jerry_shared_arraybuffer_external)


## jerry_value_is_boolean

**Summary**

Returns whether the given `jerry_value_t` is a boolean value.

**Prototype**

```c
bool
jerry_value_is_boolean (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a boolean value
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_boolean (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)

## jerry_value_is_true

**Summary**

Returns whether the given `jerry_value_t` is a true value.

**Prototype**

```c
bool
jerry_value_is_true (const jerry_value_t value);
```

- `value` - api value
- return value
    - true, if the given `jerry_value_t` is true value
    - false, otherwise

*Introduced in version 3.0*. Replaces the `jerry_get_boolean_value` method.

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_true (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)

## jerry_value_is_false

**Summary**

Returns whether the given `jerry_value_t` is a false value.

**Prototype**

```c
bool
jerry_value_is_false (const jerry_value_t value);
```

- `value` - api value
- return value
    - true, if the given `jerry_value_t` is false value
    - false, otherwise

*Introduced in version 3.0*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_false (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)

## jerry_value_is_constructor

**Summary**

Returns whether the given `jerry_value_t` is a constructor function.

**Prototype**

```c
bool
jerry_value_is_constructor (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a constructor
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_constructor (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)

## jerry_value_is_dataview

**Summary**

Returns whether the given `jerry_value_t` is a DataView object value.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_DATAVIEW`) and can be checked
  in runtime with the `JERRY_FEATURE_DATAVIEW` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
bool
jerry_value_is_dataview (const jerry_value_t value)
```

- `value` - API value
- return value
  - true, if the given `jerry_value_t` is a DataView object
  - false, otherwise

*Introduced in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t arraybuffer = jerry_arraybuffer (16);
  jerry_value_t dataview = jerry_dataview (arraybuffer, 0, 16);

  if (jerry_value_is_dataview (dataview))
  {
    // usage of dataview
  }

  jerry_value_free (dataview);
  jerry_value_free (arraybuffer);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_dataview](#jerry_dataview)


## jerry_value_is_exception

**Summary**

Returns whether the given `jerry_value_t` is an exception value.

**Prototype**

```c
bool
jerry_value_is_exception (const jerry_value_t value);
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an exception value.
  - false, otherwise

*Renamed in version 3.0 from `jerry_value_is_error` to `jerry_value_is_exception`*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_exception (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)
- [jerry_value_is_abort](#jerry_value_is_abort)

## jerry_value_is_function

**Summary**

Returns whether the given `jerry_value_t` is a function.

**Prototype**

```c
bool
jerry_value_is_function (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a function
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_function (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)


## jerry_value_is_async_function

**Summary**

Returns whether the given `jerry_value_t` is an async function.

**Prototype**

```c
bool
jerry_value_is_async_function (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an async function
  - false, otherwise

*Introduced in version 2.4*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_async_function (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)

## jerry_value_is_number

**Summary**

Returns whether the given `jerry_value_t` is a number.

**Prototype**

```c
bool
jerry_value_is_number (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a number
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_number (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)


## jerry_value_is_null

**Summary**

Returns whether the given `jerry_value_t` is a null value.

**Prototype**

```c
bool
jerry_value_is_null (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a null
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_null (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)


## jerry_value_is_object

**Summary**

Returns whether the given `jerry_value_t` is an object value.

**Prototype**

```c
bool
jerry_value_is_object (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an object
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_object (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)


## jerry_value_is_promise

**Summary**

Returns whether the given `jerry_value_t` is a promise value.

**Prototype**

```c
bool
jerry_value_is_promise (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a promise
  - false, otherwise

*Introduced in version 2.0*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_promise (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_promise](#jerry_promise)


## jerry_value_is_proxy

**Summary**

Returns whether the given `jerry_value_t` is a proxy value.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_PROXY`) and can be checked
  in runtime with the `JERRY_FEATURE_PROXY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
bool
jerry_value_is_proxy (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a proxy object
  - false, otherwise

*Introduced in version 2.3*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_proxy (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_proxy](#jerry_proxy)
- [jerry_proxy_custom](#jerry_proxy_custom)


## jerry_value_is_string

**Summary**

Returns whether the given `jerry_value_t` is a string value.

**Prototype**

```c
bool
jerry_value_is_string (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is a string
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_string (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)


## jerry_value_is_symbol

**Summary**

Returns whether the given `jerry_value_t` is a symbol value.

**Prototype**

```c
bool
jerry_value_is_symbol (const jerry_value_t value)
```

- `value` - API value
- return value
  - true, if the given `jerry_value_t` is a symbol
  - false, otherwise

*Introduced in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t string_value = jerry_string_sz ("Symbol description string");
  jerry_value_t symbol_value = jerry_symbol_with_description (string_value);

  jerry_value_free (string_value);

  if (jerry_value_is_symbol (symbol_value))
  {
    // usage of symbol_value
  }

  jerry_value_free (symbol_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_symbol](#jerry_symbol)


## jerry_value_is_bigint

**Summary**

Returns whether the given `jerry_value_t` is a bigint value.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_BIGINT`) and can be checked
  in runtime with the `JERRY_FEATURE_BIGINT` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
bool
jerry_value_is_bigint (const jerry_value_t value)
```

- `value` - API value
- return value
  - true, if the given `jerry_value_t` is a BigInt
  - false, otherwise

*Introduced in version 2.4*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t string_value = jerry_string_sz ("12345678");
  jerry_value_t bigint_value = jerry_value_to_bigint (string_value);

  jerry_value_free (string_value);

  if (jerry_value_is_bigint (bigint_value))
  {
    // usage of bigint_value
  }

  jerry_value_free (bigint_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_bigint](#jerry_bigint)
- [jerry_value_to_bigint](#jerry_value_to_bigint)


## jerry_value_is_typedarray

**Summary**

Checks whether the given `jerry_value_t` is a TypedArray object or not.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
bool
jerry_value_is_typedarray (const jerry_value_t value)
```

- `value` - object to check
- return value
  - true, if the given `jerry_value_t` is a TypedArray object.
  - false, otherwise

*Introduced in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t value = jerry_typedarray (JERRY_TYPEDARRAY_UINT16, 15);

  if (jerry_value_is_typedarray (value))
  {
    /* "value" is a typedarray. */
  }

  jerry_value_free (value);

  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_typedarray](#jerry_typedarray)


## jerry_container_type

**Summary**

Checks whether the given `jerry_value_t` is the given `jerry_container_type_t` type container object.

*Note*:
- This API function depends on a build option (`JERRY_BUILTIN_CONTAINER`) and can be checked
  runtime with the `JERRY_FEATURE_MAP, JERRY_FEATURE_SET, JERRY_FEATURE_WEAKMAP, JERRY_FEATURE_WEAKSET`
  feature enum values.
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
jerry_container_type_t
jerry_container_type (const jerry_value_t value)
```

- `value` - Container object
- return value
  - The corresponding enum value of `jerry_container_type_t`, or `JERRY_CONTAINER_TYPE_INVALID` if the container
    was not a valid container object.

*Introduced in version 2.3*.

*Renamed in version 3.0, it was previously known as `jerry_get_container_type` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"
int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t value = jerry_container (JERRY_CONTAINER_TYPE_MAP, NULL, 0);

  if (jerry_container_type (value) == JERRY_CONTAINER_TYPE_MAP)
  {
    /* "value" is a map. */
  }

  jerry_value_free (value);

  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_container](#jerry_container)
- [jerry_container_type_t](#jerry_container_type_t)


## jerry_value_is_undefined

**Summary**

Returns whether the given `jerry_value_t` is an undefined value.

**Prototype**

```c
bool
jerry_value_is_undefined (const jerry_value_t value)
```

- `value` - api value
- return value
  - true, if the given `jerry_value_t` is an undefined value
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_undefined (value))
  {
    ...
  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)

## jerry_value_type

**Summary**

Returns the JavaScript type
for a given value as a [jerry_type_t](#jerry_type_t) enum value.

This is a similar operation to the 'typeof' operator
in the standard with an exception that the 'null'
value has its own enum value.

**Prototype**

```c
jerry_type_t
jerry_value_type (const jerry_value_t value);
```

- `value` - JavaScript value to check.
- return value
  - One of the [jerry_type_t](#jerry_type_t) value.

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_value_get_type` in earlier versions.*

**Example**

```c
{
  jerry_value_t number = jerry_number (3.3);

  jerry_type_t type_info = jerry_value_type (number);

  if (type_info == JERRY_TYPE_NUMBER)
  {
    /* ... */
  }

  jerry_value_free (number);
}
```

**See also**

- [jerry_type_t](#jerry_type_t)

## jerry_object_type

**Summary**

Returns the **object** type
for a given value as a [jerry_object_type_t](#jerry_object_type_t) enum value.

Note: For non-object parameters `JERRY_OBJECT_TYPE_NONE` is returned.
Note: the returned type can be checked for more detailed type information in the following cases:
  - `JERRY_OBJECT_TYPE_CONTAINER`, *see also:* [jerry_container_get_type](#jerry_container_get_type)
  - `JERRY_OBJECT_TYPE_FUNCTION`, *see also:* [jerry_function_type](#jerry_function_type)
  - `JERRY_OBJECT_TYPE_ITERATOR`, *see also:* [jerry_iterator_type](#jerry_iterator_type)
  - `JERRY_OBJECT_TYPE_TYPEDARRAY`, *see also:* [jerry_typedarray_type](#jerry_typedarray_type)

**Prototype**

```c
jerry_object_type_t
jerry_object_type (const jerry_value_t value);
```

- `value` - JavaScript value to check.
- return value
  - One of the [jerry_object_type_t](#jerry_object_type_t) value.

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_object_get_type` in earlier versions.*

**Example**

```c
{
  jerry_value_t object = jerry_object ();

  jerry_object_type_t object_type_info = jerry_object_type (object);

  if (type_info == JERRY_OBJECT_TYPE_GENERIC)
  {
    /* ... */
  }

  jerry_value_free (object);
}
```

**See also**

- [jerry_object_type_t](#jerry_object_type_t)

## jerry_function_type

**Summary**

Returns the **function** type
for a given value as a [jerry_function_type_t](#jerry_function_type_t) enum value.

**Prototype**

```c
jerry_function_type_t
jerry_function_type (const jerry_value_t value);
```

- `value` - JavaScript value to check.
- return value
  - One of the [jerry_function_type_t](#jerry_function_type_t) value.

Note: For non-function parameters `JERRY_FUNCTION_TYPE_NONE` is returned.

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_function_get_type` in earlier versions.*

**Example**

```c
{
  const jerry_char_t script[] = "function f() {}; f";
  jerry_value_t function_object = jerry_eval (script, sizeof (script) - 1, JERRY_PARSE_NO_OPTS);

  jerry_function_type_t function_type_info = jerry_function_type (function_object);

  if (type_info == JERRY_FUNCTION_TYPE_GENERIC)
  {
    /* ... */
  }

  jerry_value_free (function_object);
}
```

**See also**

- [jerry_function_type_t](#jerry_function_type_t)

## jerry_iterator_type

**Summary**

Returns the **iterator** type
for a given value as a [jerry_iterator_type_t](#jerry_iterator_type_t) enum value.

**Prototype**

```c
jerry_iterator_type_t
jerry_iterator_type (const jerry_value_t value);
```

- `value` - JavaScript value to check.
- return value
  - One of the [jerry_iterator_type_t](#jerry_iterator_type_t) value.

Note: For non-iterator parameters `JERRY_ITERATOR_TYPE_NONE` is returned.

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_iterator_get_type` in earlier versions.*

**Example**

```c
{
  const jerry_char_t script[] = "[1, 2, 3].values()";
  jerry_value_t iterator = jerry_eval (script, sizeof (script) - 1, JERRY_PARSE_NO_OPTS);

  jerry_iterator_type_t iterator_type_info = jerry_iterator_type (iterator);

  if (type_info == JERRY_ITERATOR_TYPE_ARRAY)
  {
    /* ... */
  }

  jerry_value_free (iterator);
}
```

**See also**

- [jerry_iterator_type_t](#jerry_iterator_type_t)

## jerry_feature_enabled

**Summary**

Returns whether the specified compile time feature is enabled.

**Prototype**

```c
bool
jerry_feature_enabled (const jerry_feature_t feature);
```

- `feature` - jerry feature
- return value
  - true, if the given `jerry_feature_t` is enabled
  - false, otherwise

*Renamed in version 3.0 from `jerry_is_feature_enabled` to `jerry_feature_enabled`*.

**Example**

```c
{
  /* ... */
  jerry_feature_t feature = JERRY_FEATURE_SNAPSHOT_SAVE;

  if (jerry_feature_enabled (feature))
  {
    /* ... */
  }

}
```

**See also**

- [jerry_feature_t](#jerry_feature_t)


# Binary operations

## jerry_binary_op

**Summary**

Perform binary operation on the given operands (==, ===, <, >, etc.).

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_binary_op (jerry_binary_op_t op,
                 const jerry_value_t lhs,
                 const jerry_value_t rhs);
```

- `op` - binary operation
- `lhs` - left-hand side operand
- `rhs` - right-hand side operand
- return value
  - exception, if operation is unsuccessful or unsupported
  - the result of the binary operation on the given operands otherwise

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_binary_operation` in earlier versions.*

**Example - JERRY_BIN_OP_EQUAL**

```c
{
  jerry_value_t value1;
  jerry_value_t value2;
  ... // create or copy value
  jerry_value_t result = jerry_binary_op (JERRY_BIN_OP_EQUAL, value1, value2)

  if (!jerry_value_is_exception (result))
  {
    if (jerry_value_is_true (result))
    {
       // value1 and value2 are equal
    }
    else
    {
      // value1 and value2 are NOT equal
    }
  }
  else
  {
    ... // handle error
  }

  jerry_value_free (value1);
  jerry_value_free (value2);
  jerry_value_free (result);
}
```

**Example - JERRY_BIN_OP_INSTANCEOF**

[doctest]: # ()

```c
#include "jerryscript.h"

static jerry_value_t
my_constructor (const jerry_call_info_t *call_info_p,
                const jerry_value_t argv[],
                const jerry_length_t argc)
{
  return jerry_undefined ();
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t base_obj = jerry_object ();
  jerry_value_t constructor = jerry_function_external (my_constructor);

  /* External functions does not have a prototype by default, so we need to create one */
  jerry_value_t prototype_str = jerry_string_sz ("prototype");
  jerry_value_free (jerry_object_set (constructor, prototype_str, base_obj));
  jerry_value_free (prototype_str);

  /* Construct the instance. */
  jerry_value_t instance_val = jerry_construct (constructor, NULL, 0);

  /* Call the API function of 'instanceof'. */
  jerry_value_t is_instance = jerry_binary_op (JERRY_BIN_OP_INSTANCEOF,
                                               instance_val,
                                               constructor);
  if (!jerry_value_is_exception (is_instance)
      && jerry_value_is_true (is_instance))
  {
    /* ... */
  }

  /* Free all of the jerry values and cleanup the engine. */
  jerry_value_free (base_obj);
  jerry_value_free (constructor);
  jerry_value_free (instance_val);
  jerry_value_free (is_instance);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_binary_op_t](#jerry_binary_op_t)


# Exception manipulation functions

*Updated in version 2.0*: The exception handling and manipulation have been modified, and the old methods were replaced.

## jerry_throw_abort

**Summary**

Create (api) abort from a value.

This function creates an API abort value from an API value. The second argument defines
whether the input value must be released or not. If it is set to `true`,
then a [`jerry_value_free`](#jerry_value_free) function will be called
for the first argument, so the api value won't be available after the call of
`jerry_throw_abort`. The second argument should be false if both value
and created abort value are needed.

**Prototype**

```c
jerry_value_t
jerry_throw_abort (jerry_value_t value, bool take_ownersip);
```

- `value` - api value
- `take_ownership` - raw boolean, defines whether input value should be copied
- return value - abort exception value

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_create_abort_from_value` in earlier versions.*

**Example 1**

```c
{
  jerry_value_t value;
  ... // create or copy value

  jerry_value_t abort = jerry_throw_abort (value, true);
  // using the 'value' variable after release is invalid.

  jerry_value_free (abort);
}
```

**Example 2**

```c
{
  jerry_value_t value;
  ... // create or copy value

  jerry_value_t abort = jerry_throw_abort (value, false);
  // both 'abort' and 'value' can be used and must be released when they are no longer needed

  jerry_value_free (abort);
  jerry_value_free (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)
- [jerry_exception_value](#jerry_exception_value)
- [jerry_throw](#jerry_throw)

## jerry_throw

**Summary**

Create exception from a value.

This function creates an API exception value from an API value. The second argument defines
whether the input value should be taken by the exception or copied. If it is set to `true`,
then then the ownership of the argument value is taken, so it won't be available after the call to
`jerry_throw`. The second argument should be false if both the original value and created exception are needed.

**Prototype**

```c
jerry_value_t
jerry_throw (jerry_value_t value, bool take_ownership);
```

- `value` - api value
- `take_ownership` - raw boolean, defines whether input value should be copied
- return value - exception value

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_get_value_from_error` in earlier versions.*

**Example 1**

```c
{
  jerry_value_t value;
  ... // create or copy value

  jerry_value_t exception = jerry_throw (value, true);
  // using the 'value' variable after release is invalid.


  jerry_value_free (exception);
}
```

**Example 2**

```c
{
  jerry_value_t value;
  ... // create or copy value

  jerry_value_t exception = jerry_throw_value (value, false);
  // both 'exception' and 'value' can be used and must be released when they are no longer needed

  jerry_value_free (exception);
  jerry_value_free (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)
- [jerry_exception_value](#jerry_exception_value)
- [jerry_throw_abort](#jerry_throw_abort)

## jerry_error_type

**Summary**

Returns the type of the Error object if possible. If a non-error object is used as the input for the function the
method will return `JERRY_ERROR_NONE` indicating that the value was not an Error object.

Note: If an exception value is passed as an argument, the function will inspect the contained value instead.

**Prototype**

```c
jerry_error_t
jerry_error_type (const jerry_value_t value);
```

- `value` - api value (possible exception object)
- return value
  - JERRY_ERROR_NONE if the input is not an error object
  - one of the [jerry_error_t](#jerry_error_t) value

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_get_error_type` in earlier versions.*

**Example**

```c
{
  jerry_value_t error_obj = jerry_error_sz (JERRY_ERROR_RANGE, "error msg");
  jerry_error_t error_type = jerry_error_type (error_obj);

  // error_type is now JERRY_ERROR_RANGE.

  jerry_value_free (error_obj);
}
```

**See also**

- [jerry_error](#jerry_error)
- [jerry_value_is_exception](#jerry_value_is_exception)

## jerry_exception_value

**Summary**

Get the value contained in an exception.

Many API functions cannot be called with an exception value.
This function extracts the API value from an exception. The second argument defines
whether the input exception value should be released or not. If it is set to `true`,
then a [`jerry_value_free`](#jerry_value_free) function will be called
for the first argument, so the exception value won't be available after the call of
`jerry_exception_value`. The second argument should be false if both the exception
and its contained value are needed.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_exception_value (jerry_value_t value, bool release)
```

- `value` - exception value
- `release` - raw boolean, defines whether input value must be released
- return value - api value

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_create_error_from_value` in earlier versions.*

**Example 1**

```c
{
  jerry_value_t value;
  ... // create or copy value

  jerry_value_t exception = jerry_throw_value (value, true);
  jerry_value_t value_from_error = jerry_exception_value (exception, true);
  // using the 'exception' variable after release is invalid.

  jerry_value_free (value_from_error);
}
```

**Example 2**

```c
{
  jerry_value_t value;
  ... // create or copy value

  jerry_value_t exception = jerry_throw_value (value, true);
  jerry_value_t value_from_error = jerry_exception_value (exception, false);
  // both 'exception' and 'value_from_error' can be used and must be released when they are no longer needed

  jerry_value_free (value_from_error);
  jerry_value_free (exception);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)
- [jerry_throw](#jerry_throw)
- [jerry_throw_abort](#jerry_throw_abort)

## jerry_error_on_created

**Summary**

Set the decorator callback for newly created Error objects. The operation of the callback
is described in [jerry_error_object_created_cb_t](#jerry_error_object_created_cb_t).

**Prototype**

```c
void jerry_error_on_created (jerry_error_object_created_cb_t callback, void *user_p);
```

- `callback` - callback function, the previously set value is overwritten, and setting NULL
               disables the operation
- `user_p` - pointer passed to the callback function, can be NULL

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_set_error_object_created_callback` in earlier versions.*

**Example**

```c
static void
error_object_created_callback (const jerry_value_t error_object) /**< new error object */
                               void *user_p) /**< user pointer */
{
  (void) error_object;
  (void) user_p;
  printf ("Notification: a new error is created\n");
} /* error_object_created_callback */

void main(void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_error_on_created (error_object_created_callback, NULL);

  jerry_value_free (jerry_error_sz (JERRY_ERROR_COMMON, "Message"));

  jerry_cleanup ();
} /* main */
```

**See also**

- [jerry_error_object_created_cb_t](#jerry_error_object_created_cb_t)

## jerry_on_throw

**Summary**

The callback passed to this function is called when an exception is thrown
in ECMAScript code. The callback is not called again until the value is
caught. See: [jerry_throw_cb_t](#jerry_throw_cb_t).

*Note*:
- This API depends on a build option (`JERRY_VM_THROW`) and can be checked
  in runtime with the `JERRY_FEATURE_VM_THROW` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
void
jerry_on_throw (jerry_throw_cb_t throw_cb,
                void *user_p);
```

- `throw_cb` - callback which is called on throws (passing NULL disables this feature)
- `user_p` - user pointer passed to the `throw_cb` function

*Renamed in version 3.0, it was previously known as `jerry_set_vm_throw_callback` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

static void
vm_throw_callback (const jerry_value_t error_value, /**< captured exception */
                   void *user_p) /**< user pointer */
{
  (void) error_value;

  /* Counts the number of throws. */
  int *counter_p = (int *) user_p;
  (*counter_p)++;
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  int counter = 0;
  jerry_on_throw (vm_throw_callback, &counter);

  const jerry_char_t script[] = "try { throw new Error('1') } catch (e) { throw new Error('2') }";

  jerry_value_free (jerry_eval (script, sizeof (script) - 1, JERRY_PARSE_NO_OPTS));

  /* The counter contains 2. */

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_throw_cb_t](#jerry_throw_cb_t)
- [jerry_exception_is_captured](#jerry_exception_is_captured)
- [jerry_exception_allow_capture](#jerry_exception_allow_capture)


## jerry_exception_is_captured

**Summary**

Checks whether the callback set by [jerry_on_throw](#jerry_on_throw)
captured the exception.

*Note*:
- This API depends on a build option (`JERRY_VM_THROW`) and can be checked
  in runtime with the `JERRY_FEATURE_VM_THROW` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
bool jerry_exception_is_captured (jerry_value_t value);
```

- `value` - api value (should be an exception)
- return value
  - true, if the vm throw callback captured the exception
  - false, otherwise

*Renamed in version 3.0 from `jerry_error_is_throw_captured` to `jerry_exception_is_captured`*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

static void
vm_throw_callback (const jerry_value_t error_value, /**< captured exception */
                   void *user_p) /**< user pointer */
{
  (void) error_value;
  (void) user_p;
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  int counter = 0;
  jerry_on_throw (vm_throw_callback, &counter);

  const jerry_char_t script[] = "throw new Error()";
  jerry_value_t result_value = jerry_eval (script, sizeof (script) - 1, JERRY_PARSE_NO_OPTS);

  if (jerry_exception_is_captured (result_value))
  {
    /* Code enters here, because the vm_throw_callback function is called. */
  }

  jerry_value_free (result_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_on_throw](#jerry_on_throw)
- [jerry_exception_allow_capture](#jerry_exception_allow_capture)

## jerry_exception_allow_capture

**Summary**

Sets whether the callback set by [jerry_on_throw](#jerry_on_throw)
should capture the exception or not.

*Note*:
- This API depends on a build option (`JERRY_VM_THROW`) and can be checked
  in runtime with the `JERRY_FEATURE_VM_THROW` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
void jerry_exception_allow_capture (jerry_value_t value, bool should_capture);
```

- `value` - api value (should be an exception)
- `should_capture` - callback should capture this error

*Renamed in version 3.0, it was previously known as `jerry_error_set_throw_capture` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

static void
vm_throw_callback (const jerry_value_t error_value, /**< captured exception */
                   void *user_p) /**< user pointer */
{
  (void) error_value;
  (void) user_p;
}

static jerry_value_t
throw_exception (const jerry_call_info_t *call_info_p, /**< call info */
                 const jerry_value_t argv[], /**< argument list */
                 const jerry_length_t argc) /**< argument count */
{
  (void) call_info_p;
  (void) argv;
  (void) argc;

  jerry_value_t result_value = jerry_throw_sz (JERRY_ERROR_COMMON, "Error!");

  /* Ignore calling the vm_throw_callback function. */
  jerry_exception_allow_capture (result_value, false);
  return result_value;
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  int counter = 0;
  jerry_on_throw (vm_throw_callback, &counter);

  jerry_value_t global_object_value = jerry_current_realm ();
  jerry_value_t function_value = jerry_function_external (throw_exception);
  jerry_value_t function_name_value = jerry_string_sz ("throw_exception");

  jerry_value_free (jerry_object_set (global_object_value, function_name_value, function_value));
  jerry_value_free (function_name_value);
  jerry_value_free (function_value);
  jerry_value_free (global_object_value);

  const jerry_char_t script[] = "throw_exception()";
  jerry_value_free (jerry_eval (script, sizeof (script) - 1, JERRY_PARSE_NO_OPTS));

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_on_throw](#jerry_on_throw)
- [jerry_exception_is_captured](#jerry_exception_is_captured)


# Getter functions of 'jerry_value_t'

Get raw data from API values.

## jerry_value_as_number

**Summary**

Gets the number value of the given `jerry_value_t` parameter as a raw double.

If the argument passed is not a number `0.0` will be returned.

**Prototype**

```c
double
jerry_value_as_number (const jerry_value_t value);
```

- `value` - api value
- return value
  - the number value of the given `jerry_value_t` parameter as a raw double.
  - `0.0` if the api value passed is not a number.

*Renamed in version 3.0, it was previously known as `jerry_get_number_value` in earlier versions.*

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  if (jerry_value_is_number (value))
  {
    double raw_value = jerry_value_as_number (value);

    ... // usage of raw value

  }

  jerry_value_free (value);
}
```

**See also**

- [jerry_value_is_number](#jerry_value_is_number)
- [jerry_value_free](#jerry_value_free)


# Functions for string values

## jerry_string_size

**Summary**

Get the size of a string. Returns zero, if the value parameter is not a string.
This is effectively the number of bytes required to store the string's characters.

**Prototype**

```c
jerry_size_t jerry_string_size (const jerry_value_t value, jerry_encoding_t encoding);
```
- `value` - api value
- `encoding` - the requested [encoding](#jerry_encoding_t) for the string
- return value - number of bytes in the buffer needed to represent the string.

*Renamed in version 3.0, it was previously known as `jerry_get_string_size` in earlier versions.*

**Example**

```c
{
  const char char_array[] = "a string";
  jerry_value_t string = jerry_string_sz (char_array);

  jerry_size_t string_size = jerry_string_size (string, JERRY_ENCODING_CESU8);

  ... // usage of string_size

  jerry_value_free (string);
}
```

**See also**

- [jerry_string_sz](#jerry_string_sz)
- [jerry_string_length](#jerry_string_length)
- [jerry_validate_string](#jerry_validate_string)


## jerry_string_length

**Summary**

Get the length of a string. Returns zero, if the value parameter is not a string.

*Note*:
- The difference from [jerry_string_size](#jerry_string_size) is that it
  returns the number of bytes used for the string.
- This is **not** the number of bytes required to store the string.

**Prototype**

```c
jerry_length_t
jerry_string_length (const jerry_value_t value);
```

- `value` - api value
- return value - number of characters in the string

*Renamed in version 3.0, it was previously known as `jerry_get_string_length` in earlier versions.*

**Example**

```c
{
  const char char_array[] = "a string";
  jerry_value_t string = jerry_string_sz (char_array);

  jerry_length_t string_length = jerry_string_length (string);

  ... // usage of string_length

  jerry_value_free (string);
}
```

**See also**

- [jerry_string_sz](#jerry_string_sz)
- [jerry_string_size](#jerry_string_size)
- [jerry_validate_string](#jerry_validate_string)


## jerry_string_substr

**Summary**

Create a substring of the input string value.
Return an empty string if input value is not a string.

**Prototype**

```c
jerry_size_t jerry_string_substr (const jerry_value_t value, jerry_length_t start, jerry_length_t end);
```

- `value` - the input string value
- `start` - start position of the substring
- `end` - end position of the substring
- return value - created string

*Introduced in version 3.0. There were `jerry_substring_to_char_buffer` and `jerry_substring_to_utf8_char_buffer` in previous versions with a similar functionality.

**Example**

```c
{
  jerry_value_t str = jerry_string_sz ("Demo string");

  // Read the string into a byte buffer.
  jerry_value_t sub_str = jerry_string_substr (str, 0, 3);

  // usage of 'sub_str'

  jerry_value_free (sub_str);
  jerry_value_free (str);
}
```

**See also**

- [jerry_string_sz](#jerry_string_sz)


## jerry_string_iterate

**Summary**

Iterate over the input string value in the specified encoding, visiting each unit of the encoded string once. If the input value is not a string, the function will do nothing.


**Prototype**

```c
void jerry_string_iterate (const jerry_value_t value,
                           jerry_encoding_t encoding,
                           jerry_string_iterate_cb_t callback,
                           void *user_p);
```

- `value` - the input string value
- `encoding` - the requested [encoding](#jerry_encoding_t) for the string
- `callback` - callback function called for each byte of the encoded string, see [jerry_string_iterate_cb_t](#jerry_string_iterate_cb_t)
- `user_p` - user pointer passed to the callback function

*Introduced in version 3.0.*

**Example**

```c
static void my_str_iterator_cb_func (uint32_t value, void *user_p)
{
  // usage of the encoded byte value and the user pointer (if non-null)
}

...

void foo () {
  jerry_value_t string = jerry_string_sz ("Demo string");

  jerry_string_iterate (string, JERRY_ENCODING_UTF8, &jerryx_buffered_print, NULL);

  jerry_value_free (string);
}
```

**See also**

- [jerry_string_sz](#jerry_string_sz)
- [jerry_encoding_t](#jerry_encoding_t)
- [jerry_string_iterate_cb_t](#jerry_string_iterate_cb_t)


## jerry_string_to_buffer

**Summary**

Copy the characters of a string into a specified cesu-8 buffer.
The '\0' character could occur in the character buffer. Returns 0,
if the input value is not a string.

*Note*:
- Does not put '\0' to the end of string, the return value identifies
  the number of valid bytes in the output buffer.
- If the size of the string is larger than the size of the
  target buffer, the string will be cropped.

**Prototype**

```c
jerry_size_t
jerry_string_to_buffer (const jerry_value_t value,
                        jerry_encoding_t encoding,
                        jerry_char_t *buffer_p,
                        jerry_size_t buffer_size);
```

- `value` - input string value
- `encoding` - encoding oh the string data
- `buffer_p` - pointer to output buffer
- `buffer_size` - size of the buffer
- return value - number of bytes copied to the buffer

*Renamed in version 3.0, it was previously known as `jerry_string_to_char_buffer` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <stdio.h>
#include <stdlib.h>
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t value;
  // create or copy value
  value = jerry_string_sz ("Demo string");

  // Read the string into a byte buffer.
  jerry_size_t string_size = jerry_string_size (value, JERRY_ENCODING_CESU8);
  jerry_char_t *string_buffer_p = (jerry_char_t *) malloc (sizeof (jerry_char_t) * (string_size + 1));

  jerry_size_t copied_bytes = jerry_string_to_buffer (value, JERRY_ENCODING_CESU8, string_buffer_p, string_size);
  string_buffer_p[copied_bytes] = '\0';

  jerry_value_free (value);

  jerry_cleanup ();

  printf ("Test string: %s\n", string_buffer_p);
  free (string_buffer_p);

  return 0;
}
```

**See also**

- [jerry_encoding_t](#jerry_encoding_t)
- [jerry_string_sz](#jerry_string_sz)
- [jerry_string_size](#jerry_string_size)


## jerry_string_external_on_free

**Summary**

Sets the global callback which is called when the string data of an external
string is no longer used. It is recommended to set this function before the
first external string is created.

*Note*:
- When the callback is NULL, no function is called when an external string is freed.
- In some cases (e.g. when the string is also a magic string registered by
  [jerry_register_magic_strings](#jerry_register_magic_strings)), the callback
  is called when the string is created, not when it is released.

**Prototype**

```c
void jerry_string_external_on_free (jerry_external_string_free_cb_t callback_p);
```

- `callback_p` - callback which is called when an external string is freed.

*Renamed in version 3.0, it was previously known as `jerry_string_set_external_free_callback` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"
#include <stdio.h>

static void
external_string_free_callback (jerry_char_t *string_p, /**< string pointer */
                               jerry_size_t string_size, /**< size of the string */
                               void *user_p) /**< user pointer */
{
  printf ("External string is freed!\n");
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_string_external_on_free (external_string_free_callback);

  const char *string_p = "This is a long external string, should not be duplicated!";
  jerry_value_t external_string = jerry_string_external_sz (string_p, NULL);
  /* The external_string_free_callback is called. */
  jerry_value_free (external_string);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_external_string_free_cb_t](#jerry_external_string_free_cb_t)
- [jerry_string_user_ptr](#jerry_string_user_ptr)
- [jerry_string_external_sz](#jerry_string_external_sz)
- [jerry_string_external](#jerry_string_external)


## jerry_string_user_ptr

**Summary**

Returns the user pointer assigned to an external string.

*Note*:
- In some cases (e.g. when the string is also a magic string registered by
  [jerry_register_magic_strings](#jerry_register_magic_strings)), the
  string is a normal string without a user pointer even if it is created
  by [jerry_string_external_sz](#jerry_string_external_sz).

**Prototype**

```c
void *jerry_string_user_ptr (const jerry_value_t value,
                             bool *is_external);
```

- `value` - string value.
- `is_external` - when `is_external` is non-NULL, the referenced boolean value is set
                  to true, if `value` is an external string, and set to false otherwise
- return value
  - user pointer - if `value` argument is an external string
  - NULL - otherwise

*Renamed in version 3.0, it was previously known as `jerry_string_get_external_user_pointer` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"
#include <stdio.h>

static int user_value = 1;

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const char *string_p = "This is a long external string, should not be duplicated!";

  jerry_value_t external_string = jerry_string_external_sz (string_p, (void *) &user_value);

  bool is_external;
  void *user_p = jerry_string_user_ptr (external_string, &is_external);

  if (is_external)
  {
    /* Prints the address of user_value. */
    printf("User pointer of an external string: %p\n", user_p);
  }

  jerry_value_free (external_string);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_string_set_external_string_free_callback](#jerry_string_set_external_string_free_callback)
- [jerry_string_external_sz](#jerry_string_external_sz)
- [jerry_string_external](#jerry_string_external)


# Functions for array object values

## jerry_array_length

**Summary**

Get length of an array object. Returns zero, if the given parameter is not an array object.

**Prototype**

```c
uint32_t
jerry_array_length (const jerry_value_t value);
```

- `value` - input array value
- return value - length of the given array

*Renamed in version 3.0, it was previously known as `jerry_get_array_length` in earlier versions.*

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  uint32_t len = jerry_array_length (value);

  jerry_value_free (value);
}
```

**See also**

- [jerry_array](#jerry_array)


# Converters of 'jerry_value_t'

Functions for converting API values to another value type.

## jerry_value_to_boolean

**Summary**

Call ToBoolean operation on the api value.

**Prototype**

```c
bool
jerry_value_to_boolean (const jerry_value_t value);
```

- `value` - api value
- return value
  - true, if the logical value is true
  - false, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  bool b = jerry_value_to_boolean (value);

  jerry_value_free (value);
}

```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)

## jerry_value_to_number

**Summary**

Call ToNumber operation on the api value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_number (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted number value, if success
  - thrown exception, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  jerry_value_t number_value = jerry_value_to_number (value);

  jerry_value_free (number_value);
  jerry_value_free (value);
}

```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)

## jerry_value_to_object

**Summary**

Call ToObject operation on the api value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_object (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted object value, if success
  - thrown exception, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  jerry_value_t object_value = jerry_value_to_object (value);

  jerry_value_free (object_value);
  jerry_value_free (value);
}
```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)

## jerry_value_to_primitive

**Summary**

Call ToPrimitive operation on the api value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_primitive (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted primitive value, if success
  - thrown exception, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  jerry_value_t prim_value = jerry_value_to_primitive (value);

  jerry_value_free (prim_value);
  jerry_value_free (value);
}
```

**See also**

- [jerry_value_t](#jerry_value_t)

## jerry_value_to_string

**Summary**

Call the ToString ecma builtin operation on the api value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_string (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted string value, if success
  - thrown exception, otherwise

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  jerry_value_t string_value = jerry_value_to_string (value);

  jerry_value_free (string_value);
  jerry_value_free (value);
}
```

**See also**

- [jerry_value_to_primitive](#jerry_value_to_primitive)


## jerry_value_to_bigint

**Summary**

Call the BigInt constructor ecma builtin operation on the api value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_to_bigint (const jerry_value_t value);
```

- `value` - api value
- return value
  - converted BigInt value, if success
  - thrown exception, otherwise

*Introduced in version 2.4*.

**Example**

```c
{
  jerry_value_t value;
  ... // create or copy value

  jerry_value_t bigint_value = jerry_value_to_bigint (value);

  jerry_value_free (bigint_value);
  jerry_value_free (value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_value_is_bigint](#jerry_value_is_bigint)
- [jerry_bigint_to_digits](#jerry_bigint_to_digits)

## jerry_value_as_integer

**Summary**

Convert any number to integer number. Returns 0 if value is not number.
Based on ECMA 262 v11 7.1.5

**Prototype**

```c
double
jerry_value_as_integer (const jerry_value_t value);
```

- `value` - api value
- return
  - 0, if value is not a number
  - integer representation of the number, otherwise

*Introduced in version 2.4*.

**Example**

```c
{
  jerry_value_t number_val = jerry_number (123321);
  double number = jerry_value_as_integer (number_val);
  jerry_value_free (number_val);
}
```

## jerry_value_as_int32

**Summary**

Convert any number to int32 number. Returns 0 if value is not number.
Based on ECMA 262 v11 7.1.6

**Prototype**

```c
int32_t
jerry_value_as_int32 (const jerry_value_t value);
```

- `value` - api value
- return
  - 0, if value is not a number
  - int32 representation of the number, otherwise

*Introduced in version 2.4*.

**Example**

```c
{
  jerry_value_t number_val = jerry_number (123321);
  int32_t number = jerry_value_as_int32 (number_val);
  jerry_value_free (number_val);
}
```

## jerry_value_as_uint32

**Summary**

Convert any number to uint32 number. Returns 0 if value is not number.
Based on ECMA 262 v11 7.1.7

**Prototype**

```c
uint32_t
jerry_value_as_uint32 (const jerry_value_t value);
```

- `value` - api value
- return
  - 0, if value is not a number
  - uint32 representation of the number, otherwise

*Introduced in version 2.4*.

**Example**

```c
{
  jerry_value_t number_val = jerry_number (123321);
  uint32_t number = jerry_value_as_uint32 (number_val);
  jerry_value_free (number_val);
}
```

# Functions for module objects

These APIs all depend on module support.

## jerry_module_link

**Summary**

Link modules to their dependencies. The dependencies are resolved by a user callback.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.
- This API depends on a build option (`JERRY_MODULE_SYSTEM`) and can be checked
  in runtime with the `JERRY_FEATURE_MODULE` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
jerry_value_t jerry_module_link (const jerry_value_t module_val,
                                 jerry_module_resolve_cb_t callback, void *user_p)
```

- `module_val` - module object in unlinked state
- `callback` - user callback which is called to resolve dependencies,
               uses `jerry_port_module_resolve` when NULL is passed
- `user_p` - user pointer passed to the callback
- return
  - true - if linking is successful
  - exception - otherwise

*Introduced in version 3.0*.

**Example**

[doctest]: # (test="link")

```c
#include <jerryscript.h>

static jerry_value_t
module_resolve_callback (const jerry_value_t specifier,
                         const jerry_value_t referrer,
                         void *user_data_p)
{
  /* In this case, the specifier contains 'b.mjs', and the referrer is the module
   * created in the main() function below. Normally the specifier string should be
   * extended to a full file system path, and it should be checked whether a module
   * corresponding to this path has been loaded already. For simplicity, this function
   * returns with a new module. */

  const jerry_char_t script[] = "export var a = 5";

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_MODULE | JERRY_PARSE_HAS_SOURCE_NAME;
  parse_options.source_name = jerry_string_sz ("b.mjs");

  jerry_value_t result = jerry_parse (script, sizeof (script) - 1, &parse_options);
  jerry_value_free (parse_options.source_name);
  return result;
} /* module_resolve_callback */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "import a from 'b.mjs'";

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_MODULE | JERRY_PARSE_HAS_SOURCE_NAME;
  parse_options.source_name = jerry_string_sz ("a.mjs");

  jerry_value_t ret_value = jerry_parse (script, sizeof (script) - 1, &parse_options);
  jerry_value_free (parse_options.source_name);

  jerry_module_link (ret_value, module_resolve_callback, NULL);

  jerry_value_free (ret_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**
- [jerry_module_resolve_cb_t](#jerry_module_resolve_cb_t)

## jerry_module_evaluate

Evaluate a module and its dependencies. The module must be in linked state.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.
- This API depends on a build option (`JERRY_MODULE_SYSTEM`) and can be checked
  in runtime with the `JERRY_FEATURE_MODULE` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
jerry_value_t jerry_module_evaluate (const jerry_value_t module_val);
```

- `module_val` - module object
- return
  - result of module bytecode execution - if evaluation was successful
  - exception, otherwise

*Introduced in version 3.0*.

**Example**

[doctest]: # ()

```c
#include <jerryscript.h>
#include <stdio.h>

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "export var a = 6";

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_MODULE | JERRY_PARSE_HAS_SOURCE_NAME;
  parse_options.source_name = jerry_string_sz ("a.mjs");

  jerry_value_t module_value = jerry_parse (script, sizeof (script) - 1, &parse_options);
  jerry_value_free (parse_options.source_name);

  jerry_value_free (jerry_module_link (module_value, NULL, NULL));
  jerry_value_free (jerry_module_evaluate (module_value));

  jerry_value_free (module_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_module_link](#jerry_module_link)


## jerry_module_cleanup

**Summary**

Release known modules in the current context. If realm parameter is supplied, cleans up modules native to that realm only. This function should be called by the user application when the module database in the current context is no longer needed.

**Prototype**

```c
void jerry_module_cleanup (const jerry_value_t realm);
```

- `realm` - release only those modules whose realm value is equal to this argument.

*Introduced in version 3.0*

**Example**

```c
jerry_module_cleanup (jerry_current_realm ());
```

**See also**

- [jerry_current_realm](#jerry_current_realm)
- [jerry_module_state](#jerry_module_state)
- [jerry_module_evaluate](#jerry_module_evaluate)
- [jerry_module_link](#jerry_module_link)


## jerry_module_state

**Summary**

Returns the current status of a module. The available values
are listed in [jerry_module_state_t](#jerry_module_state_t)

*Note*:
- This API depends on a build option (`JERRY_MODULE_SYSTEM`) and can be checked
  in runtime with the `JERRY_FEATURE_MODULE` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
jerry_module_state_t jerry_module_state (const jerry_value_t module_val);
```

- `module_val` - module object
- return
  - current status - if module_val is a module
  - JERRY_MODULE_STATE_INVALID - otherwise

*Renamed in version 3.0, it was previously known as `jerry_module_get_state` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <jerryscript.h>
#include <stdio.h>

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "import a from 'b.mjs'";

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_MODULE | JERRY_PARSE_HAS_SOURCE_NAME;
  parse_options.source_name = jerry_string_sz ("a.mjs");

  jerry_value_t module_value = jerry_parse (script, sizeof (script) - 1, &parse_options);
  jerry_value_free (parse_options.source_name);

  if (jerry_module_state (module_value) == JERRY_MODULE_STATE_UNLINKED)
  {
    printf ("Module parsing has been successful\n");
  }

  jerry_value_free (module_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_module_state_t](#jerry_module_state_t)

## jerry_module_on_state_changed

**Summary**

Sets a callback which is called after a module state is changed to linked, evaluated, or error.

*Note*:
- This API depends on a build option (`JERRY_MODULE_SYSTEM`) and can be checked
  in runtime with the `JERRY_FEATURE_MODULE` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
void jerry_module_on_state_changed (jerry_module_state_changed_cb_t callback,
                                    void *user_p);
```

- `callback` - callback, which is called after the state change.
- `user_p` - pointer passed to the callback function.

*Renamed in version 3.0, it was previously known as `jerry_module_set_state_changed_callback` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <jerryscript.h>
#include <stdio.h>

static void
module_state_changed (jerry_module_state_t new_state, /**< new state of the module */
                      const jerry_value_t module_val, /**< a module whose state is changed */
                      const jerry_value_t value, /**< value argument */
                      void *user_p) /**< user pointer */
{
  (void) module_val;
  (void) value;
  (void) user_p;

  if (new_state == JERRY_MODULE_STATE_LINKED)
  {
    printf ("A module is entered into linked state\n");
  }
} /* module_state_changed */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "12";

  jerry_module_on_state_changed (module_state_changed, NULL);

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_MODULE | JERRY_PARSE_HAS_SOURCE_NAME;
  parse_options.source_name = jerry_string_sz ("a.mjs");

  jerry_value_t module_value = jerry_parse (script, sizeof (script) - 1, &parse_options);
  jerry_value_free (parse_options.source_name);

  jerry_value_free (jerry_module_link (module_value, NULL, NULL));

  jerry_value_free (module_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_module_state_t](#jerry_module_state_t)
- [jerry_module_state_changed_cb_t](#jerry_module_state_changed_cb_t)

## jerry_module_on_import_meta

**Summary**

Sets a callback which is called when an import.meta expression of a module is evaluated the first time.

*Note*:
- This API depends on a build option (`JERRY_MODULE_SYSTEM`) and can be checked
  in runtime with the `JERRY_FEATURE_MODULE` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
void jerry_module_on_import_meta (jerry_module_import_meta_cb_t callback,
                                  void *user_p);
```

- `callback` - callback, which is called when an import.meta
               expression of a module is evaluated the first time
- `user_p` - pointer passed to the callback function.

*Renamed in version 3.0, it was previously known as `jerry_module_set_import_meta_callback` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <jerryscript.h>
#include <stdio.h>

static void
module_import_meta_callback (const jerry_value_t module, /**< module */
                             const jerry_value_t meta_object, /**< import.meta object */
                             void *user_p) /**< user pointer */
{
  (void) user_p;

  /* Create a property for the meta object */
  jerry_value_t property_name_value = jerry_string_sz ("prop");
  jerry_value_t property_value = jerry_string_sz ("prop");
  jerry_value_t result_value = jerry_object_set (meta_object, property_name_value, property_value);
  jerry_value_free (result_value);
  jerry_value_free (property_value);
  jerry_value_free (property_name_value);
} /* module_import_meta_callback */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "import.meta";

  jerry_module_on_import_meta (module_import_meta_callback, NULL);

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_MODULE;

  jerry_value_t module_value = jerry_parse (script, sizeof (script) - 1, &parse_options);

  jerry_value_free (jerry_module_link (module_value, NULL, NULL));
  jerry_value_free (jerry_module_evaluate (module_value));

  jerry_value_free (module_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_module_import_meta_cb_t](#jerry_module_import_meta_cb_t)

## jerry_module_request_count

**Summary**

Returns the number of import/export requests of a module.
The requests can be queried by [jerry_module_request](#jerry_module_request).


*Note*:
- This API depends on a build option (`JERRY_MODULE_SYSTEM`) and can be checked
  in runtime with the `JERRY_FEATURE_MODULE` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
size_t jerry_module_request_count (const jerry_value_t module_val);
```

- `module_val` - module object
- return
  - number of import/export requests of a module, if `module_val` is module,
  - 0, otherwise

*Renamed in version 3.0, it was previously known as `jerry_module_get_number_of_requests` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <jerryscript.h>
#include <stdio.h>

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "export * from 'b.mjs'"
                                "import a from 'c.mjs'";

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_MODULE | JERRY_PARSE_HAS_SOURCE_NAME;
  parse_options.source_name = jerry_string_sz ("a.mjs");

  jerry_value_t module_value = jerry_parse (script, sizeof (script) - 1, &parse_options);
  jerry_value_free (parse_options.source_name);

  /* Prints 2. */
  printf ("Number of requests: %d\n", (int) jerry_module_request_count (module_value));

  jerry_value_free (module_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_module_request](#jerry_module_request)
- [jerry_parse](#jerry_parse)
- [jerry_module_link](#jerry_module_link)

## jerry_module_request

**Summary**

Returns the module request specified by the `request_index` argument. The requests
are ordered in source code occurrence. When parsing is completed, all returned values
are strings. If [jerry_module_link](#jerry_module_link) is completed successfully
all returned values are module objects instead. If linking is in progress or fails,
the successfully resolved dependencies are module objects, the rest are strings.
The number of requests can be queried by
[jerry_module_request_count](#jerry_module_request_count).

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.
- This API depends on a build option (`JERRY_MODULE_SYSTEM`) and can be checked
  in runtime with the `JERRY_FEATURE_MODULE` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
jerry_value_t jerry_module_request (const jerry_value_t module_val, size_t request_index);
```

- `module_val` - module object
- return
  - string, if the request has not been resolved yet
  - module object, if the request has been resolved successfully
  - exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_module_get_request` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <jerryscript.h>

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "export * from 'b.mjs'"
                                "import a from 'c.mjs'";
  const jerry_char_t file[] = "a.mjs";

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_MODULE | JERRY_PARSE_HAS_SOURCE_NAME;
  parse_options.source_name = jerry_string_sz ("a.mjs");

  jerry_value_t module_value = jerry_parse (script, sizeof (script) - 1, &parse_options);
  jerry_value_free (parse_options.source_name);

  jerry_value_t request_value = jerry_module_request (module_value, 0);
  /* Returns with b.mjs */
  jerry_value_free (request_value);

  request_value = jerry_module_request (module_value, 1);
  /* Returns with c.mjs */
  jerry_value_free (request_value);

  jerry_value_free (module_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_module_request_count](#jerry_module_request_count)
- [jerry_parse](#jerry_parse)
- [jerry_module_link](#jerry_module_link)

## jerry_module_namespace

Returns the namespace object of a module

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.
- This API depends on a build option (`JERRY_MODULE_SYSTEM`) and can be checked
  in runtime with the `JERRY_FEATURE_MODULE` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
jerry_value_t jerry_module_namespace (const jerry_value_t module_val);
```

- `module_val` - module object
- return
  - object, if namespace object is available
  - exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_module_get_namespace` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <jerryscript.h>
#include <stdio.h>

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "export var a = 6";

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_MODULE | JERRY_PARSE_HAS_SOURCE_NAME;
  parse_options.source_name = jerry_string_sz ("a.mjs");

  jerry_value_t module_value = jerry_parse (script, sizeof (script) - 1, &parse_options);
  jerry_value_free (parse_options.source_name);

  jerry_value_free (jerry_module_link (module_value, NULL, NULL));
  jerry_value_free (jerry_module_evaluate (module_value));

  jerry_value_t namespace_value = jerry_module_namespace (module_value);

  /* Exports can be checked. */

  jerry_value_free (namespace_value);
  jerry_value_free (module_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_module_link](#jerry_module_link)
- [jerry_module_evaluate](#jerry_module_evaluate)

## jerry_module_on_import

Sets the callback which is called when dynamic imports are resolved. The resolver
receives the `user_value` assigned to the currently executed script, which should
provide all the information that is necessary for the resolve.

*Note*:
- This API depends on a build option (`JERRY_MODULE_SYSTEM`) and can be checked
  in runtime with the `JERRY_FEATURE_MODULE` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).
- The possible return values of the callback is explained
  in [jerry_module_import_cb_t](#jerry_module_import_cb_t)

**Prototype**

```c
void
jerry_module_on_import (jerry_module_import_cb_t callback_p,
                        void *user_p)
```

- `callback_p` - a [jerry_module_import_cb_t](#jerry_module_import_cb_t) callback which handles `import()` calls
- `user_p` - user pointer passed to the callback

*Renamed in version 3.0, it was previously known as `jerry_module_set_import_callback` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <jerryscript.h>
#include <stdio.h>

typedef struct {
  jerry_value_t specifier;
  jerry_value_t user_value;
  jerry_value_t promise;
} resolve_module_task_t;

static jerry_value_t
resolve_dynamic (const jerry_value_t specifier, /**< module specifier */
                 const jerry_value_t user_value, /**< user value assigned to the script */
                 void *user_p) /**< user data */
{
  /* If the specified module has already been evaluated, this callback can
   * return with it and the promise creation is automatically done by the engine.
   * Otherwise the application usually adds a resolve task to a command queue. */

  /* This very simple command queue supports only one task. */
  resolve_module_task_t *task_p = (resolve_module_task_t *) user_p;
  task_p->specifier = jerry_value_copy (specifier);
  task_p->user_value = jerry_value_copy (user_value);

  /* This Promise should be evaluated later. */
  jerry_value_t promise = jerry_promise ();
  task_p->promise = jerry_value_copy (promise);
  return promise;
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  resolve_module_task_t task;
  jerry_module_on_import (resolve_dynamic, &task);

  const jerry_char_t script[] = "import('modules/my_module.mjs').then(\n"
                                "  function (namespace) { /* use namespace */},\n"
                                "  function (error) { /* handle error */}\n"
                                ")";

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_HAS_SOURCE_NAME | JERRY_PARSE_HAS_USER_VALUE;

  /* Resource is usually used for debugging purposes, e.g. for generating backtrace. */
  parse_options.source_name = jerry_string_sz ("dir/my_script.js");

  /* User value should provide information for resolving dynamic imports.
   * In this case it contains the full path excluding the filename. */
  parse_options.user_value = jerry_string_sz ("/home/user/dir");

  jerry_value_t script_value = jerry_parse (script, sizeof (script) - 1, &parse_options);
  jerry_value_free (parse_options.source_name);
  jerry_value_free (parse_options.user_value);
  jerry_value_free (jerry_run (script_value));
  jerry_value_free (script_value);

  /* The application resolves both the module and the promise using the specifier
   * and the user_value. In this example the specifier is modules/my_module.mjs. */
  const jerry_char_t module_script[] = "export var a = 5";

  parse_options.options = JERRY_PARSE_MODULE | JERRY_PARSE_HAS_SOURCE_NAME | JERRY_PARSE_HAS_USER_VALUE;
  parse_options.source_name = jerry_string_sz ("modules/my_module.mjs");
  parse_options.user_value = jerry_string_sz ("/home/user/dir/modules");

  jerry_value_t module_value = jerry_parse (module_script, sizeof (module_script) - 1, &parse_options);
  jerry_value_free (parse_options.source_name);
  jerry_value_free (parse_options.user_value);
  jerry_value_free (jerry_module_link (module_value, NULL, NULL));
  jerry_value_free (jerry_module_evaluate (module_value));

  /* The promise must be resolved with the namespace object, not the module. */
  jerry_value_t namespace_value = jerry_module_namespace (module_value);
  jerry_value_free (jerry_promise_resolve (task.promise, namespace_value));

  jerry_value_free (namespace_value);
  jerry_value_free (module_value);
  jerry_value_free (task.specifier);
  jerry_value_free (task.user_value);
  jerry_value_free (task.promise);

  /* Process promise handlers. */
  jerry_value_free (jerry_run_jobs ());

  jerry_cleanup ();
  return 0;
}
```

**See also**
- [jerry_module_import_cb_t](#jerry_module_import_cb_t)

## jerry_native_module

Creates a native module with a list of exports. The initial state of the module is linked.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.
- Native pointers can be used to assign private data to a native module,
  see [jerry_object_set_native_ptr](#jerry_object_set_native_ptr)
- When `callback` is `NULL`, no function is called when the module is evaluated,
  only its state is changed to evaluated.
- This API depends on a build option (`JERRY_MODULE_SYSTEM`) and can be checked
  in runtime with the `JERRY_FEATURE_MODULE` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
jerry_value_t
jerry_native_module (jerry_native_module_evaluate_cb_t callback,
                     const jerry_value_t * const exports_p,
                     size_t number_of_exports);
```

- `callback` - a [jerry_native_module_evaluate_cb_t](#jerry_native_module_evaluate_cb_t) callback
               which is called by [jerry_module_evaluate](#jerry_module_evaluate) to evaluate the native module.
- `exports_p` - list of the exported bindings of the module, must be valid string identifiers.
- `number_of_exports` - number of exports in the `exports_p` list.
- return
  - a native module, if the module is successfully created
  - exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_native_module_create` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <jerryscript.h>

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t exports[2] =
  {
    jerry_string_sz ("first_export"),
    jerry_string_sz ("second_export")
  };

  jerry_value_t native_module = jerry_native_module (NULL, exports, 2);

  jerry_value_free (exports[0]);
  jerry_value_free (exports[1]);
  jerry_value_free (native_module);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_module_link](#jerry_module_link)
- [jerry_module_evaluate](#jerry_module_evaluate)
- [jerry_native_module_get](#jerry_native_module_get)
- [jerry_native_module_set](#jerry_native_module_set)

## jerry_native_module_get

Gets the value of an export binding which belongs to a native module.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.
- This API depends on a build option (`JERRY_MODULE_SYSTEM`) and can be checked
  in runtime with the `JERRY_FEATURE_MODULE` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
jerry_value_t
jerry_native_module_get (const jerry_value_t native_module_val,
                         const jerry_value_t export_name_val);
```

- `native_module_val` - a native module object.
- `export_name_val` - string identifier of the export.
- return
  - value of the export, if success
  - exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_native_module_get_export` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <jerryscript.h>

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t export = jerry_string_sz ("an_export");

  jerry_value_t native_module = jerry_native_module (NULL, &export, 1);

  jerry_value_t value = jerry_native_module_get (native_module, export);
  jerry_value_free (value);

  jerry_value_free (export);
  jerry_value_free (native_module);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_native_module](#jerry_native_module)
- [jerry_native_module_set](#jerry_native_module_set)

## jerry_native_module_set

Sets the value of an export binding which belongs to a native module.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.
- This API depends on a build option (`JERRY_MODULE_SYSTEM`) and can be checked
  in runtime with the `JERRY_FEATURE_MODULE` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
jerry_value_t
jerry_value_t jerry_native_module_set (const jerry_value_t native_module_val,
                                       const jerry_value_t export_name_val,
                                       const jerry_value_t value_to_set)
```

- `native_module_val` - a native module object.
- `export_name_val` - string identifier of the export.
- `value_to_set` - new value of the export.
- return
  - true, if success
  - exception, otherwise

*Introduced in version 3.0*.

**Example**

[doctest]: # ()

```c
#include <jerryscript.h>

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t export = jerry_string_sz ("an_export");

  jerry_value_t native_module = jerry_native_module (NULL, &export, 1);

  jerry_value_t number = jerry_number (3.5);
  jerry_value_t value = jerry_native_module_set (native_module, export, number);
  jerry_value_free (value);
  jerry_value_free (number);

  jerry_value_free (export);
  jerry_value_free (native_module);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_native_module](#jerry_native_module)
- [jerry_native_module_get](#jerry_native_module_get)

# Functions for promise objects

These APIs are always enabled.

## jerry_promise_result

**Summary**

The function returns the result of a Promise object.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_promise_result (const jerry_value_t promise);
```

- `promise` - the input Promise object.
- return
  - The result of the Promise.
  - If the Promise is not resolved yet the result is the 'undefined' value.
  - A TypeError is returned if the input argument was not a Promise object or
    the Promise support was not built into the library.

*Introduced in version 2.2*.

*Renamed in version 3.0, it was previously known as `jerry_get_promise_result` in earlier versions.*

**Example**

[doctest]: # (test="compile")

```c
#include <jerryscript.h>

static void
example (void)
{
  // create a promise object.
  jerry_value_t promise = jerry_promise ();
  {
    // prepare the argument for the resolve or reject.
    jerry_value_t argument = jerry_number (33);

    jerry_value_t is_ok = jerry_promise_resolve (promise, argument);

    // 'is_ok' should be checked if it is an exception or not.
    // skipped in this example
    jerry_value_free (is_ok);
    jerry_value_free (argument);
  }

  jerry_value_t promise_result = jerry_promise_result (promise);
  // 'promise_result' is now the number 33.

  jerry_value_free (promise_result);
  jerry_value_free (promise);
}
```

**See also**

- [jerry_promise](#jerry_promise)
- [jerry_promise_state_t](#jerry_promise_state_t)

## jerry_promise_state

**Summary**

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_promise_state_t
jerry_promise_state (const jerry_value_t promise);
```

- `promise` - the input promise object.
- return
  - [jerry_promise_state_t](#jerry_promise_state_t)
  - `JERRY_PROMISE_STATE_NONE` is returned if the input argument was not a promise object or
    the Promise support was not built into the library.

*Introduced in version 2.2*.

*Renamed in version 3.0, it was previously known as `jerry_get_promise_state` in earlier versions.*

**Example**

[doctest]: # (test="compile")

```c
#include <jerryscript.h>

static void
example (void)
{
  // create a promise object.
  jerry_value_t promise = jerry_promise ();

  jerry_promise_state_t start_state = jerry_promise_state (promise);
  // a Promise have a default state of JERRY_PROMISE_STATE_PENDING

  {
    // prepare the argument for the resolve or reject.
    jerry_value_t argument = jerry_number (33);

    jerry_value_t is_ok = jerry_promise_resolve (promise, argument);

    // 'is_ok' should be checked if it is an exception or not.
    // skipped in this example
    jerry_value_free (is_ok);
    jerry_value_free (argument);
  }

  jerry_promise_state_t current_state = jerry_promise_state (promise);
  // at this point the Promise should be in the JERRY_PROMISE_STATE_FULFILLED state.

  jerry_value_free (promise);
}
```

**See also**

- [jerry_promise](#jerry_promise)
- [jerry_promise_state_t](#jerry_promise_state_t)

## jerry_promise_on_event

**Summary**

Sets a callback for tracking Promise and async operations.

*Note*:
- This API depends on a build option (`JERRY_PROMISE_CALLBACK`) and can be checked
  in runtime with the `JERRY_FEATURE_PROMISE_CALLBACK` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
void jerry_promise_on_event (jerry_promise_event_filter_t filters, jerry_promise_event_cb_t callback,
                             void *user_p);
```

- `filters` - combination of [jerry_promise_event_filter_t](#jerry_promise_event_filter_t) options
- `callback` - callback function, the previously set value is overwritten,
               and setting NULL disables the tracking
- `user_p` - pointer passed to the callback function, can be NULL

*Renamed in version 3.0, it was previously known as `jerry_promise_set_callback` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <stdio.h>
#include <string.h>
#include "jerryscript.h"

static void
promise_callback (jerry_promise_event_type_t event_type, /**< event type */
                  const jerry_value_t object, /**< target object */
                  const jerry_value_t value, /**< optional argument */
                  void *user_p) /**< user pointer passed to the callback */
{
  if (event_type == JERRY_PROMISE_EVENT_CREATE)
  {
    printf ("A new promise is created\n");

    if (!jerry_value_is_undefined (value))
    {
      printf ("  The Promise is created by Promise.then() built-in.\n");
    }
  }
} /* promise_callback */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_promise_on_event (JERRY_PROMISE_EVENT_FILTER_CREATE, promise_callback, NULL);

  const char *source_p = "var p = Promise.resolve(0)\n"
                         "p.then(function (v) { return v; })";
  jerry_value_free (jerry_eval ((const jerry_char_t *) source_p,
                                strlen (source_p),
                                JERRY_PARSE_NO_OPTS));

  jerry_cleanup ();
  return 0;
} /* main */
```

**See also**

- [jerry_promise](#jerry_promise)
- [jerry_promise_state_t](#jerry_promise_state_t)

## jerry_property_descriptor_to_object

**Summary**

This API function is equivalent to FromPropertyDescriptor operation defined in ECMA 262 v11 6.2.5.4.
It returns with an ECMAScript Object which represents the property attributes.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_property_descriptor_to_object (const jerry_property_descriptor_t *src_prop_desc_p)
```

- `src_prop_desc_p` - the input property descriptor.
- return
  - [jerry_value_t](#jerry_value_t)
  - object value - if success
  - exception - otherwise

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_from_property_descriptor` in earlier versions.*

**Example**

[doctest]: # (test="compile")

```c
#include <jerryscript.h>

static void
example (void)
{
  jerry_value_t prop_name = jerry_string_sz ("value");

  jerry_property_descriptor_t prop_desc = jerry_property_descriptor ();

  prop_desc.value = prop_name;
  prop_desc.flags |= JERRY_PROP_IS_VALUE_DEFINED;

  jerry_value_t from_object = jerry_property_descriptor_to_object (&prop_desc);

  jerry_value_free (prop_name);
  jerry_value_free (from_object);
  jerry_property_descriptor_free (&prop_desc);
}
```

## jerry_property_descriptor_from_object

**Summary**

This API function is equivalent to ToPropertyDescriptor operation defined in ECMA 262 v11 6.2.5.5.
It decodes the ECMAScript object and fills the fields of a JerryScript property descriptor.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_property_descriptor_from_object (jerry_value_t obj_value, jerry_property_descriptor_t *out_prop_desc_p);
```

- `obj_value` - the input object
- `src_prop_desc_p` - the output property descriptor.
- return
  - [jerry_value_t](#jerry_value_t)
  - true, if success
  - thrown exception, otherwise

*Introduced in version 2.4*.

**Example**

[doctest]: # (test="compile")

```c
#include <jerryscript.h>

static void
example (void)
{
  jerry_value_t object = jerry_object ();
  jerry_value_t prop_name = jerry_string_sz ("value");
  jerry_value_t value = jerry_boolean (true);
  jerry_property_descriptor_t prop_desc;

  jerry_value_free (jerry_object_set (object, prop_name, prop_name));

  jerry_value_free (jerry_property_descriptor_from_object (object, &prop_desc));

  jerry_value_free (object);
  jerry_value_free (prop_name);
  jerry_value_free (value);
  jerry_property_descriptor_free (&prop_desc);
}
```


## jerry_promise_resolve

**Summary**

Resolve the promise with an argument.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_promise_resolve (jerry_value_t promise, jerry_value_t argument);
```

- `promise` - the promise value
- `argument` - the argument value
- return value
  - undefined - if resolve call was successful
  - exception - otherwise

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_resolve_or_reject_promise` in earlier versions.*

**Example**

```c
{
  jerry_value_t promise = ... // create/copy a promise object.

  ...

  jerry_value_t argument = ... // prepare the argument for resolve.

  jerry_value_t is_ok = jerry_promise_resolve (promise, argument);

  if (jerry_value_is_exception (is_ok))
  {
    // handle the exception.
  }

  jerry_value_free (is_ok);
  jerry_value_free (argument);
  jerry_value_free (promise);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_value_is_exception](#jerry_value_is_exception)


## jerry_promise_reject

**Summary**

Reject the promise with an argument.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_promise_reject (jerry_value_t promise, jerry_value_t argument);
```

- `promise` - the promise value
- `argument` - the argument value
- return value
  - undefined - if reject call was successful
  - exception - otherwise

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_resolve_or_reject_promise` in earlier versions.*

**Example**

```c
{
  jerry_value_t promise = ... // create/copy a promise object.

  ...

  jerry_value_t argument = ... // prepare the argument for reject.

  jerry_value_t is_ok = jerry_promise_reject (promise, argument);

  if (jerry_value_is_exception (is_ok))
  {
    // handle the exception.
  }

  jerry_value_free (is_ok);
  jerry_value_free (argument);
  jerry_value_free (promise);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_value_is_exception](#jerry_value_is_exception)

# Functions for symbols

These APIs are always enabled.

## jerry_symbol

**Summary**

Get the well-known symbol corresponding to the given [well-known symbol id](#jerry_well_known_symbol_t).

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.
- If the symbol support is not enabled an undefined will be returned.

**Prototype**

```c
jerry_value_t
jerry_symbol (jerry_well_known_symbol_t symbol);
```

- `symbol` - [jerry_well_known_symbol_t](#jerry_well_known_symbol_t) enum value
- return value
  - undefined value - if invalid well-known symbol was requested
  - well-known symbol value, otherwise

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_get_well_known_symbol` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t array_value = jerry_array (5);
  jerry_value_t symbol_iterator = jerry_symbol (JERRY_SYMBOL_ITERATOR);
  jerry_value_t array_iterator = jerry_object_get (array_value, symbol_iterator);

  // usage of array_iterator

  jerry_value_free (array_iterator);
  jerry_value_free (symbol_iterator);
  jerry_value_free (array_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_well_known_symbol_t](#jerry_well_known_symbol_t)

## jerry_symbol_description

**Summary**

Returns with the `[[Description]]` internal property of a symbol value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_symbol_description (const jerry_value_t value);
```

- `value` - symbol value
- return value
  - string or undefined value containing the symbol's description - if success
  - thrown exception, otherwise

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_get_symbol_description` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t string_value = jerry_string_sz ("foo");
  jerry_value_t symbol_value = jerry_symbol_with_description (string_value);

  jerry_value_free (string_value);

  jerry_value_t symbol_description = jerry_symbol_description (symbol_value);

  // usage of symbol_desc_string

  jerry_value_free (symbol_description);
  jerry_value_free (symbol_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_symbol_descriptive_string](#jerry_symbol_descriptive_string)

## jerry_symbol_descriptive_string

**Summary**

Call the SymbolDescriptiveString ecma builtin operation on the API value.
Based on ECMA 262 v6 19.4.3.2.1 this is in the form of `Symbol(<description>)`.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_symbol_descriptive_string (const jerry_value_t value);
```

- `value` - symbol value
- return value
  - string value containing the symbol's descriptive string - if success
  - thrown exception, otherwise

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_get_symbol_descriptive_string` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t string_value = jerry_string_sz ("foo");
  jerry_value_t symbol_value = jerry_symbol_with_description (string_value);

  jerry_value_free (string_value);

  jerry_value_t symbol_desc_string = jerry_symbol_descriptive_string (symbol_value);

  // usage of symbol_desc_string

  jerry_value_free (symbol_desc_string);
  jerry_value_free (symbol_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_symbol_description](#jerry_symbol_description)

# Functions for BigInts

These APIs all depend on build option (`JERRY_BUILTIN_BIGINT`).

## jerry_bigint_digit_count

**Summary**

Returns the size of uint64 digits of a BigInt value. This value is the
minimum size of the buffer which can hold all digits of a BigInt value when
the digits are retrieved by `[jerry_bigint_to_digits](#jerry_bigint_to_digits)`.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_BIGINT`) and can be checked
  in runtime with the `JERRY_FEATURE_BIGINT` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
uint32_t
jerry_bigint_digit_count (jerry_value_t value)
```

- `value` - BigInt value
- return value
  - number of digits (can be zero for BigInt zero)
  - if value is not a BigInt value, it returns with zero

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_get_bigint_size_in_digits` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <stdio.h>

#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  uint64_t digits[4] = { 0x1, 0x1, 0x0, 0x0 };
  jerry_value_t bigint_value = jerry_bigint (digits, 4, true);

  /* Prints two, because the leading zeroes in digits buffer are discarded. */
  printf("size: %d\n", (int) jerry_bigint_digit_count (bigint_value));

  jerry_value_free (bigint_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_value_is_bigint](#jerry_value_is_bigint)
- [jerry_bigint_to_digits](#jerry_bigint_to_digits)


## jerry_bigint_to_digits

**Summary**

Copies the uint64 digits of a BigInt value into a buffer. This function supports any
buffer sizes. If the buffer size is smaller than the size returned by
`[jerry_bigint_digit_count](#jerry_bigint_digit_count)`, only the
least significant digits are copied into the buffer. If the buffer size is greater,
the unused digits are filled with zero.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_BIGINT`) and can be checked
  in runtime with the `JERRY_FEATURE_BIGINT` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
void
jerry_bigint_to_digits (jerry_value_t value, uint64_t *digits_p, uint32_t size, bool *sign_p)
```

- `value` - BigInt value
- `digits_p` - output buffer for digits
- `size` - size of the output buffer
- `sign_p` - the boolean passed to the function is set to false when the BigInt
             is positive, and set to true otherwise (this parameter can be NULL)

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_get_bigint_digits` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  uint64_t digits[4] = { 0x1, 0x1, 0x0, 0x0 };
  jerry_value_t bigint_value = jerry_bigint (digits, 4, true);

  uint64_t get_digits[4];
  bool sign;
  jerry_bigint_to_digits (bigint_value, get_digits, 2, &sign);

  jerry_value_free (bigint_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_value_is_bigint](#jerry_value_is_bigint)
- [jerry_bigint_digit_count](#jerry_bigint_digit_count)


# Functions for Proxy objects

These APIs all depend on build option (`JERRY_BUILTIN_PROXY`).

## jerry_proxy_target

**Summary**

Gets the target object of a Proxy object.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_PROXY`) and can be checked
  in runtime with the `JERRY_FEATURE_PROXY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
jerry_value_t
jerry_proxy_target (jerry_value_t proxy_value);
```

- `proxy_value` - Proxy object value
- return value
  - type error exception - if proxy_value is not a Proxy object
  - target object - otherwise

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_get_proxy_target` in earlier versions.*

**Example**

```c
{
  jerry_value_t target = jerry_object ();
  jerry_value_t handler = jerry_object ();
  jerry_value_t proxy = jerry_proxy (target, handler);

  jerry_value_free (target);
  jerry_value_free (handler);

  target = jerry_proxy_target (proxy);

  // ... usage of the target

  jerry_value_free (target);
  jerry_value_free (proxy);
}
```

**See also**

- [jerry_proxy](#jerry_proxy)
- [jerry_proxy_custom](#jerry_proxy_custom)
- [jerry_proxy_handler](#jerry_proxy_handler)

## jerry_proxy_handler

**Summary**

Gets the handler object of a Proxy object.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_PROXY`) and can be checked
  in runtime with the `JERRY_FEATURE_PROXY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
jerry_value_t
jerry_proxy_handler (jerry_value_t proxy_value);
```

- `proxy_value` - Proxy object value
- return value
  - type error exception - if proxy_value is not a Proxy object
  - handler object - otherwise

*Renamed in version 3.0, it was previously known as `jerry_get_proxy_handler` in earlier versions.*

**Example**

```c
{
  jerry_value_t target = jerry_object ();
  jerry_value_t handler = jerry_object ();
  jerry_value_t proxy = jerry_proxy (target, handler);

  jerry_value_free (target);
  jerry_value_free (handler);

  handler = jerry_proxy_handler (proxy);

  // ... usage of the handler

  jerry_value_free (handler);
  jerry_value_free (proxy);
}
```

**See also**

- [jerry_proxy](#jerry_proxy)
- [jerry_proxy_custom](#jerry_proxy_custom)
- [jerry_proxy_target](#jerry_proxy_target)


# Copy and free API values

## jerry_value_copy

**Summary**

Copies the specified Jerry API value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_value_copy (jerry_value_t value);
```

- `value` - api value
- return value - copied value

*Renamed in version 3.0, it was previously known as `jerry_acquire_value` in earlier versions.*

**Example**

```c
{
  jerry_value_t object_value = jerry_object ();

  jerry_value_t copied_object = jerry_value_copy (object_value);

  jerry_value_free (object_value);

  // copied_object refers to the created object and makes it
  // available after the release of 'object_value'

  jerry_value_free (copied_object);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_value_t](#jerry_value_t)


## jerry_value_free

**Summary**

Release specified Jerry API value.

**Prototype**

```c
void
jerry_value_free (jerry_value_t value);
```

- `value` - api value

*Renamed in version 3.0, it was previously known as `jerry_release_value` in earlier versions.*

**Example**

```c
{
  jerry_value_t object_value = jerry_object ();

  ...

  jerry_value_free (object_value);
}
```


# Create API values

Function for creating [API values](#jerry_value_t).

*Note*:
- Every created API value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

## jerry_array

**Summary**

Create an array object value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_array (uint32_t size);
```

 - `size` - size of array;
 - return value - value of the constructed array object

*Renamed in version 3.0, it was previously known as `jerry_create_array` in earlier versions.*

 **Example**

```c
{
    jerry_value_t array = jerry_array (10);

    ...

    jerry_value_free (array);
}
```

**See also**

- [jerry_object_set_index](#jerry_object_set_index)
- [jerry_object_get_index](#jerry_object_get_index)


## jerry_arraybuffer

**Summary**

Create a jerry_value_t representing an ArrayBuffer object.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_arraybuffer (jerry_length_t size);
```

 - `size` - size of the backing store allocated for the array buffer **in bytes**.
 - return value - the new ArrayBuffer as a `jerry_value_t`

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_create_arraybuffer` in earlier versions.*

**Example**

```c
{
  jerry_value_t buffer_value = jerry_arraybuffer (15);

  ... // use the ArrayBuffer

  jerry_value_free (buffer_value);
}
```

**See also**

- [jerry_arraybuffer_read](#jerry_arraybuffer_read)
- [jerry_arraybuffer_write](#jerry_arraybuffer_write)
- [jerry_value_is_arraybuffer](#jerry_value_is_arraybuffer)
- [jerry_value_free](#jerry_value_free)


## jerry_arraybuffer_external

**Summary**

Creates a jerry_value_t representing an ArrayBuffer object with
user specified back-buffer.

User must pass a buffer pointer which is at least `size` big.
After the object is not needed the GC will call the `free_cb`
so the user can release the buffer which was provided.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).
- If `buffer_p` is NULL, the buffer is allocated by the allocator callback passed to
  [jerry_arraybuffer_allocator](#jerry_arraybuffer_allocator)
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_arraybuffer_external (uint8_t *buffer_p,
                            const jerry_length_t size
                            void *arraybuffer_user_p);
```

- `size` - size of the buffer **in bytes**.
- `buffer_p` - the backing store used by the array buffer object.
- `arraybuffer_user_p` - user pointer assigned to the array buffer object.
- return value
  - value of the newly constructed array buffer object.

*Introduced in version 2.0*.

*Updated in version 3.0*: `free_cb` has been replaced by `arraybuffer_user_p`.

*Renamed in version 3.0, it was previously known as `jerry_create_arraybuffer_external` in earlier versions.*

**Example**

```c
{
  uint8_t buffer_p[15];
  jerry_value_t buffer_value = jerry_arraybuffer_external (buffer_p, 15, NULL);

  ... // use the array buffer

  jerry_value_free (buffer_value);
}
```

**See also**

- [jerry_arraybuffer_data](#jerry_arraybuffer_data)
- [jerry_arraybuffer_read](#jerry_arraybuffer_read)
- [jerry_arraybuffer_write](#jerry_arraybuffer_write)
- [jerry_value_is_arraybuffer](#jerry_value_is_arraybuffer)
- [jerry_value_free](#jerry_value_free)
- [jerry_object_native_free_cb_t](#jerry_object_native_free_cb_t)


## jerry_shared_arraybuffer

**Summary**

Create a jerry_value_t representing a SharedArrayBuffer object.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_SHAREDARRAYBUFFER`).
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_shared_arraybuffer (jerry_length_t size);
```

- `size` - size of the backing store allocated for the shared array buffer **in bytes**.
- return value - the new SharedArrayBuffer as a `jerry_value_t`

*Renamed in version 3.0, it was previously known as `jerry_create_shared_arraybuffer` in earlier versions.*

**Example**

```c
{
  jerry_value_t buffer_value = jerry_shared_arraybuffer (15);

  ... // use the SharedArrayBuffer

  jerry_value_free (buffer_value);
}
```

**See also**

- [jerry_arraybuffer_read](#jerry_arraybuffer_read)
- [jerry_arraybuffer_write](#jerry_arraybuffer_write)
- [jerry_value_is_shared_arraybuffer](#jerry_value_is_shared_arraybuffer)
- [jerry_value_free](#jerry_value_free)


## jerry_shared_arraybuffer_external

**Summary**

Creates a jerry_value_t representing an SharedArrayBuffer object with
user specified back-buffer.

User must pass a buffer pointer which is at least `size` big.
After the object is not needed the GC will call the `free_cb`
so the user can release the buffer which was provided.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_SHAREDARRAYBUFFER`).
- If `buffer_p` is NULL, the buffer is allocated by the allocator callback passed to
  [jerry_arraybuffer_allocator](#jerry_arraybuffer_allocator)
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_shared_arraybuffer_external (uint8_t *buffer_p,
                                   const jerry_length_t size,
                                   jerry_value_free_callback_t free_cb);
```

- `buffer_p` - the backing store used by the shared array buffer object.
- `size` - size of the buffer **in bytes**.
- `arraybuffer_user_p` - user pointer assigned to the shared array buffer object.
- return value
    - value of the newly constructed shared array buffer object.

*Renamed in version 3.0, it was previously known as `jerry_create_shared_arraybuffer_external` in earlier versions.*

**Example**

```c
{
  uint8_t buffer_p[15];
  jerry_value_t buffer_value = jerry_shared_arraybuffer_external (buffer_p, 15, NULL);

  ... // use the shared array buffer

  jerry_value_free (buffer_value);
}
```

**See also**

- [jerry_arraybuffer_data](#jerry_arraybuffer_data)
- [jerry_arraybuffer_read](#jerry_arraybuffer_read)
- [jerry_arraybuffer_write](#jerry_arraybuffer_write)
- [jerry_value_is_shared_arraybuffer](#jerry_value_is_shared_arraybuffer)
- [jerry_value_free](#jerry_value_free)
- [jerry_object_native_free_cb_t](#jerry_object_native_free_cb_t)


## jerry_boolean

**Summary**

Create a jerry_value_t representing a boolean value from the given boolean parameter.

*Note*:
- The boolean values (true/false) are fixed constants. Their values can be copied
  any number of times without calling [jerry_value_copy](#jerry_value_copy), and
  freeing it with [jerry_value_free](#jerry_value_free) is optional.

**Prototype**

```c
jerry_value_t
jerry_boolean (bool value);
```

- `value` - raw boolean value.
- return value - a `jerry_value_t` created from the given boolean argument.

*Renamed in version 3.0, it was previously known as `jerry_create_boolean` in earlier versions.*

**Example**

```c
{
  jerry_value_t boolean_value = jerry_boolean (true);

  ... // usage of the value

  jerry_value_free (boolean_value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)


## jerry_error

**Summary**

Create new JavaScript Error object with the specified error message.

Important! The `error_type` argument *must not be* `JERRY_ERROR_NONE`.
Creating an Error object with no error type is not valid.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_error (jerry_error_t error_type, jerry_value_t message);
```

- `error_type` - type of error
- `message` - error message string
- return value - constructed error object

*Renamed in version 3.0, it was previously known as `jerry_create_error` in earlier versions.*

**Example**

```c
{
  jerry_value_t message = jerry_string_sz ("error");
  jerry_value_t error_obj = jerry_error (JERRY_ERROR_COMMON, message);

  ... // usage of error_obj


  jerry_value_free (error_obj);
  jerry_value_free (message);
}
```

**See also**

- [jerry_value_is_exception](#jerry_value_is_exception)
- [jerry_exception_value](#jerry_exception_value)
- [jerry_throw](#jerry_throw)


## jerry_error_sz

**Summary**

Create new JavaScript Error object, using the a zero-terminated string as the error message.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_error_sz (jerry_error_t error_type, const char *message_p);
```

- `error_type` - type of the error
- `message_p` - value of 'message' property of the constructed error object
- return value - constructed error object

*Renamed in version 3.0, it was previously known as `jerry_create_error` in earlier versions.*

**Example**

```c
{
  jerry_value_t error_obj = jerry_error_sz (JERRY_ERROR_TYPE, "error");

  ... // usage of error_obj

  jerry_value_free (error_obj);
}
```

**See also**

- [jerry_error](#jerry_error)


## jerry_dataview

**Summary**

Create new JavaScript DataView object.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_DATAVIEW`) and can be checked
  in runtime with the `JERRY_FEATURE_DATAVIEW` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_dataview (const jerry_value_t array_buffer,
                const jerry_length_t byte_offset,
                const jerry_length_t byte_length)
```

- `array_buffer` - arrayBuffer to create DataView from
- `byte_offset` - offset in bytes, to the first byte in the buffer
- `byte_length` - number of elements in the byte array
- return value
  - value of the constructed DataView object - if success
  - exception - otherwise

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_create_dataview` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t arraybuffer = jerry_arraybuffer (16);
  jerry_value_t dataview = jerry_dataview (arraybuffer, 0, 16);

  // usage of dataview

  jerry_value_free (dataview);
  jerry_value_free (arraybuffer);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_value_is_dataview](#jerry_value_is_dataview)
- [jerry_arraybuffer](#jerry_arraybuffer)


## jerry_function_external

**Summary**

Create an external function object.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_function_external (jerry_external_handler_t handler_p);
```

- `handler_p` - pointer to native handler of the function object
- return value - value of the constructed function object

*Renamed in version 3.0, it was previously known as `jerry_create_external_function` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <stdio.h>
#include <string.h>
#include "jerryscript.h"

static jerry_value_t
handler (const jerry_call_info_t *call_info_p,
         const jerry_value_t args_p[],
         const jerry_length_t args_cnt)
{
  printf ("Native handler called!\n");

  return jerry_boolean (true);
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t func_val = jerry_function_external (handler);
  jerry_value_t glob_obj = jerry_current_realm ();

  // after this, script can invoke the native handler through "handler_field (1, 2, 3);"
  jerry_value_t prop_name = jerry_string_sz ("handler_field");
  // set property and release the return value without any check
  jerry_value_free (jerry_object_set (glob_obj, prop_name, func_val));
  jerry_value_free (prop_name);

  jerry_value_free (func_val);
  jerry_value_free (glob_obj);

  // Test the method by calling it
  const char *test_src = "handler_field ();";
  jerry_value_t ret_val = jerry_eval ((const jerry_char_t *) test_src,
                                      strlen (test_src),
                                      JERRY_PARSE_NO_OPTS);
  // release the eval result
  jerry_value_free (ret_val);
  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_external_handler_t](#jerry_external_handler_t)
- [jerry_object_set](#jerry_object_set)
- [jerry_call](#jerry_call)


## jerry_number

**Summary**

Creates a `jerry_value_t` representing a number value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_number (double value);
```

- `value` - double value from which a `jerry_value_t` will be created
- return value - a `jerry_value_t` created from the given double argument

*Renamed in version 3.0, it was previously known as `jerry_create_number` in earlier versions.*

**Example**

```c
{
  jerry_value_t number_value = jerry_number (3.14);

  ... // usage of the value

  jerry_value_free (number_value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_infinity](#jerry_infinity)
- [jerry_nan](#jerry_nan)


## jerry_infinity

**Summary**

Creates a `jerry_value_t` representing a positive or negative infinity value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_infinity (bool sign);
```

- `sign` - true for negative Infinity and false for positive Infinity
- return value - a `jerry_value_t` representing the infinity value

*Renamed in version 3.0, it was previously known as `jerry_create_number_infinity` in earlier versions.*

**Example**

```c
{
  jerry_value_t positive_inf_value = jerry_infinity (false);

  ... // usage of the positive_inf_value

  jerry_value_free (positive_inf_value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_number](#jerry_number)
- [jerry_nan](#jerry_nan)


## jerry_nan

**Summary**

Creates a `jerry_value_t` representing a not-a-number value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_nan (void);
```

- return value - a `jerry_value_t` representing the not-a-number value

*Renamed in version 3.0, it was previously known as `jerry_create_number_nan` in earlier versions.*

**Example**

```c
{
  jerry_value_t nan_value = jerry_nan ();

  ... // usage of the nan_value

  jerry_value_free (nan_value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_number](#jerry_number)
- [jerry_infinity](#jerry_infinity)


## jerry_null

**Summary**

Creates and returns a `jerry_value_t` with type null object.

*Note*:
- The null value is a fixed constant. Its value can be copied any number of times without
  calling [jerry_value_copy](#jerry_value_copy), and freeing it
  with [jerry_value_free](#jerry_value_free) is optional.

**Prototype**

```c
jerry_value_t
jerry_null (void);
```

- return value - a `jerry_value_t` representing null.

*Renamed in version 3.0, it was previously known as `jerry_create_null` in earlier versions.*

**Example**

```c
{
  jerry_value_t null_value = jerry_null ();

  ... // usage of the value

  jerry_value_free (null_value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)


## jerry_object

**Summary**

Create new JavaScript object, like with new Object().

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_object (void);
```

- return value - value of the created object

*Renamed in version 3.0, it was previously known as `jerry_create_object` in earlier versions.*

**Example**

```c
{
  jerry_value_t object_value = jerry_object ();

  ... // usage of object_value

  jerry_value_free (object_value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)


## jerry_promise

**Summary**

Create an empty promise object which can be resolved or rejected later
by calling jerry_promise_resolve or jerry_promise_reject.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_promise (void)
```

- return value - value of the newly created promise

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_create_promise` in earlier versions.*

**Example**

```c
{
  jerry_value_t p = jerry_promise ();

  ...// usage of the promise

  jerry_value_free (p);
}
```

**See also**

- [jerry_promise_resolve](#jerry_promise_resolve)
- [jerry_promise_reject](#jerry_promise_reject)
- [jerry_value_free](#jerry_value_free)


## jerry_proxy

**Summary**

Create a new Proxy object with the given target and handler.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_PROXY`) and can be checked
  in runtime with the `JERRY_FEATURE_PROXY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_proxy (const jerry_value_t target, const jerry_value_t handler);
```

- `target` - proxy target
- `handler` - proxy handler
- return exception - if the Proxy construction fails
         value of the newly created proxy object - otherwise

*Introduced in version 2.3*.

*Renamed in version 3.0, it was previously known as `jerry_create_proxy` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t target = jerry_object ();
  jerry_value_t handler = jerry_object ();
  jerry_value_t proxy = jerry_proxy (target, handler);

  jerry_value_free (target);
  jerry_value_free (handler);

  // usage of the proxy

  jerry_value_free (proxy);

  jerry_cleanup ();
}
```

**See also**

- [jerry_value_is_proxy](#jerry_value_is_proxy)
- [jerry_proxy_custom](#jerry_proxy_custom)
- [jerry_value_free](#jerry_value_free)


## jerry_proxy_custom

**Summary**

Create a new Proxy object with the given target and handler.
The behaviour of the Proxy can be specialized with an options argument.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_PROXY`) and can be checked
  in runtime with the `JERRY_FEATURE_PROXY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_proxy_custom (const jerry_value_t target,
                    const jerry_value_t handler,
                    uint32_t options);
```

- `target` - proxy target
- `handler` - proxy handler
- `options` - any combination of [jerry_proxy_custom_behavior_t](#jerry_proxy_custom_behavior_t) options
- return thrown exception - if the Proxy construction fails
         value of the newly created proxy object - otherwise

*Renamed in version 3.0, it was previously known as `jerry_create_special_proxy` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t target = jerry_object ();
  jerry_value_t handler = jerry_object ();
  jerry_value_t proxy = jerry_proxy_custom (target, handler, JERRY_PROXY_SKIP_RESULT_VALIDATION);

  jerry_value_free (target);
  jerry_value_free (handler);

  // usage of the proxy

  jerry_value_free (proxy);

  jerry_cleanup ();
}
```

**See also**

- [jerry_value_is_proxy](#jerry_value_is_proxy)
- [jerry_proxy_custom](#jerry_proxy_custom)
- [jerry_value_free](#jerry_value_free)


## jerry_string_sz

**Summary**

Create string from a zero-terminated ASCII string.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_string_sz (const char *str_p);
```

- `str_p` - non-null pointer to zero-terminated string
- return value - created string

*Renamed in version 3.0, it was previously known as `jerry_create_string` in earlier versions.*

**Example**

```c
{
  const char char_array[] = "a string";
  jerry_value_t string_value = jerry_string_sz (char_array);

  ... // usage of string_value

  jerry_value_free (string_value);
}
```

**See also**

- [jerry_string](#jerry_string)


## jerry_string

**Summary**

Create a string from a buffer using the specified encoding. The data in the buffer must be valid for the encoding.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_string (const jerry_char_t *buffer_p,
              jerry_size_t buf_size,
              jerry_encoding_t encoding)
```

- `buffer_p` - non-null pointer to buffer
- `buf_size` - size of the buffer
- `encoding` - encoding of the string data
- return value - value of the created string

*Renamed in version 3.0, it was previously known as `jerry_create_string_sz` in earlier versions.*

**Example**

```c
{
  const jerry_char_t char_array[] = "a string";
  jerry_value_t string_value = jerry_string (char_array,
                                             sizeof (char_array) - 1,
                                             JERRY_ENCODING_CESU8);

  ... // usage of string_value

  jerry_value_free (string_value);
}

```

**See also**

- [jerry_validate_string](#jerry_validate_string)
- [jerry_string_sz](#jerry_string_sz)


## jerry_string_external_sz

**Summary**

Create an external string from a zero-terminated ASCII string. The string buffer passed to the function
should not be modified until the free callback is called. This function can be used to avoid
the duplication of large strings.

*Note*:
- The free callback can be set by [jerry_string_external_on_free](#jerry_string_external_on_free)
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_string_external_sz (const char *str_p, void *user_p);
```

- `str_p` - non-null pointer to a zero-terminated string
- `user_p` - user pointer passed to the callback when the string is freed
- return value - value of the created string

*Introduced in version 2.4*.

*Updated in version 3.0*: `free_cb` is replaced by `user_p`.

*Renamed in version 3.0, it was previously known as `jerry_create_external_string` in earlier versions.*

**Example**

```c
{
  const char* string_p = "a large and immutable string: this is a story about ....";
  jerry_value_t string_value = jerry_string_external_sz (string_p, NULL);

  ... // usage of string_value

  jerry_value_free (string_value);
}
```

**See also**

- [jerry_string_external](#jerry_string_external)
- [jerry_string_external_on_free](#jerry_string_external_on_free)


## jerry_string_external

**Summary**

Create an external string from a valid CESU8 string. The string buffer passed to the function
should not be modified until the free callback is called. This function can be used to avoid
the duplication of large strings.

*Note*:
- The free callback can be set by [jerry_string_external_on_free](#jerry_string_external_on_free)
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_string_external (const jerry_char_t *str_p,
                       jerry_size_t str_size,
                       void *user_p);
```

- `str_p` - non-null pointer to string
- `str_size` - size of the string
- `user_p` - user pointer passed to the callback when the string is freed
- return value - value of the created string

*Introduced in version 2.4*.

*Updated in version 3.0*: `free_cb` is replaced by `user_p`.

*Renamed in version 3.0, it was previously known as `jerry_create_external_string_sz` in earlier versions.*

**Example**

```c
{
  const char* string_p = "a large and immutable string: this is a story about ....";
  jerry_value_t string_value = jerry_string_external ((const jerry_char_t *) string_p,
                                                      strlen (string_p),
                                                      NULL);

  ... // usage of string_value

  jerry_value_free (string_value);
}

```

**See also**

- [jerry_validate_string](#jerry_validate_string)
- [jerry_string_external_sz](#jerry_string_external_sz)
- [jerry_string_external_on_free](#jerry_string_external_on_free)


## jerry_symbol_with_description

**Summary**

Create symbol from an API value.

*Note*:
- The given argument is converted to string. This operation can throw an exception.
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_symbol_with_description (const jerry_value_t value)
```

- `value` - API value
- return value
  - value of the created symbol, if success
  - thrown exception, otherwise

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_create_symbol` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t string_value = jerry_string_sz ("Symbol description string");
  jerry_value_t symbol_value = jerry_symbol_with_description (string_value);

  // The description value is no longer needed
  jerry_value_free (string_value);

  // usage of symbol_value

  jerry_value_free (symbol_value);

  jerry_cleanup ();
}
```

**See also**

- [jerry_value_is_symbol](#jerry_value_is_symbol)
- [jerry_value_free](#jerry_value_free)


## jerry_bigint

**Summary**

Create BigInt value from uint64 digits

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_BIGINT`) and can be checked
  in runtime with the `JERRY_FEATURE_BIGINT` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_bigint (const uint64_t *digits_p, uint32_t size, bool sign)
```

- `digits_p` - array of uint64 digits, least significant digit first
- `size` - size of the `digits_p` array
- `sign` - false if the created value should be positive, and true if the created value should be negative
- return value
  - value of the created bigint, if success
  - thrown exception, otherwise

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_create_bigint` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  uint64_t digits[2] = { 0x1, 0x1 };
  jerry_value_t bigint_value = jerry_bigint (digits, 2, true);

  // usage of bigint_value

  jerry_value_free (bigint_value);

  jerry_cleanup ();
}
```

**See also**

- [jerry_value_free](#jerry_value_free)
- [jerry_value_is_bigint](#jerry_value_is_bigint)
- [jerry_bigint_to_digits](#jerry_bigint_to_digits)


## jerry_regexp_sz

**Summary**

Returns a RegExp object created from the argument of a zero-terminated UTF-8 string pattern, or an exception
if the construction of the object fails. Optional flags can be set using [jerry_regexp_flags_t](#jerry_regexp_flags_t).
These flags can be combined together with the binary OR operator or used on their own as enum values.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**
```c
jerry_value_t
jerry_regexp_sz (const jerry_char_t *pattern_p, uint16_t flags);
```

- `pattern_p` - the RegExp pattern as a zero-terminated UTF-8 string
- `flags` - optional flags for the RegExp object, see [jerry_regexp_flags_t](#jerry_regexp_flags_t)
- return value - the RegExp object as a `jerry_value_t`

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_create_regexp` in earlier versions.*

**Example**

```c
{
  jerry_char_t pattern_p = "[cgt]gggtaaa|tttaccc[acg]";
  uint16_t pattern_flags = JERRY_REGEXP_FLAG_IGNORE_CASE;

  jerry_value_t regexp = jerry_regexp_sz (pattern_p, pattern_flags);

  ...

  jerry_value_free (regexp);
}
```


## jerry_regexp

**Summary**

Returns a RegExp object from the argument pattern, or an exception if the construction of the object fails. Optional
flags can be set using [jerry_regexp_flags_t](#jerry_regexp_flags_t). These flags can be combined together with the
binary OR operator or used on their own as enum values.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**
```c
jerry_value_t
jerry_regexp (const jerry_value_t pattern, uint16_t flags);
```

- `pattern_p` - the RegExp pattern as a zero-terminated UTF-8 string
- `pattern_size` - size of the `pattern`
- `flags` - optional flags for the RegExp object, see [jerry_regexp_flags_t](#jerry_regexp_flags_t)
- return value - the RegExp object as a `jerry_value_t`

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_create_regexp_sz` in earlier versions.*

**Example**

```c
{
  jerry_char_t pattern_p = "[cgt]gggtaaa|tttaccc[acg]";
  jerry_size_t pattern_size = sizeof (pattern_p) - 1;
  jerry_value_t pattern_str = jerry_string (pattern_p, pattern_size, JERRY_ENCODING_UTF8);

  uint16_t pattern_flags = JERRY_REGEXP_FLAG_IGNORE_CASE;

  jerry_value_t regexp = jerry_regexp (pattern_str, pattern_flags);
  jerry_value_free (pattern_str);

  ...

  jerry_value_free (regexp);
}
```


## jerry_typedarray

**Summary**

Create a jerry_value_t representing an TypedArray object.

For the new object the type of the TypedArray (see: [jerry_typedarray_type_t](#jerry_typedarray_type_t))
and element count can be specified.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
jerry_value_t
jerry_typedarray (jerry_typedarray_type_t type_name, jerry_length_t item_count);
```

- `type_name` - type of TypedArray to create
- `item_count` - number of items in the new TypedArray
- return value - the new TypedArray as a `jerry_value_t`

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_create_typedarray` in earlier versions.*

**Example**

```c
{
  jerry_value_t array = jerry_typedarray (JERRY_TYPEDARRAY_UINT16, 15);

  ... // use the TypedArray

  jerry_value_free (array);
}
```

**See also**

- [jerry_typedarray_type_t](#jerry_typedarray_type_t)
- [jerry_value_is_typedarray](#jerry_value_is_typedarray)
- [jerry_value_free](#jerry_value_free)


## jerry_typedarray_with_buffer

**Summary**

Create a jerry_value_t representing an TypedArray object using
an already existing ArrayBuffer object.

For the new object the type of the TypedArray (see: [jerry_typedarray_type_t](#jerry_typedarray_type_t))
and element count can be specified.

The developer must ensure that the ArrayBuffer has the correct length for the given
type of TypedArray otherwise an exception is generated.

The JavaScript equivalent of this function is: `new %TypedArray%(arraybuffer)` where `%TypedArray%` is
one of the allowed TypedArray functions.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
jerry_value_t
jerry_typedarray_with_buffer (jerry_typedarray_type_t type_name,
                              const jerry_value_t arraybuffer);
```

- `type_name` - type of TypedArray to create
- `arraybuffer` - the ArrayBuffer to use for the new TypedArray
- return value
  - the new TypedArray as a `jerry_value_t`
  - exception if the ArrayBuffer does not have enough space for the given type of TypedArray

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_create_typedarray_for_arraybuffer` in earlier versions.*

**Example**

```c
{
  jerry_value_t buffer = jerry_array_buffer (12 * 2);
  jerry_value_t array = jerry_typedarray_with_buffer (JERRY_TYPEDARRAY_UINT16, buffer);
  jerry_value_free (buffer);

  ... // use the TypedArray

  jerry_value_free (array);
}
```

**See also**

- [jerry_typedarray_type_t](#jerry_typedarray_type_t)
- [jerry_value_is_typedarray](#jerry_value_is_typedarray)
- [jerry_value_free](#jerry_value_free)


## jerry_typedarray_with_buffer_span

**Summary**

Create a jerry_value_t representing an TypedArray object using
an already existing ArrayBuffer object and by specifying the byteOffset, and length properties.

For the new object the type of the TypedArray (see: [jerry_typedarray_type_t](#jerry_typedarray_type_t))
and element count can be specified.

The developer must ensure that the ArrayBuffer has the correct length for the given
type of TypedArray otherwise an exception is generated.

The JavaScript equivalent of this function is: `new %TypedArray%(arraybuffer, byteOffset, length)` where `%TypedArray%` is
one of the allowed TypedArray functions.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
jerry_value_t
jerry_typedarray_with_buffer_span (jerry_typedarray_type_t type_name,
                                   const jerry_value_t arraybuffer,
                                   jerry_length_t byte_offset,
                                   jerry_length_t length);
```

- `type_name` - type of TypedArray to create
- `arraybuffer` - the ArrayBuffer to use for the new TypedArray
- `byte_offset` - start offset to use for the ArrayBuffer
- `length` - number of elements to used from the ArrayBuffer (this is not the same as the byteLength)
- return value
  - the new TypedArray as a `jerry_value_t`
  - exception if the ArrayBuffer does not have enough space for the given type of TypedArray

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_create_typedarray_for_arraybuffer_sz` in earlier versions.*

**Example**

```c
{
  jerry_value_t buffer = jerry_array_buffer (12 * 2);
  jerry_value_t array = jerry_typedarray_with_buffer_span (JERRY_TYPEDARRAY_UINT16, buffer, 4, 10);
  jerry_value_free (buffer);

  ... // use the TypedArray

  jerry_value_free (array);
}
```

**See also**

- [jerry_typedarray_type_t](#jerry_typedarray_type_t)
- [jerry_value_is_typedarray](#jerry_value_is_typedarray)
- [jerry_value_free](#jerry_value_free)


## jerry_container

**Summary**

Create a jerry_value_t representing a given type container object.

*Note*:
- This method is expected to work the same way as the JavaScript Map constructor.
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_CONTAINER`) and can be checked
  in runtime with the `JERRY_FEATURE_MAP, JERRY_FEATURE_SET, JERRY_FEATURE_WEAKMAP, JERRY_FEATURE_WEAKSET`
  feature enum values.
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
jerry_value_t
jerry_container (jerry_container_type_t container_type,
                 const jerry_value_t *arguments_list_p,
                 jerry_length_t arguments_list_len);
```

- `container_type` - Type of the container to be created, see `jerry_container_type_t`.
- `arguments_list_p` - The arguments passed to the container constructor to be inserted to the container.
- `arguments_list_len` - The length of the above arguments.
- return value - the new container object as a `jerry_value_t`

*Introduced in version 2.3*.

*Renamed in version 3.0, it was previously known as `jerry_create_container` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);
  jerry_char_t src[] = "[1,2,3,4].entries()";
  jerry_value_t iterable = jerry_eval (src, sizeof (src) - 1, JERRY_PARSE_NO_OPTS);

  jerry_value_t map = jerry_container (JERRY_CONTAINER_TYPE_MAP, &iterable, 1);
  jerry_value_free (iterable);

  // use the Map

  jerry_value_free (map);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_container_type_t](#jerry_container_type_t)
- [jerry_container_type](#jerry_container_type)


## jerry_undefined

**Summary**

Creates a `jerry_value_t` representing an undefined value.

*Note*:
- The undefined value is a fixed constant. Its value can be copied any number of
  times without calling [jerry_value_copy](#jerry_value_copy), and freeing it
  with [jerry_value_free](#jerry_value_free) is optional.


**Prototype**

```c
jerry_value_t
jerry_undefined (void);
```

- return value - value of undefined

*Renamed in version 3.0, it was previously known as `jerry_create_undefined` in earlier versions.*

**Example**

```c
{
  jerry_value_t undefined_value = jerry_undefined ();

  ... // usage of the value

  jerry_value_free (undefined_value);
}
```

**See also**

- [jerry_value_free](#jerry_value_free)


## jerry_realm

**Summary**

Creates a `jerry_value_t` representing a new global object.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free)
  when it is no longer needed.
- This API depends on a build option (`JERRY_BUILTIN_REALMS`) and can be checked
  in runtime with the `JERRY_FEATURE_REALM` feature enum value.
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
jerry_value_t
jerry_realm (void);
```

- return value - realm object value

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_create_realm` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t realm_value = jerry_realm ();

  // usage of the value

  jerry_value_free (realm_value);
  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_value_free](#jerry_value_free)


# General API functions of JS objects

## jerry_object_has

**Summary**

Checks whether the object or its prototype objects have the given property.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_object_has (const jerry_value_t obj_val,
                  const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value - JavaScript value that evaluates to
  -  exception - if the operation fail
  -  true/false API value - depend on whether the property exists

*Updated in version 2.0*: The return value type is now a JavaScript value and not a primitive boolean value.

*Updated in version 2.3*: The return value can be an exception value.

*Renamed in version 3.0, it was previously known as `jerry_has_property` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_current_realm ();
  jerry_value_t prop_name = jerry_string_sz ("handler_field");

  jerry_value_t has_prop_js = jerry_object_has (global_object, prop_name);
  bool has_prop = jerry_value_is_true (has_prop_js);

  jerry_value_free (has_prop_js);
  jerry_value_free (prop_name);
  jerry_value_free (global_object);

  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_object_has_own](#jerry_object_has_own)
- [jerry_object_delete](#jerry_object_delete)


## jerry_object_has_sz

**Summary**

Checks whether the object or its prototype objects have the given property.

*Note*: Returned value must be freed with [jerry_value_free](#jerry_value_free) when it is no longer needed.

**Prototype**

```c
jerry_value_t jerry_object_has_sz (const jerry_value_t object, const char *key_p);
```

- `object` - object value
- `key_p` - property name (zero-terminated string)
- return value - JavaScript value that evaluates to
  -  exception - if the operation fail
  -  true/false API value - depend on whether the property exists

*Introduced in version 3.0*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_current_realm ();

  jerry_value_t has_prop_js = jerry_object_has_sz (global_object, "handler_field");
  bool has_prop = jerry_value_is_true (has_prop_js);

  jerry_value_free (has_prop_js);
  jerry_value_free (global_object);

  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_object_has](#jerry_object_has)


## jerry_object_has_own

**Summary**

Checks whether the object has the given property.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_object_has_own (const jerry_value_t obj_val,
                      const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value - JavaScript value that evaluates to
  -  exception - if the operation fails
  -  true/false API value - depend on whether the property exists

*Updated in version 2.0*: The return value type is now a JavaScript value and not a primitive boolean value.

*Updated in version 2.3*: The return value can be an exception value.

*Renamed in version 3.0, it was previously known as `jerry_has_own_property` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_current_realm ();
  jerry_value_t prop_name = jerry_string_sz ("handler_field");

  jerry_value_t has_prop_js = jerry_object_has_own (global_object, prop_name);
  bool has_prop = jerry_value_is_true (has_prop_js);

  jerry_value_free (has_prop_js);
  jerry_value_free (prop_name);
  jerry_value_free (global_object);

  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_object_has](#jerry_object_has)
- [jerry_object_delete](#jerry_object_delete)


## jerry_object_has_internal

**Summary**

Checks whether the object has the given internal property.

*Note*:
- Properties which were not created with [jerry_object_set_internal](#jerry_object_set_internal)
  are excluded during the operation.
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
bool
jerry_object_has_internal (const jerry_value_t obj_val,
                           const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - true, if the property exists
  - false, otherwise

*Introduced in version 2.2*.

*Renamed in version 3.0, it was previously known as `jerry_has_internal_property` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_current_realm ();
  jerry_value_t prop_name = jerry_string_sz ("hidden_property");

  bool has_internal_js_prop = jerry_object_has_internal (global_object, prop_name);

  jerry_value_free (prop_name);
  jerry_value_free (global_object);

  return 0;
}
```

**See also**

- [jerry_object_delete_internal](#jerry_object_delete_internal)
- [jerry_object_get_internal](#jerry_object_get_internal)
- [jerry_object_set_internal](#jerry_object_set_internal)


## jerry_object_delete

**Summary**

Delete a property from an object.

**Prototype**

```c
jerry_value_t
jerry_object_delete (const jerry_value_t obj_val,
                     const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - true, if property was deleted successfully
  - exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_delete_property` in earlier versions.*

**Example**

```c
{
  jerry_value_t global_object = jerry_current_realm ();
  jerry_value_t prop_name = jerry_string_sz ("my_prop");

  jerry_value_t delete_result = jerry_object_delete (global_object, prop_name);
  /* use "delete_result" */

  jerry_value_free (delete_result);
  jerry_value_free (prop_name);
  jerry_value_free (global_object);
}
```

**See also**

- [jerry_object_has](#jerry_object_has)
- [jerry_object_has_own](#jerry_object_has_own)
- [jerry_object_delete_index](#jerry_object_delete_index)
- [jerry_object_get](#jerry_object_get)


## jerry_object_delete_sz

**Summary**

Delete a property from an object.

**Prototype**

```c
jerry_value_t jerry_object_delete_sz (const jerry_value_t object, const char *key_p);
```

- `obj_val` - object value
- `key_p` - property name (zero-terminated string)
- return value
  - true, if property was deleted successfully
  - exception, otherwise

*Introduced in version 3.0*

**Example**

```c
{
  jerry_value_t global_object = jerry_current_realm ();

  jerry_value_t delete_result = jerry_object_delete_sz (global_object, "my_prop");
  /* use "delete_result" */

  jerry_value_free (delete_result);
  jerry_value_free (global_object);
}
```

**See also**

- [jerry_object_delete](#jerry_object_delete)


## jerry_object_delete_index

**Summary**

Delete indexed property from the specified object.

**Prototype**

```c
jerry_value_t
jerry_object_delete_index (const jerry_value_t obj_val,
                           uint32_t index);
```

- `obj_val` - object value
- `index` - index number
- return value
  - true value, if property was deleted successfully
  - exception, otherwise

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_delete_property_by_index` in earlier versions.*

**Example**

```c
{
  jerry_value_t object;

  ... // create or copy object

  jerry_value_t delete_result = jerry_object_delete_index (object, 5);

  jerry_value_free (delete_result);
  jerry_value_free (object);
}
```

**See also**

- [jerry_object_has](#jerry_object_has)
- [jerry_object_has_own](#jerry_object_has_own)
- [jerry_object_delete](#jerry_object_delete)
- [jerry_object_get](#jerry_object_get)
- [jerry_object_set](#jerry_object_set)
- [jerry_object_get_index](#jerry_object_get_index)
- [jerry_object_set_index](#jerry_object_set_index)

## jerry_object_delete_internal

**Summary**

Delete an internal property from an object.

*Note*:
- Properties which were not created with [jerry_object_set_internal](#jerry_object_set_internal)
  are excluded during the operation.

**Prototype**

```c
bool
jerry_object_delete_internal (const jerry_value_t obj_val,
                              const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - true, if property was deleted successfully
  - false, otherwise

*Introduced in version 2.2*.

*Renamed in version 3.0, it was previously known as `jerry_delete_internal_property` in earlier versions.*

**Example**

```c
{
  jerry_value_t global_object = jerry_current_realm ();
  jerry_value_t prop_name = jerry_string_sz ("hidden_property");

  bool delete_result = jerry_object_delete_internal (global_object, prop_name);
  /* use "delete_result" */

  jerry_value_free (prop_name);
  jerry_value_free (global_object);
}
```

**See also**

- [jerry_object_has_internal](#jerry_object_has_internal)
- [jerry_object_get_internal](#jerry_object_get_internal)
- [jerry_object_set_internal](#jerry_object_set_internal)


## jerry_object_get

**Summary**

Get value of a property to the specified object with the given name.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_object_get (const jerry_value_t obj_val,
                  const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - value of property, if success
  - thrown exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_get_property` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_current_realm ();
  jerry_value_t prop_name = jerry_string_sz ("Object");

  jerry_value_t prop_value = jerry_object_get (global_object, prop_name);

  /* use "prop_value" then release it. */

  jerry_value_free (prop_value);
  jerry_value_free (prop_name);
  jerry_value_free (global_object);

  return 0;
}
```

**See also**

- [jerry_object_has](#jerry_object_has)
- [jerry_object_has_own](#jerry_object_has_own)
- [jerry_object_delete](#jerry_object_delete)
- [jerry_object_delete_index](#jerry_object_delete_index)
- [jerry_object_set](#jerry_object_set)
- [jerry_object_get_index](#jerry_object_get_index)
- [jerry_object_set_index](#jerry_object_set_index)


## jerry_object_get_sz

**Summary**

Get value of a property to the specified object with the given name.

*Note*: Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
is no longer needed.

**Prototype**

```c
jerry_value_t jerry_object_get_sz (const jerry_value_t object, const char *key_p);
```

- `obj_val` - object value
- `key_p` - property name (zero-terminated string)
- return value
  - value of property, if success
  - thrown exception, otherwise

*Introduced in version 3.0*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_current_realm ();

  jerry_value_t prop_value = jerry_object_get_sz (global_object, "Object");

  /* use "prop_value" then release it. */

  jerry_value_free (prop_value);
  jerry_value_free (global_object);

  return 0;
}
```

**See also**

- [jerry_object_get](#jerry_object_get)


## jerry_object_get_index

**Summary**

Get value by an index from the specified object.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_object_get_index (const jerry_value_t obj_val,
                        uint32_t index);
```

- `obj_val` - object value
- `index` - index number
- return value
  - stored value on the specified index, if success
  - thrown exception, otherwise.

*Renamed in version 3.0, it was previously known as `jerry_get_property_by_index` in earlier versions.*

**Example**

```c
{
  jerry_value_t object;

  ... // create or copy object

  jerry_value_t value = jerry_object_get_index (object, 5);

  ...

  jerry_value_free (value);
  jerry_value_free (object);
}
```

**See also**

- [jerry_object_has](#jerry_object_has)
- [jerry_object_has_own](#jerry_object_has_own)
- [jerry_object_delete](#jerry_object_delete)
- [jerry_object_delete_index](#jerry_object_delete_index)
- [jerry_object_get](#jerry_object_get)
- [jerry_object_set](#jerry_object_set)
- [jerry_object_set_index](#jerry_object_set_index)

## jerry_object_find_own

**Summary**

Get the own property value of an object with the given name. The function tells
whether the property is found, and the receiver object can be specified as well.
The receiver is passed as the `this` argument for getters, and the receiver
argument for Proxy `get` traps.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it is no longer needed.
- The `found_p` argument is ignored if its value is NULL.
- The target value of `found_p` argument is set to false when the arguments are invalid, e.g. `obj_val` is not an object.

**Prototype**

```c
jerry_value_t
jerry_object_find_own (const jerry_value_t obj_val,
                       const jerry_value_t prop_name_val,
                       const jerry_value_t receiver_val,
                       bool *found_p);
```

- `obj_val` - object value
- `prop_name_val` - property name
- `receiver_val` - receiver object
- `found_p` - [out] true, if the property is found or obj_val is a Proxy object, false otherwise
- return value
  - value of property, if success
  - thrown exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_get_own_property` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"
#include "stdio.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_current_realm ();
  jerry_value_t prop_name = jerry_string_sz ("Object");

  bool found;
  jerry_value_t prop_value = jerry_object_find_own (global_object, prop_name, global_object, &found);

  if (found)
  {
    printf ("Property is found!\n");
  }

  /* use "prop_value" then release it. */

  jerry_value_free (prop_value);
  jerry_value_free (prop_name);
  jerry_value_free (global_object);

  return 0;
}
```

**See also**

- [jerry_object_get](#jerry_object_get)
- [jerry_object_get_index](#jerry_object_get_index)

## jerry_object_get_internal

**Summary**

Get value of an internal property to the specified object with the given name.

*Note*:
- Properties which were not created with [jerry_object_set_internal](#jerry_object_set_internal)
  are excluded during the operation.
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_object_get_internal (const jerry_value_t obj_val,
                           const jerry_value_t prop_name_val);
```

- `obj_val` - object value
- `prop_name_val` - property name
- return value
  - value of property, if the internal property exists
  - undefined value, if the, if the internal does not property exists
  - thrown exception, otherwise

*Introduced in version 2.2*.

*Renamed in version 3.0, it was previously known as `jerry_get_internal_property` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_current_realm ();
  jerry_value_t prop_name = jerry_string_sz ("hidden_property");

  jerry_value_t prop_value = jerry_object_get_internal (global_object, prop_name);

  /* use "prop_value" then release it. */

  jerry_value_free (prop_value);
  jerry_value_free (prop_name);
  jerry_value_free (global_object);

  return 0;
}
```

**See also**

- [jerry_object_has_internal](#jerry_object_has_internal)
- [jerry_object_delete_internal](#jerry_object_delete_internal)
- [jerry_object_set_internal](#jerry_object_set_internal)


## jerry_object_set

**Summary**

Set a property to the specified object with the given name.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_object_set (const jerry_value_t obj_val,
                  const jerry_value_t prop_name_val,
                  const jerry_value_t value_to_set)
```

- `obj_val` - object value
- `prop_name_val` - property name
- `value_to_set` - value to set
- return value
  - true, if success
  - thrown exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_set_property` in earlier versions.*

**Example**

```c
{
  jerry_value_t value_to_set;

  ... // create or copy value to set

  jerry_value_t glob_obj = jerry_current_realm ();
  jerry_value_t prop_name = jerry_string_sz ("my_prop");

  jerry_value_t set_result = jerry_object_set (glob_obj, prop_name, value_to_set);

  ... // check result of property set call

  jerry_value_free (set_result);
  jerry_value_free (prop_name);

  ...

  jerry_value_free (value_to_set);
  jerry_value_free (glob_obj);
}
```

**See also**

- [jerry_object_has](#jerry_object_has)
- [jerry_object_has_own](#jerry_object_has_own)
- [jerry_object_delete](#jerry_object_delete)
- [jerry_object_delete_index](#jerry_object_delete_index)
- [jerry_object_get](#jerry_object_get)
- [jerry_object_get_index](#jerry_object_get_index)
- [jerry_object_set_index](#jerry_object_set_index)


## jerry_object_set_sz

**Summary**

Set a property to the specified object with the given name.

*Note*: Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
is no longer needed.

**Prototype**

```c
jerry_value_t jerry_object_set_sz (jerry_value_t object, const char *key_p, const jerry_value_t value);
```

- `obj_val` - object value
- `key_p` - property name (zero-terminated string)
- `value_to_set` - value to set
- return value
  - true, if success
  - thrown exception, otherwise

*Introduced in version 3.0*

**Example**

```c
{
  jerry_value_t value_to_set;

  ... // create or copy value to set

  jerry_value_t glob_obj = jerry_current_realm ();

  jerry_value_t set_result = jerry_object_set_sz (glob_obj, "my_prop", value_to_set);

  ... // check result of property set call

  jerry_value_free (set_result);

  ...

  jerry_value_free (value_to_set);
  jerry_value_free (glob_obj);
}
```

**See also**

- [jerry_object_set](#jerry_object_set)


## jerry_object_set_index

**Summary**

Set indexed value in the specified object

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_object_set_index (const jerry_value_t obj_val,
                        uint32_t index,
                        const jerry_value_t value_to_set);
```

- `obj_val` - object value
- `index` - index number
- `value_to_set` - value to set
- return value
  - true, if field value was set successfully
  - thrown exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_set_property_by_index` in earlier versions.*

**Example**

```c
{
  jerry_value_t object;
  jerry_value_t value_to_set;

  ... // create or copy object and value to set

  jerry_value_t ret_val = jerry_object_set_index (object, 5, value_to_set);

  ...

  jerry_value_free (value_to_set);
  jerry_value_free (ret_val);
  jerry_value_free (object);
}
```

**See also**

- [jerry_object_has](#jerry_object_has)
- [jerry_object_has_own](#jerry_object_has_own)
- [jerry_object_delete](#jerry_object_delete)
- [jerry_object_delete_index](#jerry_object_delete_index)
- [jerry_object_get](#jerry_object_get)
- [jerry_object_set](#jerry_object_set)
- [jerry_object_get_index](#jerry_object_get_index)


## jerry_object_set_internal

**Summary**

Set an internal property to the specified object with the given name.

*Note*:
- The property cannot be accessed from the JavaScript context, only from the public API.
- It is different from [jerry_object_set_native_ptr](#jerry_object_set_native_ptr) in that any jerry API value
  can be hidden from the JavaScript context, not only native pointers.

**Prototype**

```c
bool
jerry_object_set_internal (const jerry_value_t obj_val,
                           const jerry_value_t prop_name_val,
                           const jerry_value_t value_to_set)
```

- `obj_val` - object value
- `prop_name_val` - property name
- `value_to_set` - value to set
- return value
  - true, if success
  - thrown exception, otherwise

*Introduced in version 2.2*.

*Renamed in version 3.0, it was previously known as `jerry_set_internal_property` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_object = jerry_current_realm ();
  jerry_value_t prop_name = jerry_string_sz ("hidden_property");
  jerry_value_t value_to_set = jerry_number (5);

  bool set_result = jerry_object_set_internal (global_object, prop_name, value_to_set);

  /* check the result of internal property set call */

  jerry_value_free (value_to_set);
  jerry_value_free (prop_name);
  jerry_value_free (global_object);

  return 0;
}
```

**See also**

- [jerry_object_has_internal](#jerry_object_has_internal)
- [jerry_object_delete_internal](#jerry_object_delete_internal)
- [jerry_object_get_internal](#jerry_object_get_internal)


## jerry_property_descriptor

**Summary**

Create a property descriptor and initialize it with default values. This means that all fields in
the `jerry_property_descriptor_t` struct will be set to zero or false depending on the field's type.

**Prototype**

```c
jerry_property_descriptor_t
jerry_property_descriptor (void);
```

*Renamed in version 3.0, it was previously known as `jerry_property_descriptor_create` in earlier versions.*

**Example**

```c
{
  jerry_property_descriptor_t prop_desc = jerry_property_descriptor ();

  ... // usage of prop_desc

  jerry_property_descriptor_free (&prop_desc);
}
```

For a more complete example see [jerry_object_define_own_prop](#jerry_object_define_own_prop).

**See also**

- [jerry_property_descriptor_t](#jerry_property_descriptor_t)
- [jerry_object_define_own_prop](#jerry_object_define_own_prop)
- [jerry_object_get_own_prop](#jerry_object_get_own_prop)
- [jerry_property_descriptor_free](#jerry_property_descriptor_free)


## jerry_object_define_own_prop

**Summary**

Define a property to the specified object with the given name.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_object_define_own_prop (const jerry_value_t obj_val,
                              const jerry_value_t prop_name_val,
                              const jerry_property_descriptor_t *prop_desc_p);
```

- `obj_val` - target object where the property should be registered
- `prop_name_val` - property name
- `prop_desc_p` - pointer to property descriptor
- return value
  - true, if success
  - thrown exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_define_own_property` in earlier versions.*

**Example**

Registering a simple value property via the `jerry_object_define_own_prop` method:

[doctest]: # (name="02.API-REFERENCE-define-property.c")

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_obj_val = jerry_current_realm ();

  // configure the property
  jerry_property_descriptor_t prop_desc = jerry_property_descriptor ();

  // create or copy value to set
  // For example:
  jerry_value_t value_to_set = jerry_number (33);

  // set the property descriptor fields:
  // set the "JERRY_PROP_IS_VALUE_DEFINED" flag to indicate the "value"
  //  field should be used during the property registration.
  // set the writable, configurable, enumerable flags to true
  prop_desc.flags |= (JERRY_PROP_IS_WRITABLE_DEFINED
                      | JERRY_PROP_IS_WRITABLE
                      | JERRY_PROP_IS_CONFIGURABLE_DEFINED
                      | JERRY_PROP_IS_CONFIGURABLE
                      | JERRY_PROP_IS_ENUMERABLE_DEFINED
                      | JERRY_PROP_IS_ENUMERABLE
                      | JERRY_PROP_IS_VALUE_DEFINED);

  // set the "value" field to the number 33
  prop_desc.value = value_to_set;

  // add the property as "my_prop" for the global object
  jerry_value_t prop_name = jerry_string_sz ("my_prop");
  jerry_value_t return_value = jerry_object_define_own_prop (global_obj_val, prop_name, &prop_desc);
  if (jerry_value_is_exception (return_value))
  {
    // there was an error
  }

  // re-define the property with the enumerable flag set to false
  prop_desc.flags &= (uint16_t) ~JERRY_PROP_IS_ENUMERABLE;
  return_value = jerry_object_define_own_prop (global_obj_val, prop_name, &prop_desc);
  if (jerry_value_is_exception (return_value))
  {
    // there was an error
  }

  // if there was no exception at this point the global object should have a "my_prop" property

  jerry_value_free (return_value);
  jerry_value_free (prop_name);

  jerry_property_descriptor_free (&prop_desc);
  jerry_value_free (global_obj_val);

  jerry_cleanup ();
  return 0;
}
```


Registering a getter/setter property via the `jerry_object_define_own_prop` method:

[doctest]: # (name="02.API-REFERENCE-define-property-getset.c")

```c
#include <stdio.h>
#include <string.h>
#include "jerryscript.h"

static int counter = 0;

static jerry_value_t
method_getter (const jerry_call_info_t *call_info_p,
               const jerry_value_t args[],
               const jerry_length_t argc)
{
  counter++;
  printf("Getter called, returning: %d\n", counter);

  return jerry_number (counter);
}

static jerry_value_t
method_setter (const jerry_call_info_t *call_info_p,
               const jerry_value_t args[],
               const jerry_length_t argc)
{
  // Note: the arguments count and type should be checked
  // in this example it is omitted!

  double new_value = jerry_value_as_number (args[0]);
  counter = (int) new_value;

  printf("Setter called, setting: %d\n", counter);

  return jerry_undefined ();
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global_obj_val = jerry_current_realm ();

  // configure the property
  jerry_property_descriptor_t prop_desc = jerry_property_descriptor ();

  // set the property descriptor fields:

  prop_desc.flags |= JERRY_PROP_IS_GET_DEFINED | JERRY_PROP_IS_SET_DEFINED;
  prop_desc.getter = jerry_function_external (method_getter);
  prop_desc.setter = jerry_function_external (method_setter);

  // add the property as "my_prop" for the global object
  jerry_value_t prop_name = jerry_string_sz ("my_prop");
  jerry_value_t return_value = jerry_object_define_own_prop (global_obj_val, prop_name, &prop_desc);
  if (jerry_value_is_exception (return_value))
  {
    // there was an error
  }

  // if there was no exception at this point the global object should have a "my_prop" property

  jerry_value_free (return_value);
  jerry_value_free (prop_name);

  jerry_property_descriptor_free (&prop_desc);
  jerry_value_free (global_obj_val);

  // run an example js code to use the getter/setters

  const char *src_p = "this.my_prop; this.my_prop; this.my_prop = 4; this.my_prop";
  jerry_value_t eval_result = jerry_eval ((const jerry_char_t *) src_p, strlen (src_p), JERRY_PARSE_NO_OPTS);

  // "eval_result" is the last result of "this.my_prop" that is "5" currently.
  double result_number = jerry_value_as_number (eval_result);
  printf("output: %lf\n", result_number);

  jerry_cleanup ();

  return result_number != 5.0;
}
```

**See also**

- [jerry_property_descriptor_t](#jerry_property_descriptor_t)
- [jerry_property_descriptor](#jerry_property_descriptor)
- [jerry_object_get_own_prop](#jerry_object_get_own_prop)
- [jerry_property_descriptor_free](#jerry_property_descriptor_free)


## jerry_object_get_own_prop

**Summary**

Construct property descriptor from specified property.

**Prototype**

```c
jerry_value_t
jerry_object_get_own_prop (const jerry_value_t obj_val,
                           const jerry_value_t prop_name_val,
                           jerry_property_descriptor_t *prop_desc_p);
```

- `obj_val` - object value
- `prop_name_val` - property name
- `prop_desc_p` - pointer to property descriptor
- return value

*Renamed in version 3.0, it was previously known as `jerry_get_own_property_descriptor` in earlier versions. Return value type is changed to `jerry_value_t`.*

**Example**

```c
{
  jerry_value_t global_obj_val = jerry_current_realm ();

  jerry_property_descriptor_t prop_desc = jerry_property_descriptor ();

  jerry_value_t prop_name = jerry_string_sz ("my_prop");
  jerry_object_get_own_prop (global_obj_val, prop_name, &prop_desc);
  jerry_value_free (prop_name);

  ... // usage of property descriptor

  jerry_property_descriptor_free (&prop_desc);
  jerry_value_free (global_obj_val);
}
```

**See also**

- [jerry_property_descriptor_t](#jerry_property_descriptor_t)
- [jerry_property_descriptor](#jerry_property_descriptor)
- [jerry_object_define_own_prop](#jerry_object_define_own_prop)
- [jerry_property_descriptor_free](#jerry_property_descriptor_free)


## jerry_property_descriptor_free

**Summary**

Free fields of property descriptor (setter, getter and value).

**Prototype**

```c
void
jerry_property_descriptor_free (const jerry_property_descriptor_t *prop_desc_p);
```

- `prop_desc_p` - pointer to property descriptor

*Introduced in version 3.0*: Replaces `jerry_free_property_descriptor_fields`.

**Example**

```c
{
  jerry_property_descriptor_t prop_desc = jerry_property_descriptor ();

  ... // usage of property descriptor

  jerry_property_descriptor_free (&prop_desc);
}
```

**See also**

- [jerry_property_descriptor](#jerry_property_descriptor)
- [jerry_object_define_own_prop](#jerry_object_define_own_prop)
- [jerry_object_get_own_prop](#jerry_object_get_own_prop)


## jerry_call

**Summary**

Call function specified by a function value. The argument values must not be exceptions.
Value of `this` parameter should be set to `undefined` for non-method calls.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_call (const jerry_value_t func_obj_val,
            const jerry_value_t this_val,
            const jerry_value_t args_p[],
            jerry_size_t args_count);
```

- `func_obj_val` - the function object to call
- `this_val` - object for 'this' binding
- `args_p` - function's call arguments
- `args_count` - number of arguments
- return value - returned jerry value of the called function

*Renamed in version 3.0, it was previously known as `jerry_call_function` in earlier versions.*

**Example**

```c
{
  jerry_value_t target_function;

  ... // create or get "target_function"

  if (jerry_value_is_function (target_function))
  {
    jerry_value_t this_val = jerry_undefined ();
    jerry_value_t ret_val = jerry_call (target_function, this_val, NULL, 0);

    if (!jerry_value_is_exception (ret_val))
    {
      ... // handle return value
    }

    jerry_value_free (ret_val);
    jerry_value_free (this_val);
  }

  jerry_value_free (target_function);
}
```

**See also**

- [jerry_is_function](#jerry_is_function)
- [jerry_function_external](#jerry_function_external)


## jerry_construct

**Summary**

Construct object, invoking specified function object as constructor.
Error flag must not be set for any arguments of this function.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_construct (const jerry_value_t func_obj_val,
                 const jerry_value_t args_p[],
                 jerry_size_t args_count);
```

- `func_obj_val` - function object to call
- `args_p` - function's call arguments
- `args_count` - number of arguments
- return value - returned value of the invoked constructor

*Renamed in version 3.0, it was previously known as `jerry_construct_object` in earlier versions.*

**Example**

```c
{
  jerry_value_t val;

  ... // receiving val

  if (jerry_is_constructor (val))
  {
    jerry_value_t ret_val = jerry_construct (val, NULL, 0);

    if (!jerry_value_is_exception (ret_val))
    {
      ... // handle return value
    }

    jerry_value_free (ret_val);
  }
}
```

**See also**

 - [jerry_is_constructor](#jerry_is_constructor)


## jerry_object_keys

**Summary**

Get keys of the specified object value.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_object_keys (const jerry_value_t obj_val);
```

- `obj_val` - object value
- return value
  - array object value, if success
  - thrown exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_get_object_keys` in earlier versions.*

**Example**

```c
{
  jerry_value_t object;
  ... // create or copy object

  jerry_value_t keys_array = jerry_object_keys (object);

  ... // usage of keys_array

  jerry_value_free (keys_array);
}
```

**See also**

- [jerry_object_get](#jerry_object_get)
- [jerry_object_set](#jerry_object_set)


## jerry_object_proto

**Summary**

Get the prototype of the specified object.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_object_proto (const jerry_value_t obj_val);
```

- `obj_val` - object value
- return value
  - object value, if success
  - null or thrown exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_get_prototype` in earlier versions.*

**Example**

```c
{
  jerry_value_t object;
  ... // create or copy object

  jerry_value_t prototype = jerry_object_proto (object);

  ... // usage of prototype object

  jerry_value_free (prototype);
  jerry_value_free (object);
}
```

**See also**

- [jerry_object_set_proto](#jerry_object_set_proto)


## jerry_object_set_proto

**Summary**

Set the prototype of the specified object.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_object_set_proto (const jerry_value_t obj_val,
                        const jerry_value_t proto_obj_val);
```

- `obj_val` - object value
- `proto_obj_val` - prototype object value
- return value
  - true, if success
  - thrown exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_set_prototype` in earlier versions.*

**Example**

```c
{
  jerry_value_t object;
  jerry_value_t prototype;

  ... // create or copy object and prototype

  jerry_value_t ret_val = jerry_object_set_proto (object, prototype);

  jerry_value_free (ret_val);
  jerry_value_free (prototype);
  jerry_value_free (object);
}
```

**See also**

- [jerry_object_proto](#jerry_object_proto)


## jerry_object_get_native_ptr

**Summary**

Get native pointer by the given type information.
The pointer and the type information are previously associated with the object by
[jerry_object_set_native_ptr](#jerry_object_set_native_ptr).

*Note*:
- `native_info_p` can be NULL

**Prototype**

```c
void *
jerry_object_get_native_ptr (const jerry_value_t object,
                             const jerry_object_native_info_t *native_info_p)
```

- `object` - object value to get native pointer from.
- `native_info_p` - native pointer's type information.
- return value
  - native pointer associated with the argument object for the given native type info

*Introduced in version 2.0*: Changed from `jerry_get_object_native_handle`.

*Renamed in version 3.0, it was previously known as `jerry_get_object_native_pointer` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "jerryscript.h"

typedef struct
{
  char *data_p;
  unsigned int length;
} buffer_native_object_t;

typedef struct
{
  int area;
  int perimeter;
} shape_native_object_t;

#define SECRET_INFO ((void *) 42)

static void
buffer_native_freecb (void *native_p,
                      jerry_object_native_info_t *info_p)
{
  (void) info_p;

  char *data_p = ((buffer_native_object_t*)native_p)->data_p;

  if (data_p != NULL)
  {
    free (data_p);
  }

  free (native_p);
}

static void
shape_native_freecb (void *native_p,
                     jerry_object_native_info_t *info_p)
{
  (void) info_p;

  free (native_p);
}

static void
destructor_freecb (void *native_p,
                   jerry_object_native_info_t *info_p)
{
  (void) native_p;
  (void) info_p;

   printf("Note: the object has been freed\n");
}

// NOTE: The address (!) of type_info acts as a way to uniquely "identify" the
// C type `buffer_native_object_t *`.
static const jerry_object_native_info_t buffer_obj_type_info =
{
  .free_cb = buffer_native_freecb
};

// NOTE: The address (!) of type_info acts as a way to uniquely "identify" the
// C type `shape_native_object_t *`.
static const jerry_object_native_info_t shape_obj_type_info =
{
  .free_cb = shape_native_freecb
};

// NOTE: The address (!) of type_info is the unique "identifier"
static const jerry_object_native_info_t destructor_obj_type_info =
{
  .free_cb = destructor_freecb
};

static void
print_buffer (char *data_p,
              unsigned int length)
{
  for (unsigned int i = 0; i < length; ++i)
  {
    printf("%c", data_p[i]);
  }

  printf("\n");
}

static void
do_stuff (jerry_value_t object)
{
  if (!jerry_object_has_native_ptr (object, &buffer_obj_type_info))
  {
    // Process the error
    return;
  }

  void *native_p = jerry_object_get_native_ptr (object, &buffer_obj_type_info);

  // It is safe to cast to buffer_native_object_t * and dereference the pointer:
  buffer_native_object_t *buffer_p = (buffer_native_object_t *) native_p;
  print_buffer (buffer_p->data_p, buffer_p->length); // Usage of buffer_p

  bool need_shape_info = true; // implementation dependent

  if (need_shape_info)
  {
    if (!jerry_object_has_native_ptr (object, &shape_obj_type_info))
    {
      // Process the error
      return;
    }

    native_p = jerry_object_get_native_ptr (object, &shape_obj_type_info);

    // It is safe to cast to shape_native_object_t * and dereference the pointer:
    shape_native_object_t *shape_p = (shape_native_object_t *) native_p;

    printf("Area: %d\tPerimeter: %d\n", shape_p->area, shape_p->perimeter); // Usage of shape_p
  }

  bool need_secret_info = true; // implementation dependent

  if (need_secret_info)
  {
    native_p = jerry_object_get_native_ptr (object, NULL);

    if (native_p == NULL)
    {
      // Process the error
      return;
    }

    printf("Secret: %d\n", (int)((uintptr_t) native_p)); // Usage of native_p

    bool deleted = jerry_object_delete_native_ptr (object, NULL);

    if (deleted)
    {
      printf("The secret is no longer available\n");
    }
  }
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t object = jerry_object ();
  buffer_native_object_t *buffer_p = (buffer_native_object_t *) malloc (sizeof (buffer_native_object_t));
  buffer_p->length = 14;
  buffer_p->data_p = (char *) malloc (buffer_p->length * sizeof (char));
  memcpy (buffer_p->data_p, "My buffer data", buffer_p->length);
  jerry_object_set_native_ptr (object, &buffer_obj_type_info, buffer_p);

  shape_native_object_t *shape_p = (shape_native_object_t *) malloc (sizeof (shape_native_object_t));
  shape_p->area = 6;
  shape_p->perimeter = 12;
  jerry_object_set_native_ptr (object, &shape_obj_type_info, shape_p);

  // The native pointer can be NULL. This gives possibly to get notified via the native type info's
  // free callback when the object has been freed by the GC.
  jerry_object_set_native_ptr (object, &destructor_obj_type_info, NULL);

  // The native type info can be NULL as well. In this case the registered property is simply freed
  // when the object is freed by the GC.
  jerry_object_set_native_ptr (object, NULL, SECRET_INFO);

  do_stuff (object);

  jerry_value_free (object);
  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_object](#jerry_object)
- [jerry_object_set_native_ptr](#jerry_object_set_native_ptr)
- [jerry_object_native_info_t](#jerry_object_native_info_t)


## jerry_object_has_native_ptr

**Summary**

Checks whether the argument object has a native pointer set for the specified native type info.

**Prototype**

```c
bool jerry_object_has_native_ptr (const jerry_value_t object, const jerry_object_native_info_t *native_info_p);
```

- `object` - object to set native pointer in
- `native_info_p` - object's native type info
- return value
  - true if the native pointer has been set,
  - false otherwise

*Introduced in version 3.0*

**Example**

See [jerry_object_get_native_ptr](#jerry_object_get_native_ptr) for a
best-practice example.

**See also**

- [jerry_object_set_native_ptr](#jerry_object_set_native_ptr)
- [jerry_object_get_native_ptr](#jerry_object_get_native_ptr)
- [jerry_object_native_info_t](#jerry_object_native_info_t)


## jerry_object_set_native_ptr

**Summary**

Set native pointer and an optional type information for the specified object.
You can get them by calling [jerry_object_get_native_ptr](#jerry_object_get_native_ptr) later.

*Note*:
- If a native pointer was already set for the object with the same type information, its value is updated.
- If a non-NULL free callback is specified in the native type information,
  it will be called by the garbage collector when the object is freed.
- If the object is only referenced via the "global" object (or one of it's "child"),
  the free callback will be invoked during the execution of `jerry_cleanup`.
- The free callback can invoke API functions.
- If possible do not store API values in native pointers, rather check
  [jerry_object_set_internal](#jerry_object_set_internal).

**Prototype**

```c
void
jerry_object_set_native_ptr (const jerry_value_t obj_val,
                             const jerry_object_native_info_t *info_p,
                             void *native_p);
```

- `obj_val` - object to set native pointer in.
- `info_p` - native pointer's type information or NULL. When used, this should
             be a long-lived pointer, usually a pointer to a
             `static const jerry_object_native_info_t` makes most sense.
- `native_p` - native pointer.

*Introduced in version 2.0*: Changed from `jerry_set_object_native_handle`.

*Renamed in version 3.0, it was previously known as `jerry_set_object_native_pointer` in earlier versions.*

**Example**

See [jerry_object_get_native_ptr](#jerry_object_get_native_ptr) for a
best-practice example.

**See also**

- [jerry_object](#jerry_object)
- [jerry_object_get_native_ptr](#jerry_object_get_native_ptr)
- [jerry_object_native_info_t](#jerry_object_native_info_t)

## jerry_object_delete_native_ptr

**Summary**

Delete the native pointer of the specified object associated with the given native type info.

*Note*:
- If the specified object has no matching native pointer for the given native type info the operation has no effect.
- The method does not invoke the free callback specified in the type info.
  If the native pointer should be freed then one must get the native pointer first and invoke the free callback manually
  before calling this method.
- This operation cannot throw an exception.

**Prototype**

```c
bool
jerry_object_delete_native_ptr (const jerry_value_t obj_val,
                                const jerry_object_native_info_t *info_p)
```

- `obj_val` - object to delete native pointer from.
- `info_p` - native pointer's type information.

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_delete_object_native_pointer` in earlier versions.*

**Example**

See [jerry_object_get_native_ptr](#jerry_object_get_native_ptr) for a
best-practice example.

**See also**

- [jerry_object](#jerry_object)
- [jerry_object_get_native_ptr](#jerry_object_get_native_ptr)
- [jerry_object_get_native_ptr](#jerry_object_set_native_ptr)
- [jerry_object_native_info_t](#jerry_object_native_info_t)


## jerry_native_ptr_init

**Summary**

Initialize the references stored in a buffer pointed by a native pointer.
The references are initialized to undefined. This function must be called
before the buffer is attached to an object by
[jerry_object_set_native_ptr](#jerry_object_set_native_ptr).

*Note*:
- The description of [jerry_object_native_info_t](#jerry_object_native_info_t)
  provides detailed information about these references.

**Prototype**

```c
void
jerry_native_ptr_init (void *native_pointer_p,
                       const jerry_object_native_info_t *native_info_p);
```

- `native_pointer_p` - a valid non-NULL pointer to a native buffer.
- `native_info_p` - native pointer's type information.

*Renamed in version 3.0, it was previously known as `jerry_native_pointer_release_references` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include <stdlib.h>
#include "jerryscript.h"

typedef struct
{
  uint32_t user_data;
  jerry_value_t a;
  jerry_value_t b;
  uint32_t user_other_data;
} user_buffer_t;

static void
native_references_free_callback (void *native_p, /**< native pointer */
                                 jerry_object_native_info_t *info_p) /**< native info */
{
  /* References must be finalized when a buffer is no longer attached. */
  jerry_native_ptr_free (native_p, info_p);
  free (native_p);
} /* native_references_free_callback */

static const jerry_object_native_info_t native_info =
{
  .free_cb = native_references_free_callback,
  .number_of_references = 2,
  .offset_of_references = offsetof(user_buffer_t, a),
};

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t object_value = jerry_object ();

  user_buffer_t *buffer_p = (user_buffer_t *) malloc (sizeof (user_buffer_t));

  /* References must be initialized before a buffer is attached. */
  jerry_native_ptr_init ((void *) buffer_p, &native_info);

  jerry_object_set_native_ptr (object_value, &native_info, (void *) buffer_p);

  /* References can be modified after the buffer is attached.
   * This example sets a self reference. */
  jerry_native_ptr_set (&buffer_p->a, object_value);

  jerry_value_free (object_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_object_set_native_ptr](#jerry_object_set_native_ptr)
- [jerry_native_ptr_free](#jerry_native_ptr_free)
- [jerry_native_ptr_set](#jerry_native_ptr_set)

## jerry_native_ptr_free

**Summary**

Release the value references stored in a buffer pointed by a native pointer.
This function must be called after a buffer is no longer attached to any
object, even if the buffer is attached to another object again. This
function also initializes the values to undefined, so calling
[jerry_native_ptr_init](#jerry_native_ptr_init)
is optional before the buffer is attached again.

*Note*:
- The description of [jerry_object_native_info_t](#jerry_object_native_info_t)
  provides detailed information about these references.

**Prototype**

```c
void
jerry_native_ptr_free (void *native_pointer_p,
                       const jerry_object_native_info_t *native_info_p);
```

- `native_pointer_p` - a valid non-NULL pointer to a native buffer.
- `native_info_p` - native pointer's type information.

*Renamed in version 3.0, it was previously known as `jerry_native_pointer_init_references` in earlier versions.*

**Example**

See the example of [jerry_native_ptr_init](#jerry_native_ptr_init).

**See also**

- [jerry_object_set_native_ptr](#jerry_object_set_native_ptr)
- [jerry_native_ptr_init](#jerry_native_ptr_init)
- [jerry_native_ptr_set](#jerry_native_ptr_set)


## jerry_native_ptr_set

**Summary**

Updates a value reference inside the area specified by the `number_of_references` and
`offset_of_references` fields in its corresponding
[jerry_object_native_info_t](#jerry_object_native_info_t) data. The area must be
part of a buffer which is currently assigned to an object.

*Note*:
- The description of [jerry_object_native_info_t](#jerry_object_native_info_t)
  provides detailed information about these references.

**Prototype**

```c
void
jerry_native_ptr_set (jerry_value_t *reference_p,
                      jerry_value_t value)
```

- `reference_p` - a valid non-NULL pointer to a reference in a native buffer.
- `value` - new value of the reference.

*Renamed in version 3.0, it was previously known as `jerry_native_pointer_set_reference` in earlier versions.*

**Example**

See the example of [jerry_native_ptr_init](#jerry_native_ptr_init).

**See also**

- [jerry_object_set_native_ptr](#jerry_object_set_native_ptr)
- [jerry_native_ptr_init](#jerry_native_ptr_init)
- [jerry_native_ptr_free](#jerry_native_ptr_free)


## jerry_object_property_names

**Summary**

Gets the property keys for the given object using the selected filters.

**Prototype**

```c
jerry_value_t
jerry_object_property_names (jerry_value_t obj_val,
                             jerry_property_filter_t filter);
```

- `obj_val` - object value
- `filter` - any combination of [jerry_property_filter_t](#jerry_property_filter_t) options
- return value
  - array containing the filtered property keys in successful operation
  - exception, otherwise

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_object_get_property_names` in earlier versions.*

**Example**

```c
{
  jerry_value_t global_object = jerry_current_realm ();
  jerry_value_t keys = jerry_object_property_names (object, JERRY_PROPERTY_FILTER_ALL);

  ... // usage of keys

  jerry_value_free (keys);
  jerry_value_free (global_object);
}
```

**See also**

- [jerry_property_filter_t](#jerry_property_filter_t)

## jerry_object_foreach

**Summary**

Applies the given function to every enumerable(!) property in the given object.

The "iterator" `foreach_p` method should return `true` value to continue the iteration.
If the method returns `false` the iteration will end.

**Prototype**

```c
bool
jerry_object_foreach (jerry_value_t obj_val,
                      jerry_object_property_foreach_cb_t foreach_p,
                      void *user_data_p);
```

- `obj_val` - object value
- `foreach_p` - foreach function, that will be applied for each property
- `user_data_p` - user data for foreach function
- return value
  - true, if object fields traversal was performed successfully, i.e.:
    - no unhandled exceptions were thrown in object fields traversal
    - object fields traversal was stopped on callback that returned false
  - false, otherwise

*Renamed in version 3.0, it was previously known as `jerry_foreach_object_property` in earlier versions.*

**Example**

[doctest]: # (name="02.API-REFERENCE-foreach-property.c")

```c
#include <stdio.h>
#include "jerryscript.h"

/* Example structure used as user data for the property iteration. */
struct iteration_data {
  int string_property_count;
};

/*
 * Example foreach function to print out property names.
 */
static bool
foreach_function (const jerry_value_t prop_name,
                  const jerry_value_t prop_value,
                  void *user_data_p)
{
  if (jerry_value_is_string (prop_name)) {
    jerry_char_t string_buffer[128];
    jerry_size_t copied_bytes = jerry_string_to_buffer (prop_name,
                                                        JERRY_ENCODING_UTF8,
                                                        string_buffer,
                                                        sizeof (string_buffer) - 1);
    string_buffer[copied_bytes] = '\0';

    printf ("Property: %s\n", string_buffer);

    struct iteration_data *data = (struct iteration_data *) user_data_p;
    data->string_property_count++;
  }

  /* return true to continue iteration */
  return true;
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  /* Construct an example object with a single property. */
  jerry_value_t object = jerry_object ();
  {
    jerry_value_t test_property = jerry_string_sz ("DemoProp");
    jerry_value_t test_value = jerry_number (3);
    /* By default all properties added to an object are enumerable. */
    jerry_value_t set_result = jerry_object_set (object, test_property, test_value);
    /* The `set_result` should be checked if it is an exception or not. */
    jerry_value_free (set_result);
    jerry_value_free (test_value);
    jerry_value_free (test_property);
  }

  /* Iterate on the object's properties with the given user data. */
  struct iteration_data user_data = { 0 };

  bool iteration_result = jerry_object_foreach (object, foreach_function, &user_data);
  /* Check and process the `iteration_result` if required. */

  jerry_value_free (object);

  jerry_cleanup ();

  return user_data.string_property_count == 0;
}
```

**See also**

- [jerry_object_property_foreach_cb_t](#jerry_object_property_foreach_cb_t)

## jerry_foreach_live_object

**Summary**

Iterate over all objects available in the engine.

The "iterator" `foreach_p` method should return `true` value to continue the search.
If the method returns `false` the search for the object is finished.

*Note*:
- Values obtained in `foreach_p` must be retained using [jerry_value_copy](#jerry_value_copy).

**Prototype**

```c
bool
jerry_foreach_live_object (jerry_foreach_live_object_cb_t foreach_p,
                           void *user_data_p);
```

- `foreach_p` - function that will be invoked for each object.
- `user_data_p` - User data to pass to the function.
- return value
  - `true`, if the search function terminated the traversal by returning `false`
  - `false`, if the end of the list of objects was reached

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_objects_foreach` in earlier versions.*

**Example**

[doctest]: # (name="02.API-REFERENCE-objects-foreach.c")

```c
#include <stdio.h>
#include "jerryscript.h"

/* Create a custom structure to guide the search and store the result. */
typedef struct
{
  jerry_value_t property_name;
  jerry_value_t result;
} find_my_object_info_t;

/*
 * Find the first object with the given property.
 */
static bool
find_my_object (const jerry_value_t candidate,
                void *user_data_p)
{
  find_my_object_info_t *info_p = (find_my_object_info_t *) user_data_p;

  /* Check if the given object has the required property. */
  jerry_value_t has_property = jerry_object_has (candidate, info_p->property_name);
  bool object_found = jerry_value_is_true (has_property);

  if (object_found)
  {
    /* We found it, so we copy the value and record it. */
    info_p->result = jerry_value_copy (candidate);
  }

  jerry_value_free (has_property);

  /* If the object was not found continue the search. */
  return !object_found;
} /* find_my_object */

int
main (void)
{
  int return_value = 0;

  /* Initialize JerryScript engine. */
  jerry_init (JERRY_INIT_EMPTY);

  /* Create the test object. */
  {
    jerry_value_t test_object = jerry_object ();

    {
      jerry_value_t test_property = jerry_string_sz ("DemoProp");
      jerry_value_t test_value = jerry_number (3);
      jerry_value_t set_result = jerry_object_set (test_object, test_property, test_value);
      /* The `set_result` should be checked if it is an exception or not. */
      jerry_value_free (set_result);
      jerry_value_free (test_value);
      jerry_value_free (test_property);
    }

    {
      /* Register the test object into the global object. */
      jerry_value_t global_object = jerry_current_realm ();
      jerry_value_t demo_property = jerry_string_sz ("DemoObject");
      jerry_value_t set_result = jerry_object_set (global_object, demo_property, test_object);
      /* The `set_result` should be checked if it is an exception or not. */
      jerry_value_free (set_result);
      jerry_value_free (demo_property);
      jerry_value_free (global_object);
    }

    jerry_value_free (test_object);
  }

  /* Look up the test object base on a property name. */
  find_my_object_info_t search_info =
  {
    .property_name = jerry_string_sz ("DemoProp")
  };

  if (jerry_foreach_live_object (find_my_object, &search_info))
  {
    /* The search was successful. Do something useful with search_info.result. */
    // ...
    printf ("Object found\n");

    /* Release the found object after we're done using it. */
    jerry_value_free (search_info.result);
  }
  else
  {
    /* The search has failed. */
    printf ("Object not found\n");

    return_value = 1;
  }

  jerry_value_free (search_info.property_name);

  /* Engine cleanup */
  jerry_cleanup ();
  return return_value;
}
```

**See also**

- [jerry_foreach_live_object_cb_t](#jerry_foreach_live_object_cb_t)

## jerry_foreach_live_object_with_info

**Summary**

Iterate over all objects in the engine matching a certain native data type.

The "iterator" `foreach_p` method should return `true` value to continue the search.
If the method returns `false` the search for the object is finished.

*Note*:
- Values obtained in `foreach_p` must be retained using [jerry_value_copy](#jerry_value_copy).

**Prototype**

```c
bool
jerry_foreach_live_object_with_info (const jerry_object_native_info_t *native_info_p,
                                     jerry_foreach_live_object_with_info_cb_t foreach_p,
                                     void *user_data_p);
```

- `native_info_p` - native pointer's type information.
- `foreach_p` - function that will be invoked for each object.
- return value
  - `true`, if the search function terminated the traversal by returning `false`
  - `false`, if the end of the list of objects was reached

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_objects_foreach_by_native_info` in earlier versions.*

**Example**

[doctest]: # (name="02.API-REFERENCE-objects-foreach-nativeptr.c")

```c
#include <stdio.h>
#include <stdlib.h>
#include "jerryscript.h"

typedef struct
{
  int foo;
  bool bar;
} native_obj_t;

typedef struct
{
  jerry_value_t found_object;
  native_obj_t *found_native_data_p;

  int match_foo_value;
} find_object_data_t;

static void native_freecb (void *native_p,
                           jerry_object_native_info_t *info_p)
{
  (void) info_p;

  /* `native_p` was allocated via malloc. */
  free (native_p);
} /* native_freecb */

/*
 * NOTE: The address (!) of type_info acts as a way to uniquely "identify" the
 * C type `native_obj_t *`.
 */
static const jerry_object_native_info_t native_obj_type_info =
{
  .free_cb = native_freecb
};

/*
 * Function creating JS object that is "backed" by a `native_obj_t`.
 */
static void
add_object_with_nativeptr (int foo_value)
{
  // construct object and native_set value:
  jerry_value_t test_object = jerry_object ();
  native_obj_t *native_obj_p = malloc (sizeof (*native_obj_p));
  native_obj_p->foo = foo_value;
  native_obj_p->bar = true;

  jerry_object_set_native_ptr (test_object, &native_obj_type_info, native_obj_p);

  /* Register the test object into the global object. */
  jerry_value_t global_object = jerry_current_realm ();
  jerry_value_t demo_property = jerry_string_sz ("DemoObject");
  jerry_value_t set_result = jerry_object_set (global_object, demo_property, test_object);
  /* The `set_result` should be checked if it is an exception or not. */
  jerry_value_free (set_result);
  jerry_value_free (demo_property);
  jerry_value_free (global_object);

  jerry_value_free (test_object);
} /* create_object_with_nativeptr */

/*
 * Example native method that searches for a JavaScript object
 * with a `native_obj_type_info` has the correct value.
 */
static bool
find_object (const jerry_value_t candidate, void *data_p, void *user_data_p)
{
  find_object_data_t *find_data_p = (find_object_data_t *) user_data_p;
  native_obj_t *native_obj_p = (native_obj_t *) data_p;

  if (find_data_p->match_foo_value == native_obj_p->foo)
  {
    /* If the object was found, copy it and store it in the user data. */
    find_data_p->found_object = jerry_value_copy (candidate);
    find_data_p->found_native_data_p = native_obj_p;

    /* Stop traversing over the objects. */
    return false;
  }

  /* Indicate that the object was not found, so traversal must continue. */
  return true;
} /* find_object */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  add_object_with_nativeptr (4);
  add_object_with_nativeptr (3);
  add_object_with_nativeptr (2);

  find_object_data_t find_data =
  {
    .match_foo_value = 3,
  };

  if (jerry_foreach_live_object_with_info (&native_obj_type_info, find_object, &find_data))
  {
    /* The object was found and is now stored in `find_data.found_object`. After using it, it must be released. */
    printf ("Object found, native foo value: %d\n", find_data.found_native_data_p->foo);

    jerry_value_free (find_data.found_object);
  }
  else
  {
    printf ("Object not found\n");
  }

  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_object](#jerry_object)
- [jerry_object_set_native_ptr](#jerry_object_set_native_ptr)
- [jerry_object_get_native_ptr](#jerry_object_get_native_ptr)
- [jerry_object_native_info_t](#jerry_object_native_info_t)
- [jerry_foreach_live_object](#jerry_foreach_live_object)


# Input validator functions

## jerry_validate_string

**Summary**

Check if a given character buffer is a valid in the specified encoding string.

*Note*:
- Calling this method is safe in any time. It can be called
  even before engine initialization.

**Prototype**

```c
bool
jerry_validate_string (const jerry_char_t *buffer_p, /**< string data */
                       jerry_size_t buf_size, /**< string size */
                       jerry_encoding_t encoding); /**< encoding */
```

- `buffer_p` - input string buffer.
- `buf_size` - input string buffer size in bytes.
- `encoding` - string encoding
- return value
  - true, if the provided string was a valid UTF-8 string.
  - false, if the string is not valid as an UTF-8 string.

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_is_valid_cesu8_string` or `jerry_is_valid_utf8_string` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  const jerry_char_t script[] = "print ('Hello, World!');";
  const jerry_size_t script_size = sizeof (script) - 1;

  if (!jerry_validate_string (script, script_size, JERRY_ENCODING_CESU8))
  {
    return 1;
  }

  return 0;
}
```

**See also**

- [jerry_string](#jerry_string)
- [jerry_string_size](#jerry_string_size)
- [jerry_string_length](#jerry_string_length)
- [jerry_string_to_buffer](#jerry_string_to_buffer)

# Dynamic memory management functions

## jerry_heap_alloc

**Summary**

Allocate memory on the engine's heap.

*Note*:
- This function may take away memory from the executed JavaScript code.
  If any other dynamic memory allocation API is available (e.g., libc malloc), it
  should be used instead.

**Prototype**

```c
void *jerry_heap_alloc (size_t size);
```

- `size`: size of the memory block.
- return value: non-NULL pointer, if the memory is successfully allocated,
                NULL otherwise.

*Introduced in version 2.0*.

**See also**

- [jerry_heap_free](#jerry_heap_free)

## jerry_heap_free

**Summary**

Free memory allocated on the engine's heap.

**Prototype**

```c
void jerry_heap_free (void *mem_p, size_t size);
```

- `mem_p`: value returned by `jerry_heap_alloc`.
- `size`: same size as passed to `jerry_heap_alloc`.

*Introduced in version 2.0*.

**See also**

- [jerry_heap_alloc](#jerry_heap_alloc)


# Snapshot functions

## jerry_generate_snapshot

**Summary**

Generate snapshot from the specified source code.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.
- This API depends on a build option (`JERRY_SNAPSHOT_SAVE`) and can be checked in runtime with
  the `JERRY_FEATURE_SNAPSHOT_SAVE` feature enum value, see [jerry_feature_enabled](#jerry_feature_enabled).
  If the feature is not enabled the function will return an exception.

**Prototype**

```c
jerry_value_t
jerry_generate_snapshot (jerry_value_t compiled_code,
                         uint32_t generate_snapshot_opts,
                         uint32_t *buffer_p,
                         size_t buffer_size);
```

- `compiled_code` - compiled script or function (see: [jerry_parse](#jerry_parse)).
- `generate_snapshot_opts` - any combination of [jerry_generate_snapshot_opts_t](#jerry_generate_snapshot_opts_t) flags.
- `buffer_p` - output buffer (aligned to 4 bytes) to save snapshot to.
- `buffer_size` - the output buffer's size in bytes.
- return value
  - the size of the generated snapshot in bytes as number value, if it was generated successfully (i.e. there
    are no syntax errors in source code, buffer size is sufficient, and snapshot support is enabled in
    current configuration through JERRY_SNAPSHOT_SAVE)
  - thrown exception, otherwise.

*Introduced in version 2.0*.

*Updated in version 3.0*: The `source_p`, `source_size`, `source_name_p`,
                                       and `source_name_length` arguments are replaced by `compiled_code`
                                       which should contain a compiled ECMAScript script / function.
                                       The `jerry_generate_function_snapshot` is now removed and can be reproduced
                                       by calling `jerry_parse` with function arguments and using this method
                                       (see [jerry_exec_snapshot](#jerry_exec_snapshot)).
**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  static uint32_t global_mode_snapshot_buffer[256];
  const jerry_char_t script_to_snapshot[] = "(function () { return 'string from snapshot'; }) ();";

  jerry_value_t parse_result = jerry_parse (script_to_snapshot,
                                            sizeof (script_to_snapshot) - 1,
                                            NULL);

  size_t buffer_size = sizeof (global_mode_snapshot_buffer) / sizeof (uint32_t);
  jerry_value_t generate_result = jerry_generate_snapshot (parse_result,
                                                           0,
                                                           global_mode_snapshot_buffer,
                                                           buffer_size);
  jerry_value_free (parse_result);

  if (!jerry_value_is_exception (generate_result))
  {
    size_t snapshot_size = (size_t) jerry_value_as_number (generate_result);
  }

  jerry_value_free (generate_result);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_parse](#jerry_parse)
- [jerry_parse_value](#jerry_parse_value)
- [jerry_exec_snapshot](#jerry_exec_snapshot)
- [jerry_parse_options_t](#jerry_parse_options_t)


## jerry_exec_snapshot

**Summary**

Execute/load snapshot from the specified buffer.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.
- This API depends on a build option (`JERRY_SNAPSHOT_EXEC`) and can be checked in runtime with
  the `JERRY_FEATURE_SNAPSHOT_EXEC` feature enum value, see [jerry_feature_enabled](#jerry_feature_enabled).
  If the feature is not enabled the function will return an exception.

**Prototype**

```c
jerry_value_t
jerry_exec_snapshot (const uint32_t *snapshot_p,
                     size_t snapshot_size,
                     size_t func_index,
                     uint32_t exec_snapshot_opts,
                     const jerry_exec_snapshot_option_values_t *options_values_p);
```

- `snapshot_p` - pointer to snapshot.
- `snapshot_size` - size of snapshot in bytes.
- `func_index` - index of executed function.
- `exec_snapshot_opts` - any combination of [jerry_exec_snapshot_opts_t](#jerry_exec_snapshot_opts_t) flags.
- `options_values_p` - additional loading options, can be NULL if not used. The fields are described in
                       [jerry_exec_snapshot_option_values_t](#jerry_exec_snapshot_option_values_t).
- return value
  - result of bytecode, if run was successful.
  - thrown exception, otherwise (an exception is reported if the snapshot execution feature is not enabled).

*Updated in version 2.0*: Added `func_index` and `exec_snapshot_opts` arguments. Removed the `copy_bytecode` last argument.

*Updated in version 3.0*: Added `options_p` argument.

**Example 1**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  static uint32_t snapshot_buffer[256];

  /* 1st example: global mode snapshot. */
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script_to_snapshot[] = "(function () { return 'string from snapshot'; }) ();";

  jerry_value_t parse_result = jerry_parse (script_to_snapshot,
                                            sizeof (script_to_snapshot) - 1,
                                            NULL);

  size_t buffer_size = sizeof (snapshot_buffer) / sizeof (uint32_t);
  jerry_value_t generate_result = jerry_generate_snapshot (parse_result,
                                                           0,
                                                           snapshot_buffer,
                                                           buffer_size);
  jerry_value_free (parse_result);

  /* 'generate_result' variable should be checked whether it contains an exception. */

  size_t snapshot_size = (size_t) jerry_value_as_number (generate_result);
  jerry_value_free (generate_result);

  jerry_cleanup ();
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t res = jerry_exec_snapshot (snapshot_buffer,
                                           snapshot_size,
                                           0,
                                           0,
                                           NULL);

  /* 'res' now contains 'string from snapshot' */
  jerry_value_free (res);

  jerry_cleanup ();
  return 0;
}
```

**Example 2**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  static uint32_t snapshot_buffer[256];

  /* 2nd example: function snapshot. */
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t function_to_snapshot[] = "return a + b;";

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_HAS_ARGUMENT_LIST;
  parse_options.argument_list = jerry_string_sz ("a, b");

  jerry_value_t parse_result = jerry_parse (function_to_snapshot,
                                            sizeof (function_to_snapshot) - 1,
                                            &parse_options);

  size_t buffer_size = sizeof (snapshot_buffer) / sizeof (uint32_t);
  jerry_value_t generate_result = jerry_generate_snapshot (parse_result,
                                                           0,
                                                           snapshot_buffer,
                                                           buffer_size);
  jerry_value_free (parse_result);
  jerry_value_free (parse_options.argument_list);

  /* 'generate_result' variable should be checked whether it contains an exception. */

  size_t snapshot_size = (size_t) jerry_value_as_number (generate_result);
  jerry_value_free (generate_result);

  jerry_cleanup ();
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t func = jerry_exec_snapshot (snapshot_buffer,
                                            snapshot_size,
                                            0,
                                            JERRY_SNAPSHOT_EXEC_LOAD_AS_FUNCTION,
                                            NULL);
  /* 'func' can be used now as a function object. */

  jerry_value_t this_value = jerry_undefined ();
  jerry_value_t args[2];
  args[0] = jerry_number (1.0);
  args[1] = jerry_number (2.0);

  jerry_value_t res = jerry_call (func, this_value, args, 2);

  /* 'res' now contains the value 3 as a jerry_value_t. */
  jerry_value_free (res);
  jerry_value_free (args[0]);
  jerry_value_free (args[1]);
  jerry_value_free (this_value);
  jerry_value_free (func);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_generate_snapshot](#jerry_generate_snapshot)

## jerry_merge_snapshots

**Summary**

Merge multiple snapshots into a single buffer

**Prototype**

```c
size_t jerry_merge_snapshots (const uint32_t **inp_buffers_p,
                              size_t *inp_buffer_sizes_p,
                              size_t number_of_snapshots,
                              uint32_t *out_buffer_p,
                              size_t out_buffer_size,
                              const char **error_p);
```

- `inp_buffers_p` - array of (pointers to start of) input buffers
- `inp_buffer_sizes_p` - array of input buffer sizes
- `number_of_snapshots` - number of snapshots
- `out_buffer_p` - output buffer
- `out_buffer_size` - output buffer size
- `error_p` - error description
- return value
  - length of the merged snapshot file
  - 0 on error

**Example**

```c
  static uint32_t snapshot_buffer_0[SNAPSHOT_BUFFER_SIZE];
  static uint32_t snapshot_buffer_1[SNAPSHOT_BUFFER_SIZE];
  size_t snapshot_sizes[2];
  static uint32_t merged_snapshot_buffer[SNAPSHOT_BUFFER_SIZE];

  const jerry_char_t code_to_snapshot1[] = "var a = 'hello'; 123";

  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t parse_result = jerry_parse (code_to_snapshot1, sizeof (code_to_snapshot1) - 1, NULL);
  jerry_value_t generate_result = jerry_generate_snapshot (parse_result, 0, snapshot_buffer_0, SNAPSHOT_BUFFER_SIZE);
  jerry_value_free (parse_result);

  snapshot_sizes[0] = (size_t) jerry_value_as_number (generate_result);
  jerry_value_free (generate_result);

  jerry_cleanup ();

  const jerry_char_t code_to_snapshot2[] = "var b = 'hello'; 456";

  jerry_init (JERRY_INIT_EMPTY);

  parse_result = jerry_parse (code_to_snapshot2, sizeof (code_to_snapshot2) - 1, NULL);
  generate_result = jerry_generate_snapshot (parse_result, 0, snapshot_buffer_1, SNAPSHOT_BUFFER_SIZE);
  jerry_value_free (parse_result);

  snapshot_sizes[1] = (size_t) jerry_value_as_number (generate_result);
  jerry_value_free (generate_result);

  jerry_cleanup ();

  jerry_init (JERRY_INIT_EMPTY);

  const char *error_p;
  const uint32_t *snapshot_buffers[2];

  snapshot_buffers[0] = snapshot_buffer_0;
  snapshot_buffers[1] = snapshot_buffer_1;

  static uint32_t snapshot_buffer_0_bck[SNAPSHOT_BUFFER_SIZE];
  static uint32_t snapshot_buffer_1_bck[SNAPSHOT_BUFFER_SIZE];

  memcpy (snapshot_buffer_0_bck, snapshot_buffer_0, SNAPSHOT_BUFFER_SIZE);
  memcpy (snapshot_buffer_1_bck, snapshot_buffer_1, SNAPSHOT_BUFFER_SIZE);

  size_t merged_size = jerry_merge_snapshots (snapshot_buffers,
                                              snapshot_sizes,
                                              2,
                                              merged_snapshot_buffer,
                                              SNAPSHOT_BUFFER_SIZE,
                                              &error_p);

  jerry_cleanup ();
```

**See also**

- [jerry_generate_snapshot](#jerry_generate_snapshot)
- [jerry_exec_snapshot](#jerry_exec_snapshot)


## jerry_get_literals_from_snapshot

**Summary**

Collect the used literals from the given snapshot and save them into a buffer in list or C format.
None of these literals are magic strings. In C format only valid identifiers are collected.

*Note*:
- This API depends on a build option (`JERRY_SNAPSHOT_SAVE`) and can be checked in runtime with
  the `JERRY_FEATURE_SNAPSHOT_SAVE` feature enum value, see [jerry_feature_enabled](#jerry_feature_enabled).
  If the feature is not enabled the function will return zero.

**Prototype**

```c
size_t
jerry_get_literals_from_snapshot (const uint32_t *snapshot_p,
                                  size_t snapshot_size,
                                  jerry_char_t *lit_buf_p,
                                  size_t lit_buf_size,
                                  bool is_c_format);
```

- `snapshot_p` - input snapshot buffer.
- `snapshot_size` - size of snapshot in bytes.
- `lit_buf_p` - buffer to save literals to.
- `lit_buf_size` - the buffer's size.
- `is_c_format` - the output format would be C-style (true) or a simple list (false).
- return value
  - the size of the literal-list, if it was generated successfully (i.e. the list of literals isn't empty,
    and literal-save support is enabled in current configuration through `JERRY_SNAPSHOT_SAVE`)
  - 0 otherwise.

*Introduced in version 2.0*.

**Example**

[doctest]: # (test="link")

```c
#include <stdio.h>
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  static jerry_char_t literal_buffer[256];
  static uint32_t snapshot_buffer[256];
  const jerry_char_t script_for_literal_save[] = "var obj = { a:'aa', bb:'Bb' }";

  jerry_value_t parse_result = jerry_parse (script_for_literal_save,
                                            sizeof (script_for_literal_save) - 1,
                                            NULL);

  size_t buffer_size = sizeof (snapshot_buffer) / sizeof (uint32_t);
  jerry_value_t generate_result = jerry_generate_snapshot (parse_result,
                                                           0,
                                                           snapshot_buffer,
                                                           buffer_size);
  jerry_value_free (parse_result);

  size_t snapshot_size = (size_t) jerry_value_as_number (generate_result);
  jerry_value_free (generate_result);

  const size_t literal_size = jerry_get_literals_from_snapshot (snapshot_buffer,
                                                                snapshot_size,
                                                                literal_buffer,
                                                                256,
                                                                true);

  if (literal_size != 0)
  {
    FILE *literal_file_p = fopen ("literals.h", "wb");
    fwrite (literal_buffer, sizeof (uint8_t), literal_size, literal_file_p);
    fclose (literal_file_p);
  }

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_init](#jerry_init)
- [jerry_cleanup](#jerry_cleanup)
- [jerry_register_magic_strings](#jerry_register_magic_strings)


# Backtrace functions

## jerry_backtrace

**Summary**

Get backtrace. The backtrace is an array of strings where
each string contains the position of the corresponding frame.
The array length is zero if the backtrace is not available.

This function is typically called from native callbacks.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.
- This feature depends on build option (`JERRY_LINE_INFO`) and can be checked
  in runtime with the `JERRY_FEATURE_LINE_INFO` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
jerry_value_t
jerry_backtrace (uint32_t max_depth);
```

- `max_depth` - backtrace collection stops after reaching this value, 0 = unlimited
- return value
  - a newly constructed JS array

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_get_backtrace` in earlier versions.*

**Example**

[doctest]: # (name="02.API-REFERENCE-jsbacktrace.c")

```c
#include <stdio.h>
#include <string.h>
#include "jerryscript.h"

static jerry_value_t
backtrace_handler (const jerry_call_info_t *call_info_p,
                   const jerry_value_t args_p[],
                   const jerry_length_t args_count)
{
  if (!jerry_feature_enabled (JERRY_FEATURE_LINE_INFO))
  {
    printf ("Line info disabled, no backtrace will be printed\n");
    return jerry_undefined ();
  }

  /* If the line info feature is disabled an empty array will be returned. */
  jerry_value_t backtrace_array = jerry_backtrace (5);
  uint32_t array_length = jerry_array_length (backtrace_array);

  for (uint32_t idx = 0; idx < array_length; idx++)
  {
    jerry_value_t property = jerry_object_get_index (backtrace_array, idx);

    jerry_char_t string_buffer[64];
    jerry_size_t copied_bytes = jerry_string_to_buffer (property,
                                                        JERRY_ENCODING_UTF8,
                                                        string_buffer,
                                                        sizeof (string_buffer) - 1);
    string_buffer[copied_bytes] = '\0';
    printf(" %d: %s\n", idx, string_buffer);

    jerry_value_free (property);
  }

  jerry_value_free (backtrace_array);

  return jerry_undefined ();
} /* backtrace_handler */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global = jerry_current_realm ();

  /* Register the "capture_backtrace" method. */
  {
    jerry_value_t func = jerry_function_external (backtrace_handler);
    jerry_value_t name = jerry_string_sz ("backtrace");
    jerry_value_t result = jerry_object_set (global, name, func);
    jerry_value_free (result);
    jerry_value_free (name);
    jerry_value_free (func);
  }

  jerry_value_free (global);

  const char *source = ("function f() { g (); }\n"
                        "function g() { h (); }\n"
                        "function h() { backtrace (); }\n"
                        "f ();\n");

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_HAS_SOURCE_NAME;
  parse_options.source_name = jerry_string_sz ("demo_memory.js");

  jerry_value_t program = jerry_parse ((const jerry_char_t *) source,
                                       strlen (source),
                                       &parse_options);
  jerry_value_free (parse_options.source_name);

  if (!jerry_value_is_exception (program))
  {
    jerry_value_t run_result = jerry_run (program);
    jerry_value_free (run_result);
  }

  jerry_value_free (program);
  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_function_external](#jerry_function_external)


## jerry_backtrace_capture

**Summary**

Low-level function to capture each backtrace frame. The captured frame data
is passed to a callback function. To improve performance, the majority of
the frame data is not initialized when the callback function is called. The
initialization of these fields can be done later by helper functions such
as [jerry_frame_location](#jerry_frame_location).

**Prototype**

```c
void
jerry_backtrace_capture (jerry_backtrace_cb_t callback, void *user_p);
```

- `callback` - a [jerry_backtrace_cb_t](#jerry_backtrace_cb_t) callback
               which is called for each captured frame
- `user_p` - pointer passed to the `callback` function, can be NULL

*Introduced in version 3.0*.

**Example**

[doctest]: # (name="02.API-REFERENCE-jscapturebacktrace.c")

```c
#include <stdio.h>
#include <string.h>
#include "jerryscript.h"

static bool
backtrace_callback (jerry_frame_t *frame_p,
                    void *user_p)
{
  printf (" A stack frame is captured\n");
  return true;
}

static jerry_value_t
backtrace_handler (const jerry_call_info_t *call_info_p,
                   const jerry_value_t args_p[],
                   const jerry_length_t args_count)
{
  (void) call_info_p;
  (void) args_p;
  (void) args_count;

  jerry_backtrace_capture (&backtrace_callback, NULL);

  return jerry_undefined ();
} /* backtrace_handler */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global = jerry_current_realm ();

  /* Register the "dump_backtrace" method. */
  {
    jerry_value_t func = jerry_function_external (backtrace_handler);
    jerry_value_t name = jerry_string_sz ("backtrace");
    jerry_value_t result = jerry_object_set (global, name, func);
    jerry_value_free (result);
    jerry_value_free (name);
    jerry_value_free (func);
  }

  jerry_value_free (global);

  const char *source = ("function f() { g (); }\n"
                        "function g() { h (); }\n"
                        "function h() { backtrace (g); }\n"
                        "f ();\n");

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_HAS_SOURCE_NAME;
  parse_options.source_name = jerry_string_sz ("demo_backtrace.js");

  jerry_value_t program = jerry_parse ((const jerry_char_t *) source,
                                       strlen (source),
                                       &parse_options);
  jerry_value_free (parse_options.source_name);

  if (!jerry_value_is_exception (program))
  {
    jerry_value_t run_result = jerry_run (program);
    jerry_value_free (run_result);
  }

  jerry_value_free (program);
  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_backtrace_capture](#jerry_backtrace_capture)
- [jerry_frame_type](#jerry_frame_type)
- [jerry_frame_location](#jerry_frame_location)
- [jerry_frame_callee](#jerry_frame_callee)
- [jerry_frame_this](#jerry_frame_this)
- [jerry_frame_is_strict](#jerry_frame_is_strict)


## jerry_frame_type

**Summary**

Returns with the type of the backtrace frame. This function can only be called
from the callback function of [jerry_backtrace_capture](#jerry_backtrace_capture),
and the value becomes invalid after the callback returns.

**Prototype**

```c
jerry_frame_type_t
jerry_frame_type (jerry_frame_t *frame_p);
```

- `frame_p` - a frame passed to the [jerry_backtrace_cb_t](#jerry_backtrace_cb_t) callback
- return value
  - frame type listed in [jerry_frame_type_t](#jerry_frame_type_t)

*Renamed in version 3.0, it was previously known as `jerry_backtrace_get_frame_type` in earlier versions.*

**Example**

See the example of [jerry_backtrace_capture](#jerry_backtrace_capture)
with the following callback function:

```c
static bool
backtrace_callback (jerry_frame_t *frame_p,
                    void *user_p)
{
  switch (jerry_frame_type (frame_p))
  {
    case JERRY_BACKTRACE_FRAME_JS:
    {
      printf (" ECMAScript frame\n");
      break;
    }
    default:
    {
      printf (" Other frame\n");
      break;
    }
  }

  return true;
}
```

**See also**

- [jerry_backtrace_capture](#jerry_backtrace_capture)


## jerry_frame_location

**Summary**

Initialize and return with the location private field of a backtrace
frame. If the location is not available, the returned value is NULL.
This function can only be called from the callback function of
[jerry_backtrace_capture](#jerry_backtrace_capture), and the value
becomes invalid after the callback returns.

*Note*:
- Location information can only be retrieved if JERRY_FEATURE_LINE_INFO feature is
  enabled. Otherwise the function always returns with NULL.
- The returned data must not be modified, and does not need to be freed.
  Any cleanup is done automatically after the callback is returned.

**Prototype**

```c
const jerry_frame_location_t *
jerry_frame_location (jerry_frame_t *frame_p);
```

- `frame_p` - a frame passed to the [jerry_backtrace_cb_t](#jerry_backtrace_cb_t) callback
- return value
  - pointer to the location private field if the location is available,
  - NULL otherwise

*Renamed in version 3.0, it was previously known as `jerry_backtrace_get_location` in earlier versions.*

**Example**

See the example of [jerry_backtrace_capture](#jerry_backtrace_capture)
with the following callback function:

```c
static bool
backtrace_callback (jerry_frame_t *frame_p,
                    void *user_p)
{
  const jerry_frame_location_t *location_p;
  location_p = jerry_frame_location (frame_p);

  if (location_p == NULL)
  {
    printf ("No location info is available\n");
    return true;
  }

  jerry_char_t string_buffer[64];
  jerry_size_t copied_bytes = jerry_string_to_buffer (location_p->source_name,
                                                      JERRY_ENCODING_UTF8
                                                      string_buffer,
                                                      sizeof (string_buffer) - 1);
  string_buffer[copied_bytes] = '\0';
  printf(" %s:%d:%d\n", string_buffer, (int) location_p->line, (int) location_p->column);
  return true;
}
```

**See also**

- [jerry_backtrace_capture](#jerry_backtrace_capture)


## jerry_frame_callee

**Summary**

Initialize and return with the called function private field of a backtrace frame.
The backtrace frame is created for running the code bound to this function. This
function can only be called from the callback function of
[jerry_backtrace_capture](#jerry_backtrace_capture), and the value becomes invalid
after the callback returns.

*Note*:
- The returned data must not be modified, and does not need to be freed.
  Any cleanup is done automatically after the callback is returned.

**Prototype**

```c
const jerry_value_t *
jerry_frame_callee (jerry_frame_t *frame_p);
```

- `frame_p` - a frame passed to the [jerry_backtrace_cb_t](#jerry_backtrace_cb_t) callback
- return value
  - pointer to the called function if the function is available,
  - NULL otherwise

*Renamed in version 3.0, it was previously known as `jerry_backtrace_get_function` in earlier versions.*

**Example**

See the example of [jerry_backtrace_capture](#jerry_backtrace_capture)
with the following callback function:

```c
static bool
backtrace_callback (jerry_frame_t *frame_p,
                    void *user_p)
{
  jerry_value_t *function_p = jerry_frame_callee (frame_p);

  if (function_p != NULL)
  {
    printf ("Called function is available");
    return true;
  }

  printf ("Called function is NOT available");
  return true;
}
```

**See also**

- [jerry_backtrace_capture](#jerry_backtrace_capture)


## jerry_frame_this

**Summary**

Initialize and return with the 'this' binding private field of a backtrace frame.
The 'this' binding is a hidden value passed to the called function. As for arrow
functions, the 'this' binding is assigned at function creation. This getter
function can only be called from the callback function of
[jerry_backtrace_capture](#jerry_backtrace_capture), and the value becomes invalid
after the callback returns.

*Note*:
- The returned data must not be modified, and does not need to be freed.
  Any cleanup is done automatically after the callback is returned.

**Prototype**

```c
const jerry_value_t *
jerry_frame_this (jerry_frame_t *frame_p);
```

- `frame_p` - a frame passed to the [jerry_backtrace_cb_t](#jerry_backtrace_cb_t) callback
- return value
  - pointer to the 'this' binding if the binding is available,
  - NULL otherwise

*Renamed in version 3.0, it was previously known as `jerry_backtrace_get_this` in earlier versions.*

**Example**

See the example of [jerry_backtrace_capture](#jerry_backtrace_capture)
with the following callback function:

```c
static bool
backtrace_callback (jerry_frame_t *frame_p,
                    void *user_p)
{
  jerry_value_t *this_p = jerry_frame_this (frame_p);

  if (this_p != NULL)
  {
    printf ("The 'this' binding is available");
    return true;
  }

  printf ("The 'this' binding is NOT available");
  return true;
}
```

**See also**

- [jerry_backtrace_capture](#jerry_backtrace_capture)


## jerry_frame_is_strict

**Summary**

Returns true, if the code bound to the backtrace frame is strict mode
code. This function can only be called from the callback function of
[jerry_backtrace_capture](#jerry_backtrace_capture), and the value
becomes invalid after the callback returns.

**Prototype**

```c
bool
jerry_frame_is_strict (jerry_frame_t *frame_p);
```

- `frame_p` - a frame passed to the [jerry_backtrace_cb_t](#jerry_backtrace_cb_t) callback
- return value
  - true, if strict mode code is bound to the frame
  - false, otherwise

*Renamed in version 3.0, it was previously known as `jerry_backtrace_is_strict` in earlier versions.*

**Example**

See the example of [jerry_backtrace_capture](#jerry_backtrace_capture)
with the following callback function:

```c
static bool
backtrace_callback (jerry_frame_t *frame_p,
                    void *user_p)
{
  if (jerry_frame_is_strict (frame_p))
  {
    printf ("Strict mode code is running");
    return true;
  }

  printf ("Non-strict mode code is running");
  return true;
}
```

**See also**

- [jerry_backtrace_capture](#jerry_backtrace_capture)


# Miscellaneous functions

## jerry_log

**Summary**

 Log a zero-terminated formatted message with the specified log level.

**Prototype**

```c
void jerry_log (jerry_log_level_t level, const char *format_p, ...);
```

- `level` - message log level, see [jerry_log_level_t](#jerry_log_level_t)
- `format_p` - format string
  Supported format specifiers:
    `%s`: zero-terminated string
    `%c`: character
    `%u`: unsigned integer
    `%d`: decimal integer
    `%x`: unsigned hexadecimal
  Width and padding sub-modifiers are also supported.
- rest - variables to substitue into the format string

*Introduced in version 3.0*.

**Example**

```c
jerry_log (JERRY_LOG_LEVEL_DEBUG,
           "My debug message with a number param: %d.\n", 42);
```

**See also**

- [jerry_log_level_t](#jerry_log_level_t)
- [jerry_log_set_level](#jerry_log_set_level)


## jerry_log_set_level

**Summary**

Set the log level of the engine.
Log messages with lower significance than the current log level will be ignored by `jerry_log`.

**Prototype**

```c
void jerry_log_set_level (jerry_log_level_t level);
```

- `level` - log level to be set, see [jerry_log_level_t](#jerry_log_level_t)

*Introduced in version 3.0*.

**Example**

```c
jerry_log_set_level (JERRY_LOG_LEVEL_WARNING);
```

**See also**

- [jerry_log_level_t](#jerry_log_level_t)
- [jerry_log](#jerry_log)


## jerry_halt_handler

**Summary**

The callback passed to this function is periodically called when
JerryScript executes an ECMAScript program.

If the callback returns with undefined value the ECMAScript execution continues.
Otherwise the result is thrown by the engine. The callback function might be
called again even if it threw an exception. In this case the function must throw the
same exception again.

To reduce the CPU overhead of constantly checking the termination
condition the callback is called when a backward jump is executed
or an exception is caught. Setting the `interval` to a greater
than `1` value reduces this overhead further. If its value is N
only every Nth event (backward jump, etc.) trigger the next check.

*Note*:
- This API depends on a build option (`JERRY_VM_HALT`) and can be checked
  in runtime with the `JERRY_FEATURE_VM_EXEC_STOP` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
void
jerry_halt_handler (uint32_t interval,
                    jerry_halt_cb_t callback
                    void *user_p);
```

- `interval` - interval of calling the `callback` function
- `callback` - periodically called callback (passing NULL disables this feature)
- `user_p` - user pointer passed to the `callback` function

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_set_vm_exec_stop_callback` in earlier versions.*

**Example**

[doctest]: # (test="link")

```c
#include "jerryscript.h"

static jerry_value_t
vm_exec_stop_callback (void *user_p)
{
  int *countdown_p = (int *) user_p;

  while (*countdown_p > 0)
  {
    (*countdown_p)--;
    return jerry_undefined ();
  }

  // The value will be automatically wrapped into an exception.
  return jerry_string_sz ("Abort script");
}

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  int countdown = 10;
  jerry_halt_handler (16, vm_exec_stop_callback, &countdown);

  // Infinite loop.
  const jerry_char_t script[] = "while(true) {}";

  jerry_value_t parsed_code = jerry_parse (script, sizeof (script) - 1, NULL);
  jerry_value_free (jerry_run (parsed_code));
  jerry_value_free (parsed_code);
  jerry_cleanup ();
}
```

**See also**

- [jerry_halt_cb_t](#jerry_halt_cb_t)

## jerry_source_name

**Summary**

Get the source name (usually a file name) of the currently executed script or the given function object.

This function is typically called from native callbacks.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.
- This feature depends on build option (`JERRY_LINE_INFO`) and can be checked
  in runtime with the `JERRY_FEATURE_LINE_INFO` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
jerry_value_t
jerry_source_name (jerry_value_t value);
```
- `value` - api value to obtain the source name from
- return string value constructed from
  - the currently executed function object's source name, if the given value is undefined
  - source name of the function object, if the given value is a function object
  - "<anonymous>", otherwise

*Introduced in version 2.2*.

*Renamed in version 3.0, it was previously known as `jerry_get_resource_name` in earlier versions.*

**Example**

[doctest]: # (name="02.API-REFERENCE-jssourcename.c")

```c
#include <stdio.h>
#include <string.h>
#include "jerryscript.h"

static jerry_value_t
source_name_handler (const jerry_call_info_t *call_info_p,
                     const jerry_value_t args_p[],
                     const jerry_length_t args_count)
{
  jerry_value_t undefined_value = jerry_undefined ();
  jerry_value_t source_name = jerry_source_name (args_count > 0 ? args_p[0] : undefined_value);
  jerry_value_free (undefined_value);

  return source_name;
} /* source_name_handler */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t global = jerry_current_realm ();

  /* Register the sourceName" method. */
  {
    jerry_value_t func = jerry_function_external (source_name_handler);
    jerry_value_t name = jerry_string_sz ("sourceName");
    jerry_value_t result = jerry_object_set (global, name, func);
    jerry_value_free (result);
    jerry_value_free (name);
    jerry_value_free (func);
  }

  jerry_value_free (global);

  const jerry_char_t source[] = "function myFunction() { return sourceName() }; myFunction()";

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_HAS_SOURCE_NAME;
  parse_options.source_name = jerry_string_sz ("demo.js");

  jerry_value_t program = jerry_parse (source,
                                       sizeof (source) - 1,
                                       &parse_options);
  jerry_value_free (parse_options.source_name);

  if (!jerry_value_is_exception (program))
  {
    /* `run_result` contains "demo.js" */
    jerry_value_t run_result = jerry_run (program);

    /* usage of `run_result` */

    jerry_value_free (run_result);
  }

  jerry_value_free (program);
  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_function_external](#jerry_function_external)

## jerry_source_user_value

**Summary**

Returns the user value assigned to a script / module / function. This value is
set by the parser when the JERRY_PARSE_HAS_USER_VALUE flag is set in the `options`
member of the [jerry_parse_options_t](#jerry_parse_options_t) structure.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_source_user_value (const jerry_value_t value);
```
- `value` - script / module / function value which executes ECMAScript
            code (native modules / functions do not have user value).
- return
  - user value - if available,
  - undefined - otherwise

*Renamed in version 3.0, it was previously known as `jerry_get_user_value` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "function abc() {} abc";

  jerry_value_t user_value = jerry_object ();

  jerry_parse_options_t parse_options;
  parse_options.options = JERRY_PARSE_HAS_USER_VALUE;
  parse_options.user_value = user_value;

  jerry_value_t parsed_code = jerry_parse (script, sizeof (script) - 1, &parse_options);
  jerry_value_free (user_value);

  /* The jerry_source_user_value returns the object which
   * was created by jerry_object before. */

  user_value = jerry_source_user_value (parsed_code);
  jerry_value_free (parsed_code);

  jerry_value_free (user_value);
  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_parse](#jerry_parse)
- [jerry_generate_snapshot](#jerry_generate_snapshot)
- [jerry_exec_snapshot](#jerry_exec_snapshot)

## jerry_function_is_dynamic

**Summary**

Checks whether an ECMAScript code is compiled by eval like (eval, new Function,
[jerry_eval](#jerry_eval), etc.) command.

**Prototype**

```c
bool jerry_function_is_dynamic (const jerry_value_t value);
```
- `value` - script / module / function value which executes ECMAScript code
- return
  - true - if code is compiled by eval like command
  - false - otherwise

*Renamed in version 3.0, it was previously known as `jerry_is_eval_code` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "eval('(function (a) { return a; })')";

  jerry_value_t script_value = jerry_parse (script, sizeof (script) - 1, NULL);
  jerry_value_t function_value = jerry_run (script_value);
  jerry_value_free (script_value);

  if (jerry_function_is_dynamic (function_value))
  {
    /* Code enters here. */
  }

  jerry_value_free (function_value);
  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_parse](#jerry_parse)
- [jerry_generate_snapshot](#jerry_generate_snapshot)
- [jerry_exec_snapshot](#jerry_exec_snapshot)

## jerry_source_info

**Summary**

Returns a newly created source info structure corresponding to the passed script/module/function.
The function is lower level than `toString()` operation, but provides more contextual information.

*Note*:
- Returned value must be freed with [jerry_source_info_free](#jerry_source_info_free) when it
  is no longer needed.
- This API depends on a build option (`JERRY_FUNCTION_TO_STRING`) and can be checked
  in runtime with the `JERRY_FEATURE_FUNCTION_TO_STRING` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
jerry_source_info_t *jerry_source_info (const jerry_value_t value);
```
- `value` - script / module / function value which executes JavaScript
            code (native modules / functions do not have source info).
- return
  - source info - a newly created source info, if at least one field is available,
  - NULL - otherwise

*Renamed in version 3.0, it was previously known as `jerry_get_source_info` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t script[] = "function abc() {} abc";

  jerry_value_t parsed_code = jerry_parse (script, sizeof (script) - 1, NULL);

  jerry_source_info_t *source_info_p = jerry_source_info (parsed_code);
  jerry_value_free (parsed_code);

  if (source_info_p != NULL)
  {
    /* Check the information provided by jerry_source_info. */
  }

  jerry_source_info_free (source_info_p);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_source_info_free](#jerry_source_info_free)
- [jerry_source_info_t](#jerry_source_info_t)

## jerry_source_info_free

**Summary**

Frees the the source info structure returned by [jerry_source_info](#jerry_source_info).

*Note*:
- This API depends on a build option (`JERRY_FUNCTION_TO_STRING`) and can be checked
  in runtime with the `JERRY_FEATURE_FUNCTION_TO_STRING` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
void jerry_source_info_free (jerry_source_info_t *source_info_p)
```
- `source_info_p` - source info structure returned by [jerry_source_info](#jerry_source_info)

*Renamed in version 3.0, it was previously known as `jerry_free_source_info` in earlier versions.*

**Example**

See [jerry_source_info](#jerry_source_info)

**See also**

- [jerry_source_info](#jerry_source_info)
- [jerry_source_info_t](#jerry_source_info_t)


# Functions for realm objects

These APIs all depend on build option (`JERRY_BUILTIN_REALMS`).

## jerry_set_realm

**Summary**

Replaces the currently active realm (including the global object) with another realm.
The replacement should be temporary, and the original realm must be restored after
the tasks are completed. During the replacement, the realm must be referenced
by the application (i.e. the gc must not reclaim it). This is also true to
the returned previously active realm, so there is no need to free the value
after the restoration. The function can only fail if realms are not supported
or the passed argument is not a realm. In this case the returned exception must
be freed by [jerry_value_free](#jerry_value_free).

This function is useful to parse a script, create a native function, load a snapshot
or create an exception in another realm. Each ECMAScript code runs in the realm
which was active when the code was parsed or loaded regardless of the current realm.

*Note*:
- This feature depends on build option (`JERRY_BUILTIN_REALMS`) and can be checked
  in runtime with the `JERRY_FEATURE_REALM` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
jerry_value_t
jerry_set_realm (jerry_value_t realm_value);
```
- `realm_value` - the new realm value
- return
  - previous realm value - if the passed value is a realm
  - exception - otherwise

*Introduced in version 2.4*.

**Example**

```c
{
  jerry_value_t realm_value = jerry_realm ();

  jerry_value_t old_realm = jerry_set_realm (realm_value);

  ... // usage of the realm

  jerry_set_realm (old_realm);
}
```

**See also**

- [jerry_realm](#jerry_realm)

## jerry_realm_this

**Summary**

Gets the 'this' binding of a realm. The 'this' binding is always an object.
By default the 'this' binding is the same as the realm object and can be
changed by [jerry_realm_set_this](#jerry_realm_set_this).

*Note*:
- This feature depends on build option (`JERRY_BUILTIN_REALMS`) and can be checked
  in runtime with the `JERRY_FEATURE_REALM` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
jerry_value_t
jerry_realm_this (jerry_value_t realm_value)
```
- `realm_value` - realm value
- return
  - type error exception- if realm_value is not a realm
  - 'this' binding object - otherwise

*Introduced in version 2.4*.

*Renamed in version 3.0, it was previously known as `jerry_realm_get_this` in earlier versions.*

**Example**

```c
{
  jerry_value_t realm_value = jerry_realm ();

  jerry_value_t this_value = jerry_realm_this (realm_value);

  ... // usage of the this_value

  jerry_value_free (this_value);
  jerry_value_free (realm_value);
}
```

**See also**

- [jerry_realm](#jerry_realm)
- [jerry_realm_set_this](#jerry_realm_set_this)

## jerry_realm_set_this

**Summary**

Sets the 'this' binding of a realm. This function must be called before executing
any script on the realm. Otherwise the operation is undefined.

*Note*:
- This feature depends on build option (`JERRY_BUILTIN_REALMS`) and can be checked
  in runtime with the `JERRY_FEATURE_REALM` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
jerry_value_t
jerry_realm_set_this (jerry_value_t realm_value, jerry_value_t this_value)
```
- `realm_value` - realm value
- `this_value` - new this value
- return
  - type error exception- if realm_value is not a realm or this_value is not object
  - true - otherwise

*Introduced in version 2.4*.

**Example**

```c
{
  jerry_value_t realm_value = jerry_realm ();

  jerry_value_t old_realm = jerry_set_realm (realm_value);
  /* The prototype of the object comes from the new realm. */
  jerry_value_t this_value = jerry_object ();
  jerry_set_realm (old_realm);

  jerry_value_t result = jerry_realm_set_this (realm_value, this_value);

  ... // usage of the realm
}
```

**See also**

- [jerry_realm](#jerry_realm)
- [jerry_set_realm](#jerry_set_realm)
- [jerry_realm_this](#jerry_realm_this)

# ArrayBuffer and TypedArray functions

These APIs all depend on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
see: [jerry_feature_enabled](#jerry_feature_enabled).

## jerry_arraybuffer_size

**Summary**

Get the byte length property of the ArrayBuffer or SharedArrayBuffer. This is the
same value which was passed to the ArrayBuffer constructor call.

**Prototype**

```c
jerry_length_t
jerry_arraybuffer_size (const jerry_value_t value);
```

- `value` - ArrayBuffer object
- return value
  - size of the ArrayBuffer in bytes
  - 0 if the `value` parameter is not an ArrayBuffer

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_get_arraybuffer_byte_length` in earlier versions.*

**Example**

```c
{
  jerry_value_t buffer = jerry_arraybuffer (15);
  jerry_length_t length = jerry_arraybuffer_size (buffer);
  // length should be 15

  jerry_value_free (buffer);
}
```

**See also**
- [jerry_arraybuffer](#jerry_arraybuffer)


## jerry_arraybuffer_read

**Summary**

Copy the portion of the ArrayBuffer or SharedArrayBuffer into a user provided buffer.
The start offset of the read operation can be specified.

The number bytes to be read can be specified via the `buf_size`
parameter. It is not possible to read more than the length of
the ArrayBuffer.

Function returns the number of bytes read from the ArrayBuffer
(and written to the buffer parameter). This value is
calculated in the following way: `min(array buffer length - offset, buf_size)`.

**Prototype**

```c
jerry_length_t
jerry_arraybuffer_read (const jerry_value_t value,
                        jerry_length_t offset,
                        uint8_t *buf_p,
                        jerry_length_t buf_size);
```

- `value` - ArrayBuffer to read from
- `offset` - start offset of the read operation
- `buf_p` - buffer to read the data to
- `buf_size` - maximum number of bytes to read into the buffer
- return value
  - number of bytes written into the buffer (read from the ArrayBuffer)
  - 0 if the `value` is not an ArrayBuffer object
  - 0 if the `buf_size` is zero or there is nothing to read

*Introduced in version 2.0*.

**Example**

```c
{
  uint8_t data[20];
  jerry_value_t buffer;
  // ... create the ArrayBuffer or copy it from somewhere.

  jerry_value_t bytes_read;

  // read 10 bytes from the start of the ArrayBuffer.
  bytes_read = jerry_arraybuffer_read (buffer, 0, data, 10);
  // read the next 10 bytes
  bytes_read += jerry_arraybuffer_read (buffer, bytes_read, data + bytes_read, 10);

  // process the data variable

  jerry_value_free (buffer);
}
```

**See also**

- [jerry_arraybuffer](#jerry_arraybuffer)
- [jerry_arraybuffer_write](#jerry_arraybuffer_write)
- [jerry_arraybuffer_size](#jerry_arraybuffer_size)


## jerry_arraybuffer_write

**Summary**

Copy the contents of a buffer into the ArrayBuffer or SharedArrayBuffer.
The start offset of the write operation can be specified.

The number bytes to be written can be specified via the `buf_size`
parameter. It is not possible to write more than the length of
the ArrayBuffer.

Function returns the number of bytes written into the ArrayBuffer
(and read from the buffer parameter). This value is
calculated in the following way: `min(array buffer length - offset, buf_size)`.

**Prototype**

```c
jerry_length_t
jerry_arraybuffer_write (const jerry_value_t value,
                         jerry_length_t offset,
                         const uint8_t *buf_p,
                         jerry_length_t buf_size);
```

- `value` - ArrayBuffer to write to
- `offset` - start offset of the write operation
- `buf_p` - buffer to read the data from
- `buf_size` - maximum number of bytes to write into the ArrayBuffer
- return value
  - number of bytes written into the ArrayBuffer (read from the buffer parameter)
  - 0 if the `value` is not an ArrayBuffer object
  - 0 if the `buf_size` is zero or there is nothing to write

*Introduced in version 2.0*.

**Example**

```c
{
  uint8_t data[20];

  // fill the data with values
  for (int i = 0; i < 20; i++)
  {
    data[i] = (uint8_t) (i * 2);
  }

  jerry_value_t buffer;
  // ... create the ArrayBuffer or copy it from somewhere.

  jerry_value_t bytes_written;

  // write 10 bytes from to the start of the ArrayBuffer.
  bytes_written = jerry_arraybuffer_write (buffer, 0, data, 10);
  // read the next 10 bytes
  bytes_written += jerry_arraybuffer_write (buffer, bytes_written, data + bytes_written, 10);

  // use the ArrayBuffer

  jerry_value_free (buffer);
}
```

**See also**

- [jerry_arraybuffer](#jerry_arraybuffer)
- [jerry_arraybuffer_write](#jerry_arraybuffer_write)
- [jerry_arraybuffer_size](#jerry_arraybuffer_size)


## jerry_arraybuffer_data

**Summary**

The function allows access to the contents of the Array Buffer directly.

**WARNING!** This operation is for expert use only! The programmer must
ensure that the returned memory area is used correctly. That is
there is no out of bounds reads or writes. The lifetime of the underlying
data buffer is managed by the ArrayBuffer value. Make sure to copy the
value with [`jerry_value_copy`](#jerry_value_copy) if the data
buffer is needed later.

**Prototype**

```c
uint8_t *
jerry_arraybuffer_data (const jerry_value_t value);
```

- `value` - Array Buffer object.
- return value
  - pointer to the Array Buffer's data area.
  - NULL if the `value` is:
    - not an ArrayBuffer object
    - an external ArrayBuffer has been detached

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_get_arraybuffer_pointer` in earlier versions.*

**Example**

```c
{
  // create the ArrayBuffer
  jerry_value_t buffer = jerry_arraybuffer (16);

  uint8_t *const data = jerry_arraybuffer_data (buffer);

  for (int i = 0; i < 16; i++)
  {
    data[i] = (uint8_t) (i + 4);
  }

  // use the Array Buffer

  // release buffer as it is not needed after this point
  jerry_value_free (buffer);
}
```

**See also**

- [jerry_arraybuffer_external](#jerry_arraybuffer_external)

## jerry_arraybuffer_is_detachable

**Summary**

Get if the ArrayBuffer is detachable.

**Prototype**

```c
bool
jerry_arraybuffer_is_detachable (const jerry_value_t value);
```

- `value` - ArrayBuffer to be detached
- return
  - true if the arraybuffer is detachable
  - false otherwise

*Introduced in version 2.2*.

*Renamed in version 3.0, it was previously known as `jerry_is_arraybuffer_detachable` in earlier versions.*

**Example**

```c
{
  // create the ArrayBuffer
  jerry_value_t buffer = jerry_arraybuffer (16);

  bool is_detachable = jerry_arraybuffer_is_detachable (buffer);

  // release buffer as it is not needed after this point
  jerry_value_free (buffer);
}
```

**See also**

- [jerry_arraybuffer_detach](#jerry_arraybuffer_detach)

## jerry_arraybuffer_detach

**Summary**

Detach the underlying data block from ArrayBuffer and set its bytelength to 0.

This operation requires the ArrayBuffer to be an external buffer created by
`jerry_arraybuffer_external`.

*Note*:
- If the ArrayBuffer has been created with `jerry_arraybuffer_external` the optional free callback is
  called on a successful detach operation

**Prototype**

```c
jerry_value_t
jerry_arraybuffer_detach (const jerry_value_t value);
```

- `value` - ArrayBuffer to be detached
- return
  - null value if success
  - exception otherwise

*Introduced in version 2.2*.

*Renamed in version 3.0, it was previously known as `jerry_detach_arraybuffer` in earlier versions.*

**Example**

```c
{
  uint8_t buf[1];
  jerry_size_t length = 1;
  // create the ArrayBuffer
  jerry_value_t buffer = jerry_arraybuffer (length, buf, NULL);

  jerry_value_t res = jerry_arraybuffer_detach (buffer);

  // release buffer as it is not needed after this point
  jerry_value_free (res);
  jerry_value_free (buffer);
}
```

**See also**

- [jerry_arraybuffer_is_detachable](#jerry_arraybuffer_is_detachable)

## jerry_arraybuffer_has_buffer

**Summary**

Checks whether a buffer is currently allocated for an array buffer or typed array.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).

**Prototype**

```c
bool
jerry_arraybuffer_has_buffer (const jerry_value_t value);
```

- `value` - array buffer or typed array value.
- return
  - true, if a buffer is allocated for an array buffer or typed array
  - false, otherwise

*Introduced in version 3.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t array_buffer_value = jerry_arraybuffer (1024 * 1024);

  /* By default, the backing store of large array buffers
   * is allocated when it is used the first time. */

  if (!jerry_arraybuffer_has_buffer (array_buffer_value))
  {
    /* Code enters here in this case. */
  }

  jerry_value_free (array_buffer_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_arraybuffer_external](#jerry_arraybuffer_external)
- [jerry_shared_arraybuffer_external](#jerry_shared_arraybuffer_external)
- [jerry_arraybuffer_heap_allocation_limit](#jerry_arraybuffer_heap_allocation_limit)
- [jerry_arraybuffer_allocator](#jerry_arraybuffer_allocator)

## jerry_arraybuffer_heap_allocation_limit

**Summary**

Array buffers which size is less or equal than the limit passed to this
function are allocated in a single memory block. The allocator callbacks set by
[jerry_arraybuffer_set_allocation_callbacks](#jerry_arraybuffer_set_allocation_callbacks)
are not called for these array buffers.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).
- The default limit is 256 bytes.
- When an array buffer is allocated in a single memory block, its
  backing store is not freed when the array buffer is detached.
- This limit does not affect shared array buffers, their backing store is always
  allocated by the allocator callback.

**Prototype**

```c
void
jerry_arraybuffer_heap_allocation_limit (const jerry_length_t allocation_limit);
```

- `allocation_limit` - maximum size of compact allocation.

*Renamed in version 3.0, it was previously known as `jerry_arraybuffer_set_compact_allocation_limit` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_arraybuffer_heap_allocation_limit (1);

  jerry_value_t array_buffer_value = jerry_arraybuffer (1);

  if (jerry_arraybuffer_has_buffer (array_buffer_value))
  {
    /* Code enters here because the backing store
     * is allocated during buffer creation. */
  }

  jerry_value_free (array_buffer_value);

  array_buffer_value = jerry_arraybuffer (2);

  if (jerry_arraybuffer_has_buffer (array_buffer_value))
  {
    /* Code does not enter here because the backing store
     * is allocated when it is used the first time. */
  }

  jerry_value_free (array_buffer_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_arraybuffer_has_buffer](#jerry_arraybuffer_has_buffer)
- [jerry_arraybuffer_allocator](#jerry_arraybuffer_allocator)

## jerry_arraybuffer_allocator

**Summary**

Set callbacks for allocating and freeing backing stores for array buffer objects.

*Note*:
- This API depends on a build option (`JERRY_BUILTIN_TYPEDARRAY`) and can be checked
  in runtime with the `JERRY_FEATURE_TYPEDARRAY` feature enum value,
  see: [jerry_feature_enabled](#jerry_feature_enabled).
- This function is recommended to be called after [jerry_init](#jerry_init) before
  any array buffer is allocated.
- The callbacks can be NULL to use the default callbacks. The default `allocate_callback`
  allocates memory using [jerry_heap_alloc](#jerry_heap_alloc) and the default
  `free_callback` frees memory using [jerry_heap_free](#jerry_heap_free).


**Prototype**

```c
void
jerry_arraybuffer_allocator (jerry_arraybuffer_allocate_cb_t allocate_callback,
                             jerry_arraybuffer_free_cb_t free_callback,
                             void *user_p)
```

- `allocate_callback` - callback for allocating array buffer memory.
- `free_callback` - callback for freeing array buffer memory.
- `user_p` - user pointer passed to the callbacks.

*Renamed in version 3.0, it was previously known as `jerry_arraybuffer_set_allocator_callbacks` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

static uint8_t global_buffer[64];

static void
array_buffer_free_cb (jerry_arraybuffer_type_t buffer_type, /**< type of the array buffer object */
                      uint8_t *buffer_p, /**< pointer to the allocated buffer */
                      uint32_t buffer_size, /**< size of the allocated buffer */
                      void *arraybuffer_user_p, /**< user pointer assigned to the array buffer object */
                      void *user_p) /**< user pointer passed to jerry_arraybuffer_set_allocation_callbacks */
{
  (void) buffer_type;
  (void) user_p;

  /* As for this example, only the free callback is redirected. This callback
   * function does not free the memory if the arraybuffer_user_p is non-NULL. */

  if (arraybuffer_user_p == NULL)
  {
    jerry_heap_free (buffer_p, buffer_size);
  }
} /* array_buffer_free_cb */

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_arraybuffer_allocator (NULL, array_buffer_free_cb, NULL);

  /* The buffer of the array buffer object is allocated by the default
   * allocator using jerry_heap_alloc and freed by array_buffer_free_cb. */

  const jerry_char_t script[] = "var result = new uint32Array(1024); result[0] = 1; result";
  jerry_value_t array_buffer_value = jerry_eval (script, sizeof (script) - 1, JERRY_PARSE_NO_OPTS);
  jerry_value_free (array_buffer_value);

  /* The buffer of the array buffer object has a non-NULL
   * arraybuffer_user_p value, so it is not freed by array_buffer_free_cb. */

  array_buffer_value = jerry_arraybuffer_external (global_buffer, sizeof (global_buffer), global_buffer);
  jerry_value_free (array_buffer_value);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_arraybuffer_has_buffer](#jerry_arraybuffer_has_buffer)
- [jerry_arraybuffer_allocator](#jerry_arraybuffer_allocator)

## jerry_dataview_buffer

**Summary**

Get the ArrayBuffer object used by a DataView object.
Additionally returns the byteLength and byteOffset properties
of the DataView object.

For the returned ArrayBuffer the [jerry_value_free](#jerry_value_free)
must be called when it is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_dataview_buffer (const jerry_value_t value,
                       jerry_length_t *byteOffset,
                       jerry_length_t *byteLength);
```

- `value` - DataView to get the ArrayBuffer from
- `byteOffset` - (Optional) returns the start offset of the ArrayBuffer for the DataView
- `byteLength` - (Optional) returns the number of bytes used from the ArrayBuffer for the DataView
- return
  - DataView object's underlying ArrayBuffer object
  - TypeError if the `value` is not a DataView object

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_get_dataview_buffer` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t arraybuffer = jerry_arraybuffer (16);
  jerry_value_t dataview = jerry_dataview (arraybuffer, 0, 16);
  jerry_length_t byteOffset = 0;
  jerry_length_t byteLength = 0;
  jerry_value_t buffer = jerry_dataview_buffer (dataview, &byteOffset, &byteLength);

  // buffer is an ArrayBuffer object and ArrayBuffer operations can be performed on it
  // byteOffset is 0
  // byteLength is 16

  // usage of buffer

  jerry_value_free (buffer);
  jerry_value_free (dataview);
  jerry_value_free (arraybuffer);

  jerry_cleanup ();
}
```

**See also**

- [jerry_dataview](#jerry_dataview)


## jerry_typedarray_type

**Summary**

Get the type of the TypedArray.

The returned type is one of the [jerry_typedarray_type_t](#jerry_typedarray_type_t)
enum value.

**Prototype**

```c
jerry_typedarray_type_t
jerry_typedarray_type (jerry_value_t value);
```

- `value` - TypedArray object to query for type.
- return
  - the type of the TypedArray
  - JERRY_TYPEDARRAY_INVALID if the object was not a TypedArray

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_get_typedarray_type` in earlier versions.*

**Example**

```c
{
  jerry_typedarray_type_t expected_type = JERRY_TYPEDARRAY_UINT32;
  jerry_value_t typedarray = jerry_typedarray (expected_class, 25);

  jerry_typedarray_type_t type = jerry_typedarray_type (typedarray);

  // 'type' is now JERRY_TYPEDARRAY_UINT32

  jerry_value_free (typedarray);
}
```

**See also**

- [jerry_typedarray](#jerry_typedarray)
- [jerry_typedarray_type_t](#jerry_typedarray_type_t)


## jerry_typedarray_length

**Summary**

Get the element count of the TypedArray as specified during creation.

This is not the same as the byteLength property of a TypedArray object.

**Prototype**

```
jerry_length_t
jerry_typedarray_length (jerry_value_t value);
```

- `value` - TypedArray object to query
- return
  - length (element count) of the TypedArray object
  - 0 if the object is not a TypedArray

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_get_typedarray_length` in earlier versions.*

**Example**

```c
{
  jerry_value_t array = jerry_typedarray (JERRY_TYPEDARRAY_INT32, 21);

  jerry_length_t element_count = jerry_typedarray_length (array);

  // element_count is now 21.

  jerry_value_free (array);
}
```

**See also**

- [jerry_typedarray](#jerry_typedarray)


## jerry_typedarray_buffer

**Summary**

Get the ArrayBuffer object used by a TypedArray object.
Additionally returns the byteLength and byteOffset properties
of the TypedArray object.

For the returned ArrayBuffer the [jerry_value_free](#jerry_value_free)
must be called.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_typedarray_buffer (jerry_value_t value,
                         jerry_length_t *byteOffset,
                         jerry_length_t *byteLength);
```

- `value` - TypedArray to get the ArrayBuffer from
- `byteOffset` - (Optional) returns the start offset of the ArrayBuffer for the TypedArray
- `byteLength` - (Optional) returns the number of bytes used from the ArrayBuffer for the TypedArray
- return
  - TypedArray object's underlying ArrayBuffer object
  - TypeError if the `value` is not a TypedArray object

*Introduced in version 2.0*.

*Renamed in version 3.0, it was previously known as `jerry_get_typedarray_buffer` in earlier versions.*

**Example**

```c
{
  jerry_value_t array = jerry_typedarray (JERRY_TYPEDARRAY_INT16, 11);

  jerry_length_t byteLength = 0;
  jerry_length_t byteOffset = 0;
  jerry_value_t buffer = jerry_typedarray_buffer (array, &byteOffset, &byteLength);

  // buffer is an ArrayBuffer object and ArrayBuffer operations can be performed on it
  // byteLength is 11 * 2 (2 as the TypedArray stores Int16 that is 2 byte elements)
  // byteOffset is 0

  jerry_value_free (buffer);
  jerry_value_free (array);
}
```

**See also**

- [jerry_typedarray](#jerry_typedarray)

# JSON functions

## jerry_json_parse

**Summary**

Parses a CESU-8 or UTF-8 encoded string as a JSON string, creating a JavaScript value. The behaviour is equivalent with
the "JSON.parse(string)" JS call.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_json_parse (const jerry_char_t *string_p,
                  jerry_size_t string_size);
```

- `string_p` - pointer to a JSON string.
- `string_size` - size of the string.
- return
  - `jerry_value_t` containing a JavaScript value.
  - exception value in case of any parse error.

*Introduced in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  /* Initialize engine */
  jerry_init (JERRY_INIT_EMPTY);

  const jerry_char_t data[] = "{\"name\": \"John\", \"age\": 5}";
  jerry_value_t obj = jerry_json_parse (data, sizeof (data) - 1);

  /* "obj" now contains and object created from the "data" JSON string. */

  jerry_value_free (obj);

  /* Cleanup engine */
  jerry_cleanup ();

  return 0;
}
```

## jerry_json_stringify

**Summary**

Create a JSON string value from a JavaScript value. The behaviour is equivalent with
the "JSON.stringify(input_value)" JS call.

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.

**Prototype**

```c
jerry_value_t
jerry_json_stringify (const jerry_value_t input_value);
```

- `input_value` - a `jerry_value_t` to stringify.
- return
  - `jerry_value_t` containing a JSON string.
  - exception value in case of any stringification error.

*Introduced in version 2.0*.

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  /* Initialize engine */
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t obj = jerry_object ();
  {
    jerry_value_t key = jerry_string_sz ("name");
    jerry_value_t value = jerry_string_sz ("John");
    jerry_value_free (jerry_object_set (obj, key, value));
    jerry_value_free (key);
    jerry_value_free (value);
  }

  jerry_value_t stringified = jerry_json_stringify (obj);

  /* "stringified" now contains a JSON string */

  jerry_value_free (stringified);
  jerry_value_free (obj);

  /* Cleanup engine */
  jerry_cleanup ();

  return 0;
}
```

# Container Functions

## jerry_container_to_array

**Summary**

Return a new array containing elements from a Container, or a Container Iterator. Sets the is_key_value_p to true.
if the container object contains key-value structure and false if not.

*Note*:
- The return value will be an empty array if the Map/Set or Iterator object was empty or finished.
- This API function depends on a build option (`JERRY_BUILTIN_CONTAINER`) and can be checked
  runtime with the `JERRY_FEATURE_MAP, JERRY_FEATURE_SET, JERRY_FEATURE_WEAKMAP, JERRY_FEATURE_WEAKSET`
  feature enum values.
  see: [jerry_feature_enabled](#jerry_feature_enabled).


*Introduced in version 3.0*.

**Prototype**

```c
jerry_value_t
jerry_container_to_array (jerry_value_t value,
                          bool *is_key_value_p);
```

- `value` - Map/Set or iterator object
- `is_key_value` - Will be set to `true` if the given container has key-value pairs, `false` otherwise.
- return value
  - jerry_value_t containing an array of values from the Map/Set or iterator object
  - exception if the `value` is nor a Container or a Container Iterator.
  - `undefined` if the `value` is undefined/null.
**Example**

[doctest]: # ()

```c
#include "jerryscript.h"
int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_char_t src[] = "var map = new Map(); map.set(1,2); map.entries()";
  jerry_value_t iterable = jerry_eval (src, sizeof (src) - 1, JERRY_PARSE_NO_OPTS);

  bool is_key_value_container = false;
  jerry_value_t buffer_from_map = jerry_container_to_array (iterable, &is_key_value_container);

  /*
    The buffer_from_map contains two elements: 1 and 2, which is the key/value pair of the only item in the set.
    is_key_value set to true, as the original is a key-value structure (a Map Iterator)
  */

  jerry_value_free (iterable);
  jerry_value_free (buffer_from_map);

  jerry_cleanup ();

  return 0;
}
```

**See also**

- [jerry_container](#jerry_container)
- [jerry_container_type_t](#jerry_container_type_t)


## jerry_container_op

**Summary**

Perform container operation on the given operands (add, delete, set, etc.).

*Note*:
- Returned value must be freed with [jerry_value_free](#jerry_value_free) when it
  is no longer needed.
- This API function depends on a build option (`JERRY_BUILTIN_CONTAINER`) and can be checked
  runtime with the `JERRY_FEATURE_MAP` , `JERRY_FEATURE_SET` , `JERRY_FEATURE_WEAKMAP` , `JERRY_FEATURE_WEAKSET`
  feature enum values.
  see: [jerry_feature_enabled](#jerry_feature_enabled).


**Prototype**

```c
jerry_value_t
jerry_container_op (jerry_container_op_t operation,
                    jerry_value_t container,
                    jerry_value_t *arguments,
                    uint32_t arguments_number)
```

 - `operation` - container operation
 - `container` - this value
 - `arguments` - array of arguments
 - `arguments_number` - number of arguments
 - result if the operation is successful
 - exception, otherwise

*Renamed in version 3.0, it was previously known as `jerry_container_operation` in earlier versions.*

**Example**

[doctest]: # ()

```c
#include "jerryscript.h"

int
main (void)
{
  jerry_init (JERRY_INIT_EMPTY);

  jerry_value_t map = jerry_container (JERRY_CONTAINER_TYPE_MAP, NULL, 0);
  jerry_value_t key_str = jerry_string_sz ("number");
  jerry_value_t number = jerry_number (10);
  jerry_value_t args[2] = {key_str, number};

  jerry_value_t result = jerry_container_op (JERRY_CONTAINER_OP_SET, map, args, 2);
  jerry_value_free (result);

  result = jerry_container_op (JERRY_CONTAINER_OP_SIZE, map, NULL, 0);
  jerry_value_free (result);

  result = jerry_container_op (JERRY_CONTAINER_OP_CLEAR, map, NULL, 0);
  jerry_value_free (result);

  jerry_value_free (map);
  jerry_value_free (key_str);
  jerry_value_free (number);

  jerry_cleanup ();
  return 0;
}
```

**See also**

- [jerry_container](#jerry_container)
- [jerry_container_type_t](#jerry_container_type_t)
