/*
 * This file is part of the source code of the software program
 * Vampire. It is protected by applicable
 * copyright laws.
 *
 * This source code is distributed under the licence found here
 * https://vprover.github.io/license.html
 * and in the source directory
 */
/**
 * @file Statistics.hpp
 * Defines proof-search statistics
 *
 * @since 02/01/2008 Manchester
 */

#ifndef __Statistics__
#define __Statistics__

#include <ostream>

#include "Forwards.hpp"
#include "Lib/Timer.hpp"
#include "Debug/Assertion.hpp"

extern const char *VERSION_STRING;

namespace Kernel {
class Unit;
}

namespace Shell {

using namespace Kernel;

/** termination reason */
enum class TerminationReason {
  /** refutation found */
  REFUTATION,
  /** satisfiability detected (saturated set built) */
  SATISFIABLE,
  /** saturation terminated but an incomplete strategy was used */
  REFUTATION_NOT_FOUND,
  /** inappropriate strategy **/
  INAPPROPRIATE,
  /** unknown termination reason */
  UNKNOWN,
  /** time limit reached */
  TIME_LIMIT,
  /** instruction limit reached */
  INSTRUCTION_LIMIT,
  /** memory limit reached */
  MEMORY_LIMIT,
  /** activation limit reached */
  ACTIVATION_LIMIT
};

enum class ExecutionPhase {
  /** Whatever happens before we start parsing the problem */
  INITIALIZATION,
  PARSING,
  /** Scanning for properties to be passed to preprocessing */
  PROPERTY_SCANNING,
  NORMALIZATION,
  SHUFFLING,
  SINE_SELECTION,
  INCLUDING_THEORY_AXIOMS,
  PREPROCESS_1,
  PREDIACTE_DEFINITION_MERGING,
  PREDICATE_DEFINITION_INLINING,
  UNUSED_PREDICATE_DEFINITION_REMOVAL,
  BLOCKED_CLAUSE_ELIMINATION,
  TWEE,
  ANSWER_LITERAL,
  PREPROCESS_2,
  NEW_CNF,
  NAMING,
  PREPROCESS_3,
  CLAUSIFICATION,
  FUNCTION_DEFINITION_ELIMINATION,
  INEQUALITY_SPLITTING,
  EQUALITY_RESOLUTION_WITH_DELETION,
  EQUALITY_PROXY,
  GENERAL_SPLITTING,
  SATURATION,
  /** The actual run of the conflict resolution algorithm */
  SOLVING,
  /** The actual run of the SAT solver*/
  SAT_SOLVING,
  PREPROCESSING,
  /** Whatever happens after the saturation algorithm finishes */
  FINALIZATION,
  FMB_PREPROCESSING,
  FMB_CONSTRAINT_GEN,
  FMB_SOLVING,
  UNKNOWN_PHASE
};

/**
 * Class Statistics
 * @since 02/01/2008 Manchester
 */
class Statistics {
public:

  void print(std::ostream& out);
  void explainRefutationNotFound(std::ostream& out);

  // Input
  /** number of input clauses */
  unsigned inputClauses = 0;
  /** number of input formulas */
  unsigned inputFormulas = 0;

  // Preprocessing
  /** number of formula names introduced during preprocessing */
  unsigned formulaNames = 0;
  /** number of skolem functions (also predicates in FOOL) introduced during skolemization */
  unsigned skolemFunctions = 0;
  /** number of initial clauses */
  unsigned initialClauses = 0;
  /** number of inequality splittings performed */
  unsigned splitInequalities = 0;
  /** number of pure predicates */
  unsigned purePredicates = 0;
  /** number of trivial predicates */
  unsigned trivialPredicates = 0;
  /** number of unused predicate definitions */
  unsigned unusedPredicateDefinitions = 0;
  /** number of eliminated function definitions */
  unsigned eliminatedFunctionDefinitions = 0;
  /** number of introduced function definitions */
  unsigned introducedFunctionDefinitions = 0;
  /** number of formulas selected by SInE selector */
  unsigned selectedBySine = 0;
  /** number of iterations before SInE reached fixpoint */
  unsigned sineIterations = 0;
  /** number of detected blocked clauses */
  unsigned blockedClauses = 0;

  // Generating inferences
  /** number of clauses generated by factoring*/
  unsigned factoring = 0;
  /** number of clauses generated by binary resolution*/
  unsigned resolution = 0;
  /** number of clauses generated by unit resulting resolution*/
  unsigned urResolution = 0;
  /** number of clauses generated by constrained resolution */
  unsigned cResolution = 0;
  /** number of clauses generated by forward superposition*/
  unsigned forwardSuperposition = 0;
  /** number of clauses generated by backward superposition*/
  unsigned backwardSuperposition = 0;
  /** number of clauses generated by self superposition*/
  unsigned selfSuperposition = 0;
  /** number of clauses generated by self superposition*/
  unsigned cSelfSuperposition = 0;
  /** number of clauses generated by constrained forward superposition*/
  unsigned cForwardSuperposition = 0;
  /** number of clauses generated by constrained backward superposition*/
  unsigned cBackwardSuperposition = 0;
  /** number of clauses generated by equality factoring*/
  unsigned equalityFactoring = 0;
  /** number of clauses generated by equality resolution*/
  unsigned equalityResolution = 0;
  /** number of clauses generated by forward extensionality resolution*/
  unsigned forwardExtensionalityResolution = 0;
  /** number of clauses generated by backward extensionality resolution*/
  unsigned backwardExtensionalityResolution = 0;
  /** number of theory inst simp **/
  unsigned theoryInstSimp = 0;
  /** number of theoryInstSimp candidates **/
  unsigned theoryInstSimpCandidates = 0;
  /** number of theoryInstSimp tautologies **/
  unsigned theoryInstSimpTautologies = 0;
  /** number of theoryInstSimp solutions lost as we could not represent them **/
  unsigned theoryInstSimpLostSolution = 0;
  /** number of theoryInstSimp application where an empty substitution was applied */
  unsigned theoryInstSimpEmptySubstitution = 0;
  /** number of induction applications **/
  unsigned maxInductionDepth = 0;
  unsigned structInduction = 0;
  unsigned structInductionInProof = 0;
  unsigned intInfInduction = 0;
  unsigned intInfInductionInProof = 0;
  unsigned intFinInduction = 0;
  unsigned intFinInductionInProof = 0;
  unsigned intDBInduction = 0;
  unsigned intDBInductionInProof = 0;
  unsigned intInfUpInduction = 0;
  unsigned intInfUpInductionInProof =0;
  unsigned intFinUpInduction = 0;
  unsigned intFinUpInductionInProof = 0;
  unsigned intDBUpInduction = 0;
  unsigned intDBUpInductionInProof = 0;
  unsigned intInfDownInduction = 0;
  unsigned intInfDownInductionInProof = 0;
  unsigned intFinDownInduction = 0;
  unsigned intFinDownInductionInProof = 0;
  unsigned intDBDownInduction = 0;
  unsigned intDBDownInductionInProof = 0;
  unsigned inductionApplication = 0;
  unsigned inductionApplicationInProof = 0;
  unsigned generalizedInductionApplication = 0;
  unsigned generalizedInductionApplicationInProof = 0;
  unsigned nonGroundInductionApplication = 0;
  unsigned nonGroundInductionApplicationInProof = 0;
  /** number of argument congruences */
  unsigned argumentCongruence = 0;
  unsigned narrow = 0;
  unsigned forwardSubVarSup = 0;
  unsigned backwardSubVarSup = 0;
  unsigned selfSubVarSup = 0;
  unsigned negativeExtensionality = 0;
  unsigned primitiveInstantiations = 0;
  unsigned choiceInstances = 0;
  unsigned proxyEliminations = 0;
  unsigned leibnizElims = 0;
  unsigned booleanSimps = 0;
  // Redundant inferences
  unsigned skippedSuperposition = 0;
  unsigned skippedResolution = 0;
  unsigned skippedEqualityResolution = 0;
  unsigned skippedEqualityFactoring = 0;
  unsigned skippedFactoring = 0;
  unsigned skippedInferencesDueToOrderingConstraints = 0;
  unsigned skippedInferencesDueToAvatarConstraints = 0;
  unsigned skippedInferencesDueToLiteralConstraints = 0;
  // Simplifying inferences
  /** number of duplicate literals deleted */
  unsigned duplicateLiterals = 0;
  /** number of literals s |= s deleted */
  unsigned trivialInequalities = 0;
  /** number of forward subsumption resolutions */
  unsigned forwardSubsumptionResolution = 0;
  /** number of backward subsumption resolutions */
  unsigned backwardSubsumptionResolution = 0;
  /** number of forward demodulations */
  unsigned forwardDemodulations = 0;
  /** number of forward demodulations into equational tautologies */
  unsigned forwardDemodulationsToEqTaut = 0;
  /** number of backward demodulations */
  unsigned backwardDemodulations = 0;
  /** number of backward demodulations into equational tautologies */
  unsigned backwardDemodulationsToEqTaut = 0;
  /** number of forward subsumption demodulations */
  unsigned forwardSubsumptionDemodulations = 0;
  /** number of forward subsumption demodulations into equational tautologies */
  unsigned forwardSubsumptionDemodulationsToEqTaut = 0;
  /** number of backward subsumption demodulations */
  unsigned backwardSubsumptionDemodulations = 0;
  /** number of backward subsumption demodulations into equational tautologies */
  unsigned backwardSubsumptionDemodulationsToEqTaut = 0;
  /** number of forward literal rewrites */
  unsigned forwardLiteralRewrites = 0;
  /** number of condensations */
  unsigned condensations = 0;
  /** number of global subsumptions */
  unsigned globalSubsumption = 0;
  /** number of interpreted simplifications */
  unsigned interpretedSimplifications = 0;

  /** how often did asg not simplify correctly. */
  unsigned asgViolations = 0;
  /** applications of asg */
  unsigned asgCnt = 0;

  /** how often did gve not simplify correctly. */
  unsigned gveViolations = 0;
  /** applications of gve */
  unsigned gveCnt = 0;

  /** number of evaluations that resulted in a incomparable literal */
  unsigned evaluationIncomp = 0;
  /** number of evaluations that resulted in a greater literal */
  unsigned evaluationGreater = 0;
  /** number of simplifications by PolynomialNormalizer */
  unsigned evaluationCnt = 0;

  /** number of machine arithmetic overflows within the inequality resolution calculus specific rules */
  unsigned alascaVarElimKNonZeroCnt = 0;
  unsigned alascaVarElimKSum = 0;
  unsigned alascaVarElimKMax = 0;

  /** number of (proper) inner rewrites */
  unsigned innerRewrites = 0;
  /** number of inner rewrites into equational tautologies */
  unsigned innerRewritesToEqTaut = 0;
  /** number of equational tautologies discovered by CC */
  unsigned deepEquationalTautologies = 0;

  // Deletion inferences
  /** number of tautologies A \/ ~A */
  unsigned simpleTautologies = 0;
  /** number of equational tautologies s=s */
  unsigned equationalTautologies = 0;
  /** number of forward subsumed clauses */
  unsigned forwardSubsumed = 0;
  /** number of backward subsumed clauses */
  unsigned backwardSubsumed = 0;
  /** number of forward ground joinable clauses */
  unsigned forwardGroundJoinable = 0;

  /** statistics of term algebra rules */
  unsigned taDistinctnessSimplifications = 0;
  unsigned taDistinctnessTautologyDeletions = 0;
  unsigned taInjectivitySimplifications = 0;
  unsigned taNegativeInjectivitySimplifications = 0;
  unsigned taAcyclicityGeneratedDisequalities = 0;

  // Saturation
  unsigned activations = 0; // NOTE: This is not a mere stat, it is also used for LRS estimation!

  /** all clauses ever occurring in the unprocessed queue */
  unsigned generatedClauses = 0;
  /** all passive clauses */
  unsigned passiveClauses = 0;
  /** all active clauses */
  unsigned activeClauses = 0;
  /** all extensionality clauses */
  unsigned extensionalityClauses = 0;

  unsigned discardedNonRedundantClauses = 0;

  unsigned inferencesBlockedForOrderingAftercheck = 0;

  bool smtReturnedUnknown = false;
  bool smtDidNotEvaluate = false;

  unsigned inferencesSkippedDueToColors = 0;

  /** passive clauses at the end of the saturation algorithm run */
  unsigned finalPassiveClauses = 0;
  /** active clauses at the end of the saturation algorithm run */
  unsigned finalActiveClauses = 0;
  /** extensionality clauses at the end of the saturation algorithm run */
  unsigned finalExtensionalityClauses = 0;
  unsigned splitClauses = 0;
  unsigned splitComponents = 0;

  /** Number of clauses generated for the SAT solver */
  unsigned satClauses = 0;
  /** Number of unit clauses generated for the SAT solver */
  unsigned unitSatClauses = 0;
  /** Number of binary clauses generated for the SAT solver */
  unsigned binarySatClauses = 0;

  unsigned satSplits = 0;
  unsigned satSplitRefutations = 0;

  unsigned smtFallbacks = 0;

  /** Number of pure variables eliminated by SAT solver */
  unsigned satPureVarsEliminated = 0;


  friend std::ostream& operator<<(std::ostream& out, TerminationReason const& self)
  {
    switch (self) {
      case TerminationReason::REFUTATION:
        return out << "REFUTATION";
      case TerminationReason::SATISFIABLE:
        return out << "SATISFIABLE";
      case TerminationReason::REFUTATION_NOT_FOUND:
        return out << "REFUTATION_NOT_FOUND";
      case TerminationReason::INAPPROPRIATE:
        return out << "INAPPROPRIATE";
      case TerminationReason::UNKNOWN:
        return out << "UNKNOWN";
      case TerminationReason::TIME_LIMIT:
        return out << "TIME_LIMIT";
      case TerminationReason::INSTRUCTION_LIMIT:
        return out << "INSTRUCTION_LIMIT";
      case TerminationReason::MEMORY_LIMIT:
        return out << "MEMORY_LIMIT";
      case TerminationReason::ACTIVATION_LIMIT:
        return out << "ACTIVATION_LIMIT";
    }
    ASSERTION_VIOLATION
  }
  /** termination reason */
  TerminationReason terminationReason = TerminationReason::UNKNOWN;
  /** refutation, if any */
  Kernel::Unit *refutation = nullptr;
  /** the saturated set of clauses, if any */
  Kernel::UnitList *saturatedSet = nullptr;
  /** if problem is satisfiable and we obtained a model, contains its
   * representation; otherwise it is an empty string */
  std::string model;

  ExecutionPhase phase = ExecutionPhase::INITIALIZATION;

private:
  static const char* phaseToString(ExecutionPhase p);
}; // class Statistics

} // namespace Shell

#endif
