"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmalgamatorSession = exports.ContinuedEvent = exports.StoppedEvent = void 0;
/*********************************************************************
 * Copyright (c) 2021 Kichwa Coders Canada Inc. and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
// import * as cp from 'child_process';
// import * as os from 'os';
// import * as path from 'path';
const debugadapter_1 = require("@vscode/debugadapter");
const AmalgamatorClient_1 = require("./AmalgamatorClient");
class StoppedEvent extends debugadapter_1.Event {
    constructor(reason, threadId, preserveFocusHint, allThreadsStopped) {
        super('stopped');
        this.body = {
            reason,
            threadId,
            allThreadsStopped,
            preserveFocusHint,
        };
    }
}
exports.StoppedEvent = StoppedEvent;
class ContinuedEvent extends debugadapter_1.Event {
    constructor(threadId, allThreadsContinued) {
        super('continued');
        this.body = { threadId, allThreadsContinued };
    }
}
exports.ContinuedEvent = ContinuedEvent;
class AmalgamatorSession extends debugadapter_1.LoggingDebugSession {
    constructor() {
        super();
        /* child processes XXX: A type that represents the union of the following datastructures? */
        this.childDaps = [];
        /**
         * This is a map of the start/end addresses or the instructionPointerReference that client sees -> child DAP index, child DAP addresses
         *
         * It is needed to workaround for problems:
         *  1. VSCode assuming that the instructionPointerReference has the same format as DisassembledInstruction.address
         *  even though the spec doesn't say so.
         *  See the spec at https://microsoft.github.io/debug-adapter-protocol/specification#Types_StackFrame
         *  The problem has been reported at https://github.com/microsoft/vscode/issues/164875
         *  2. VSCode/debug adapter protocol does not support multiple memory spaces.
         *  The problem has been reported at https://github.com/microsoft/vscode/issues/164877
         * Solution:
         *  Based on elements: start addresses or end addresses or the instructionPointerReference to determine
         *  the child dap to be handled.
         * Note:
         *  1. This should be updated after problems are resolved
         *  2. Limit of the solution is this can work incorrectly when child daps have same start addresses
         *  or end addresses or the instructionPointerReference.
         */
        this.addressMap = new Map();
        this.childDapNames = [];
        this.breakpointHandles = new debugadapter_1.Handles();
        this.frameHandles = new debugadapter_1.Handles();
        this.variableHandles = new debugadapter_1.Handles();
        this.logger = debugadapter_1.logger;
    }
    initializeRequest(response, args) {
        this.initializeRequestArgs = args;
        /**
         * TODO: When combinging a bunch of cdt-gdb-adapters this is fine as these are the same settings
         * as in GDBDebugSession.initializeRequest -- but when we combine unrelated debuggers then we
         * need to know what the individual child adapters supports are - but at this point in the
         * launch sequence we can't know that as that information is in LaunchRequestArguments. This
         * can be partially solved (as hints to frontend) with CapabilitiesEvent.
         */
        response.body = response.body || {};
        response.body.supportsConfigurationDoneRequest = true;
        response.body.supportsSetVariable = true;
        response.body.supportsConditionalBreakpoints = true;
        response.body.supportsHitConditionalBreakpoints = true;
        response.body.supportsLogPoints = true;
        response.body.supportsFunctionBreakpoints = true;
        //  response.body.supportsSetExpression = true;
        response.body.supportsDisassembleRequest = true;
        response.body.supportsReadMemoryRequest = true;
        response.body.supportsWriteMemoryRequest = true;
        this.sendResponse(response);
    }
    launchRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                debugadapter_1.logger.setup(args.verbose ? debugadapter_1.Logger.LogLevel.Verbose : debugadapter_1.Logger.LogLevel.Warn, args.logFile || false);
                for (const child of args.children) {
                    if (child.delay) {
                        debugadapter_1.logger.verbose(`waiting ${child.delay}ms before starting ${child.debugAdapterRuntime} ${child.debugAdapterExecutable}`);
                        yield new Promise((res) => setTimeout(res, child.delay));
                    }
                    const dc = yield this.createChild(child, this.childDaps.length);
                    this.childDaps.push(dc);
                    this.childDapNames.push(child.name ? child.name : '');
                }
                this.sendEvent(new debugadapter_1.InitializedEvent());
                this.sendResponse(response);
            }
            catch (err) {
                // TODO cleanup already done launches
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    startAmalgamatorClient(child, index) {
        return __awaiter(this, void 0, void 0, function* () {
            debugadapter_1.logger.verbose(`creating debug adapter ${child.debugAdapterRuntime} ${child.debugAdapterExecutable}`);
            const dc = new AmalgamatorClient_1.AmalgamatorClient(child.debugAdapterRuntime, child.debugAdapterExecutable, 'unused');
            // TODO get startup sequence right here (i.e. wait for child's InitializedEvent)
            dc.on('output', (event) => {
                const e = event;
                if (e.body.category == 'stdout') {
                    let output = e.body.output.trimEnd();
                    if (output.startsWith('To client: ')) {
                        output =
                            'To amalgamator: ' +
                                output.substr('To client: '.length);
                        debugadapter_1.logger.verbose(output);
                    }
                    else if (output.startsWith('From client: ')) {
                        output =
                            'From amalgamator: ' +
                                output.substr('From client: '.length);
                        debugadapter_1.logger.verbose(output);
                    }
                    this.sendEvent(e);
                }
                else {
                    this.sendEvent(e);
                }
            }).on('stopped', (event) => __awaiter(this, void 0, void 0, function* () {
                const e = event;
                const reason = e.body.reason;
                const intiatingThreadId = e.body.threadId;
                const threadMap = yield this.getThreadMap();
                let stoppedDapIndex = -1;
                // First send the event for the stopped thread
                threadMap.forEach(([childDapIndex, childId], clientId) => {
                    if (childDapIndex === index && childId == intiatingThreadId) {
                        this.sendEvent(new StoppedEvent(reason, clientId, false, false));
                        stoppedDapIndex = childDapIndex;
                    }
                });
                // then send the event for all the other stopped threads in the same child
                if (e.body.allThreadsStopped) {
                    threadMap.forEach(([childDapIndex, childId], clientId) => {
                        if (childDapIndex === stoppedDapIndex &&
                            childId != intiatingThreadId) {
                            this.sendEvent(new StoppedEvent(reason, clientId, true, false));
                        }
                    });
                }
            }));
            yield dc.start();
            yield dc.initializeRequest(this.initializeRequestArgs);
            return dc;
        });
    }
    createChild(child, index) {
        return __awaiter(this, void 0, void 0, function* () {
            const dc = yield this.startAmalgamatorClient(child, index);
            if (child.request === 'attach') {
                yield dc.attachRequest(child.arguments);
            }
            else {
                yield dc.launchRequest(child.arguments);
            }
            return dc;
        });
    }
    setBreakPointsRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            const responses = yield Promise.all(this.childDaps.map((dc) => dc.setBreakpointsRequest(args)));
            response.body = responses[0].body;
            //breakpointHandles
            // XXX: assert that respones[?].body.breakpoints.length == args.breakpoints.length
            // XXX: assert that the non-deprecated args.breakpoints is used (and not args.lines)
            // TODO: Handle the case where a breakpoint is resolved to different things on different childDaps
            //       see https://github.com/microsoft/debug-adapter-protocol/issues/13 and https://github.com/eclipse-cdt/cdt-gdb-adapter/issues/64
            if (args.breakpoints) {
                for (let i = 0; i < args.breakpoints.length; i++) {
                    responses.forEach((response, i) => {
                        response.body.breakpoints.forEach((bp) => {
                            if (bp.id) {
                                bp.id = this.breakpointHandles.create([
                                    this.childDaps[i],
                                    bp.id,
                                ]);
                            }
                        });
                    });
                    // choose the first verified response
                    const selected = responses.find((r) => r.body.breakpoints[i].verified);
                    if (selected) {
                        response.body.breakpoints[i] = selected.body.breakpoints[i];
                    }
                }
            }
            this.sendResponse(response);
        });
    }
    configurationDoneRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            yield Promise.all(this.childDaps.map((dc) => __awaiter(this, void 0, void 0, function* () { return yield dc.configurationDoneRequest(args); })));
            // TODO handle case that child returns an error (for every request!)
            this.sendResponse(response);
        });
    }
    /**
     * This is a map of the thread ID that client sees -> child DAP index, child DAP thread id
     */
    getThreadInfo(clientThreadId) {
        return __awaiter(this, void 0, void 0, function* () {
            const threadMap = yield this.getThreadMap();
            const threadInfo = threadMap.get(clientThreadId);
            if (!threadInfo) {
                const msg = `Missing information on client threadId ${clientThreadId} from threadMap: ${threadMap}`;
                debugadapter_1.logger.error(msg);
                // TODO how best to handle this error
                return [0, 0];
            }
            else {
                return threadInfo;
            }
        });
    }
    getThreadMap() {
        return __awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, _reject) => {
                this.getThreadMapInternal().then(([threadMap]) => resolve(threadMap));
            });
        });
    }
    getThreadMapInternal() {
        if (this.threadMapInProcess === undefined) {
            return this.collectChildTheads();
        }
        return this.threadMapInProcess;
    }
    collectChildTheads() {
        this.threadMapInProcess = new Promise((resolve, _reject) => {
            const threads = [];
            Promise.all(this.childDaps.map((dc) => dc.threadsRequest())).then((responses) => {
                const threadMap = new Map();
                let clientId = 1000;
                responses.forEach((r, i) => {
                    r.body.threads.forEach((t) => {
                        threads.push({
                            id: clientId,
                            name: `${this.childDapNames[i]
                                ? this.childDapNames[i] + ': '
                                : ''} ${t.name}`,
                            running: 'running' in t
                                ? t.running
                                : undefined,
                        });
                        threadMap.set(clientId, [i, t.id]);
                        clientId++;
                    });
                });
                resolve([threadMap, threads]);
            });
        });
        return this.threadMapInProcess;
    }
    threadsRequest(response) {
        return __awaiter(this, void 0, void 0, function* () {
            const [, threads] = yield this.collectChildTheads();
            response.body = {
                threads,
            };
            this.sendResponse(response);
        });
    }
    stackTraceRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const [childIndex, childId] = yield this.getThreadInfo(args.threadId);
                args.threadId = childId;
                const childDap = this.childDaps[childIndex];
                const childResponse = yield childDap.stackTraceRequest(args);
                const frames = childResponse.body.stackFrames;
                // XXX: When does frameHandles get reset as we don't have a "stopped all"
                frames.forEach((frame) => {
                    frame.id = this.frameHandles.create([childDap, frame.id]);
                    if (frame.instructionPointerReference) {
                        this.addressMap.set(frame.instructionPointerReference, childIndex);
                    }
                });
                response.body = childResponse.body;
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    scopesRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            const [childDap, childFrameId] = this.frameHandles.get(args.frameId);
            const scopes = yield childDap.scopesRequest({ frameId: childFrameId });
            scopes.body.scopes.forEach((scope) => (scope.variablesReference = this.variableHandles.create([
                childDap,
                scope.variablesReference,
            ])));
            response.body = scopes.body;
            this.sendResponse(response);
        });
    }
    variablesRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            const [childDap, childVariablesReference] = this.variableHandles.get(args.variablesReference);
            args.variablesReference = childVariablesReference;
            const variables = yield childDap.variablesRequest(args);
            variables.body.variables.forEach((variable) => {
                if (variable.variablesReference) {
                    variable.variablesReference = this.variableHandles.create([
                        childDap,
                        variable.variablesReference,
                    ]);
                }
            });
            response.body = variables.body;
            this.sendResponse(response);
        });
    }
    setVariableRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            const [childDap, childVariablesReference] = this.variableHandles.get(args.variablesReference);
            args.variablesReference = childVariablesReference;
            const variables = yield childDap.setVariableRequest(args);
            response.body = variables.body;
            this.sendResponse(response);
        });
    }
    evaluateRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            if (args.frameId) {
                try {
                    const [childDap, childFrameId] = this.frameHandles.get(args.frameId);
                    args.frameId = childFrameId;
                    const evaluate = yield childDap.evaluateRequest(args);
                    response.body = evaluate.body;
                    this.sendResponse(response);
                }
                catch (err) {
                    this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
                }
            }
            else {
                this.sendErrorResponse(response, 1, 'Cannot get evaluate expression');
            }
        });
    }
    disassembleRequest(response, args) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            if (args.memoryReference) {
                response.body = {
                    instructions: [],
                };
                try {
                    this.childDapIndex = this.addressMap.has(args.memoryReference)
                        ? this.addressMap.get(args.memoryReference)
                        : this.childDapIndex;
                    if (this.childDapIndex !== undefined) {
                        const disassemble = yield this.childDaps[this.childDapIndex].disassembleRequest(args);
                        response.body = disassemble.body;
                        const instructions = (_a = disassemble.body) === null || _a === void 0 ? void 0 : _a.instructions;
                        if (instructions !== undefined) {
                            this.addressMap.set(instructions[0].address, this.childDapIndex);
                            this.addressMap.set(instructions[instructions.length - 1].address, this.childDapIndex);
                            this.sendResponse(response);
                        }
                        else {
                            this.sendErrorResponse(response, 1, 'Cannot get disassembled data');
                        }
                    }
                    else {
                        this.sendErrorResponse(response, 1, 'Cannot determine the index of the child Dap');
                    }
                }
                catch (err) {
                    this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
                }
            }
            else {
                this.sendErrorResponse(response, 1, 'Cannot get disassembled data');
            }
        });
    }
    nextRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            const [childIndex, childId] = yield this.getThreadInfo(args.threadId);
            args.threadId = childId;
            const childDap = this.childDaps[childIndex];
            const childResponse = yield childDap.nextRequest(args);
            response.body = childResponse.body;
            this.sendResponse(response);
        });
    }
    stepInRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            const [childIndex, childId] = yield this.getThreadInfo(args.threadId);
            args.threadId = childId;
            const childDap = this.childDaps[childIndex];
            const childResponse = yield childDap.stepInRequest(args);
            response.body = childResponse.body;
            this.sendResponse(response);
        });
    }
    stepOutRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            const [childIndex, childId] = yield this.getThreadInfo(args.threadId);
            args.threadId = childId;
            const childDap = this.childDaps[childIndex];
            const childResponse = yield childDap.stepOutRequest(args);
            response.body = childResponse.body;
            this.sendResponse(response);
        });
    }
    continueRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            const [childIndex, childId] = yield this.getThreadInfo(args.threadId);
            args.threadId = childId;
            const childDap = this.childDaps[childIndex];
            const childResponse = yield childDap.continueRequest(args);
            response.body = childResponse.body;
            if (this.childDaps.length > 1) {
                if (childResponse.body === undefined) {
                    response.body = {};
                }
                response.body.allThreadsContinued = false;
            }
            this.sendResponse(response);
        });
    }
    pauseRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            const [childIndex, childId] = yield this.getThreadInfo(args.threadId);
            args.threadId = childId;
            const childDap = this.childDaps[childIndex];
            const childResponse = yield childDap.pauseRequest(args);
            response.body = childResponse.body;
            this.sendResponse(response);
        });
    }
    customRequest(command, response, args) {
        const _super = Object.create(null, {
            customRequest: { get: () => super.customRequest }
        });
        return __awaiter(this, void 0, void 0, function* () {
            if (command === 'cdt-amalgamator/getChildDapNames') {
                response.body = {
                    children: this.childDapNames,
                };
                this.sendResponse(response);
            }
            else if (command === 'cdt-amalgamator/Memory') {
                if (typeof args.address !== 'string') {
                    throw new Error(`Invalid type for 'address', expected string, got ${typeof args.address}`);
                }
                if (typeof args.length !== 'number') {
                    throw new Error(`Invalid type for 'length', expected number, got ${typeof args.length}`);
                }
                if (typeof args.child !== 'number') {
                    throw new Error(`Invalid type for 'child', expected number, got ${typeof args.child}`);
                }
                const childResponse = yield this.childDaps[args.child].customRequest('cdt-gdb-adapter/Memory', args);
                response.body = childResponse.body;
                this.sendResponse(response);
            }
            else if (command === 'cdt-amalgamator/resumeAll' ||
                command === 'cdt-amalgamator/suspendAll') {
                const [, threads] = yield this.collectChildTheads();
                for (const thread of threads) {
                    const [childIndex, childId] = yield this.getThreadInfo(thread.id);
                    const childDap = this.childDaps[childIndex];
                    if (thread.running === false &&
                        command === 'cdt-amalgamator/resumeAll') {
                        yield childDap.continueRequest({ threadId: childId });
                        this.sendEvent(new ContinuedEvent(thread.id, false));
                    }
                    else if (thread.running === true &&
                        command === 'cdt-amalgamator/suspendAll') {
                        yield childDap.pauseRequest({ threadId: childId });
                        this.sendEvent(new StoppedEvent('SIGINT', thread.id, true, false));
                    }
                }
                this.sendResponse(response);
            }
            else {
                return _super.customRequest.call(this, command, response, args);
            }
        });
    }
}
exports.AmalgamatorSession = AmalgamatorSession;
//# sourceMappingURL=AmalgamatorSession.js.map