"use strict";
/*********************************************************************
 * Copyright (c) 2023 Renesas Electronics Corporation and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const path = require("path");
const utils_1 = require("./utils");
const os_1 = require("os");
const debugTargetAdapter = 'debugTargetAdapter.js';
describe('launch with environment', function () {
    let dc;
    const lineTags = {
        'STOP HERE': 0,
    };
    before(function () {
        (0, utils_1.resolveLineTagLocations)(path.join(utils_1.testProgramsDir, 'vars_env.c'), lineTags);
    });
    const showGDBEnv = (name) => __awaiter(this, void 0, void 0, function* () {
        var _a, _b, _c;
        const value = yield (dc === null || dc === void 0 ? void 0 : dc.send('cdt-gdb-tests/executeCommand', {
            command: `show environment ${name}`,
        }));
        return (_c = (_b = (_a = value === null || value === void 0 ? void 0 : value.body) === null || _a === void 0 ? void 0 : _a.console) === null || _b === void 0 ? void 0 : _b[1]) === null || _c === void 0 ? void 0 : _c.trim();
    });
    const getAPPEnv = (vars, name) => {
        const variable = vars.body.variables.find((i) => i.name === name);
        if (!variable) {
            throw new Error(`Variable not found : ${name}`);
        }
        return variable.value;
    };
    const runForEnvironmentTest = (adapter, test, environment, targetEnvironment) => __awaiter(this, void 0, void 0, function* () {
        dc = yield (0, utils_1.standardBeforeEach)(adapter);
        yield dc.launchRequest((0, utils_1.fillDefaults)(test, {
            program: path.join(utils_1.testProgramsDir, 'vars_env'),
            environment: environment,
            target: {
                environment: targetEnvironment,
            },
        }));
        const bpResp = yield dc.setBreakpointsRequest({
            source: {
                name: 'vars_env.c',
                path: path.join(utils_1.testProgramsDir, 'vars_env.c'),
            },
            breakpoints: [
                {
                    column: 1,
                    line: lineTags['STOP HERE'],
                },
            ],
        });
        (0, chai_1.expect)(bpResp.body.breakpoints.length).eq(1);
        (0, chai_1.expect)(bpResp.body.breakpoints[0].verified).eq(true);
        (0, chai_1.expect)(bpResp.body.breakpoints[0].message).eq(undefined);
        yield dc.configurationDoneRequest();
        yield dc.waitForEvent('stopped');
        yield (dc === null || dc === void 0 ? void 0 : dc.send('cdt-gdb-tests/executeCommand', {
            command: `set print addr off`,
        }));
        const scope = yield (0, utils_1.getScopes)(dc);
        const vr = scope.scopes.body.scopes[0].variablesReference;
        const vars = yield dc.variablesRequest({ variablesReference: vr });
        const APP_PROC = {
            ENV_PATH: getAPPEnv(vars, 'path'),
            ENV_VARTEST1: getAPPEnv(vars, 'test1'),
            ENV_VARTEST2: getAPPEnv(vars, 'test2'),
            ENV_VARTEST3: getAPPEnv(vars, 'test3'),
            ENV_VARTEST4: getAPPEnv(vars, 'test4'),
            ENV_TEST_VAR: getAPPEnv(vars, 'envtest'),
        };
        // Output of the "show variable <VARNAME>" command
        // (gets value of 'undefined' in any unexpected error occured in test)
        const GDB_PROC = {
            SHOW_VARTEST1: yield showGDBEnv('VARTEST1'),
            SHOW_VARTEST2: yield showGDBEnv('VARTEST2'),
            SHOW_VARTEST3: yield showGDBEnv('VARTEST3'),
            SHOW_VARTEST4: yield showGDBEnv('VARTEST4'),
            SHOW_ENV_TEST_VAR: yield showGDBEnv('ENV_TEST_VAR'),
        };
        return {
            APP_PROC,
            GDB_PROC,
        };
    });
    afterEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            // dc could be undefined if test is skipped.
            yield (dc === null || dc === void 0 ? void 0 : dc.stop());
            dc = undefined;
        });
    });
    it('sets environment variables passed to the process', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const environment = {
                VARTEST1: 'TEST1',
                VARTEST2: 'TEST2',
                VARTEST3: 'TEST3',
                VARTEST4: 'TEST4',
            };
            const { APP_PROC, GDB_PROC } = yield runForEnvironmentTest(undefined, this.test, environment);
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST1).to.equals('"TEST1"');
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST2).to.equals('"TEST2"');
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST3).to.equals('"TEST3"');
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST4).to.equals('"TEST4"');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST1).to.equals('VARTEST1 = TEST1');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST2).to.equals('VARTEST2 = TEST2');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST3).to.equals('VARTEST3 = TEST3');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST4).to.equals('VARTEST4 = TEST4');
        });
    });
    it('checks setting environment variables with debugAdapter', function () {
        return __awaiter(this, void 0, void 0, function* () {
            if (utils_1.isRemoteTest) {
                // checks setting environment variables with debugTargetAdapter is the test for when isRemoteTest
                this.skip();
            }
            const environment = {
                VARTEST1: 'TEST1',
                VARTEST2: 'TEST2',
            };
            const targetEnvironment = {
                VARTEST1: 'TEST1_SOMEDIFFERENT_VALUE',
                VARTEST2: 'TEST2_SOMEDIFFERENT_VALUE',
                VARTEST3: 'TEST3_SOMEDIFFERENT_VALUE',
                VARTEST4: 'TEST4_SOMEDIFFERENT_VALUE',
            };
            const { APP_PROC, GDB_PROC } = yield runForEnvironmentTest(undefined, this.test, environment, targetEnvironment);
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST1).to.equals('"TEST1"');
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST2).to.equals('"TEST2"');
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST3).to.equals(''); // NULL
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST4).to.equals(''); // NULL
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST1).to.equals('VARTEST1 = TEST1');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST2).to.equals('VARTEST2 = TEST2');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST3).to.equals('Environment variable "VARTEST3" not defined.');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST4).to.equals('Environment variable "VARTEST4" not defined.');
        });
    });
    it('checks setting environment variables with debugTargetAdapter', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const environment = {
                VARTEST1: 'TEST1',
                VARTEST2: 'TEST2',
            };
            const targetEnvironment = {
                VARTEST1: 'TEST1_SOMEDIFFERENT_VALUE',
                VARTEST2: 'TEST2_SOMEDIFFERENT_VALUE',
                VARTEST3: 'TEST3_SOMEDIFFERENT_VALUE',
                VARTEST4: 'TEST4_SOMEDIFFERENT_VALUE',
            };
            const { APP_PROC, GDB_PROC } = yield runForEnvironmentTest(debugTargetAdapter, this.test, environment, targetEnvironment);
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST1).to.equals('"TEST1_SOMEDIFFERENT_VALUE"');
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST2).to.equals('"TEST2_SOMEDIFFERENT_VALUE"');
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST3).to.equals('"TEST3_SOMEDIFFERENT_VALUE"');
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST4).to.equals('"TEST4_SOMEDIFFERENT_VALUE"');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST1).to.equals('VARTEST1 = TEST1');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST2).to.equals('VARTEST2 = TEST2');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST3).to.equals('Environment variable "VARTEST3" not defined.');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST4).to.equals('Environment variable "VARTEST4" not defined.');
        });
    });
    it('unsets when target environment variables sets null with debugTargetAdapter', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const environment = {
                VARTEST1: 'TEST1',
                VARTEST2: 'TEST2',
                VARTEST3: 'TEST3',
                VARTEST4: 'TEST4',
            };
            const targetEnvironment = {
                VARTEST1: 'TEST1_SOMEDIFFERENT_VALUE',
                VARTEST2: 'TEST2_SOMEDIFFERENT_VALUE',
                VARTEST3: null,
                VARTEST4: null,
            };
            const { APP_PROC, GDB_PROC } = yield runForEnvironmentTest(debugTargetAdapter, this.test, environment, targetEnvironment);
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST1).to.equals('"TEST1_SOMEDIFFERENT_VALUE"');
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST2).to.equals('"TEST2_SOMEDIFFERENT_VALUE"');
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST3).to.equals(''); // NULL
            (0, chai_1.expect)(APP_PROC.ENV_VARTEST4).to.equals(''); // NULL
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST1).to.equals('VARTEST1 = TEST1');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST2).to.equals('VARTEST2 = TEST2');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST3).to.equals('VARTEST3 = TEST3');
            (0, chai_1.expect)(GDB_PROC.SHOW_VARTEST4).to.equals('VARTEST4 = TEST4');
        });
    });
    it('ensures that path is not null', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const { APP_PROC } = yield runForEnvironmentTest(undefined, this.test);
            (0, chai_1.expect)(APP_PROC.ENV_PATH).not.to.equals('');
        });
    });
    it('ensures that new entries could be injected to path', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const pathToAppend = __dirname;
            const currentPathValue = process.env.PATH;
            const environment = {
                PATH: `${pathToAppend}${path.delimiter}${currentPathValue}`,
            };
            const { APP_PROC } = yield runForEnvironmentTest(undefined, this.test, environment);
            (0, chai_1.expect)(APP_PROC.ENV_PATH).not.to.equals('');
            const entries = APP_PROC.ENV_PATH.substring(1, APP_PROC.ENV_PATH.length - 1).split(path.delimiter);
            if ((0, os_1.platform)() === 'win32') {
                // Win32 test platform auto inject another folder to the front of the list.
                // So we have a little bit different test here.
                const winEntries = entries.map((i) => i.replace(/\\\\/g, '\\'));
                (0, chai_1.expect)(winEntries, 'Path does not include appended folder').to.includes(pathToAppend);
            }
            else {
                (0, chai_1.expect)(entries[0]).to.equals(pathToAppend);
            }
        });
    });
    it('ensures that ENV_TEST_VAR is not null', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const { APP_PROC } = yield runForEnvironmentTest(undefined, this.test);
            (0, chai_1.expect)(APP_PROC.ENV_TEST_VAR).not.to.equals('');
        });
    });
    it('check setting null will delete the variable', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const environment = {
                ENV_TEST_VAR: null,
            };
            const { APP_PROC } = yield runForEnvironmentTest(undefined, this.test, environment);
            (0, chai_1.expect)(APP_PROC.ENV_TEST_VAR).to.equals('');
        });
    });
});
//# sourceMappingURL=launchWithEnvironment.spec.js.map