"use strict";
// *****************************************************************************
// Copyright (C) 2021 SAP SE or an SAP affiliate company and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var QuickCommandService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QuickCommandService = exports.CLOSE_QUICK_OPEN = exports.CLEAR_COMMAND_HISTORY = exports.quickCommand = void 0;
const inversify_1 = require("inversify");
const keybinding_1 = require("../keybinding");
const common_1 = require("../../common");
const context_key_service_1 = require("../context-key-service");
const core_preferences_1 = require("../core-preferences");
const quick_access_1 = require("./quick-access");
const quick_input_service_1 = require("./quick-input-service");
const widgets_1 = require("../widgets");
exports.quickCommand = {
    id: 'workbench.action.showCommands'
};
exports.CLEAR_COMMAND_HISTORY = common_1.Command.toDefaultLocalizedCommand({
    id: 'clear.command.history',
    label: 'Clear Command History'
});
exports.CLOSE_QUICK_OPEN = {
    id: 'workbench.action.closeQuickOpen'
};
let QuickCommandService = QuickCommandService_1 = class QuickCommandService {
    constructor() {
        // The list of exempted commands not to be displayed in the recently used list.
        this.exemptedCommands = [
            exports.CLEAR_COMMAND_HISTORY,
        ];
        this.recentItems = [];
        this.otherItems = [];
        this.contexts = new Map();
    }
    registerQuickAccessProvider() {
        this.quickAccessRegistry.registerQuickAccessProvider({
            getInstance: () => this,
            prefix: QuickCommandService_1.PREFIX,
            placeholder: '',
            helpEntries: [{ description: 'Quick Command', needsEditor: false }]
        });
    }
    reset() {
        const { recent, other } = this.getCommands();
        this.recentItems = [];
        this.otherItems = [];
        this.recentItems.push(...recent.map(command => this.toItem(command)));
        this.otherItems.push(...other.map(command => this.toItem(command)));
    }
    getPicks(filter, token) {
        const items = [];
        // Update the list of commands by fetching them from the registry.
        this.reset();
        const recentItems = (0, quick_input_service_1.filterItems)(this.recentItems.slice(), filter);
        const otherItems = (0, quick_input_service_1.filterItems)(this.otherItems.slice(), filter);
        if (recentItems.length > 0) {
            items.push({ type: 'separator', label: common_1.nls.localizeByDefault('recently used') }, ...recentItems);
        }
        if (otherItems.length > 0) {
            if (recentItems.length > 0) {
                items.push({ type: 'separator', label: common_1.nls.localizeByDefault('other commands') });
            }
            items.push(...otherItems);
        }
        return items;
    }
    toItem(command) {
        const label = (command.category) ? `${command.category}: ` + command.label : command.label;
        const iconClasses = this.getItemIconClasses(command);
        const activeElement = window.document.activeElement;
        const originalLabel = command.originalLabel || command.label;
        const originalCategory = command.originalCategory || command.category;
        let detail = originalCategory ? `${originalCategory}: ${originalLabel}` : originalLabel;
        if (label === detail) {
            detail = undefined;
        }
        return {
            label,
            detail,
            iconClasses,
            alwaysShow: !!this.commandRegistry.getActiveHandler(command.id),
            keySequence: this.getKeybinding(command),
            execute: () => {
                activeElement.focus({ preventScroll: true });
                this.commandRegistry.executeCommand(command.id);
                this.commandRegistry.addRecentCommand(command);
            }
        };
    }
    getKeybinding(command) {
        const keybindings = this.keybindingRegistry.getKeybindingsForCommand(command.id);
        if (!keybindings || keybindings.length === 0) {
            return undefined;
        }
        try {
            return this.keybindingRegistry.resolveKeybinding(keybindings[0]);
        }
        catch (error) {
            return undefined;
        }
    }
    getItemIconClasses(command) {
        const toggledHandler = this.commandRegistry.getToggledHandler(command.id);
        if (toggledHandler) {
            return (0, widgets_1.codiconArray)('check');
        }
        return undefined;
    }
    pushCommandContext(commandId, when) {
        const contexts = this.contexts.get(commandId) || [];
        contexts.push(when);
        this.contexts.set(commandId, contexts);
        return common_1.Disposable.create(() => {
            const index = contexts.indexOf(when);
            if (index !== -1) {
                contexts.splice(index, 1);
            }
        });
    }
    /**
     * Get the list of valid commands.
     *
     * @param commands the list of raw commands.
     * @returns the list of valid commands.
     */
    getValidCommands(raw) {
        const valid = [];
        raw.forEach(command => {
            if (command.label) {
                const contexts = this.contexts.get(command.id);
                if (!contexts || contexts.some(when => this.contextKeyService.match(when))) {
                    valid.push(command);
                }
            }
        });
        return valid;
    }
    /**
     * Get the list of recently used and other commands.
     *
     * @returns the list of recently used commands and other commands.
     */
    getCommands() {
        // Get the list of recent commands.
        const recentCommands = this.commandRegistry.recent;
        // Get the list of all valid commands.
        const allCommands = this.getValidCommands(this.commandRegistry.commands);
        // Get the max history limit.
        const limit = this.corePreferences['workbench.commandPalette.history'];
        // Build the list of recent commands.
        let rCommands = [];
        if (limit > 0) {
            rCommands.push(...recentCommands.filter(r => !this.exemptedCommands.some(c => common_1.Command.equals(r, c)) &&
                allCommands.some(c => common_1.Command.equals(r, c))));
            if (rCommands.length > limit) {
                rCommands = rCommands.slice(0, limit);
            }
        }
        // Build the list of other commands.
        const oCommands = allCommands.filter(c => !rCommands.some(r => common_1.Command.equals(r, c)));
        // Normalize the list of recent commands.
        const recent = this.normalize(rCommands);
        // Normalize, and sort the list of other commands.
        const other = this.sort(this.normalize(oCommands));
        return { recent, other };
    }
    /**
     * Normalizes a list of commands.
     * Normalization includes obtaining commands that have labels, are visible, and are enabled.
     *
     * @param commands the list of commands.
     * @returns the list of normalized commands.
     */
    normalize(commands) {
        return commands.filter((a) => a.label && (this.commandRegistry.isVisible(a.id) && this.commandRegistry.isEnabled(a.id)));
    }
    /**
     * Sorts a list of commands alphabetically.
     *
     * @param commands the list of commands.
     * @returns the list of sorted commands.
     */
    sort(commands) {
        return commands.sort((a, b) => common_1.Command.compareCommands(a, b));
    }
};
QuickCommandService.PREFIX = '>';
__decorate([
    (0, inversify_1.inject)(context_key_service_1.ContextKeyService),
    __metadata("design:type", Object)
], QuickCommandService.prototype, "contextKeyService", void 0);
__decorate([
    (0, inversify_1.inject)(common_1.CommandRegistry),
    __metadata("design:type", common_1.CommandRegistry)
], QuickCommandService.prototype, "commandRegistry", void 0);
__decorate([
    (0, inversify_1.inject)(core_preferences_1.CorePreferences),
    __metadata("design:type", Object)
], QuickCommandService.prototype, "corePreferences", void 0);
__decorate([
    (0, inversify_1.inject)(quick_access_1.QuickAccessRegistry),
    __metadata("design:type", Object)
], QuickCommandService.prototype, "quickAccessRegistry", void 0);
__decorate([
    (0, inversify_1.inject)(keybinding_1.KeybindingRegistry),
    __metadata("design:type", keybinding_1.KeybindingRegistry)
], QuickCommandService.prototype, "keybindingRegistry", void 0);
QuickCommandService = QuickCommandService_1 = __decorate([
    (0, inversify_1.injectable)()
], QuickCommandService);
exports.QuickCommandService = QuickCommandService;
//# sourceMappingURL=quick-command-service.js.map