"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReferenceLinkMap = exports.LinkDefinitionSet = exports.MdLinkKind = exports.HrefKind = void 0;
var HrefKind;
(function (HrefKind) {
    HrefKind[HrefKind["External"] = 0] = "External";
    HrefKind[HrefKind["Internal"] = 1] = "Internal";
    HrefKind[HrefKind["Reference"] = 2] = "Reference";
})(HrefKind || (exports.HrefKind = HrefKind = {}));
var MdLinkKind;
(function (MdLinkKind) {
    /** Standard Markdown link syntax: `[text][ref]` or `[text](http://example.com)` */
    MdLinkKind[MdLinkKind["Link"] = 1] = "Link";
    /** Link definition: `[def]: http://example.com` */
    MdLinkKind[MdLinkKind["Definition"] = 2] = "Definition";
    /** Auto link: `<http://example.com>` */
    MdLinkKind[MdLinkKind["AutoLink"] = 3] = "AutoLink";
})(MdLinkKind || (exports.MdLinkKind = MdLinkKind = {}));
/**
 * A map that lets you look up definitions by reference name.
 */
class LinkDefinitionSet {
    #map = new ReferenceLinkMap();
    constructor(links) {
        for (const link of links) {
            if (link.kind === MdLinkKind.Definition) {
                if (!this.#map.has(link.ref.text)) {
                    this.#map.set(link.ref.text, link);
                }
            }
        }
    }
    [Symbol.iterator]() {
        return this.#map[Symbol.iterator]();
    }
    lookup(ref) {
        return this.#map.lookup(ref);
    }
}
exports.LinkDefinitionSet = LinkDefinitionSet;
/**
 * A store of link reference names.
 *
 * Correctly normalizes reference names.
 */
class ReferenceLinkMap {
    #map = new Map();
    set(ref, link) {
        this.#map.set(this.#normalizeRefName(ref), link);
    }
    lookup(ref) {
        return this.#map.get(this.#normalizeRefName(ref));
    }
    has(ref) {
        return this.#map.has(this.#normalizeRefName(ref));
    }
    [Symbol.iterator]() {
        return this.#map.values();
    }
    /**
     * Normalizes a link reference. Link references are case-insensitive, so this lowercases the reference so you can
     * correctly compare two normalized references.
     */
    #normalizeRefName(ref) {
        return ref.normalize().trim().toLowerCase();
    }
}
exports.ReferenceLinkMap = ReferenceLinkMap;
//# sourceMappingURL=documentLink.js.map