/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util.bkd;

import java.io.IOException;
import java.util.Arrays;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.PointValues;
import org.apache.lucene.store.ByteArrayDataInput;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.MathUtil;
import org.apache.lucene.util.RamUsageEstimator;
import org.apache.lucene.util.bkd.DocIdsWriter;

/*
 * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
 */
public final class BKDReader
extends PointValues
implements Accountable {
    final int leafNodeOffset;
    final int numDims;
    final int bytesPerDim;
    final int numLeaves;
    final IndexInput in;
    final int maxPointsInLeafNode;
    final byte[] minPackedValue;
    final byte[] maxPackedValue;
    final long pointCount;
    final int docCount;
    final int version;
    protected final int packedBytesLength;
    final byte[] packedIndex;
    private final byte[] splitPackedValues;
    final int bytesPerIndexEntry;
    final long[] leafBlockFPs;

    public BKDReader(IndexInput in) throws IOException {
        this.version = CodecUtil.checkHeader(in, "BKD", 0, 5);
        this.numDims = in.readVInt();
        this.maxPointsInLeafNode = in.readVInt();
        this.bytesPerDim = in.readVInt();
        this.bytesPerIndexEntry = this.numDims == 1 && this.version >= 3 ? this.bytesPerDim : this.bytesPerDim + 1;
        this.packedBytesLength = this.numDims * this.bytesPerDim;
        this.numLeaves = in.readVInt();
        assert (this.numLeaves > 0);
        this.leafNodeOffset = this.numLeaves;
        this.minPackedValue = new byte[this.packedBytesLength];
        this.maxPackedValue = new byte[this.packedBytesLength];
        in.readBytes(this.minPackedValue, 0, this.packedBytesLength);
        in.readBytes(this.maxPackedValue, 0, this.packedBytesLength);
        for (int dim = 0; dim < this.numDims; ++dim) {
            if (Arrays.compareUnsigned(this.minPackedValue, dim * this.bytesPerDim, dim * this.bytesPerDim + this.bytesPerDim, this.maxPackedValue, dim * this.bytesPerDim, dim * this.bytesPerDim + this.bytesPerDim) <= 0) continue;
            throw new CorruptIndexException("minPackedValue " + new BytesRef(this.minPackedValue) + " is > maxPackedValue " + new BytesRef(this.maxPackedValue) + " for dim=" + dim, in);
        }
        this.pointCount = in.readVLong();
        this.docCount = in.readVInt();
        if (this.version >= 4) {
            int numBytes = in.readVInt();
            this.packedIndex = new byte[numBytes];
            in.readBytes(this.packedIndex, 0, numBytes);
            this.leafBlockFPs = null;
            this.splitPackedValues = null;
        } else {
            this.splitPackedValues = new byte[this.bytesPerIndexEntry * this.numLeaves];
            in.readBytes(this.splitPackedValues, 0, this.splitPackedValues.length);
            long[] leafBlockFPs = new long[this.numLeaves];
            long lastFP = 0L;
            for (int i = 0; i < this.numLeaves; ++i) {
                long delta = in.readVLong();
                leafBlockFPs[i] = lastFP + delta;
                lastFP += delta;
            }
            if (this.numDims == 1 && this.numLeaves > 1) {
                int levelCount = 2;
                while (true) {
                    if (this.numLeaves >= levelCount && this.numLeaves <= 2 * levelCount) {
                        int lastLevel = 2 * (this.numLeaves - levelCount);
                        assert (lastLevel >= 0);
                        if (lastLevel == 0) break;
                        long[] newLeafBlockFPs = new long[this.numLeaves];
                        System.arraycopy(leafBlockFPs, lastLevel, newLeafBlockFPs, 0, leafBlockFPs.length - lastLevel);
                        System.arraycopy(leafBlockFPs, 0, newLeafBlockFPs, leafBlockFPs.length - lastLevel, lastLevel);
                        leafBlockFPs = newLeafBlockFPs;
                        break;
                    }
                    levelCount *= 2;
                }
            }
            this.leafBlockFPs = leafBlockFPs;
            this.packedIndex = null;
        }
        this.in = in;
    }

    long getMinLeafBlockFP() {
        if (this.packedIndex != null) {
            return new ByteArrayDataInput(this.packedIndex).readVLong();
        }
        long minFP = Long.MAX_VALUE;
        for (long fp : this.leafBlockFPs) {
            minFP = Math.min(minFP, fp);
        }
        return minFP;
    }

    private int getTreeDepth() {
        return MathUtil.log(this.numLeaves, 2) + 2;
    }

    @Override
    public void intersect(PointValues.IntersectVisitor visitor) throws IOException {
        this.intersect(this.getIntersectState(visitor), this.minPackedValue, this.maxPackedValue);
    }

    @Override
    public long estimatePointCount(PointValues.IntersectVisitor visitor) {
        return this.estimatePointCount(this.getIntersectState(visitor), this.minPackedValue, this.maxPackedValue);
    }

    private void addAll(IntersectState state, boolean grown) throws IOException {
        long maxPointCount;
        if (!grown && (maxPointCount = (long)this.maxPointsInLeafNode * (long)state.index.getNumLeaves()) <= Integer.MAX_VALUE) {
            state.visitor.grow((int)maxPointCount);
            grown = true;
        }
        if (state.index.isLeafNode()) {
            assert (grown);
            if (state.index.nodeExists()) {
                this.visitDocIDs(state.in, state.index.getLeafBlockFP(), state.visitor);
            }
        } else {
            state.index.pushLeft();
            this.addAll(state, grown);
            state.index.pop();
            state.index.pushRight();
            this.addAll(state, grown);
            state.index.pop();
        }
    }

    public IntersectState getIntersectState(PointValues.IntersectVisitor visitor) {
        IndexTree index = this.packedIndex != null ? new PackedIndexTree() : new LegacyIndexTree();
        return new IntersectState(this.in.clone(), this.numDims, this.packedBytesLength, this.maxPointsInLeafNode, visitor, index);
    }

    public void visitLeafBlockValues(IndexTree index, IntersectState state) throws IOException {
        int count = this.readDocIDs(state.in, index.getLeafBlockFP(), state.scratchDocIDs);
        this.visitDocValues(state.commonPrefixLengths, state.scratchPackedValue1, state.scratchPackedValue2, state.in, state.scratchDocIDs, count, state.visitor);
    }

    private void visitDocIDs(IndexInput in, long blockFP, PointValues.IntersectVisitor visitor) throws IOException {
        in.seek(blockFP);
        int count = in.readVInt();
        if (this.version < 1) {
            DocIdsWriter.readInts32(in, count, visitor);
        } else {
            DocIdsWriter.readInts(in, count, visitor);
        }
    }

    int readDocIDs(IndexInput in, long blockFP, int[] docIDs) throws IOException {
        in.seek(blockFP);
        int count = in.readVInt();
        if (this.version < 1) {
            DocIdsWriter.readInts32(in, count, docIDs);
        } else {
            DocIdsWriter.readInts(in, count, docIDs);
        }
        return count;
    }

    void visitDocValues(int[] commonPrefixLengths, byte[] scratchPackedValue1, byte[] scratchPackedValue2, IndexInput in, int[] docIDs, int count, PointValues.IntersectVisitor visitor) throws IOException {
        int compressedDim;
        this.readCommonPrefixes(commonPrefixLengths, scratchPackedValue1, in);
        if (this.numDims != 1 && this.version >= 5) {
            byte[] minPackedValue = scratchPackedValue1;
            byte[] maxPackedValue = scratchPackedValue2;
            System.arraycopy(minPackedValue, 0, maxPackedValue, 0, this.packedBytesLength);
            this.readMinMax(commonPrefixLengths, minPackedValue, maxPackedValue, in);
            PointValues.Relation r = visitor.compare(minPackedValue, maxPackedValue);
            if (r == PointValues.Relation.CELL_OUTSIDE_QUERY) {
                return;
            }
            visitor.grow(count);
            if (r == PointValues.Relation.CELL_INSIDE_QUERY) {
                for (int i = 0; i < count; ++i) {
                    visitor.visit(docIDs[i]);
                }
                return;
            }
        } else {
            visitor.grow(count);
        }
        int n = compressedDim = this.version < 2 ? -1 : this.readCompressedDim(in);
        if (compressedDim == -1) {
            this.visitRawDocValues(commonPrefixLengths, scratchPackedValue1, in, docIDs, count, visitor);
        } else {
            this.visitCompressedDocValues(commonPrefixLengths, scratchPackedValue1, in, docIDs, count, visitor, compressedDim);
        }
    }

    private void readMinMax(int[] commonPrefixLengths, byte[] minPackedValue, byte[] maxPackedValue, IndexInput in) throws IOException {
        for (int dim = 0; dim < this.numDims; ++dim) {
            int prefix = commonPrefixLengths[dim];
            in.readBytes(minPackedValue, dim * this.bytesPerDim + prefix, this.bytesPerDim - prefix);
            in.readBytes(maxPackedValue, dim * this.bytesPerDim + prefix, this.bytesPerDim - prefix);
        }
    }

    private void visitRawDocValues(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in, int[] docIDs, int count, PointValues.IntersectVisitor visitor) throws IOException {
        for (int i = 0; i < count; ++i) {
            for (int dim = 0; dim < this.numDims; ++dim) {
                int prefix = commonPrefixLengths[dim];
                in.readBytes(scratchPackedValue, dim * this.bytesPerDim + prefix, this.bytesPerDim - prefix);
            }
            visitor.visit(docIDs[i], scratchPackedValue);
        }
    }

    private void visitCompressedDocValues(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in, int[] docIDs, int count, PointValues.IntersectVisitor visitor, int compressedDim) throws IOException {
        int i;
        int runLen;
        int compressedByteOffset = compressedDim * this.bytesPerDim + commonPrefixLengths[compressedDim];
        int n = compressedDim;
        commonPrefixLengths[n] = commonPrefixLengths[n] + 1;
        for (i = 0; i < count; i += runLen) {
            scratchPackedValue[compressedByteOffset] = in.readByte();
            runLen = Byte.toUnsignedInt(in.readByte());
            for (int j = 0; j < runLen; ++j) {
                for (int dim = 0; dim < this.numDims; ++dim) {
                    int prefix = commonPrefixLengths[dim];
                    in.readBytes(scratchPackedValue, dim * this.bytesPerDim + prefix, this.bytesPerDim - prefix);
                }
                visitor.visit(docIDs[i + j], scratchPackedValue);
            }
        }
        if (i != count) {
            throw new CorruptIndexException("Sub blocks do not add up to the expected count: " + count + " != " + i, in);
        }
    }

    private int readCompressedDim(IndexInput in) throws IOException {
        byte compressedDim = in.readByte();
        if (compressedDim < -1 || compressedDim >= this.numDims) {
            throw new CorruptIndexException("Got compressedDim=" + compressedDim, in);
        }
        return compressedDim;
    }

    private void readCommonPrefixes(int[] commonPrefixLengths, byte[] scratchPackedValue, IndexInput in) throws IOException {
        for (int dim = 0; dim < this.numDims; ++dim) {
            int prefix;
            commonPrefixLengths[dim] = prefix = in.readVInt();
            if (prefix <= 0) continue;
            in.readBytes(scratchPackedValue, dim * this.bytesPerDim, prefix);
        }
    }

    private void intersect(IntersectState state, byte[] cellMinPacked, byte[] cellMaxPacked) throws IOException {
        PointValues.Relation r = state.visitor.compare(cellMinPacked, cellMaxPacked);
        if (r != PointValues.Relation.CELL_OUTSIDE_QUERY) {
            if (r == PointValues.Relation.CELL_INSIDE_QUERY) {
                this.addAll(state, false);
            } else if (state.index.isLeafNode()) {
                if (state.index.nodeExists()) {
                    int count = this.readDocIDs(state.in, state.index.getLeafBlockFP(), state.scratchDocIDs);
                    this.visitDocValues(state.commonPrefixLengths, state.scratchPackedValue1, state.scratchPackedValue2, state.in, state.scratchDocIDs, count, state.visitor);
                }
            } else {
                int splitDim = state.index.getSplitDim();
                assert (splitDim >= 0) : "splitDim=" + splitDim;
                assert (splitDim < this.numDims);
                byte[] splitPackedValue = state.index.getSplitPackedValue();
                BytesRef splitDimValue = state.index.getSplitDimValue();
                assert (splitDimValue.length == this.bytesPerDim);
                assert (Arrays.compareUnsigned(cellMinPacked, splitDim * this.bytesPerDim, splitDim * this.bytesPerDim + this.bytesPerDim, splitDimValue.bytes, splitDimValue.offset, splitDimValue.offset + this.bytesPerDim) <= 0) : "bytesPerDim=" + this.bytesPerDim + " splitDim=" + splitDim + " numDims=" + this.numDims;
                assert (Arrays.compareUnsigned(cellMaxPacked, splitDim * this.bytesPerDim, splitDim * this.bytesPerDim + this.bytesPerDim, splitDimValue.bytes, splitDimValue.offset, splitDimValue.offset + this.bytesPerDim) >= 0) : "bytesPerDim=" + this.bytesPerDim + " splitDim=" + splitDim + " numDims=" + this.numDims;
                System.arraycopy(cellMaxPacked, 0, splitPackedValue, 0, this.packedBytesLength);
                System.arraycopy(splitDimValue.bytes, splitDimValue.offset, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
                state.index.pushLeft();
                this.intersect(state, cellMinPacked, splitPackedValue);
                state.index.pop();
                System.arraycopy(splitPackedValue, splitDim * this.bytesPerDim, splitDimValue.bytes, splitDimValue.offset, this.bytesPerDim);
                System.arraycopy(cellMinPacked, 0, splitPackedValue, 0, this.packedBytesLength);
                System.arraycopy(splitDimValue.bytes, splitDimValue.offset, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
                state.index.pushRight();
                this.intersect(state, splitPackedValue, cellMaxPacked);
                state.index.pop();
            }
        }
    }

    private long estimatePointCount(IntersectState state, byte[] cellMinPacked, byte[] cellMaxPacked) {
        PointValues.Relation r = state.visitor.compare(cellMinPacked, cellMaxPacked);
        if (r == PointValues.Relation.CELL_OUTSIDE_QUERY) {
            return 0L;
        }
        if (r == PointValues.Relation.CELL_INSIDE_QUERY) {
            return (long)this.maxPointsInLeafNode * (long)state.index.getNumLeaves();
        }
        if (state.index.isLeafNode()) {
            return (this.maxPointsInLeafNode + 1) / 2;
        }
        int splitDim = state.index.getSplitDim();
        assert (splitDim >= 0) : "splitDim=" + splitDim;
        assert (splitDim < this.numDims);
        byte[] splitPackedValue = state.index.getSplitPackedValue();
        BytesRef splitDimValue = state.index.getSplitDimValue();
        assert (splitDimValue.length == this.bytesPerDim);
        assert (Arrays.compareUnsigned(cellMinPacked, splitDim * this.bytesPerDim, splitDim * this.bytesPerDim + this.bytesPerDim, splitDimValue.bytes, splitDimValue.offset, splitDimValue.offset + this.bytesPerDim) <= 0) : "bytesPerDim=" + this.bytesPerDim + " splitDim=" + splitDim + " numDims=" + this.numDims;
        assert (Arrays.compareUnsigned(cellMaxPacked, splitDim * this.bytesPerDim, splitDim * this.bytesPerDim + this.bytesPerDim, splitDimValue.bytes, splitDimValue.offset, splitDimValue.offset + this.bytesPerDim) >= 0) : "bytesPerDim=" + this.bytesPerDim + " splitDim=" + splitDim + " numDims=" + this.numDims;
        System.arraycopy(cellMaxPacked, 0, splitPackedValue, 0, this.packedBytesLength);
        System.arraycopy(splitDimValue.bytes, splitDimValue.offset, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
        state.index.pushLeft();
        long leftCost = this.estimatePointCount(state, cellMinPacked, splitPackedValue);
        state.index.pop();
        System.arraycopy(splitPackedValue, splitDim * this.bytesPerDim, splitDimValue.bytes, splitDimValue.offset, this.bytesPerDim);
        System.arraycopy(cellMinPacked, 0, splitPackedValue, 0, this.packedBytesLength);
        System.arraycopy(splitDimValue.bytes, splitDimValue.offset, splitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
        state.index.pushRight();
        long rightCost = this.estimatePointCount(state, splitPackedValue, cellMaxPacked);
        state.index.pop();
        return leftCost + rightCost;
    }

    @Override
    public long ramBytesUsed() {
        if (this.packedIndex != null) {
            return this.packedIndex.length;
        }
        return RamUsageEstimator.sizeOf(this.splitPackedValues) + RamUsageEstimator.sizeOf(this.leafBlockFPs);
    }

    @Override
    public byte[] getMinPackedValue() {
        return (byte[])this.minPackedValue.clone();
    }

    @Override
    public byte[] getMaxPackedValue() {
        return (byte[])this.maxPackedValue.clone();
    }

    @Override
    public int getNumDimensions() {
        return this.numDims;
    }

    @Override
    public int getBytesPerDimension() {
        return this.bytesPerDim;
    }

    @Override
    public long size() {
        return this.pointCount;
    }

    @Override
    public int getDocCount() {
        return this.docCount;
    }

    public boolean isLeafNode(int nodeID) {
        return nodeID >= this.leafNodeOffset;
    }

    public abstract class IndexTree
    implements Cloneable {
        protected int nodeID;
        protected int level;
        protected int splitDim;
        protected final byte[][] splitPackedValueStack;

        protected IndexTree() {
            int treeDepth = BKDReader.this.getTreeDepth();
            this.splitPackedValueStack = new byte[treeDepth + 1][];
            this.nodeID = 1;
            this.level = 1;
            this.splitPackedValueStack[this.level] = new byte[BKDReader.this.packedBytesLength];
        }

        public void pushLeft() {
            this.nodeID *= 2;
            ++this.level;
            if (this.splitPackedValueStack[this.level] == null) {
                this.splitPackedValueStack[this.level] = new byte[BKDReader.this.packedBytesLength];
            }
        }

        public abstract IndexTree clone();

        public void pushRight() {
            this.nodeID = this.nodeID * 2 + 1;
            ++this.level;
            if (this.splitPackedValueStack[this.level] == null) {
                this.splitPackedValueStack[this.level] = new byte[BKDReader.this.packedBytesLength];
            }
        }

        public void pop() {
            this.nodeID /= 2;
            --this.level;
            this.splitDim = -1;
        }

        public boolean isLeafNode() {
            return this.nodeID >= BKDReader.this.leafNodeOffset;
        }

        public boolean nodeExists() {
            return this.nodeID - BKDReader.this.leafNodeOffset < BKDReader.this.leafNodeOffset;
        }

        public int getNodeID() {
            return this.nodeID;
        }

        public byte[] getSplitPackedValue() {
            assert (!this.isLeafNode());
            assert (this.splitPackedValueStack[this.level] != null) : "level=" + this.level;
            return this.splitPackedValueStack[this.level];
        }

        public int getSplitDim() {
            assert (!this.isLeafNode());
            return this.splitDim;
        }

        public abstract BytesRef getSplitDimValue();

        public abstract long getLeafBlockFP();

        public int getNumLeaves() {
            int leftMostLeafNode;
            for (leftMostLeafNode = this.nodeID; leftMostLeafNode < BKDReader.this.leafNodeOffset; leftMostLeafNode *= 2) {
            }
            int rightMostLeafNode = this.nodeID;
            while (rightMostLeafNode < BKDReader.this.leafNodeOffset) {
                rightMostLeafNode = rightMostLeafNode * 2 + 1;
            }
            int numLeaves = rightMostLeafNode >= leftMostLeafNode ? rightMostLeafNode - leftMostLeafNode + 1 : rightMostLeafNode - leftMostLeafNode + 1 + BKDReader.this.leafNodeOffset;
            assert (numLeaves == this.getNumLeavesSlow(this.nodeID)) : numLeaves + " " + this.getNumLeavesSlow(this.nodeID);
            return numLeaves;
        }

        private int getNumLeavesSlow(int node) {
            if (node >= 2 * BKDReader.this.leafNodeOffset) {
                return 0;
            }
            if (node >= BKDReader.this.leafNodeOffset) {
                return 1;
            }
            int leftCount = this.getNumLeavesSlow(node * 2);
            int rightCount = this.getNumLeavesSlow(node * 2 + 1);
            return leftCount + rightCount;
        }
    }

    public static final class IntersectState {
        final IndexInput in;
        final int[] scratchDocIDs;
        final byte[] scratchPackedValue1;
        final byte[] scratchPackedValue2;
        final int[] commonPrefixLengths;
        final PointValues.IntersectVisitor visitor;
        public final IndexTree index;

        public IntersectState(IndexInput in, int numDims, int packedBytesLength, int maxPointsInLeafNode, PointValues.IntersectVisitor visitor, IndexTree indexVisitor) {
            this.in = in;
            this.visitor = visitor;
            this.commonPrefixLengths = new int[numDims];
            this.scratchDocIDs = new int[maxPointsInLeafNode];
            this.scratchPackedValue1 = new byte[packedBytesLength];
            this.scratchPackedValue2 = new byte[packedBytesLength];
            this.index = indexVisitor;
        }
    }

    private final class LegacyIndexTree
    extends IndexTree {
        private long leafBlockFP;
        private final byte[] splitDimValue;
        private final BytesRef scratch;

        public LegacyIndexTree() {
            this.splitDimValue = new byte[BKDReader.this.bytesPerDim];
            this.scratch = new BytesRef();
            this.setNodeData();
            this.scratch.bytes = this.splitDimValue;
            this.scratch.length = BKDReader.this.bytesPerDim;
        }

        @Override
        public LegacyIndexTree clone() {
            LegacyIndexTree index = new LegacyIndexTree();
            index.nodeID = this.nodeID;
            index.level = this.level;
            index.splitDim = this.splitDim;
            index.leafBlockFP = this.leafBlockFP;
            index.splitPackedValueStack[index.level] = (byte[])this.splitPackedValueStack[index.level].clone();
            return index;
        }

        @Override
        public void pushLeft() {
            super.pushLeft();
            this.setNodeData();
        }

        @Override
        public void pushRight() {
            super.pushRight();
            this.setNodeData();
        }

        private void setNodeData() {
            if (this.isLeafNode()) {
                this.leafBlockFP = BKDReader.this.leafBlockFPs[this.nodeID - BKDReader.this.leafNodeOffset];
                this.splitDim = -1;
            } else {
                this.leafBlockFP = -1L;
                int address = this.nodeID * BKDReader.this.bytesPerIndexEntry;
                if (BKDReader.this.numDims == 1) {
                    this.splitDim = 0;
                    if (BKDReader.this.version < 3) {
                        assert (BKDReader.this.splitPackedValues[address] == 0);
                        ++address;
                    }
                } else {
                    this.splitDim = BKDReader.this.splitPackedValues[address++] & 0xFF;
                }
                System.arraycopy(BKDReader.this.splitPackedValues, address, this.splitDimValue, 0, BKDReader.this.bytesPerDim);
            }
        }

        @Override
        public long getLeafBlockFP() {
            assert (this.isLeafNode());
            return this.leafBlockFP;
        }

        @Override
        public BytesRef getSplitDimValue() {
            assert (!this.isLeafNode());
            return this.scratch;
        }

        @Override
        public void pop() {
            super.pop();
            this.leafBlockFP = -1L;
        }
    }

    private final class PackedIndexTree
    extends IndexTree {
        private final ByteArrayDataInput in;
        private final long[] leafBlockFPStack;
        private final int[] leftNodePositions;
        private final int[] rightNodePositions;
        private final int[] splitDims;
        private final boolean[] negativeDeltas;
        private final byte[][] splitValuesStack;
        private final BytesRef scratch;

        public PackedIndexTree() {
            int treeDepth = BKDReader.this.getTreeDepth();
            this.leafBlockFPStack = new long[treeDepth + 1];
            this.leftNodePositions = new int[treeDepth + 1];
            this.rightNodePositions = new int[treeDepth + 1];
            this.splitValuesStack = new byte[treeDepth + 1][];
            this.splitDims = new int[treeDepth + 1];
            this.negativeDeltas = new boolean[BKDReader.this.numDims * (treeDepth + 1)];
            this.in = new ByteArrayDataInput(BKDReader.this.packedIndex);
            this.splitValuesStack[0] = new byte[BKDReader.this.packedBytesLength];
            this.readNodeData(false);
            this.scratch = new BytesRef();
            this.scratch.length = BKDReader.this.bytesPerDim;
        }

        @Override
        public PackedIndexTree clone() {
            PackedIndexTree index = new PackedIndexTree();
            index.nodeID = this.nodeID;
            index.level = this.level;
            index.splitDim = this.splitDim;
            index.leafBlockFPStack[this.level] = this.leafBlockFPStack[this.level];
            index.leftNodePositions[this.level] = this.leftNodePositions[this.level];
            index.rightNodePositions[this.level] = this.rightNodePositions[this.level];
            index.splitValuesStack[index.level] = (byte[])this.splitValuesStack[index.level].clone();
            System.arraycopy(this.negativeDeltas, this.level * BKDReader.this.numDims, index.negativeDeltas, this.level * BKDReader.this.numDims, BKDReader.this.numDims);
            index.splitDims[this.level] = this.splitDims[this.level];
            return index;
        }

        @Override
        public void pushLeft() {
            int nodePosition = this.leftNodePositions[this.level];
            super.pushLeft();
            System.arraycopy(this.negativeDeltas, (this.level - 1) * BKDReader.this.numDims, this.negativeDeltas, this.level * BKDReader.this.numDims, BKDReader.this.numDims);
            assert (this.splitDim != -1);
            this.negativeDeltas[this.level * BKDReader.this.numDims + this.splitDim] = true;
            this.in.setPosition(nodePosition);
            this.readNodeData(true);
        }

        @Override
        public void pushRight() {
            int nodePosition = this.rightNodePositions[this.level];
            super.pushRight();
            System.arraycopy(this.negativeDeltas, (this.level - 1) * BKDReader.this.numDims, this.negativeDeltas, this.level * BKDReader.this.numDims, BKDReader.this.numDims);
            assert (this.splitDim != -1);
            this.negativeDeltas[this.level * BKDReader.this.numDims + this.splitDim] = false;
            this.in.setPosition(nodePosition);
            this.readNodeData(false);
        }

        @Override
        public void pop() {
            super.pop();
            this.splitDim = this.splitDims[this.level];
        }

        @Override
        public long getLeafBlockFP() {
            assert (this.isLeafNode()) : "nodeID=" + this.nodeID + " is not a leaf";
            return this.leafBlockFPStack[this.level];
        }

        @Override
        public BytesRef getSplitDimValue() {
            assert (!this.isLeafNode());
            this.scratch.bytes = this.splitValuesStack[this.level];
            this.scratch.offset = this.splitDim * BKDReader.this.bytesPerDim;
            return this.scratch;
        }

        private void readNodeData(boolean isLeft) {
            this.leafBlockFPStack[this.level] = this.leafBlockFPStack[this.level - 1];
            if (!isLeft) {
                int n = this.level;
                this.leafBlockFPStack[n] = this.leafBlockFPStack[n] + this.in.readVLong();
            }
            if (this.isLeafNode()) {
                this.splitDim = -1;
            } else {
                int code = this.in.readVInt();
                this.splitDims[this.level] = this.splitDim = code % BKDReader.this.numDims;
                int prefix = (code /= BKDReader.this.numDims) % (1 + BKDReader.this.bytesPerDim);
                int suffix = BKDReader.this.bytesPerDim - prefix;
                if (this.splitValuesStack[this.level] == null) {
                    this.splitValuesStack[this.level] = new byte[BKDReader.this.packedBytesLength];
                }
                System.arraycopy(this.splitValuesStack[this.level - 1], 0, this.splitValuesStack[this.level], 0, BKDReader.this.packedBytesLength);
                if (suffix > 0) {
                    int firstDiffByteDelta = code / (1 + BKDReader.this.bytesPerDim);
                    if (this.negativeDeltas[this.level * BKDReader.this.numDims + this.splitDim]) {
                        firstDiffByteDelta = -firstDiffByteDelta;
                    }
                    int oldByte = this.splitValuesStack[this.level][this.splitDim * BKDReader.this.bytesPerDim + prefix] & 0xFF;
                    this.splitValuesStack[this.level][this.splitDim * BKDReader.this.bytesPerDim + prefix] = (byte)(oldByte + firstDiffByteDelta);
                    this.in.readBytes(this.splitValuesStack[this.level], this.splitDim * BKDReader.this.bytesPerDim + prefix + 1, suffix - 1);
                }
                int leftNumBytes = this.nodeID * 2 < BKDReader.this.leafNodeOffset ? this.in.readVInt() : 0;
                this.leftNodePositions[this.level] = this.in.getPosition();
                this.rightNodePositions[this.level] = this.leftNodePositions[this.level] + leftNumBytes;
            }
        }
    }
}

