/*
 * Copyright (c) 2011-2013, 2021 Eike Stepper (Loehne, Germany) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    Eike Stepper - initial API and implementation
 *    Simon McDuff - bug 201266
 *    Simon McDuff - bug 230832
 */
package org.eclipse.emf.cdo.internal.common.revision;

import org.eclipse.emf.cdo.common.branch.CDOBranch;
import org.eclipse.emf.cdo.common.id.CDOID;
import org.eclipse.emf.cdo.common.id.CDOIDUtil;
import org.eclipse.emf.cdo.common.revision.CDOIDAndBranch;
import org.eclipse.emf.cdo.common.revision.CDORevision;
import org.eclipse.emf.cdo.spi.common.revision.InternalCDORevisionCache;

import org.eclipse.emf.ecore.EClass;

import java.util.Map;

/**
 * @author Eike Stepper
 */
public class CDORevisionCacheBranching extends CDORevisionCacheAuditing
{
  private final Map<CDOID, TypeAndRefCounter> typeMap = CDOIDUtil.createMap();

  public CDORevisionCacheBranching()
  {
  }

  @Override
  public InternalCDORevisionCache instantiate(CDORevision revision)
  {
    return new CDORevisionCacheBranching();
  }

  @Override
  public EClass getObjectType(CDOID id)
  {
    synchronized (revisionLists)
    {
      TypeAndRefCounter typeCounter = typeMap.get(id);
      if (typeCounter != null)
      {
        return typeCounter.getType();
      }

      return null;
    }
  }

  @Override
  protected void typeRefIncrease(CDOID id, EClass type)
  {
    TypeAndRefCounter typeCounter = typeMap.get(id);
    if (typeCounter == null)
    {
      typeCounter = new TypeAndRefCounter(type);
      typeMap.put(id, typeCounter);
    }

    typeCounter.increase();
  }

  @Override
  protected void typeRefDecrease(CDOID id)
  {
    TypeAndRefCounter typeCounter = typeMap.get(id);
    if (typeCounter != null && typeCounter.decreaseAndGet() == 0)
    {
      typeMap.remove(id);
    }
  }

  @Override
  protected void typeRefDispose()
  {
    typeMap.clear();
  }

  @Override
  protected CDOID getID(Object key)
  {
    return ((CDOIDAndBranch)key).getID();
  }

  @Override
  protected boolean isKeyInBranch(Object key, CDOBranch branch)
  {
    return ((CDOIDAndBranch)key).getBranch() == branch;
  }

  @Override
  protected Object createKey(CDOID id, CDOBranch branch)
  {
    return CDOIDUtil.createIDAndBranch(id, branch);
  }

  /**
   * @author Eike Stepper
   */
  private static final class TypeAndRefCounter
  {
    private final EClass type;

    private int refCounter;

    public TypeAndRefCounter(EClass type)
    {
      this.type = type;
    }

    public EClass getType()
    {
      return type;
    }

    public void increase()
    {
      ++refCounter;
    }

    public int decreaseAndGet()
    {
      return --refCounter;
    }
  }
}
