package org.bouncycastle.crypto.params;

import org.bouncycastle.math.ec.ECPoint;

/**
 * Represents the public key parameters for the Elliptic Curve-based Certificateless
 * Signature Infrastructure (ECCSI) scheme as defined in RFC 6507.
 * <p>
 * This class encapsulates the Public Validation Token (PVT) required for verifying
 * ECCSI signatures. The PVT is cryptographically bound to a user's identity and
 * generated by the Key Management Service (KMS) as part of the key material.
 *
 * <p>Per RFC 6507 Section 5.1:
 * <ul>
 *   <li>The PVT is derived from the user's identity and KMS secret material</li>
 *   <li>Used during signature verification to validate the signer's identity</li>
 *   <li>Does not require certificates for authentication</li>
 * </ul>
 *
 * @see <a href="https://datatracker.ietf.org/doc/html/rfc6507">RFC 6507:  Elliptic Curve-Based Certificateless
 *  * Signatures for Identity-Based Encryption (ECCSI)</a>
 */

public class ECCSIPublicKeyParameters
    extends AsymmetricKeyParameter
{
    private final ECPoint pvt;

    /**
     * Constructs {@code ECCSIPublicKeyParameters} with the provided Public Validation Token (PVT).
     */
    public ECCSIPublicKeyParameters(ECPoint pvt)
    {
        super(false);
        this.pvt = pvt;
    }

    /**
     * Returns the Public Validation Token (PVT) for signature verification.
     * <p>
     * The PVT is used in conjunction with the KMS Public Authentication Key (KPAK)
     * to verify signatures per RFC 6507 Section 5.2.2.
     *
     * @return The PVT as an elliptic curve point in uncompressed format
     */
    public final ECPoint getPVT()
    {
        return pvt;
    }
}
