/*****************************************************************************
 * Copyright (c) 2023, 2025 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Issue GL-9
 *****************************************************************************/

package org.eclipse.papyrus.model2doc.markup.uml.template2structure.internal.mapping;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.model2doc.core.logger.IModel2DocLogger;
import org.eclipse.papyrus.model2doc.emf.documentstructure.BodyPart;
import org.eclipse.papyrus.model2doc.emf.documentstructure.InsertedGeneratedFile;
import org.eclipse.papyrus.model2doc.emf.template2structure.mapping.IMappingService;
import org.eclipse.papyrus.model2doc.markup.emf.template2structure.mapping.AbstractMarkdownToHtmlFileMapper;
import org.eclipse.papyrus.model2doc.markup.uml.documentstructuretemplate.CommentInMarkupToFile;
import org.eclipse.papyrus.model2doc.markup.uml.documentstructuretemplate.MarkupUMLDocumentStructureTemplatePackage;
import org.eclipse.uml2.uml.Comment;
import org.eclipse.uml2.uml.Element;

/**
 * This mapper convert the body of {@link Comment} written in markdown on in html into an html file.
 */
public class CommentInMarkupToHtmlFileMapper extends AbstractMarkdownToHtmlFileMapper<CommentInMarkupToFile> {

	/**
	 * Constructor.
	 *
	 * @param inputEClass
	 * @param outputClass
	 */
	public CommentInMarkupToHtmlFileMapper() {
		super(MarkupUMLDocumentStructureTemplatePackage.eINSTANCE.getCommentInMarkupToFile(), BodyPart.class);
	}

	/**
	 *
	 * @see org.eclipse.papyrus.model2doc.emf.template2structure.mapping.AbstractTemplateToStructureMapper#doMap(org.eclipse.papyrus.model2doc.emf.template2structure.mapping.IMappingService, org.eclipse.emf.ecore.EObject, org.eclipse.emf.ecore.EObject,
	 *      java.lang.Class, IModel2DocLogger)
	 *
	 * @param <T>
	 * @param mappingService
	 * @param commentInMarkupToFile
	 * @param semanticModelElement
	 * @param expectedReturnedClass
	 * @return
	 */
	@Override
	protected <T> List<T> doMap(final IMappingService mappingService, final CommentInMarkupToFile commentInMarkupToFile, final EObject semanticModelElement, final Class<T> expectedReturnedClass, final IModel2DocLogger logger) {
		if (false == commentInMarkupToFile.generateBranch(semanticModelElement)) {
			return Collections.emptyList();
		}
		if (false == commentInMarkupToFile.isGenerate() || false == semanticModelElement instanceof Element) {
			// This element can't have children, so if isGenerate() returns false, we go out!
			return Collections.emptyList();
		}
		final List<T> generatedElements = new ArrayList<>();

		final Iterator<Comment> commentIter = commentInMarkupToFile.getMatchingComments(semanticModelElement).iterator();

		while (commentIter.hasNext()) {
			final Comment comment = commentIter.next();
			final InsertedGeneratedFile generatedFile = STRUCTURE_EFACTORY.createInsertedGeneratedFile();
			final String elementID = getXMIID(comment);
			final StringBuilder fileNameBuilder = new StringBuilder();
			fileNameBuilder.append(comment.eClass().getName()).append(elementID);


			generatedFile.setFilePath(createHTMLFile(commentInMarkupToFile, comment.getBody(), fileNameBuilder.toString(), logger));
			generatedElements.add(expectedReturnedClass.cast(generatedFile));
		}

		return buildMapperResult(commentInMarkupToFile, semanticModelElement, expectedReturnedClass, generatedElements);
	}

}
