/*******************************************************************************
 * Copyright (c) 2006 - 2010 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.internal.ui.source;

import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.SafeRunner;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.mint.CodeGenStatus;
import org.eclipse.emf.mint.IMemberAnnotationListener;
import org.eclipse.emf.mint.IMemberAnnotationManager;
import org.eclipse.emf.mint.MemberAnnotationChangedEvent;
import org.eclipse.emf.mint.MintCore;
import org.eclipse.emf.mint.internal.ui.MintUI;
import org.eclipse.jdt.core.IInitializer;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.resource.ColorRegistry;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.util.SafeRunnable;
import org.eclipse.jface.viewers.IDecoration;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ILightweightLabelDecorator;
import org.eclipse.jface.viewers.LabelProviderChangedEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.progress.WorkbenchJob;
import org.eclipse.ui.themes.IThemeManager;

public class GeneratedElementDecorator implements ILightweightLabelDecorator,
		IMemberAnnotationListener, IPropertyChangeListener {

	private enum EffectiveCodeGenStatus {
		NONE, GENERATED, GENERATED_NOT, GENERATED_NOT_CHILDREN;
	}

	private final ListenerList listeners = new ListenerList();

	private IMemberAnnotationManager manager;

	private IThemeManager themeManager;

	private Color generatedColor;

	private Color generatedNotColor;

	private Color generatedNotChildrenColor;

	private ColorJob colorJob;

	private final Object colorLock = new Object();

	private final UpdateJob updateJob = new UpdateJob();

	public GeneratedElementDecorator() {
		manager = MintCore.getInstance().getMemberAnnotationManager();
		manager.addMemberAnnotationListener(this);
		themeManager = MintUI.getDefault().getWorkbench().getThemeManager();
		themeManager.addPropertyChangeListener(this);
	}

	public void decorate(Object element, IDecoration decoration) {
		if (!(element instanceof IMember))
			return;

		IMember member = (IMember) element;
		CodeGenStatus status = manager.getCodeGenStatus(member);
		EffectiveCodeGenStatus effectiveStatus;
		switch (status) {
		case GENERATED:
			effectiveStatus = EffectiveCodeGenStatus.GENERATED;
			break;
		case GENERATED_NOT:
			effectiveStatus = EffectiveCodeGenStatus.GENERATED_NOT;
			break;
		default:
			effectiveStatus = EffectiveCodeGenStatus.NONE;
		}

		if (status != CodeGenStatus.GENERATED_NOT) {
			try {
				for (IJavaElement child : member.getChildren()) {
					if (child instanceof IInitializer)
						continue;

					CodeGenStatus childStatus = manager
							.getCodeGenStatus((IMember) child);
					if (childStatus == CodeGenStatus.GENERATED_NOT) {
						effectiveStatus = EffectiveCodeGenStatus.GENERATED_NOT_CHILDREN;
						break;
					}
				}
			} catch (JavaModelException e) {
				MintCore.getInstance().logError(
						"Could not get element's children.", e); //$NON-NLS-1$
			}
		}

		decorate(effectiveStatus, decoration);
	}

	private void decorate(EffectiveCodeGenStatus status, IDecoration decoration) {
		Color color = null;
		switch (status) {
		case GENERATED:
			color = getGeneratedColor();
			break;
		case GENERATED_NOT:
			color = getGeneratedNotColor();
			break;
		case GENERATED_NOT_CHILDREN:
			color = getGeneratedNotChildrenColor();
			break;
		}

		if (color != null)
			decoration.setForegroundColor(color);
	}

	public void addListener(ILabelProviderListener listener) {
		listeners.add(listener);
	}

	public void dispose() {
		listeners.clear();
		if (!updateJob.cancel())
			try {
				updateJob.join();
			} catch (InterruptedException e) {
				// ignore
			}

		synchronized (colorLock) {
			if (colorJob != null && !colorJob.cancel())
				try {
					colorJob.join();
				} catch (InterruptedException e) {
					// ignore
				}
		}

		themeManager.removePropertyChangeListener(this);
		themeManager = null;
		manager.removeMemberAnnotationListener(this);
		manager = null;
	}

	public boolean isLabelProperty(Object element, String property) {
		return false;
	}

	public void removeListener(ILabelProviderListener listener) {
		listeners.remove(listener);
	}

	protected void fireLabelProviderChangedEvent(
			final LabelProviderChangedEvent event) {
		Object[] l = listeners.getListeners();
		for (int i = 0; i < l.length; ++i) {
			final ILabelProviderListener listener = (ILabelProviderListener) l[i];
			SafeRunner.run(new SafeRunnable() {
				public void run() throws Exception {
					listener.labelProviderChanged(event);
				}
			});
		}
	}

	public void memberAnnotationChanged(MemberAnnotationChangedEvent event) {
		HashSet<IMember> elements = new HashSet<IMember>(event.getChanges()
				.keySet());

		for (IMember member : event.getChanges().keySet()) {
			IType parent = member.getDeclaringType();
			if (parent != null)
				elements.add(parent);
		}

		if (Display.getCurrent() == null) {
			updateJob.addElements(elements);
		} else {
			fireLabelProviderChangedEvent(new LabelProviderChangedEvent(this,
					elements.toArray()));
		}
	}

	public void propertyChange(PropertyChangeEvent event) {
		if (MintUI.GENERATED_COLOR.equals(event.getProperty())
				|| MintUI.GENERATED_NOT_COLOR.equals(event.getProperty())
				|| MintUI.GENERATED_NOT_CHILDREN_COLOR.equals(event
						.getProperty())) {
			synchronized (colorLock) {
				if (colorJob == null)
					return;
			}

			colorJob.schedule();
		}
	}

	private Color getGeneratedColor() {
		synchronized (colorLock) {
			if (generatedColor == null) {
				if (colorJob == null)
					colorJob = new ColorJob();
			}
		}

		try {
			colorJob.join();
		} catch (InterruptedException e) {
			// ignore
		}

		return generatedColor;
	}

	private Color getGeneratedNotColor() {
		synchronized (colorLock) {
			if (generatedNotColor == null) {
				if (colorJob == null)
					colorJob = new ColorJob();
			}
		}

		try {
			colorJob.join();
		} catch (InterruptedException e) {
			// ignore
		}

		return generatedNotColor;
	}

	private Color getGeneratedNotChildrenColor() {
		synchronized (colorLock) {
			if (generatedNotChildrenColor == null) {
				if (colorJob == null)
					colorJob = new ColorJob();
			}
		}

		try {
			colorJob.join();
		} catch (InterruptedException e) {
			// ignore
		}

		return generatedNotChildrenColor;
	}

	private class UpdateJob extends WorkbenchJob {

		private final Object elementsLock = new Object();

		private HashSet<Object> elements;

		public UpdateJob() {
			super("update"); //$NON-NLS-1$
			setSystem(true);
		}

		public void addElements(Collection<?> elements) {
			synchronized (elementsLock) {
				if (this.elements == null) {
					this.elements = new HashSet<Object>(elements);
					schedule();
				} else if (this.elements.addAll(elements)) {
					schedule();
				}
			}
		}

		@Override
		public IStatus runInUIThread(IProgressMonitor monitor) {
			LinkedList<Object> elements;
			synchronized (this.elements) {
				elements = new LinkedList<Object>(this.elements);
				this.elements = null;
			}

			if (monitor.isCanceled())
				return Status.CANCEL_STATUS;

			if (!elements.isEmpty())
				fireLabelProviderChangedEvent(new LabelProviderChangedEvent(
						GeneratedElementDecorator.this, elements.toArray()));

			return Status.OK_STATUS;
		}
	}

	private class ColorJob extends WorkbenchJob {

		private boolean refresh;

		public ColorJob() {
			super(Messages.GeneratedElementDecorator_ThemeJob);
			setSystem(true);
			schedule();
		}

		@Override
		public IStatus runInUIThread(IProgressMonitor monitor) {
			if (themeManager == null)
				return Status.CANCEL_STATUS;

			ColorRegistry registry = themeManager.getCurrentTheme()
					.getColorRegistry();
			synchronized (colorLock) {
				generatedColor = registry.get(MintUI.GENERATED_COLOR);
				generatedNotColor = registry.get(MintUI.GENERATED_NOT_COLOR);
				generatedNotChildrenColor = registry
						.get(MintUI.GENERATED_NOT_CHILDREN_COLOR);
			}

			if (refresh)
				fireLabelProviderChangedEvent(new LabelProviderChangedEvent(
						GeneratedElementDecorator.this));

			refresh = true;
			return Status.OK_STATUS;
		}
	}
}