/*******************************************************************************
 * Copyright (c) 2007 - 2009 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.internal;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.ISafeRunnable;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.SafeRunner;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.mint.CodeGenStatus;
import org.eclipse.emf.mint.IMemberAnnotationListener;
import org.eclipse.emf.mint.IMemberAnnotationManager;
import org.eclipse.emf.mint.MemberAnnotationChangedEvent;
import org.eclipse.emf.mint.MintCore;
import org.eclipse.jdt.core.ElementChangedEvent;
import org.eclipse.jdt.core.IBuffer;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IElementChangedListener;
import org.eclipse.jdt.core.IInitializer;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaElementDelta;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IOpenable;
import org.eclipse.jdt.core.IParent;
import org.eclipse.jdt.core.ISourceRange;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;

public class MemberAnnotationManager implements IMemberAnnotationManager,
		IElementChangedListener {

	private static final Debug debug = new Debug("MemberAnnotationManager"); //$NON-NLS-1$

	private static final Debug debugCleaner = new Debug(
			"MemberAnnotationManager/CacheCleaner"); //$NON-NLS-1$

	private static final Pattern GENERATED = Pattern
			.compile("@generated([\\s]+(?i)NOT)?"); //$NON-NLS-1$

	private final ListenerList listeners = new ListenerList();

	private final Map<IMember, CodeGenStatus> statusCache = new HashMap<IMember, CodeGenStatus>();

	private final CacheCleaner cacheCleaner = new CacheCleaner();

	public MemberAnnotationManager() {
		JavaCore.addElementChangedListener(this);
	}

	public void addMemberAnnotationListener(IMemberAnnotationListener listener) {
		listeners.add(listener);
	}

	public void removeMemberAnnotationListener(
			IMemberAnnotationListener listener) {
		listeners.remove(listener);
	}

	public CodeGenStatus getCodeGenStatus(IMember member) {
		synchronized (statusCache) {
			CodeGenStatus status = statusCache.get(member);
			if (status == null) {
				status = getCodeGenStatusChecked(member);
				cacheStatus(member, status);
			}

			return status;
		}
	}

	private CodeGenStatus getCodeGenStatusChecked(IMember member) {
		CodeGenStatus value = CodeGenStatus.NONE;
		try {
			value = computeCodeGenStatus(member);
		} catch (JavaModelException e) {
			MintCore.getInstance().logError(
					Messages.MemberAnnotationManager_ErrorCodegenStatus, e);
		} catch (IOException e) {
			MintCore.getInstance().logError(
					Messages.MemberAnnotationManager_ErrorCodegenStatus, e);
		}

		return value;
	}

	public void elementChanged(ElementChangedEvent event) {
		IJavaElementDelta delta = event.getDelta();
		uncacheRemovedElements(delta);

		if (event.getType() == ElementChangedEvent.POST_CHANGE)
			return;

		if ((delta.getKind() & IJavaElementDelta.CHANGED) == 0)
			return;

		IJavaElement element = delta.getElement();
		if (element instanceof ICompilationUnit) {
			int flags = delta.getFlags();
			if ((flags & IJavaElementDelta.F_AST_AFFECTED) != 0) {
				ArrayList<IMember> members = new ArrayList<IMember>();
				collectMembers(element, members);
				Map<IMember, CodeGenStatus> changes = new HashMap<IMember, CodeGenStatus>();
				computeChanges(members, changes);
				if (!changes.isEmpty()) {
					fireMemberAnnotationChangedEvent(new MemberAnnotationChangedEvent(
							this, changes));

					if (debug.enabled)
						debug.trace("changes: %s", changes); //$NON-NLS-1$
				}
			}
		}
	}

	private void uncacheRemovedElements(IJavaElementDelta delta) {
		switch (delta.getKind()) {
		case IJavaElementDelta.CHANGED:
			if ((delta.getFlags() & IJavaElementDelta.F_CLOSED) != 0)
				cacheCleaner.schedule();
			else
				for (IJavaElementDelta childDelta : delta.getAffectedChildren())
					uncacheRemovedElements(childDelta);

			break;
		case IJavaElementDelta.REMOVED:
			cacheCleaner.schedule();
			break;
		}
	}

	private void collectMembers(IJavaElement element,
			Collection<IMember> collector) {
		if (element instanceof IMember)
			collector.add((IMember) element);

		if (!(element instanceof IParent))
			return;

		IJavaElement[] children;
		try {
			children = ((IParent) element).getChildren();
		} catch (JavaModelException e) {
			MintCore.getInstance().logError(
					Messages.MemberAnnotationManager_ErrorJavaModel, e);
			return;
		}

		for (IJavaElement child : children) {
			if (child instanceof IInitializer)
				continue;

			collectMembers(child, collector);
		}
	}

	private void computeChanges(Collection<IMember> members,
			Map<IMember, CodeGenStatus> changes) {
		synchronized (statusCache) {
			for (IMember member : members) {
				CodeGenStatus oldStatus = statusCache.get(member);
				if (oldStatus == null)
					continue;

				CodeGenStatus status = getCodeGenStatusChecked(member);
				if (status != oldStatus) {
					changes.put(member, status);
					cacheStatus(member, status);
				}
			}
		}
	}

	private void fireMemberAnnotationChangedEvent(
			final MemberAnnotationChangedEvent event) {
		for (Object obj : listeners.getListeners()) {
			final IMemberAnnotationListener listener = (IMemberAnnotationListener) obj;
			SafeRunner.run(new ISafeRunnable() {

				public void run() throws Exception {
					listener.memberAnnotationChanged(event);
				}

				public void handleException(Throwable e) {
					MintCore
							.getInstance()
							.logError(
									Messages.MemberAnnotationManager_ErrorNotifyListener,
									e);
				}
			});
		}
	}

	private CodeGenStatus computeCodeGenStatus(IMember member)
			throws JavaModelException, IOException {
		CodeGenStatus result = CodeGenStatus.NONE;
		if (!member.exists())
			return result;

		String javadoc = getJavadoc(member);
		if (javadoc == null)
			return result;

		Matcher m = GENERATED.matcher(javadoc);
		if (m.find())
			result = m.group(1) == null ? CodeGenStatus.GENERATED
					: CodeGenStatus.GENERATED_NOT;

		return result;
	}

	private String getJavadoc(IMember member) throws JavaModelException {
		IOpenable openable = member.getOpenable();
		if (openable == null)
			return null;

		IBuffer buf = openable.getBuffer();
		if (buf == null)
			return null;

		ISourceRange range = member.getJavadocRange();
		if (range != null)
			return buf.getText(range.getOffset(), range.getLength());

		return null;
	}

	private void cacheStatus(IMember member, CodeGenStatus status) {
		if (!member.isReadOnly()) {
			statusCache.put(member, status);

			if (debug.enabled)
				debug.trace("cached %s (%s)", member, status); //$NON-NLS-1$
		}
	}

	public void dispose() {
		cacheCleaner.cancel();
		JavaCore.removeElementChangedListener(this);
		listeners.clear();
	}

	private class CacheCleaner extends Job {

		public CacheCleaner() {
			super(Messages.MemberAnnotationManager_JobCacheCleaner);
			setSystem(true);
			setPriority(DECORATE);
		}

		protected IStatus run(IProgressMonitor monitor) {
			synchronized (statusCache) {
				for (Iterator<IMember> i = statusCache.keySet().iterator(); i
						.hasNext();) {
					if (monitor.isCanceled())
						return Status.CANCEL_STATUS;

					IMember member = i.next();
					if (!member.exists()) {
						i.remove();

						if (debugCleaner.enabled)
							debugCleaner.trace("removed %s", member); //$NON-NLS-1$
					}
				}
			}

			return Status.OK_STATUS;
		}
	}
}
