/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "Camera.h"
#include "TimeFunction.h"
#include <glm/gtx/rotate_vector.hpp>
#include <cstring>
#include <cmath>

namespace libgltf
{

const float PI = float(std::atan(1.0) * 4.0);

CPhysicalCamera::CPhysicalCamera()
    : fSensitivity(0.1f)
    , flength(0.0)
    , vModelCenterPos(0.0f)
    , mViewMatrix(1.0f)
    , mProjection(1.0f)
    , mbAerialView(false)
    , vSpeed(1.0f)
    , bMove_camera_flag(false)
    , mMoveTime(0.0)
    , mLastTimeStamp(0.0)
{
}

void CPhysicalCamera::moveCamera(double x, double y, double z,
                                 double time)
{
    if (std::abs(x) > 0.0001 || std::abs(y) > 0.0001 || std::abs(z) > 0.0001)
    {
        if (std::abs(time) > 0.0001)
        {
            bMove_camera_flag = true;
            mMoveTime = time;
            vSpeed = glm::vec3(x, y, z) / static_cast<float>(time);
            mLastTimeStamp = time::getCurrentTime();
        }
        else
        {
            if( mbAerialView )
            {
                glm::vec3 vEye, vView, vUp;
                getCameraPosVectors(&vEye, &vView, &vUp);
                mViewMatrix = glm::lookAt(vEye + glm::vec3(x, y, z), vView, vUp);
            }
            else
            {
                mViewMatrix = glm::translate(mViewMatrix, glm::vec3(-x, -y, -z));
            }
        }
    }
}

void CPhysicalCamera::setViewMatrix(const glm::mat4& rViewMatrix)
{
    mViewMatrix = rViewMatrix;
}

const glm::mat4& CPhysicalCamera::getViewMatrix() const
{
    return mViewMatrix;
}

void CPhysicalCamera::setAerialView( bool bAerialView )
{
    mbAerialView = bAerialView;
}

void CPhysicalCamera::getCameraPosVectors(glm::vec3* pEye, glm::vec3* pView, glm::vec3* pUp) const
{
    // Calculate eye, view and up verctor from the view matrix
    glm::mat4 inverseViewMatrix = glm::inverse(mViewMatrix);
    if( pEye || pView )
    {
        glm::vec3 vEye;
        vEye.x = inverseViewMatrix[3][0];
        vEye.y = inverseViewMatrix[3][1];
        vEye.z = inverseViewMatrix[3][2];

        if( pEye )
            *pEye = vEye;

        if( pView )
        {
            if( mbAerialView )
            {
                *pView = vModelCenterPos;
            }
            else
            {
                *pView = vEye + glm::vec3(-inverseViewMatrix[2][0],-inverseViewMatrix[2][1],-inverseViewMatrix[2][2]);
            }
        }
    }

    if( pUp )
    {
        pUp->x = inverseViewMatrix[1][0];
        pUp->y = inverseViewMatrix[1][1];
        pUp->z = inverseViewMatrix[1][2];
        *pUp = glm::normalize(*pUp);
    }
}

void CPhysicalCamera::setPerspective(const glm::mat4& Perspective)
{
    mProjection = Perspective;
}

const glm::mat4& CPhysicalCamera::getPerspective() const
{
    return mProjection;
}

void CPhysicalCamera::updateViewMatrix()
{
    if (bMove_camera_flag)
    {
        const double dCurrentTime = time::getCurrentTime();
        const double dTimeDifference = time::diffTime(dCurrentTime, mLastTimeStamp);
        mLastTimeStamp = dCurrentTime;
        glm::vec3 vMove;
        if( mMoveTime > dTimeDifference )
        {
            vMove = vSpeed * static_cast<float>(dTimeDifference);
        }
        else
        {
            bMove_camera_flag = false;
            vMove = vSpeed * mMoveTime;
        }

        if( mbAerialView )
        {
            glm::vec3 vEye, vView, vUp;
            getCameraPosVectors(&vEye, &vView, &vUp);
            mViewMatrix = glm::lookAt(vEye + vMove, vView, vUp);
        }
        else
        {
            mViewMatrix = glm::translate(mViewMatrix, -vMove);
        }
        mMoveTime -= dTimeDifference;
    }
}

void CPhysicalCamera::rotateObjectMouse(double horizontal, double vertical,
                                        double planar)
{
    if (std::abs(horizontal) > planar || std::abs(vertical) > 0.0001)
    {
        glm::vec3 vOrigView;
        getCameraPosVectors(0, &vOrigView, 0);
        mViewMatrix = glm::translate(mViewMatrix, vOrigView);

        glm::vec3 vEye, vView, vUp;
        getCameraPosVectors(&vEye, &vView, &vUp);

        // Rotate horizontally
        float deltaX = (float)(-horizontal * fSensitivity * 0.5);
        mViewMatrix = glm::rotate(mViewMatrix, deltaX, vUp);

        // Rotate vertically
        float deltaY = (float)(vertical * fSensitivity * 0.5);
        mViewMatrix = glm::rotate(mViewMatrix, deltaY, glm::cross(vEye-vView,vUp));

        mViewMatrix = glm::translate(mViewMatrix, -vOrigView);
    }
}

void CPhysicalCamera::rotateCamera(double horizontal, double vertical,
                                   double planar)
{
    if (std::abs(horizontal) > planar || std::abs(vertical) > 0.0001)
    {
        glm::vec3 vOrigEye;
        getCameraPosVectors(&vOrigEye, 0, 0);
        mViewMatrix = glm::translate(mViewMatrix, vOrigEye);

        glm::vec3 vEye, vView, vUp;
        getCameraPosVectors(&vEye, &vView, &vUp);

        // Rotate horizontally
        float deltaX = (float)(-horizontal * fSensitivity * 0.5);
        mViewMatrix = glm::rotate(mViewMatrix, deltaX, vUp);

        // Rotate vertically
        float deltaY = (float)(vertical * fSensitivity * 0.5);
        mViewMatrix = glm::rotate(mViewMatrix, deltaY, glm::cross(vEye-vView,vUp));

        mViewMatrix = glm::translate(mViewMatrix, -vOrigEye);
    }
}

} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
