use ui;
use graphics;
use core:geometry;
use progvis:data;

/**
 * An element that can be drawn to the screen. Saves its position from the last paint so that it is
 * possible to interact with it at a later stage without having to re-implement the layout logic.
 */
class Drawable on Render {
	init() {
		init() {
			pos = Point();
		}
	}

	// Current position.
	Point pos;

	// Current size. Computed on demand.
	Size size() {
		if (cachedSize) {
			cachedSize;
		} else {
			Size s = computeSize();
			cachedSize = s;
			s;
		}
	}

	// Compute the bounding rectangle of this data element.
	Rect rect() {
		Rect(pos, size);
	}

	// Draw.
	void draw(Graphics g, Bool active) : abstract;

	// Draw links.
	void drawLinks(Graphics g, Bool active) : abstract;

	// Called when this object is no longer reachable to determine if it is ready to be removed.
	Bool remove() { true; }

	// Get the data that corresponds to this drawable, if any.
	Data? srcData() : abstract;

	// Update the data in this drawable.
	void update(Workspace:Update ws) : abstract;

	// Cached size.
	private Size? cachedSize;

	// Set the size.
	assign size(Size s) {
		cachedSize = s;
		s;
	}

	// Request a size change.
	void resize(Size s) {
		cachedSize = s;
	}

	// Clear the size.
	protected void invalidateSize() {
		cachedSize = null;
	}

	// Override to compute the size.
	protected Size computeSize() { Size(); }

	// Find a sub-drawable, if applicable. 'pt' is relative to the world coordinates.
	Drawable findAt(Point pt) {
		this;
	}

	// Called when the mouse is pressed inside this drawable (return true for repaint).
	Bool mouseClicked(Point pt, Bool down) { false; }

	// Called when the mouse is moved inside this drawable. "origin" is the location (in world space) where it was pressed.
	Bool mouseMoved(Point pt, Point origin) { false; }

	// Get the contained drawable, if we're a wrapper around an object.
	Drawable contained() { this; }

	// Is this element docked in the main window?
	Bool docked() { false; }

}

/**
 * Drawable that draws a border around the contents.
 */
class WithBorder extends Drawable {
	void draw(Graphics g, Bool active) : override {
		Rect r = rect;
		g.fill(r, dataBg);
		g.draw(r, dataFg);
	}
}


/**
 * Common object for all drawables that wrap other drawables.
 */
class WrapDrawable extends Drawable {
	// Create.
	init(Drawable wrap) {
		init() {
			wrap = wrap;
		}
	}

	// Wrapped object.
	Drawable wrap;

	// Same data?
	Data? srcData() : override {
		wrap.srcData();
	}

	// Update.
	void update(Workspace:Update ws) : override {
		wrap.update(ws);
	}

	// Draw.
	void draw(Graphics g, Bool active) : override {
		wrap.pos = pos;
		wrap.draw(g, active);
	}

	// Draw links.
	void drawLinks(Graphics g, Bool active) : override {
		wrap.drawLinks(g, active);
	}

	// Set and get size.
	Size size() : override {
		wrap.size();
	}

	assign size(Size s) : override {
		wrap.size(s);
	}

	// Remove?
	Bool remove() : override {
		wrap.remove();
	}

	// Get contained element.
	Drawable contained() : override {
		wrap.contained();
	}

	// Get sub-element.
	Drawable findAt(Point pt) : override {
		wrap.findAt(pt);
	}

}

// Insert a new wrap object at the innermost position.
Drawable inject(Drawable into, fn(Drawable)->WrapDrawable insert) on Render {
	if (into as WrapDrawable) {
		WrapDrawable curr = into;
		while (next = curr.wrap as WrapDrawable)
			curr = next;

		curr.wrap = insert.call(curr.wrap);
		into;
	} else {
		insert.call(into);
	}
}


/**
 * Drawable class that wraps another drawable to give it a caption.
 */
class WithCaption extends WrapDrawable {
	init(Drawable wrap, Str caption) {
		Text c(caption, captionFont);

		init(wrap) {
			caption = c;
			captionSz = c.size + compositeBorder*2;
		}
	}

	Size size() : override {
		Size sz = wrap.size();
		sz.w = max(sz.w, captionSz.w);
		sz.h += captionSz.h;
		sz;
	}

	assign size(Size s) : override {
		wrap.size(Size(s.w, s.h - captionSz.h));
		s;
	}

	void draw(Graphics g, Bool active) : override {
		Size cSz = captionSz;
		Size wSz = wrap.size;
		cSz.w = max(cSz.w, wSz.w);
		g.fill(Rect(pos, cSz), if (active) { captionActiveBg; } else { captionBg; });
		if (unreachable)
			g.fill(Rect(pos, cSz), captionUnreachableOverlay);
		g.draw(caption, captionFg, pos + compositeBorder);
		g.draw(Rect(pos, cSz), captionFg);

		wrap.size(Size(cSz.w, wSz.h));

		wrap.pos = pos + Size(0, captionSz.h);
		wrap.draw(g, active);
	}

	// Caption text.
	private Text caption;

	// Size of the caption area.
	protected Size captionSz;

	// Unreachable?
	protected Bool unreachable;

	// Get the size of the caption.
	Size captionSize() { captionSz; }

	// Return the sub-drawable if the click is there.
	Drawable findAt(Point pt) : override {
		if (wrap.rect.contains(pt))
			wrap.findAt(pt);
		else
			this;
	}

	// See if not reachable?
	void update(Workspace:Update ws) : override {
		unreachable = ws.currentUnreachable();
		super:update(ws);
	}
}


/**
 * Movable version of the with caption.
 */
class Movable extends WithCaption {
	init(Drawable wrap, Str caption) {
		init(wrap, caption) {}
	}

	private Point? mouseOrigin;

	Bool mouseClicked(Point pt, Bool down) {
		if (down) {
			mouseOrigin = pt - pos;
		} else {
			mouseOrigin = null;
		}
		false;
	}

	Bool mouseMoved(Point pt, Point origin) {
		if (mouseOrigin) {
			pos = pt - mouseOrigin;
			return true;
		}

		false;
	}
}


/**
 * Extension of the WithCaption class that allows docking to the viewport.
 */
class Dockable extends WithCaption {
	init(Drawable wrap, Str caption) {
		init(wrap, caption) {
			lockedPath = lockDecoration();
			unlockedPath = unlockDecoration();
			locked = true;
		}
	}

	void draw(Graphics g, Bool active) : override {
		if (anchorSize) {
			Float newHeight = wrap.size.h;
			if (!locked)
				pos.y += anchorSize - newHeight;
			this.anchorSize = newHeight;
		}

		super:draw(g, active);

		Rect button = buttonRect();
		g.draw(button, captionFg);

		Path decoration = if (locked) { lockedPath; } else { unlockedPath; };

		Rect bound = decoration.bound;
		g.push();
		g.transform(translate(button.p0 - bound.p0 + (button.size - bound.size) / 2));
		// Scale as well?
		g.fill(decoration, captionFg);
		g.pop();

		if (!locked) {
			Point edge = rect.p1;
			for (Int i = 5; i < resizeZone; i += 6) {
				g.line(edge - Point(i.float, 0), edge - Point(0, i.float), captionFg);
			}
		}
	}

	private Rect buttonRect() {
		Size sz = size();
		Point p(pos.x + sz.w - captionSz.h, pos.y);
		Rect(p, Size(captionSz.h)).shrink(Size(4));
	}

	// Anchor this item to the bottom left rather than the top left. This means that if the wrapped
	// item changes its size, the bottom left corner is stationary rather than the top left.
	Dockable anchorBottom() {
		anchorSize = wrap.size().h;
		this;
	}

	// Allow resizing this rectangle.
	Dockable allowResize() {
		resizeZone = 30;
		this;
	}

	private Path lockedPath;
	private Path unlockedPath;
	private Int resizeZone;
	private Bool locked;
	private Point? moveOrigin;
	private Point? resizeOrigin;
	private Float? anchorSize;

	Bool mouseClicked(Point pt, Bool down) : override {
		if (down) {
			Rect button = buttonRect();
			if (button.contains(pt)) {
				locked = !locked;
				return true;
			} else if (pt.y <= wrap.pos.y) {
				moveOrigin = pt - pos;
			} else {
				resizeOrigin = rect.p1 - pt;
			}
		} else {
			moveOrigin = null;
			resizeOrigin = null;
		}

		wrap.mouseClicked(pt, down);
	}

	Bool mouseMoved(Point pt, Point origin) : override {
		if (moveOrigin) {
			var newPos = pt - moveOrigin;
			if (locked) {
				if ((newPos - pos).lengthSq > 1000) {
					pos = newPos;
					locked = false;
				}
			} else {
				pos = pt - moveOrigin;
			}
			return true;
		} else if (resizeOrigin) {
			Size sz = (pt - wrap.pos) + resizeOrigin;
			sz = sz.max(captionSz + Size(resizeZone.float));
			wrap.resize(sz);
			if (anchorSize.any)
				anchorSize = wrap.size.h;
			return true;
		}

		false;
	}

	Drawable findAt(Point pt) : override {
		Rect wrapRect = wrap.rect;
		if (wrapRect.contains(pt)) {
			if (!locked & ((wrapRect.p1 - pt).taxiLength < resizeZone.float)) {
				return this;
			} else {
				wrap.findAt(pt);
			}
		} else {
			this;
		}
	}

	Bool docked() : override {
		locked;
	}

}
