use parser;
use core:io;

Buffer? Bytes(RecursiveBinaryState state, Nat count) {
	Nat from = state.pos;
	Nat to = from + count;
	if (to > state.input.count) {
		state.error("Expected ${count} bytes, but only ${state.input.count - from} remain.");
		return null;
	}

	state.pos = to;
	state.input.cut(from, to);
}

Str? Chars(RecursiveStrState state, Nat count) {
	if (count == 0)
		return "";

	// Step most codepoints in C++.
	var start = state.pos;
	var iter = start + (count - 1);
	var end = iter + 1;

	// If we failed to go the last step, then we did not have enough.
	if (iter == end) {
		state.error("Expected ${count} codepoints, but reached end of the input too early.");
		return null;
	}

	state.pos = end;
	state.input.cut(start, end);
}

private Nat findIndent(RecursiveStrState state, Nat tabWidth) {
	Nat width = 0;
	Str:Iter pos = state.pos;
	do {
		var ch = pos.v;
		if (ch == ' ') {
			width++;
		} else if (ch == '\t') {
			width += tabWidth;
		} else {
			state.pos = pos;
			return width;
		}

		++pos;
	}
}

Nat? Indent(RecursiveStrState state) {
	IndentTab(state, 4);
}

Nat? IndentTab(RecursiveStrState state, Nat tabWidth) {
	findIndent(state, tabWidth);
}

Nat? MinIndent(RecursiveStrState state, Nat indent) {
	MinIndentTab(state, 4, indent);
}

Nat? MinIndentTab(RecursiveStrState state, Nat tabWidth, Nat indent) {
	Nat i = findIndent(state, tabWidth);
	if (i < indent)
		return null;
	else
		return i;
}

Bool ExactIndent(RecursiveStrState state, Nat indent) {
	ExactIndentTab(state, 4, indent);
}

Bool ExactIndentTab(RecursiveStrState state, Nat tabWidth, Nat indent) {
	Nat consumed = 0;
	Str:Iter pos = state.pos;
	Str:Iter end = state.input.end;
	while (consumed < indent) {
		if (pos == end)
			return false;

		var ch = pos.v;
		if (ch == ' ')
			consumed++;
		else if (ch == '\t')
			consumed += tabWidth;
		else {
			return false;
		}

		++pos;
	}

	state.pos = pos;
	return true;
}

Bool enough(RecursiveBinaryState state, Nat count) {
	if (state.pos + count > state.input.count) {
		state.error("Expected ${count} bytes, but only ${state.input.count} remain.");
		return false;
	} else {
		return true;
	}
}

Byte? Nat8(RecursiveBinaryState this) {
	if (!enough(this, 1))
		return null;

	return input[pos++];
}

Nat? Nat16LE(RecursiveBinaryState this) {
	if (!enough(this, 2))
		return null;

	Nat result = input[pos++].nat;
	result |= input[pos++].nat << 8b;
	return result;
}

Nat? Nat16BE(RecursiveBinaryState this) {
	if (!enough(this, 2))
		return null;

	Nat result = input[pos++].nat << 8;
	result |= input[pos++].nat;
	return result;
}

Nat? Nat32LE(RecursiveBinaryState this) {
	if (!enough(this, 4))
		return null;

	Nat result = input[pos++].nat;
	result |= input[pos++].nat << 8;
	result |= input[pos++].nat << 16;
	result |= input[pos++].nat << 24;
	return result;
}

Nat? Nat32BE(RecursiveBinaryState this) {
	if (!enough(this, 4))
		return null;

	Nat result = input[pos++].nat << 24;
	result |= input[pos++].nat << 16;
	result |= input[pos++].nat << 8;
	result |= input[pos++].nat;
	return result;
}

Int? Int16LE(RecursiveBinaryState this) {
	unless (v = Nat16LE(this))
		return null;

	// Sign extend.
	if (v > 0x8000)
		v |= 0xFFFF0000;
	v.int;
}

Int? Int16BE(RecursiveBinaryState this) {
	unless (v = Nat16BE(this))
		return null;

	// Sign extend.
	if (v > 0x8000)
		v |= 0xFFFF0000;
	v.int;
}

Int? Int32LE(RecursiveBinaryState this) {
	unless (v = Nat32BE(this))
		return null;
	v.int;
}

Int? Int32BE(RecursiveBinaryState this) {
	unless (v = Nat32BE(this))
		return null;
	v.int;
}
