/*
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * This routine does a series of sanity checks on a pointer.  Returns
 * zero if no errors.  no-zero if something is wrong.  It also prints
 * a bogosity message on an error.
 */
#include "memdebug.h"

/*
 * Incase the (l)user doesn't #include<memdebug.h>, this will
 * allow them to make calls without the macro, and from
 * within other memdebug functions.
 */
int memdebug_ptrchk(void *ptr)
{
	return memdebug_traced_ptrchk(ptr, 0, 0);
}

int memdebug_traced_ptrchk(void *ptr, char *file, int line)
{
	memdebug_mhead *head;
	memdebug_mtail *tail;
	int rc = 0;
	int i;
		
	DPRINTF("checking pointer 0x%p\n", ptr);

	if (ptr == 0)
	{
		file_info_print("\n\nPOINTER CHECK: ptr is NULL ", file, line);
		return -1;
	}

	/* Find and check the memory block header.  */
	head = (memdebug_mhead*)ptr - 1;
	if (head->magic == FREE_MAGIC)
	{
		file_info_print("\n\nPOINTER CHECK:"
				" block has been free()'d. ",
				file, line);
		file_info_print("at ", file, line);
		memdebug_bogosity(head);
		rc = -1;
	}
	if (head->magic != HEAD_MAGIC)
	{
		memdebug_printf("\n\nPOINTER CHECK:"
				" Bogus pointer (head magic wrong.) ");
		file_info_print("@ ", file, line);
		memdebug_bogosity(head);
		return -1;
	}
	for (i = 0; i < MHEAD_DEADBEEF; i++)
	{
		if (head->deadbeef[i] != 0xdeadbeef)
		{
			memdebug_printf("\n\nPOINTER CHECK:"
					" Data written below beginning of block. ");
			file_info_print("@ ", file, line);
			memdebug_bogosity(head);
			rc = -1;
		}
	}

	/* Find and check the memory block tailer.  */
	tail = (memdebug_mtail*)((void*)(head + 1) + head->size);
	if (tail->magic != TAIL_MAGIC)
	{
		memdebug_printf("\n\nPOINTER CHECK:"
				" Trailer trashed or not where expected. ");
		file_info_print("@ ", file, line);
		memdebug_bogosity(head);
		rc = -1;
	}
	if (tail->size != head->size)
	{
		memdebug_printf("\n\nPOINTER CHECK:"
				" Head and tail sizes checks don't match. ");
		file_info_print("@ ", file, line);
		memdebug_bogosity(head);
		rc = -1;
	}
	for (i = 0; i < MTAIL_DEADBEEF; i++)
	{
		if (tail->deadbeef[i] != 0xdeadbeef)
		{
			memdebug_printf("\n\nPOINTER CHECK:"
					" Data written past end of block. ");
			file_info_print("@ ", file, line);
			memdebug_bogosity(head);
			rc = -1;
		}
	}

	return rc;
}

