/*
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Default implementation of memory routines.
 */

#include <flux/c/malloc.h>
#include <flux/c/stdlib.h>
#include <flux/fdev.h>
#include <flux/lmm.h>
#include <flux/x86/base_vm.h>
#include <flux/x86/pc/phys_lmm.h>

#define AUTO_SIZE_LIST_SIZE	200

struct auto_size_region {
	void *addr;
	unsigned size;
};

/*
 * List of auto-sized regions.
 *
 * XXX: Needs to be dynamic.
 */
static struct auto_size_region auto_size_list[AUTO_SIZE_LIST_SIZE];

/*
 * Allocate memory.
 */
void *
fdev_mem_alloc(unsigned size, int flags, unsigned align)
{
	int i, lmm_flags, align_bits;
	unsigned n;
	void *p;

	if (size == 0)
		return (NULL);
	align_bits = 0;
	if (align > 1) {
		for (n = align; n > 1; n >>= 1, align_bits++)
			;
	}
	lmm_flags = 0;
	if (flags & FDEV_ISA_DMA_MEMORY)
		lmm_flags |= LMMF_16MB;
	p = lmm_alloc_aligned(&malloc_lmm, size, lmm_flags, align_bits, 0);
	if (align > 1 && ((unsigned)p & (align - 1)) != 0)
		panic("%s:%d: memory not aligned", __FILE__, __LINE__);
	if ((flags & FDEV_AUTO_SIZE) == 0 || !p)
		return (p);
	for (i = 0; i < AUTO_SIZE_LIST_SIZE; i++) {
		if (!auto_size_list[i].addr) {
			auto_size_list[i].addr = p;
			auto_size_list[i].size = size;
			return (p);
		}
	}
	lmm_free(&malloc_lmm, p, size);
	return (NULL);
}

/*
 * Free memory.
 */
void
fdev_mem_free(void *p, int flags, unsigned size)
{
	int i;

	if (flags & FDEV_AUTO_SIZE) {
		for (i = 0; i < AUTO_SIZE_LIST_SIZE; i++) {
			if (auto_size_list[i].addr == p) {
				size = auto_size_list[i].size;
				auto_size_list[i].addr = 0;
				break;
			}
		}
	}
	lmm_free(&malloc_lmm, p, size);
}

/*
 * Map physical memory into kernel virtual space.
 */
int
fdev_map_phys_mem(vm_offset_t pa, vm_size_t size, void **addr, int flags)
{
	*addr = (void *)phystokv(pa);
	return (0);
}
