/* 
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Example mini-kernel that boots from a MultiBoot boot loader
 * and prints out the multiboot info and info about the cpu.
 */

#include <flux/x86/pc/reset.h>
#include <flux/x86/pc/direct_cons.h>
#include <flux/x86/pc/base_multiboot.h>
#include <flux/x86/multiboot.h>
#include <flux/x86/base_cpu.h>
#include <flux/debug.h>
#include <stdio.h>

/*
 * This function defines our kernel "console device";
 * the calls to printf() below will resolve to calls to putchar().
 * Our implementation simply writes characters to the local PC display.
 * If printing a character at a time was too inefficient for some reason
 * (not the case here, but could be the case in other kernels),
 * we could also override the default puts() function,
 * which printf() uses to print entire lines of text.
 */
int putchar(int c)
{
	direct_cons_putchar(c);
	return c;
}

/*
 * This function defines how to "exit" the kernel;
 * the return from main() below will eventually wind up here.
 * This implementation of _exit() simply reboots the machine
 * after giving the user a chance to see any screen output.
 */
void _exit(int rc)
{
	printf("\nPress any key to reboot...\n");
	direct_cons_getchar();
	pc_reset();
}

/*
 * The MultiBoot startup code will call this function
 * after setting up the base environment.
 * The kernel command line string passed by the boot loader
 * will have been parsed into separate argv option strings,
 * and options of the form FOO=BAR will be separated out
 * into environment variables,
 * which can be accessed using getenv() as in ordinary programs.
 */
int main(int argc, char **argv)
{
	extern char **environ;
	unsigned i;

	printf("Hello World!\n");

	printf("\nI was given this command line and environment:\n");
	for (i = 0; i < argc; i++)
		printf("  argv[%d]: `%s'\n", i, argv[i]);
	for (i = 0; environ[i]; i++)
		printf("  environ[%d]: `%s'\n", i, environ[i]);

	multiboot_info_dump();

	printf("\nI'm running on a...\n");
	cpu_info_dump(&base_cpuid);

	return 0;	/* just to kill warning... */
}

