/*
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 *
 */

/*
 * This is the critical_section code.  It is NOT designed to provide
 * a locking mechanism for general use; however, it does provide a
 * simple recursive lock which can be used for debugging and speed-
 * insensitive code (such as puts/putc).  Critical sections should 
 * have their own locks, rather than all sharing a common lock.
 */

#include <flux/smp/smp.h>
#include <flux/x86/smp/smp.h>
#include <flux/base_critical.h>
#include <flux/x86/proc_reg.h>
#include <flux/x86/smp/linux-smp.h>

/* This also acts as the spin lock; -1 means unlocked. */
static volatile unsigned critical_cpu_id = -1;

static unsigned critical_nest_count;
static unsigned critical_saved_eflags;

extern unsigned int num_processors;

void base_critical_enter(void)
{
	unsigned old_eflags = get_eflags();
	unsigned cpu_id = (num_processors > 1) ? smp_find_cur_cpu() : 0;

	/* First make sure we get no interference from interrupt activity. */
	cli();

	/* If we already own the lock, just increment the count and return. */
	if (critical_cpu_id == cpu_id) {
		critical_nest_count++;
		return;
	}

	/* Lock the global spin lock, waiting if another processor has it. */
	asm volatile("1: movl $-1,%%eax; lock; cmpxchgl %0,(%1); jne 1b"
		     : : "r" (cpu_id), "r" (&critical_cpu_id) : "eax");

	critical_nest_count = 0;
	critical_saved_eflags = old_eflags;
}

void base_critical_leave(void)
{
	unsigned old_eflags;

	if (critical_nest_count > 0) {
		critical_nest_count--;
		return;
	}

	old_eflags = critical_saved_eflags;

	/* Unlock the global spin lock. */
	asm volatile("movl %0,(%1)"
		     : : "r" (-1), "r" (&critical_cpu_id));

	set_eflags(old_eflags);
}

