/*
 * Copyright (c) 1994 The University of Utah and
 * the Computer Systems Laboratory (CSL).  All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * This file implements the core routines in setting up the fence posts,
 * checking them, and complaining when something goes wrong.
 */

#include <stdio.h>
#include <assert.h>
#include <flux/c/string.h>
#include <flux/lmm.h>
#include <stdarg.h>

#include "memdebug.h"

memdebug_mhead memdebug_all_head;

#if NO_MEM_FATAL
#define RETURN_NULL panic("%s:%d malloc failure.\n", file ? file : "N/A", line)
#else
#define RETURN_NULL {file_info_print("Ran out of memory @", file, line); return 0;}
#endif

/*
 * static file_info_print()
 *
 * Consistently print out the file info, if we have any.
 */
void 
file_info_print(const char *str, const char *file, int line)
{
	if (file)
		memdebug_printf("%s %s::%d\n",
				str, file, line);
	else
		memdebug_printf("%s unknown location\n", str);
}

/*
 * memdebug_bogosity()
 *
 * For error messages generated in this package.  Prints the backtrace
 * stored with each allocation.
 */
void 
memdebug_bogosity(memdebug_mhead *head)
{
	if (head)
	{
		unsigned i;

		file_info_print("Block allocated by", head->file, head->line);
		memdebug_printf("Block addr= %p, a %d byte malloc.\nBacktrace:\n",
				head + 1, head->size);
		
		for (i = 0; i < MHEAD_BTLEN; i++)
		{
			if (head->backtrace[i] == 0)
				break;
			/*
			 * This is formated to match the output of
			 * stack_trace.c
			 */
			memdebug_printf("%d] eip=0x%08x\n", i, head->backtrace[i]); 
		}
	}
	/* memdebug_printf("\n"); */

	/* XXX fflush(stdout); */
}

/*
 * memdebug_malloc()
 *
 * Do a memory allocation, setup the fence posts, and record all the
 * information.
 * This is written for the most restrictive case, an smemalign().
 * All of the other memory allocations can be written in terms of this.
 * (And they are..)
 */
void* 
memdebug_alloc(size_t alignment, size_t bytes, char initval, char caller,
	       char *file, int line)
{
	memdebug_mhead *head = NULL;
	memdebug_mtail *tail = NULL;
	size_t req_size;
	size_t req_align;
	unsigned align_shift;
	unsigned *ebp;
	unsigned i;

	assert((caller == SMALLOC_CALLER)
	       || (caller == MALLOC_CALLER));

	DPRINTF(" align= %d, size= %d, init= %2.2x\n",
		alignment, bytes, initval);
	DPRINTF(" file = %s, line = %d\n",
		file ? file : "N/A", file ? line : 0);
	if (bytes <= 0)
	{
		memdebug_printf("\n\nMEMDEBUG BOGOSITY: Tried to allocate chunk with size %d", 
				bytes);
		file_info_print(" at", file, line);
		RETURN_NULL;
	}

	if (alignment < 0)
	{
		memdebug_printf("\n\nMEMDEBUG BOGOSITY: Tried to allocate chunk with alignment %d",
				alignment);
		file_info_print(" at", file, line);
		RETURN_NULL;
	}
	
	
	/*
	 * Compute size of allocation with header and tail.  Figure
	 * out the alignment (if its less than the size of the
	 * memdebug_head, and non-zero, we have to align to the 
	 * aligned memdebug_mhead size.
	 */
	req_size = bytes + sizeof(memdebug_mhead) + sizeof(memdebug_mhead);

	/* Double check the value of NP2_SZ_MHEAD */
	assert(((sizeof(memdebug_mhead) << 1) > NP2_SZ_MHEAD)
	       && ((sizeof(memdebug_mhead) >> 1) < NP2_SZ_MHEAD));

	if (alignment && (alignment < NP2_SZ_MHEAD))
		req_align = NP2_SZ_MHEAD;
	else
		req_align = alignment;

	assert((req_align == 0)  
	       || ((int)req_align - (int)sizeof(memdebug_mhead)) > 0);
			
	/* Find the alignment shift in bits.  XXX use proc_ops.h  */
	for (align_shift = 0; (1 << align_shift) < req_align; align_shift++)
		/* nothing */;

	DPRINTF("req_size= %d, req_align= %d\n",
		req_size, req_align);
	DPRINTF("(align= %d; min_align= %d)\n",
		alignment, (int)NP2_SZ_MHEAD);
	DPRINTF("align_shift= %d, hd offset = %d\n",
		align_shift, req_align - sizeof(memdebug_mhead));
	mem_lock();
	while (!(head = lmm_alloc_aligned(&malloc_lmm, req_size, 0, align_shift,
					  req_align ? req_align - sizeof(memdebug_mhead) : 0)))
	{
		mem_unlock();
		DPRINTF("alloc failed, morecore()'ing\n");
		if (!morecore(req_size * 2))
		{
			RETURN_NULL;
		}
		mem_lock();
	}
	mem_unlock();

	/* Initialize the memory header.  */
	head->magic = (unsigned)(HEAD_MAGIC);
	head->size = bytes;
	head->flags = NO_FLAGS;
	for (i = 0; i < MHEAD_DEADBEEF; i++)
		head->deadbeef[i] = 0xdeadbeef;
	head->file = file;
	head->line = line;
	if (caller == SMALLOC_CALLER)
		head->flags |= SMALLOC_CREATED;
	else
		head->flags &= ~SMALLOC_CREATED;


	/*
	 * Add it to the global memory list. Use mem locks to keep it
	 * thread-safe.
	 */
	mem_lock();
	if (memdebug_all_head.next == 0)
	{
		memdebug_all_head.next = memdebug_all_head.prev = &memdebug_all_head;
	}
	head->next = &memdebug_all_head;
	head->prev = memdebug_all_head.prev;
	head->prev->next = head;
	memdebug_all_head.prev = head;
	mem_unlock();
	
	/*
	 * Create a backtrace of the stack and store it in the header.  
	 * XXX this is x86 specific.  The machine dependant
	 * stack_trace.c should be modified to return a trace log.
	 */
	asm volatile("movl %%ebp,%0" : "=r" (ebp));
	for (i = 0; i < MHEAD_BTLEN; i++)
	{
		if (ebp)
		{
			head->backtrace[i] = ebp[1];
			ebp = (unsigned*)ebp[0];
		}
		else
			head->backtrace[i] = 0;
	}

	/* Initialize the memory tailer.  */
	tail = (memdebug_mtail*)((void*)(head + 1) + bytes);
	for (i = 0; i < MTAIL_DEADBEEF; i++)
		tail->deadbeef[i] = 0xdeadbeef;
	tail->size = bytes;
	tail->magic = (unsigned)(TAIL_MAGIC);

	/* Initialize the memory block itself,
	   either to garbage or to zeros, as appropriate.  */
	memset(head + 1, initval, bytes);

	DPRINTF("Done\n");
	return head + 1;
}

/*
 * memdebug_free()
 *
 * returns -1 if there was an error.  0 otherwise.
 */
int
memdebug_free(void* mem, int wipeval, char caller, int blocksize, char *file, int line)
{
	memdebug_mhead *head;
	memdebug_mtail *tail;

	if (memdebug_traced_ptrchk(mem, file, line) != 0)
	{
		file_info_print("NOTE: detected during ``free()'' @ ", file, line);
		return -1;
	}

	head = (memdebug_mhead*)mem - 1;
	tail = (memdebug_mtail*)((void*)(head + 1) + head->size);

	DPRINTF("Doing free()-specific checks.\n");
	/*
	 * Make sure the user isn't smalloc()'ing and free()'ing or
	 * vice-versa.
	 */
	if ((caller == SMALLOC_CALLER) &&
	    (!(head->flags & SMALLOC_CREATED)))
	{
		memdebug_printf("\n\nMEMDEBUG BOGOSITY:"
				" sfree() of block allocated by malloc() ");
		file_info_print("@ ", file, line);
		memdebug_bogosity(head);
		/* Don't return, we can continue and do the correct thing. */
	}
	else if ((caller == MALLOC_CALLER)
		 && (head->flags & SMALLOC_CREATED))
	{
		memdebug_printf("\n\nMEMDEBUG BOGOSITY:"
				" free() of block allocated by smalloc() ");
		file_info_print("@ ", file, line);
		memdebug_bogosity(head);
		/* Don't return, we can continue and do the correct thing. */
	}

	/*
	 * If this is an sfree() call, then check to make sure the
	 * user is passing in the correct size.
	 */
	if ((blocksize != -1) && (blocksize != head->size))
	{
		memdebug_printf("\n\nMEMDEBUG BOGOSITY:"
				" sfree()'ing a block with incorrect size ");
		file_info_print("@ ", file, line);
		memdebug_bogosity(head);
		/* Don't return, we can continue and do the correct thing. */
	}

	/* Mark the block freed.  */
	head->magic = FREE_MAGIC;
	tail->magic = FREE_MAGIC;

	/*
	 * Fill the freed block with bogus data to reveal reads from
	 * freed memory blocks.
	 */
	memset(mem, wipeval, head->size);

	/* Remove the block from the allocated list.  Keep thread-safe.*/
	mem_lock();
	head->next->prev = head->prev;
	head->prev->next = head->next;
	mem_unlock();
	
	/* Free it.  */
	mem_lock();
	lmm_free(&malloc_lmm, head,
		 head->size + sizeof(memdebug_mhead) + sizeof(memdebug_mtail));
	mem_unlock();

	DPRINTF("finished.\n");
	
	return 0;
}


