/* 
 * Copyright (c) 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */
/*
 * Routine to determine the minimum or least-common-denominator
 * of the contents of two cpu_info structures.
 * Typically used on SMP systems to determine the basic feature set
 * common across all processors in the system regardless of type.
 */

#include <string.h>
#include <flux/x86/cpuid.h>

void cpu_info_min(struct cpu_info *id, const struct cpu_info *id2)
{
	unsigned i, j;

	/*
	 * First calculate the minimum family, model, and stepping values.
	 * These values are treated as if concatenated into one big number,
	 * family being "most significant" and stepping being "least".
	 */
	if (id2->family < id->family) {
		id->family = id2->family;
		id->model = id2->model;
		id->stepping = id2->stepping;
	} else if (id2->family == id->family) {
		if (id2->model < id->model) {
			id->model = id2->model;
			id->stepping = id2->stepping;
		} else if (id2->model == id->model) {
			if (id2->stepping < id->stepping)
				id->stepping = id2->stepping;
		}
	}

	/*
	 * Don't bother doing anything with cpu_info.type;
	 * it's not really useful to software anyway.
	 */

	/* Take the minimum of the feature flags... */
	id->feature_flags &= id2->feature_flags;

	/*
	 * If we ever see the day when an SMP system
	 * contains multiple processors from different vendors,
	 * just clear the vendor ID string to "unknown".
	 */
	if (memcmp(id->vendor_id, id2->vendor_id, 12) != 0)
		memset(id->vendor_id, 0, 12);

	/*
	 * Take the minimum of the cache config descriptors,
	 * eliminating any that aren't in both processor descriptions.
	 */
	for (i = 0; i < sizeof(id->cache_config); i++) {
		unsigned char id_desc = id->cache_config[i];
		if (id_desc != 0) {
			for (j = 0; ; j++) {
				if (j >= sizeof(id2->cache_config)) {
					/* Not found in id2 - remove from id */
					id->cache_config[i] = 0;
					break;
				}
				if (id2->cache_config[j] == id_desc) {
					/* Found in id2 - leave in id */
					break;
				}
			}
		}
	}
}

